/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	_startup;
/*
------------------------------------------------------------------------------
		Startup code

	This is a not very elegant solution to the problem of locating the
	startup entry point.  Since the compiler looks for '__startup__' in
	the machine unit, we just jump to the real starting point.  
	Eventually, I will eliminate the extra call with some more kludges.
 */
__startup__:	() =
	{
	__realStartup();
	}
/*
------------------------------------------------------------------------------
		Portable intrinsics

	These are the built-in portable functions of the compiler:
 */
abs:		public	(x: int) int = $1;
memScan:	public	(s: pointer, c: byte, len: size_t) pointer = $5;
memCopy:	public	(dest: pointer, src: pointer, len: size_t) = $6;
memSet:		public	(s: pointer, c: byte, len: size_t) = $7;
rotateLeft:	public	(value: unsigned, shiftAmt: byte)
				unsigned = $8;
rotateRight:	public	(value: unsigned, shiftAmt: byte)
				unsigned = $9;
fabs:		public	(x: extended) extended = $10;
exchangeByte:	public	(x: * byte, y: byte) byte = $11;
exchangeInt:	public	(x: * int, y: int) int = $12;
exchangeShort:	public	(x: * signed[16], y: short) short = $13;
roundInt:	public	(x: extended) extended = $14;
_cvtBCD:	public	(x: extended, buf: ref byte) = $15;
bitScanForward:	public	(u: unsigned) int =
	{
	_EAX = u;
	_emit(0x0F, 0xBC, 0xC0);		// BSR EAX,EAX
	return _EAX;
	}

memCompare:	public	(ps1: pointer, ps2: pointer, n: size_t) int =
	{
	s1:	ref byte;
	s2:	ref byte;

	s1 = ps1;
	s2 = ps2;
	while	(n > 0){
		n--;
		if	(*s1 < *s2)
			return -1;
		else if	(*s1 > *s2)
			return 1;
		s1++;
		s2++;
		}
	return 0;
	}

memMove:	public	(dst: pointer, src: pointer, len: size_t) =
	{
	_emit(upDirection);
	if	(src < dst){
		_ESI = int(src) + len - 1;
		_EDI = int(dst) + len - 1;
		_emit(downDirection);
		}
	else	{
		_ESI = int(src);
		_EDI = int(dst);
		}
	_ECX = len;
	_emit(copyBytes);
	_emit(upDirection);
	}

upDirection:	const	byte  = 0xfc;
downDirection:	const	byte  = 0xfd;
copyBytes:	const	unsigned[16] = 0xa4f3;

bsearch:	public	(key: pointer, base: pointer, nelem: int, width: int,
				fcmp: * (pointer, pointer) int) pointer =
	{
	kmin:	* char;
	probe:	* char;
	i:	int;
	j:	int;

	kmin = base;
	while	(nelem > 0){
		i = nelem >> 1;
		probe = kmin + i * width;
		j = (*fcmp)(key, probe);
		if	(j == 0)
			return(probe);
		else if	(j < 0)
			nelem = i;
		else	{
			kmin = probe + width;
			nelem = nelem - i - 1;
			}
		}
	return(0);
	}

exit:	public	(code: exit_t) =
	{
	ip:	* _cleanupVector;
	i:	int;

	--_threadCount_;
	if	(_threadCount_ == 0){
		for	(ip = _activeCleanup_ - 1;
				ip >= _cleanup_;
				ip--)
			code = (*ip)(code);
		}
	_EBX = code;
	_emit(0x9a, _null, _GDT_EXIT);
	}

exit_t:	public	type	unsigned[32] = {
	EX_SUCCESS	= 0,
	EX_FAIL		= 1,
	EX_CORE		= 0x80000000,
	EX_CAUSE	= 0x0F000000,
	EX_NORMAL	= 0x00000000,
	EX_ABORT	= 0x01000000,
	EX_RAISE	= 0x02000000,
	EX_LEVEL	= 0x00FFFFFF,	/* Ignore the high order bits */
	};

abort:	public	(code: exit_t) =
	{
	--_threadCount_;
	_EBX = code;
	_emit(0x9a, _null, _GDT_ABORT);
	}
/*
------------------------------------------------------------------------------
		Built-in types

	These are the built-in types of the language:
 */
bit:		public	type	unsigned[1];
byte:		public	type	unsigned[8];
signedByte:	public	type	signed[8];
unsignedByte:	public	type	unsigned[8];
char:		public	type	unsigned[8];
short:		public	type	signed[16];
int:		public	type	signed[32];
long:		public	type	signed[32];
unsignedShort:	public	type	unsigned[16];
unsignedLong:	public	type	unsigned[32];
single:		public	type	float[32];
double:		public	type	float[64];
extended:	public	type	float[80];

size_t:		public	type	unsigned;
ptrDiff_t:	public	type	int;
tick_t:		public	type	unsigned;
time_t:		public	type	unsigned;

boolean:	public	type	char = {
	FALSE,
	TRUE
	};
/*
------------------------------------------------------------------------------
		Built-in constants
 */
NULL:		public	const	int = 0;

NaN:		public	const	single = 0x1p-97081;	// magic exponent
Infinity:	public	const	single = 0x1p97081;	// magic exponent

/*
------------------------------------------------------------------------------
		Built-in floating point
 */
fexcept_t:	public	type	short = {
	FE_INVALID 	= 0x01,
	FE_DENORMAL 	= 0x02,		// Intel 80x86 only
	FE_DIVBYZERO	= 0x04,
	FE_OVERFLOW 	= 0x08,
	FE_UNDERFLOW 	= 0x10,
	FE_INEXACT 	= 0x20,

	FE_ALL_EXCEPT	= 0x3F
	};

frounds_t:	public	type	short = {
	FE_TONEAREST 	= 0x000,
	FE_UPWARD	= 0x400,
	FE_DOWNWARD	= 0x800,
	FE_TOWARDZERO	= 0xC00
	};

fenv_t:	public	type	packed { public:

		// Data members are Intel only

	control:	unsigned[32];
	status:		unsigned[32];
	tag:		unsigned[32];
	fip:		unsigned[32];
	fcs:		unsigned[16];
	opcode:		unsigned[16];
	foo:		unsigned[32];
	fos:		unsigned[32];

		// Methods are portable

load:	(f: fenv_t) =
	{
	if	(self == &_FENV){
		_EAX = unsigned(&f);
		_emit(0xD9, 0x20);		// fldenv [eax]
		}
	else
		*self = f;
	}

save:	() fenv_t =
	{
	if	(self == &_FENV){
		f:	fenv_t;

		_EAX = unsigned(&f);
		_emit(0xD9, 0x30);		// fstenv [eax]
//		return f;
		}
//	else
//		return *self;
	}

update:	(f: fenv_t) =
	{
	f setRaised(self raised());
	if	(self == &_FENV){
		_EAX = unsigned(&f);
		_emit(0xD9, 0x20);		// fldenv [eax]
		}
	else
		*self = f;
	}

mask:	(e: fexcept_t) boolean =
	{
	if	(e & FE_ALL_EXCEPT != e)
		return FALSE;
	if	(self == &_FENV){
		x:	short;

		_EAX = unsigned(&x);
		_emit(0xD9, 0x38);		// fstcw [eax]
		x |= e;
		_EAX = unsigned(&x);
		_emit(0xD9, 0x28);		// fldcw [eax]
		}
	else
		control |= e;
	return TRUE;
	}

unmask:	(e: fexcept_t) boolean =
	{
	if	(e & FE_ALL_EXCEPT != e)
		return FALSE;
	if	(self == &_FENV){
		x:	short;

		_EAX = unsigned(&x);
		_emit(0xD9, 0x38);		// fstcw [eax]
		x &= ~e;
		_EAX = unsigned(&x);
		_emit(0xD9, 0x28);		// fldcw [eax]
		}
	else
		control &= ~e;
	return TRUE;
	}

raised:	() fexcept_t =
	{
	if	(self == &_FENV){
		_emit(0xDF, 0xE0);		// fstsw ax
		return _EAX & FE_ALL_EXCEPT;
		}
	else
		return status & FE_ALL_EXCEPT;
	}

setRaised:	(e: fexcept_t) boolean =
	{
	if	(e & FE_ALL_EXCEPT != e)
		return FALSE;
	if	(self == &_FENV){
		f:	fenv_t;

		_EAX = unsigned(&f);
		_emit(0xD9, 0x30);		// fstenv [eax]
		f.status &= ~FE_ALL_EXCEPT;
		f.status |= e;
		_EAX = unsigned(&f);
		_emit(0xD9, 0x20);		// fldenv [eax]
		}
	else	{
		status &= ~FE_ALL_EXCEPT;
		status |= e;
		}
	return TRUE;
	}

raise:	(e: fexcept_t) boolean =
	{
	if	(e & FE_ALL_EXCEPT != e)
		return FALSE;
	if	(self == &_FENV){
		f:	fenv_t;

		_EAX = unsigned(&f);
		_emit(0xD9, 0x30);		// fstenv [eax]
		f.status |= e;
		_EAX = unsigned(&f);
		_emit(0xD9, 0x20);		// fldenv [eax]
		}
	else
		status |= e;
	return TRUE;
	}

clear:	(e: fexcept_t) boolean =
	{
	if	(e & FE_ALL_EXCEPT != e)
		return FALSE;
	if	(self == &_FENV){
		f:	fenv_t;

		_EAX = unsigned(&f);
		_emit(0xD9, 0x30);		// fstenv [eax]
		f.status &= ~e;
		_EAX = unsigned(&f);
		_emit(0xD9, 0x20);		// fldenv [eax]
		}
	else
		status &= ~e;
	return TRUE;
	}

round:	(r: frounds_t) boolean =
	{
	if	(r & 0xC00 != r)
		return FALSE;
	if	(self == &_FENV){
		x:	short;

		_EAX = unsigned(&x);
		_emit(0xD9, 0x38);		// fstcw [eax]
		x &= ~0xC00;
		x |= r;
		_EAX = unsigned(&x);
		_emit(0xD9, 0x28);		// fldcw [eax]
		}
	else	{
		control &= ~0xC00;
		control |= r;
		}
	return TRUE;
	}

roundsAs:	() frounds_t =
	{
	if	(self == &_FENV){
		x:	short;

		_EAX = unsigned(&x);
		_emit(0xD9, 0x38);		// fstcw [eax]
		return x & 0xC00;
		}
	else
		return control & 0xC00;
	}

	};

fclass_t:	public	type	int = {
	FP_NAN,
	FP_INFINITE,
	FP_NORMAL,
	FP_SUBNORMAL,
	FP_ZERO
	};

fpclassify:	public	(x: extended) fclass_t =
	{
	if	(x !<>= 0.0)
		return FP_NAN;
	if	(x !<> 0.0)
		return FP_ZERO;
	x = fabs(x);
	if	(x !<> Infinity)
		return FP_INFINITE;
	if	(x !>= 0x1p-16383)
		return FP_SUBNORMAL;
	else
		return FP_NORMAL;
	}

_FENV:			public	fenv_t;
fDefaultEnvironment:	public	fenv_t = [ 0x37F, 0, 0xFFFF ];
/*
------------------------------------------------------------------------------
		entry and cleanup

	The following section describes the entry and cleanup tables and
	supporting data.  The memory for the tables is laid out as follows:

			_entry_		_entryVector
					_entryVector
					...
					_entryVector
			_cleanup_	_cleanupVector
					_cleanupVector
					...
			_endCleanup_

	The code asssumes these tables are contiguous in memory, though
	the exact location is unimportant.  The linker will magically
	assign real addresses to each of the symbols after it builds the
	tables themselves.
 */	
_entry_:		public	[1] _entryVector = $0;
_cleanup_:		public	[1] _cleanupVector = $1;
_endCleanup_:		public	[1] _cleanupVector = $2;
_activeCleanup_:	public	* _cleanupVector = _cleanup_;
/*
	The linker builds the entry vectors automatically corresponding to
	the list of entry functions in a project.
 */ 
_entryVector:	public	type	{
	public:

	func:	* ();
	exit:	* _cleanupVector;
	};
/*
	The linker builds the cleanup vectors automatically corresponding to
	the list of cleanup functions in a project.
 */
_cleanupVector:	public	type	* (int) int;

/*
------------------------------------------------------------------------------
		Command line arguments

	These variables define the command line arguments to be passed into
	a program.  These are defined, even for kernel contexts, but in the
	kernel they are always zero.  Note that the linker will drop them out
	if they are never used, so the kernel won't really link them in.
 */
ArgumentCount:		public	int;
_ArgumentVector:	public	[:] char;
CommandPath:		public	[:] char;
ArgumentTrap:		trap;

getNextArgument:	public	() [:] char =
	{
	s:	[:] char;

	if	(ArgumentCount == 0)
		ArgumentTrap raise();
	ArgumentCount--;
	if	(ArgumentCount == 0)
		return _ArgumentVector;

	i:	int;
	xp:	ref char;

	xp = memScan(_ArgumentVector, 0, |_ArgumentVector);
	if	(_FLAGS & 0x40)
		i = (xp - _ArgumentVector) - 1;
	else
		ArgumentTrap raise();
	s = _ArgumentVector[:i];
	_ArgumentVector = _ArgumentVector[i + 1:];
	return s;
	}
/*
------------------------------------------------------------------------------
		Exceptions

	Exception handling is done using a linked list of exception frames
	that are actually laced through the execution stack of a thread.
	The head of that list is the _Thread->exceptionFrame pointer.
 */
trap:	public	type	{
	private:

	stuff:		byte;			// give it some substance

	public:
/*
	Note: this code is VERY VERY VERY sensitive to the code generated
	by the compiler.  The essential problem is to be sure that no 
	references to EBP are generated inside the loop, that the value of
	xf is not needed after the call to the exception handler and that
	the two pop instructions correspond to the register variables picked
	by the compiler for vx and self respectively (also that the 
	arguments to the handler are passed on the stack in the exact
	order the pop's expect.
 */
raise:	(...) =
	{
	va:	varArgs;
	vx:	unsigned;

	va = ...;
	vx = *ref unsigned(&va);
	while	(_Thread->exceptionFrame){
		xf:	ref __exceptionFrame;

		xf = _Thread->exceptionFrame;
		_Thread->exceptionFrame = xf->next;
		_EBP = xf->bp;
		xf->addr(self, vx);
		_emit(0x5b);			// pop EBX (vx)
		_emit(0x59);			// pop ECX (self)
		}
	_emit(0x9a, _null, _GDT_RAISE);
	}

};
/*
	This is the object type that an exception clause can name to store
	exception context information for later review.
 */
exceptionContext:	public	type	packed	{
	public:

	raised:			ref trap;
	context:		varArgs;
	initialContext:	private	varArgs;

returnAddress:	() pointer =
	{
	return (*ref ref pointer(&initialContext))[-1];
	}

	};

__exceptionFrame:	public	type	packed {
	public:

	next:		ref __exceptionFrame;
	addr:		ref (ref trap, unsigned);
	bp:		unsigned;
	sp:		unsigned;
	};
/*
------------------------------------------------------------------------------
		External message interface object

	This must be the base class for any type defining any gate functions.
	The compiler will create a typeDescriptor and will automatically
	set the value of the '__td__' member to point to it.
 */
external:	public	type	{
	visible:

	__td__:			ref typeDescriptor;
	MessageHeader:		ref messageHeader;
	objectId:		ref far external;

	public:

		// Uncatchable kill

kill:			gate!	() = { abort(int(&kill)); }

		// Synchronous interrupts

illegalInstruction:	gate!	() = { abort(int(&illegalInstruction)); }
memoryError:		gate!	() = { abort(int(&memoryError)); }
pageFault:		gate!	() = { abort(int(&pageFault)); }
arrayBounds:		gate!	() = { abort(int(&arrayBounds)); }
systemCallError:	gate!	() = { abort(int(&systemCallError)); }
mathError:		gate!	() = { abort(int(&mathError)); }
integerOverflow:	gate!	() = { abort(int(&integerOverflow)); }
rejectedMessage:	gate!	(code: int) = { RejectTrap raise(code); }
interruptedMessage:	gate!	() = { InterruptTrap raise(); }

		// Asynchronous interrupts next

powerFailure:		gate!	() = { abort(int(&powerFailure)); }
hangup:			gate!	() = { abort(int(&hangup)); }
attention:		gate!	() = { abort(int(&attention)); }
quit:			gate!	() = { abort(int(&quit)); }
brokenSend:		gate!	() = { abort(int(&brokenSend)); }
alarmExpired:		gate!	() = { abort(int(&alarmExpired)); }

start:	gate	() =
	{
	}

childExit:	gate	(exitCode: exit_t) =
	{
	}

//	These following two functions return TRUE if the operation is
//	allowed, FALSE otherwise

delete:	gate	() boolean =
	{
	return TRUE;
	}

close:	gate	() boolean =
	{
	return TRUE;
	}

copy:	gate	() ref far external =
	{
	return 0;
	}

dup:	gate	() =
	{
	}

__spare1:	gate	() =
	{
	}

__spare2:	gate	() =
	{
	}

__spare3:	gate	() =
	{
	}

__spare4:	gate	() =
	{
	}

__spare5:	gate	() =
	{
	}

__spare6:	gate	() =
	{
	}

__spare7:	gate	() =
	{
	}

__spare8:	gate	() =
	{
	}

loop:	() =
	{
	wait(pointer(-1));
	}

wait:	(func: pointer) =
	{
	buf:		ref byte;
	mark:		ref byte;
	f:		ref gateDescriptor;
	i, fparms:	int;
	index:		unsigned[32];
	len:		int;
	mh:		messageHeader;

	mark = pointer(_ESP);
	mh.next = MessageHeader;
	for	(;;){
		_ESP = unsigned(mark);
		i = _receive(&mh);
		if	(i < 0)
			continue;
		MessageHeader = &mh;
		index = mh.func - __td__->funcBase;
		if	(index >= __td__->gateCount){
			_reject(mh.sequence, ERRINVALIDFUNC);
			continue;
			}
		f = &__td__->funcs[index];
		if	(mh.rights & f->accessMask == 0){
			_reject(mh.sequence, ERRPERMISSION);
			continue;
			}
		len = (f->parms + 3) & ~3;
		if	(len > IMMED_MAX)
			len = IMMED_MAX;
		_ESP -= len;
		buf = pointer(_ESP);
		_readText(mh.sequence, 0, buf, len);
		try	{
			f->fptr(self, buf);
			}
		except	{
//		case	_RejectTrap:
//			break;
//
//		default:
			MessageHeader = mh.next;
			_reject(mh.sequence, ERREXCEPTION);
			continue;
			}
		_reply(mh.sequence, 0, 0);
		if	(unsigned(func) == mh.func){
			break;
			}
		}
	MessageHeader = mh.next;
	}

_signal:	(seq: _sequence_t) =
	{
	buf:		ref byte;
	mark:		ref byte;
	f:		ref gateDescriptor;
	len, i, fparms:	int;
	index:		unsigned[32];
	mh:		messageHeader;

	_EDX = int(&mh);
	_BX = seq;
	_emit(0x9a, _null, _GDT_SIGNALINFO);
	i = _EAX;
	if	(i < 0)
		return;
	mh.next = MessageHeader;
	MessageHeader = &mh;
	index = mh.func - __td__->funcBase;
	if	(index >= __td__->gateCount){
		_reject(mh.sequence, ERRINVALIDFUNC);
		return;
		}
	f = &__td__->funcs[index];
	if	(mh.rights & f->accessMask == 0){
		_reject(mh.sequence, ERRPERMISSION);
		return;
		}
	len = (f->parms + 3) & ~3;
	if	(len > IMMED_MAX)
		len = IMMED_MAX;
	_ESP -= len;
	buf = pointer(_ESP);
	_readText(mh.sequence, 0, buf, len);
	try	{
		f->fptr(self, buf);
		MessageHeader = mh.next;
		_reply(mh.sequence, 0, 0);
		}
	except	{
		MessageHeader = mh.next;
		_reject(mh.sequence, ERREXCEPTION);
		continue;
		}
	}

receive:	(mh: ref messageHeader) int =
	{
	i:	int;

	i = _receive(mh);
	if	(i < 0)
		return i;
	mh->next = MessageHeader;
	MessageHeader = mh;
	}

discardMessage:	() =
	{
	if	(MessageHeader)
		MessageHeader = MessageHeader->next;
	}

kernelLocal:	(mh: ref messageHeader, buf: pointer) =
	{
	index:	unsigned[32];
	fd:	ref gateDescriptor;

	mh->next = MessageHeader;
	MessageHeader = mh;
	index = mh->func - __td__->funcBase;
	if	(index >= __td__->gateCount){
		_reject(mh->sequence, ERRINVALIDFUNC);
		return;
		}
	fd = &__td__->funcs[index];
	if	(mh->rights & fd->accessMask == 0){
		_reject(mh->sequence, ERRPERMISSION);
		return;
		}
	try	{
		fd->fptr(self, buf);
		}
	except	{
//	case	_RejectTrap:
//		break;
//
//	default:
		_reject(mh->sequence, ERREXCEPTION);
		}
	MessageHeader = mh->next;
	}

kernelRejected:	() =
	{
	index:	unsigned[32];
	fd:	ref gateDescriptor;
	mh:	messageHeader;

	mh.rights = 0xFFFF;
	mh.sender = 0;
	mh.expected = 0;
	mh.len = 0;
	mh.func = int(&rejectedMessage);
	mh.next = MessageHeader;
	MessageHeader = &mh;
	index = mh.func - __td__->funcBase;
	fd = &__td__->funcs[index];
	try	{
		fd->fptr(self, 0);
		}
	except	{
//	case	_RejectTrap:
//		break;
//
//	default:
		_reject(mh.sequence, ERREXCEPTION);
		}
	MessageHeader = mh.next;
	}

kernelRemote:	(mh: ref messageHeader, 
				buf: pointer, firstPacketLength: int) =
	{
	index:	unsigned[32];
	fparms:	int;
	fd:	ref gateDescriptor;

	mh->next = MessageHeader;
	MessageHeader = mh;
	index = mh->func - __td__->funcBase;
	if	(index >= __td__->gateCount){
		_reject(mh->sequence, ERRINVALIDFUNC);
		return;
		}
	fd = &__td__->funcs[index];
	if	(mh->rights & fd->accessMask == 0){
		_reject(mh->sequence, ERRPERMISSION);
		return;
		}
	if	(fd->parms == unsigned[16](~0) || 
		 mh->len < fd->parms)
		fparms = mh->len;
	else
		fparms = fd->parms;
	if	(fparms > firstPacketLength){
		_ESP -= fparms;
		_ESP &= ~3;
		buf = pointer(_ESP);
		_readText(mh->sequence, 0, buf, fparms);
		}
	try	{
		fd->fptr(self, buf);
		}
	except	{
//	case	_RejectTrap:
//		break;
//
//	default:
		_reject(mh->sequence, ERREXCEPTION);
		}
	MessageHeader = mh->next;
	}

replyGeneric:	(buf: pointer, len: int) =
	{
	_reply(MessageHeader->sequence, buf, len);
	}

replyPartial:	(buf: pointer, len: int) =
	{
	_replyPartial(MessageHeader->sequence, buf, len);
	}

readText:	(offset: unsigned, buf: pointer, len: int) int =
	{
	return _readText(MessageHeader->sequence, offset, buf, len);
	}

discardText:	() =
	{
	_discardText(MessageHeader->sequence);
	}

reject:	(code: int) =
	{
	_reject(MessageHeader->sequence, code);
	}

publish:	(nm: [:] char, jr: accessRights_t, userRights: accessRights_t,
			groupRights: accessRights_t,
			worldRights: accessRights_t) ref far external =
	{
	objectId = _publish(nm, self, jr, userRights, groupRights, worldRights);
	return objectId;
	}

jobPublish:	(nm: [:] char) ref far external =
	{
	objectId = _publish(nm, self, AR_ANY, AR_NONE, AR_NONE, AR_NONE);
	return objectId;
	}

	};

probeObject:	public	(o: ref far external) ref far external =
	{
	_EBX = int(o);
	_emit(0x9a, _null, _GDT_PROBEOBJECT);
	}

RejectTrap:	public	trap;			// You were rejected
InterruptTrap:	public	trap;			// You were interrupted
//_RejectTrap:	public	trap;			// For control rendesvous

IMMED_MAX:	public	const	int = 256;

messageHeader:	public	type	packed	{
	public:

	next:		ref messageHeader;
	id:		messageId;		// system message id
	rights:		unsigned[16];
	sequence:	_sequence_t;
	sender:		ref far external;	// sender's object id
	expected:	unsigned[32];		// expected length of reply
	len:		unsigned[32];		// length of send
	func:		unsigned[32];		// function called
	};

objectId:	public	type	signed[32];
messageId:	public	type	signed[32];

errorCode:	public	type	int = {
	SUCCESS,			// Success
	ERRINVALIDFUNC		= -1,	// Invalid function number
	ERRPERMISSION		= -2,	// Permission denied
	ERREXCEPTION		= -3,	// Exception raised
	};

interruptFrame_t:	public	type	packed	{
	public:
	gs:			unsigned[16];
	fs:			unsigned[16];
	es:			unsigned[16];
	ds:			unsigned[16];
	edi:			unsigned[32];
	esi:			unsigned[32];
	ebp:			unsigned[32];
	esp:			unsigned[32];
	ebx:			unsigned[32];
	edx:			unsigned[32];
	ecx:			unsigned[32];
	eax:			unsigned[32];
	eip:			unsigned[32];
	cs:			unsigned[16];
	extra:			_sequence_t;
	eflags:			unsigned[32];

calledFromUserMode:	() boolean =
	{
	if	(eflags & 0x20000 ||			// v8086 mode
		 cs & RING == USER_RING)		// ring == 3
		return TRUE;
	else
		return FALSE;
	}

	};

_sequence_t:	public	type	unsigned[16];

USER_RING:	const	int = 3;
RING:		const	int = 0x0003;

MessageTrap:	public	trap;
/*
	The _threadCount_ is the count of the total number of separate
	process threads currently alive in this arena.  The cleanup functions
	are only called when the number drops to zero.
 */
_Thread:	public	ref threadContext = &_Thread0;
_Thread0:		threadContext = [ 0, 0, 0, &Heap ];
ThreadTrap:	public	trap;

threadContext:	public	type	{
	public:

	exceptionFrame:	ref __exceptionFrame;
	myThread:	ref external;
	myStack:	ref byte;
	currentHeap:	ref heap;

declareObject:	(p: ref external) =
	{
//	if	(myThread)
//		ThreadTrap raise();
	myThread = p;
	}

	};

_threadCount_:		public	int = 1;

thread:	public	(ex: ref external) ref far external =
	{
	stack:	ref byte;
	t:	ref threadContext;

	stack = new [THREAD_STACK + sizeof threadContext] byte;
	t = ref threadContext(stack);
	stack += sizeof threadContext;
	t->myThread = ex;
	t->exceptionFrame = 0;
	t->myStack = stack;
	t->currentHeap = &Heap;
	ex->objectId = _threadLaunch(ex, t, stack + THREAD_STACK, 
							&threadStartup);
	_threadCount_++;
	return ex->objectId;
	}
/*
	This is where threads start.  The value of _Thread points at the
	running thread.
 */
threadStartup:	() =
	{
	_Thread->myThread loop();
	}

_latchValues_t:	public	type	int =
	{
	_LATCH_OPEN,		// open
	_LATCH_CLOSED,		// closed, but no one waiting
	_LATCH_WAITERS		// closed, processes waiting
	};

lock:	public	type	{
	latch:	_latchValues_t;

	public:

enter:	() =
	{
	_lock_enter(&latch);
	}

leave:	() =
	{
	_lock_leave(&latch);
	}

	};

THREAD_STACK:	const	int = 0x2000;		// use an 8K stack
/*
	The funcBase value is designed to insure that the minimum function
	value is this number.  The ALYS kernel reserves message function
	id's below 0x100 for system control messages.  The compiler is
	allowed to set this value (to make future compatibility possibly
	easier).  For example, if I change the fence point on reserved id's.

	The gateCount is just the count of the number of functions in the 
	table.

	The function descriptors point to the gate functions themselves
	(which accept a special function interface).

	In effect, other the the self pointer, all other parameters are 
	stored indirectly, with only a pointer to the argument brick on the
	stack.

	The parms value is set to some value other than 0xFFFF to indicate
	a maximum argument list size.  If the supplied message has more data
	than the function expects, the excess is disregarded.

	The accessMask is a sixteen bit mask that describes the set of
	access rights bits that have permission to use this function.  The
	value has a bit on if that right is needed to use the function.
 */
typeDescriptor:	type	packed	{
	public:

			unsigned[32];
	funcBase:	unsigned[32];
			unsigned[16];
	gateCount:	unsigned[16];
	funcs:		[] gateDescriptor;
	};

gateDescriptor:	type	packed	{
	public:

	fptr:		ref (ref external, pointer) int;
	parms:		unsigned[16];
	accessMask:	accessRights_t;
	};

accessRights_t:	public	type	unsigned[16];

AR_ANY:		public	const	accessRights_t = ~0;
AR_NONE:	public	const	accessRights_t = 0;
/*
------------------------------------------------------------------------------
		Heap Management

	The heap begins at the end of the static data region of the program.
	It is located by _heapbase_, which is set by the linker.  The
	_brklvl variable is the current top of heap marker.  It moves as the
	heap is manipulated.

	Multiple heap support is provided through the two variables: Heap
	and CurrentHeap.  The arena heap is defined by Heap, while whatever
	is the currently active subheap is CurrentHeap.  Note that CurrentHeap
	is usually set to the same value as Heap.
 */
_brklvl:		public	pointer;
_heapbase_:		public	[1] char = $3;

alloc:	public	(n: unsigned) pointer =
	{
	return _Thread->currentHeap alloc(n);
	}

free:	public	(p: pointer) =
	{
	_Thread->currentHeap free(p);
	}

Heap:	public	inherit heap {
	public:

alloc:	dynamic	(size: size_t) pointer =
	{
	return _alloc(size);
	}

free:	dynamic	(block: pointer) =
	{
	_free(block);
	}

freeListSize:	dynamic	() size_t =
	{
	return _freeListSize();
	}

	};

currentHeap:	public	() ref heap =
	{
	return _Thread->currentHeap;
	}

heap:	public	type	{
	public:

activate:	() ref heap =
	{
	ch:	ref heap;

	ch = _Thread->currentHeap;
	_Thread->currentHeap = self;
	return ch;
	}

alloc:	dynamic	(size_t) pointer =
	{
	MemTrap raise(H_ALLOC);
	}

free:	dynamic	(pointer) =
	{
	MemTrap raise(H_FREE);
	}

freeAll:	dynamic	() =
	{
	MemTrap raise(H_FREEALL);
	}

freeListSize:	dynamic	() size_t =
	{
	MemTrap raise(H_FREELISTSIZE);
	}

	};

MemTrap:	public	trap;

memTrapCauses:	public	type	int = {
	H_ALLOC,
	H_FREE,
	H_FREEALL,
	H_FREELISTSIZE,
	H_NOMEMORY,
	H_CORRUPT
	};
/*
------------------------------------------------------------------------------
		Miscellany

	These are various handy values for the kernel or for other processing.

	The _IDT_ and _GDT_ are the IDT and GDT tables for the kernel.  They
	are reserved in the image and initially filled in by the boot program.
 */
_IDT_:			public	[8 * 256] char = $5;
_GDT_:			public	[_GDT_END] char = $4;
/*
------------------------------------------------------------------------------
		Variable argument lists
 */
varArgs:	public	type	{
	nextArgument:	ref byte;

nextArg:	public	(dest: pointer, len: unsigned) =
	{
	memCopy(dest, nextArgument, len);
	nextArgument += (len + 3) & ~3;		// advance by groups of 4
	}

	};

/*
------------------------------------------------------------------------------
		Character conversion

	These functions convert alphabetic characters to upper or lower case.
 */
tolower:	public	(c: int) int =
	{
	if	(isupper(c))
		return _tolower(c);
	else
		return c;
	}

toupper:	public	(c: int) int =
	{
	if	(islower(c))
		return _toupper(c);
	else
		return c;
	}

_tolower:	public	(c: int) int =
	{
	return c + 'a' - 'A';
	}

_toupper:	public	(c: int) int =
	{
	return c + 'A' - 'a';
	}

/*
------------------------------------------------------------------------------
		Character classification

	These functions classify an integer according to whether it is a
	particular class of character.  The integer must be either a
	representable character or -1 (EOF).  Each function returns non-zero
	if the character fits the predicate being tested, zero otherwise.
 */

isalnum:	public	(c: int) int =
	{
	return(Ctype[c + 1] & (IS_DIG | IS_UPP | IS_LOW));
	}

isalpha:	public	(c: int) int =
	{
	return(Ctype[c + 1] & (IS_UPP | IS_LOW));
	}

isascii:	public	(c: int) int =
	{
	return(c < 0x80);
	}

iscntrl:	public	(c: int) int =
	{
	return(Ctype[c + 1] & IS_CTL);
	}

isdigit:	public	(c: int) int =
	{
	return(Ctype[c + 1] & IS_DIG);
	}

isgraph:	public	(c: int) int =
	{
	return unsigned(c - 0x21) <= 0x7e - 0x21;
	}

islower:	public	(c: int) int =
	{
	return(Ctype[c + 1] & IS_LOW);
	}

isprint:	public	(c: int) int =
	{
	return unsigned(c - 0x20) <= 0x7e - 0x20;
	}

ispunct:	public	(c: int) int =
	{
	return(Ctype[c + 1] & IS_PUN);
	}

isspace:	public	(c: int) int =
	{
	return(Ctype[c + 1] & IS_SP);
	}

isupper:	public	(c: int) int =
	{
	return(Ctype[c + 1] & IS_UPP);
	}

isxdigit:	public	(c: int) int =
	{
	return(Ctype[c + 1] & (IS_DIG | IS_HEX));
	}

isFileChar:	public	(c: byte) int =
	{
	if	(c >= 128)
		return 1;
	return(Ctype[c + 1] & (IS_FNM | IS_DIG | IS_LOW | IS_UPP));
	}

charMasks:	type	char = {
			IS_SP  = 0x01,		/* is space */
			IS_DIG = 0x02,		/* is digit */
			IS_UPP = 0x04,		/* is upper case */
			IS_LOW = 0x08,		/* is lower case */
			IS_HEX = 0x10,		/* [A-F] or [a-f] */
			IS_CTL = 0x20,		/* Control */
			IS_PUN = 0x40,		/* punctuation */
			IS_FNM = 0x80		// DOS filename char
		};

Ctype:	public	[257] charMasks = [
	0,

	IS_CTL,		IS_CTL,		IS_CTL,		IS_CTL,
	IS_CTL,		IS_CTL,		IS_CTL,		IS_CTL,
	IS_CTL,		IS_CTL|IS_SP,	IS_SP|IS_CTL,	IS_CTL,
	IS_CTL,		IS_CTL,		IS_CTL,		IS_CTL,

	IS_CTL,		IS_CTL,		IS_CTL,		IS_CTL,
	IS_CTL,		IS_CTL,		IS_CTL,		IS_CTL,
	IS_CTL,		IS_CTL,		IS_CTL,		IS_CTL,
	IS_CTL,		IS_CTL,		IS_CTL,		IS_CTL,

	IS_SP|IS_FNM,	IS_PUN|IS_FNM,	IS_PUN,		IS_PUN|IS_FNM,
	IS_PUN|IS_FNM,	IS_PUN|IS_FNM,	IS_PUN|IS_FNM,	IS_PUN|IS_FNM,
	IS_PUN|IS_FNM,	IS_PUN|IS_FNM,	IS_PUN|IS_FNM,	IS_PUN,
	IS_PUN,		IS_PUN|IS_FNM,	IS_PUN,		IS_PUN,

	IS_DIG,		IS_DIG,		IS_DIG,		IS_DIG,
	IS_DIG,		IS_DIG,		IS_DIG,		IS_DIG,
	IS_DIG,		IS_DIG,		IS_PUN,		IS_PUN,
	IS_PUN,		IS_PUN,		IS_PUN,		IS_PUN|IS_FNM,

	IS_PUN|IS_FNM,	IS_UPP|IS_HEX,	IS_HEX|IS_UPP,	IS_UPP|IS_HEX,
	IS_UPP|IS_HEX,	IS_UPP|IS_HEX,	IS_UPP|IS_HEX,	IS_UPP,
	IS_UPP,		IS_UPP,		IS_UPP,		IS_UPP,
	IS_UPP,		IS_UPP,		IS_UPP,		IS_UPP,

	IS_UPP,		IS_UPP,		IS_UPP,		IS_UPP,
	IS_UPP,		IS_UPP,		IS_UPP,		IS_UPP,
	IS_UPP,		IS_UPP,		IS_UPP,		IS_PUN,
	IS_PUN,		IS_PUN,		IS_PUN|IS_FNM,	IS_PUN|IS_FNM,

	IS_PUN|IS_FNM,	IS_LOW|IS_HEX,	IS_HEX|IS_LOW,	IS_LOW|IS_HEX,
	IS_LOW|IS_HEX,	IS_LOW|IS_HEX,	IS_LOW|IS_HEX,	IS_LOW,
	IS_LOW,		IS_LOW,		IS_LOW,		IS_LOW,
	IS_LOW,		IS_LOW,		IS_LOW,		IS_LOW,

	IS_LOW,		IS_LOW,		IS_LOW,		IS_LOW,
	IS_LOW,		IS_LOW,		IS_LOW,		IS_LOW,
	IS_LOW,		IS_LOW,		IS_LOW,		IS_PUN|IS_FNM,
	IS_PUN,		IS_PUN|IS_FNM,	IS_PUN|IS_FNM,	IS_CTL,

	0, 0		// ...
	];

/*
------------------------------------------------------------------------------
		Non-portable built-in functions

	These are machine specific built-in functions for the 80386:
 */
_softInterrupt:	public	(intr: byte) = $0;
_outportByte:	public	(port: unsigned[16], value: byte) = $2;
_inportByte:	public	(port: unsigned[16]) byte = $3;
_emit:		public	(...) = $4;

/*
------------------------------------------------------------------------------
		80386 machine registers

	These are the 80386 machine registers:
 */
_AL:	public	__reg__ byte		= $0;
_AH:	public	__reg__ byte		= $1;
_DL:	public	__reg__ byte		= $2;
_DH:	public	__reg__ byte		= $3;
_BL:	public	__reg__ byte		= $4;
_BH:	public	__reg__ byte		= $5;
_CL:	public	__reg__ byte		= $6;
_CH:	public	__reg__ byte		= $7;

_SI:	public	__reg__ unsigned[16]	= $8;
_DI:	public	__reg__ unsigned[16]	= $9;
_BP:	public	__reg__ unsigned[16]	= $10;
_SP:	public	__reg__ unsigned[16]	= $11;

_ES:	public	__reg__ unsigned[16]	= $12;
_DS:	public	__reg__ unsigned[16]	= $13;
_CS:	public	__reg__ unsigned[16]	= $14;
_SS:	public	__reg__ unsigned[16]	= $15;

//_FS:	public	__reg__ unsigned[16]	= $16;
//_GS:	public	__reg__ unsigned[16]	= $17;

_AX:	public	__reg__ unsigned[16]	= $0;
_BX:	public	__reg__ unsigned[16]	= $4;
_CX:	public	__reg__ unsigned[16]	= $6;
_DX:	public	__reg__ unsigned[16]	= $2;

_ESI:	public	__reg__ unsigned	= $8;
_EDI:	public	__reg__ unsigned	= $9;
_EBP:	public	__reg__ unsigned	= $10;
_ESP:	public	__reg__ unsigned	= $11;

_EAX:	public	__reg__ unsigned	= $0;
_EBX:	public	__reg__ unsigned	= $4;
_ECX:	public	__reg__ unsigned	= $6;
_EDX:	public	__reg__ unsigned	= $2;

_FLAGS:	public	__reg__ unsigned	= $0xFF;
/*
------------------------------------------------------------------------------
		ALYS Kernel Interface

	These constants define the specific selectors used for ALYS kernel
	entry points in the GDT.  Rather than mediating system calls through
	software interrupts, ALYS uses gate functions.  The following 
	constants define the GDT selectors.  The selector ranges are chosen
	so that selectos 0x0000 - 0x00F8 are reserved for kernel internal
	selectors.  Selectors 0x0100 - 0x03F8 are the kernel entry points.
	Selectors 0x0400 - 0x7F8 are reserved for TSS and LDT selectors.  The
	number of these selectors declared here actually limits the size of 
	the process table.
 */
_KERNEL_PROCS:	public	const	int = 32;	
					// Number of processes in ALYS kernel

_null:		public	const	unsigned[32] = 0;

_gdtSelectors:	public	type	unsigned[16] = {

	_GDT_KERNEL_CODE	= 0x0008,
	_GDT_KERNEL_DATA	= 0x0010,
	_GDT_KERNEL_PHYSICAL	= 0x0018,

		// Message calls

	_GDT_PUBLISH		= 0x0100,
	_GDT_SEND_ANY		= 0x0108,
	_GDT_SEND_INT		= 0x0110,
	_GDT_SEND_VOID		= 0x0118,
	_GDT_RECEIVE		= 0x0120,
	_GDT_READTEXT		= 0x0128,
	_GDT_DISCARDTEXT	= 0x0130,
	_GDT_REPLY		= 0x0138,
	_GDT_REPLYPARTIAL	= 0x0140,
	_GDT_REJECT		= 0x0148,

		// System calls

	_GDT_ALARM		= 0x0150,
	_GDT_GROW		= 0x0158,
	_GDT_EXIT		= 0x0160,
	_GDT_FORKARENA		= 0x0168,
	_GDT_DISCARDARENA	= 0x0170,
	_GDT_CATCHSIGNAL	= 0x0178,
	_GDT_SETSIGNALTHRESHOLD = 0x0180,
	_GDT_PAUSE		= 0x0188,
	_GDT_MAKELOCAL		= 0x0190,
	_GDT_MAKEANYLOCAL	= 0x0198,
	_GDT_SETFORKACTION	= 0x01a0,
	_GDT_SIGNALINFO		= 0x01a8,
	_GDT_ABORT		= 0x01b0,
	_GDT_RAISE		= 0x01b8,
	_GDT_PROBEOBJECT	= 0x01c0,
	_GDT_NEWJOB		= 0x01c8,
	_GDT_MYJOB		= 0x01d0,

		// Thread management code

	_GDT_THREADLAUNCH	= 0x01d8,
	_GDT_KERNELBLOCK	= 0x01e0,
	_GDT_KERNELUNBLOCK	= 0x01e8,
	_GDT_KERNELDOWN		= 0x01f0,
	_GDT_KERNELUP		= 0x01f8,

	_GDT_TSS_BASE		= 0x0400,
	_GDT_TSS_TOP		= _GDT_TSS_BASE + 0x10 * _KERNEL_PROCS,
	_GDT_END		= _GDT_TSS_TOP
	};
