/**********************************************************************
 *  
 *  ckbdintf.c
 *
 *  copyright (c) 1988,89,90,91 J. Alan Eldridge
 *
 *  C functions to interface to the PC keyboard BIOS
 *  
 *  11/02/91 JAE modified to work with GNU C++ (DJ Delorie's port)
 *
 *********************************************************************/

#include "curses.h"

#include "pckbscan.h"

#ifdef  __GNUC__
#include    "aedef.h"
#include    <pc.h>
#define MSC50 0
#define TURBOC 0
#define AZTEC 0
#endif

#if (MSC50 | TURBOC)
#include <dos.h>
#endif  /* (MSC50 | TURBOC) */

#if AZTEC
#include "aztecdos.h"
#endif  /* AZTEC */

/**********************************************************************
 *  
 *  useful constants
 *  
 *********************************************************************/

#define K_SYNONYM   64  /* size of synonym table (set to 0 to disable) */

#define KBD_INTR        0x16    /* ROM BIOS keyboard interrupt # */
#define ZEROFLAG        0x40    /* mask to check CPU zero flag */

#define KBD_GET         0x00    /* standard kbd read function # */
#define KBD_STATUS      0x01    /* standard kbd status function # */

#define EXT_KBD_GET     0x10    /* extended kbd read function # */
#define EXT_KBD_STATUS  0x11    /* extended kbd status function # */

#define AT_MACHINE_ID   0xfc    /* ROM id code for AT and higher */

/**********************************************************************
 *  
 *  static variables
 *  
 *********************************************************************/

static  UCHAR   getfunc = KBD_GET;      /* BIOS function # to read chars */
static  UCHAR   askfunc = KBD_STATUS;   /* BIOS function # for kb status */

#if K_SYNONYM

static int syncnt = 0;  /* cnt of entries in synonym table */

static struct {         /* map one key value to another */
    int oldval;         /* value as originally mapped */
    int newval;         /* new value to return instead */
} syntable[K_SYNONYM];

#endif /* K_SYNONYM */

/* lookup table to map from BIOS to my key representation */

typedef struct {
    int biosval;
    int newval;
} KEY_DEF;

#define NELEMS(arr) (sizeof(arr)/sizeof(arr[0]))

static KEY_DEF  keytable[] = {
    /* unmodified keys */
    { NULL_CHAR,    K_NULL  },
    { FUNC_1,       K_F1    },
    { FUNC_1 + 1,   K_F2    },
    { FUNC_1 + 2,   K_F3    },
    { FUNC_1 + 3,   K_F4    },
    { FUNC_1 + 4,   K_F5    },
    { FUNC_1 + 5,   K_F6    },
    { FUNC_1 + 6,   K_F7    },
    { FUNC_1 + 7,   K_F8    },
    { FUNC_1 + 8,   K_F9    },
    { FUNC_1 + 9,   K_F10   },
    { FUNC_11,      K_F11   },
    { FUNC_11 + 1,  K_F12   },
    { HOME,         K_HOME  },
    { UP,           K_UP    },
    { PGUP,         K_PGUP  },
    { LEFT,         K_LEFT  },
    { RIGHT,        K_RIGHT },
    { ENDKEY,       K_END   },
    { DOWN,         K_DOWN  },
    { PGDN,         K_PGDN  },
    { INSRT,        K_INS   },
    { DEL,          K_DEL   },
    /* shift key down */
    { FUNC_1S,      K_F1 | K_SHIFT  },
    { FUNC_1S + 1,  K_F2 | K_SHIFT  },
    { FUNC_1S + 2,  K_F3 | K_SHIFT  },
    { FUNC_1S + 3,  K_F4 | K_SHIFT  },
    { FUNC_1S + 4,  K_F5 | K_SHIFT  },
    { FUNC_1S + 5,  K_F6 | K_SHIFT  },
    { FUNC_1S + 6,  K_F7 | K_SHIFT  },
    { FUNC_1S + 7,  K_F8 | K_SHIFT  },
    { FUNC_1S + 8,  K_F9 | K_SHIFT  },
    { FUNC_1S + 9,  K_F10 | K_SHIFT },
    { FUNC_11S,     K_F11 | K_SHIFT },
    { FUNC_11S + 1, K_F12 | K_SHIFT },
    { SHIFT_TAB,    K_BACKTAB       },
    /* alt key down */
    { FUNC_1A,      K_F1 | K_ALT    },
    { FUNC_1A + 1,  K_F2 | K_ALT    },
    { FUNC_1A + 2,  K_F3 | K_ALT    },
    { FUNC_1A + 3,  K_F4 | K_ALT    },
    { FUNC_1A + 4,  K_F5 | K_ALT    },
    { FUNC_1A + 5,  K_F6 | K_ALT    },
    { FUNC_1A + 6,  K_F7 | K_ALT    },
    { FUNC_1A + 7,  K_F8 | K_ALT    },
    { FUNC_1A + 8,  K_F9 | K_ALT    },
    { FUNC_1A + 9,  K_F10 | K_ALT   },
    { FUNC_11A,     K_F11 | K_ALT   },
    { FUNC_11A + 1, K_F12 | K_ALT   },
    { ALT_A,        'a' | K_ALT     },
    { ALT_B,        'b' | K_ALT     },
    { ALT_C,        'c' | K_ALT     },
    { ALT_D,        'd' | K_ALT     },
    { ALT_E,        'e' | K_ALT     },
    { ALT_F,        'f' | K_ALT     },
    { ALT_G,        'g' | K_ALT     },
    { ALT_H,        'h' | K_ALT     },
    { ALT_I,        'i' | K_ALT     },
    { ALT_J,        'j' | K_ALT     },
    { ALT_K,        'k' | K_ALT     },
    { ALT_L,        'l' | K_ALT     },
    { ALT_M,        'm' | K_ALT     },
    { ALT_N,        'n' | K_ALT     },
    { ALT_O,        'o' | K_ALT     },
    { ALT_P,        'p' | K_ALT     },
    { ALT_Q,        'q' | K_ALT     },
    { ALT_R,        'r' | K_ALT     },
    { ALT_S,        's' | K_ALT     },
    { ALT_T,        't' | K_ALT     },
    { ALT_U,        'u' | K_ALT     },
    { ALT_V,        'v' | K_ALT     },
    { ALT_W,        'w' | K_ALT     },
    { ALT_X,        'x' | K_ALT     },
    { ALT_Y,        'y' | K_ALT     },
    { ALT_Z,        'z' | K_ALT     },
    { ALT_1,        '1' | K_ALT     },
    { ALT_1 + 1,    '2' | K_ALT     },
    { ALT_1 + 2,    '3' | K_ALT     },
    { ALT_1 + 3,    '4' | K_ALT     },
    { ALT_1 + 4,    '5' | K_ALT     },
    { ALT_1 + 5,    '6' | K_ALT     },
    { ALT_1 + 6,    '7' | K_ALT     },
    { ALT_1 + 7,    '8' | K_ALT     },
    { ALT_1 + 8,    '9' | K_ALT     },
    { ALT_1 + 9,    '0' | K_ALT     },
    { ALT_MINUS,    '-' | K_ALT     },
    { ALT_EQUAL,    '=' | K_ALT     },
    /* control key down */
    { FUNC_1C,      K_F1 | K_CTL    },
    { FUNC_1C + 1,  K_F2 | K_CTL    },
    { FUNC_1C + 2,  K_F3 | K_CTL    },
    { FUNC_1C + 3,  K_F4 | K_CTL    },
    { FUNC_1C + 4,  K_F5 | K_CTL    },
    { FUNC_1C + 5,  K_F6 | K_CTL    },
    { FUNC_1C + 6,  K_F7 | K_CTL    },
    { FUNC_1C + 7,  K_F8 | K_CTL    },
    { FUNC_1C + 8,  K_F9 | K_CTL    },
    { FUNC_1C + 9,  K_F10 | K_CTL   },
    { FUNC_11C,     K_F11 | K_CTL   },
    { FUNC_11C + 1, K_F12 | K_CTL   },
    { C_PRTSC,      K_PRTSC | K_CTL },
    { C_LEFT,       K_LEFT  | K_CTL },
    { C_RIGHT,      K_RIGHT | K_CTL },
    { C_END,        K_END   | K_CTL },
    { C_PGDN,       K_PGDN  | K_CTL },
    { C_HOME,       K_HOME  | K_CTL },
    { C_PGUP,       K_PGUP  | K_CTL }
};
    
#define KEYTBL_SIZE NELEMS(keytable)

/**********************************************************************
 * 
 *  _kb_init()
 *  
 *  initialize the keyboard interface
 *  
 *  no harm will come if this is not called, but certain keys
 *  on the AT (e.g., F11 and F12) cannot be read otherwise
 *  
 *********************************************************************/

void
_kb_init(void)
{
#ifndef __GNUC__
    if (__MACHID.model == AT_MACHINE_ID) {
        /* if it's an AT type machine or higher ... */
        if (__MACHID.submodel) {  /* if it's not submodel 0 ... */
            getfunc = EXT_KBD_GET;      /* we can use extended kb  ... */
            askfunc = EXT_KBD_STATUS;   /* read and status functions */
        }
    }
#endif
}

/**********************************************************************
 *  
 *  _kb_read()
 *  
 *  get a character from the BIOS
 *  
 *********************************************************************/

int
_kb_read(void)
{
#ifndef __GNUC__
    union REGS  reg_st;
    
    reg_st.h.ah = getfunc;
    int86(KBD_INTR, &reg_st, &reg_st);
    return reg_st.x.ax;
#else
    int c = getxkey();
    if (c > 255) {
        unsigned char ah = c >> 8;
        unsigned char al = c & 0xff;
        
        ah = (ah == 1) ? 0 : 0xE0;
        
        c = ah | (al << 8);
    }
    return c;
#endif    
}    

/**********************************************************************
 *  
 *  _kb_look()
 *  
 *  returns 1 if there's a character waiting, 0 if not
 *  
 *********************************************************************/

int
_kb_look(void)
{
#ifndef __GNUC__
    union REGS  reg_st;
    
    reg_st.h.ah = askfunc;
#if TURBOC
    int86(KBD_INTR, &reg_st, &reg_st);
    return !(reg_st.x.flags & ZEROFLAG);
#endif  /* TURBOC */
#if AZTEC
    return !(int86(KBD_INTR, &reg_st, &reg_st) & ZEROFLAG);
#endif  /* AZTEC */
#if MSC50
    return _bios_keybrd(askfunc) != 0;
#endif  /* MSC50 */
#else
    return kbhit();
#endif
}

#if K_SYNONYM

/**********************************************************************
 *  
 *  _kb_syn(oldval, newval)
 *  
 *  creates a "synonym" for a key
 *  
 *  if oldval would have been returned, return newval instead
 *  
 *********************************************************************/

int
_kb_syn(
    int oldval,
    int newval)
{
    if (syncnt < K_SYNONYM) {
        syntable[syncnt].oldval = oldval;
        syntable[syncnt++].newval = newval;
        return 0;
    } else
        return -1;
}

static int
synxlat(int oldval)
{
    int n;
    
    for (n = 0; n < syncnt; n++) 
        if (syntable[n].oldval == oldval)
            return syntable[n].newval;

    return oldval;
}

#else

#define synxlat(c)  c   /* make this a no-op */

#endif  /* K_SYNONYM */

/**********************************************************************
 *  
 *  _kb_mapc(c)
 *  
 *  given a key code returned by the BIOS, return the Curses!
 *  representation of that key, translating a synonym if found
 *  
 *  a key that doesn't make sense is mapped to K_UNKNOWN
 *  
 *********************************************************************/

int
_kb_mapc(int c)
{
    int al = c & 0xff;
    int ah = (unsigned int)c >> 8;
    
    if (al == 0x00 || al == 0xe0) {
        int n; 
        
        for (n = 0; n < KEYTBL_SIZE; n++)
            if (keytable[n].biosval == ah)
                return synxlat(keytable[n].newval);

        return K_UNKNOWN;
    } else
        return al;
}

/**********************************************************************
 *  
 *  _kb_getc()
 *  
 *  low level keyboard call that does character mapping
 *  
 *********************************************************************/

int
_kb_getc(void)
{
    return _kb_mapc(_kb_read());
}
