-------------------------------------------------------
-- The functions package contains functions for converting and performing
--   arithmetic operations with the bit_vector type.
--   The package includes:
--    + : an overloaded operator for bit_vector addition
--    - : an overloaded operator for bit_vector subtraction
--    * : an overloaded operator for bit_vector multiplication
--    / : an overloaded operator for bit_vector division
--    negate : two's complement of a bit_vector
--    to_integer : convert a bit_vector to an integer
--    to_bit : convert an integer to a bit_vector
--   All operations assume the left most bit to be most significant.
-------------------------------------------------------

package functions is
  -- overloaded functions for bit_vector operations
  function "+" (a,b: bit_vector) return bit_vector;
  function negate (a: bit_vector) return bit_vector;
  function "-" (a,b: bit_vector) return bit_vector;
  function "*" (a,b: bit_vector) return bit_vector;
  function "/" (a,b: bit_vector) return bit_vector;

  -- conversion functions for bit_vector operations
  function to_integer (a: bit_vector) return integer;
  function to_bit (size: integer; num: integer) return bit_vector;
end functions;


-------------------------------------------------------
-- functions package body
-------------------------------------------------------

package body functions is
  -----------------------
  -- "+" : bit vector addition
  -----------------------
  function "+" (a,b: bit_vector) return bit_vector is
    alias av : bit_vector (1 to a'length) is a;
    alias bv : bit_vector (1 to b'length) is b;
    variable sum: bit_vector (1 to a'length);
    variable carry: bit;
  begin
    assert a'length=b'length 
	report "Operands to overloaded '+' operator with different lengths"
	severity failure;

    carry:='0';         -- carry in

    for i in a'length downto 1 loop
      sum(i):=av(i) xor bv(i) xor carry;
      carry:=(av(i) and bv(i)) or (av(i) and carry) or (bv(i) and carry);
    end loop;

    return sum;
  end;

  -----------------------
  -- negate : bit vector two's complement
  -----------------------
  function negate (a: bit_vector) return bit_vector is
    alias av : bit_vector (1 to a'length) is a;
    variable neg_a: bit_vector (1 to a'length);
    variable carry: bit;
  begin
    carry:='1';         -- carry in will add one to neg_a

    for i in a'length downto 1 loop
      neg_a(i):=(not av(i)) xor carry;
      carry:=(not av(i)) and carry;
    end loop;
  
    return(neg_a);
  end negate;

  -----------------------
  -- "-" : bit vector subtraction
  -----------------------
  function "-" (a,b: bit_vector) return bit_vector is
    variable sum: bit_vector (1 to a'length);
    variable neg_b: bit_vector (1 to b'length);
    variable carry: bit;
  begin
    assert a'length=b'length 
	report "Operands to overloaded '-' operator with different lengths"
	severity failure;

    neg_b:=negate(b);
    sum:=a+neg_b;
    return sum;
  end;

  -----------------------
  -- "*" : bit vector multiplication
  -----------------------
  function "*" (a,b: bit_vector) return bit_vector is
    variable a_int, b_int: integer;
  begin
    assert a'length=b'length 
	report "Operands to overloaded '*' operator with different lengths"
	severity failure;

    -- this is faster than using an boolean multiplication algorithm
    a_int:=to_integer(a);
    b_int:=to_integer(b);
    return to_bit(a'length,a_int*b_int);
  end;

  -----------------------
  -- "/" : bit vector division
  -----------------------
  function "/" (a,b: bit_vector) return bit_vector is
    variable a_int, b_int: integer;
  begin
    assert a'length=b'length 
	report "Operands to overloaded '/' operator with different lengths"
	severity failure;

    -- this is faster than using an boolean division algorithm
    a_int:=to_integer(a);
    b_int:=to_integer(b);
    return to_bit(a'length,a_int/b_int);
  end;

  -----------------------
  -- to_integer : converts a bit vector to a signed integer
  -----------------------
  function to_integer (a: in bit_vector) return integer is
    alias av: bit_vector (1 to a'length) is a;
    variable temp : bit_vector (1 to a'length);
    variable ret,d: integer;
  begin
    if av(1)='1' then
      temp:=negate(a);
    else
      temp:=a;
    end if;

    d:=1;
    ret:=0;
    for i in temp'length downto 1 loop
      if (temp(i)='1') then
        ret:=ret+d;
      end if;
      d:=d*2;
    end loop;

    if av(1)='1' then
      return -ret;
    else
      return ret;
    end if;
  end;

  -----------------------
  -- to_natural : converts a bit vector to a non-negative integer
  -----------------------
  function to_natural (a: in bit_vector) return integer is
    alias av: bit_vector (1 to a'length) is a;
    variable ret,d: integer;
  begin
    d:=1;
    ret:=0;

    for i in a'length downto 1 loop
      if (av(i)='1') then
        ret:=ret+d;
      end if;
      d:=d*2;
    end loop;

    return ret;
  end;

  -----------------------
  -- to_bit : converts an integer to a bit vector
  -----------------------
  function to_bit (size: in integer; num: in integer) return bit_vector is
    variable ret: bit_vector (1 to size);
    variable a: integer;
  begin
    a:=abs num;
    for i in size downto 1 loop
      if ((a rem 2)=1) then
        ret(i):='1';
      else 
        ret(i):='0';
      end if;
      a:=a/2;
    end loop;

    if num < 0 then
      return negate(ret);
    else
      return ret;
    end if;
  end;

end functions;
