/*****************************************************************
Module name: Setup.C
Programmer : Jeffrey M. Richter.
*****************************************************************/

#include <windows.h>
#include <windowsx.h>
#include <ver.h>
#include <lzexpand.h>
#include <dos.h>
#include <dir.h>
#include <direct.h>
#include <io.h>
#include <string.h>
#include "Setup.H"
#include "SetupInf.H"
#include "Meter.h"

BOOL WINAPI CreateDstDirTree (HWND hDlgStatus);
BOOL WINAPI CopyAllFiles (HWND hDlgStatus);
BOOL WINAPI CreatePMInfo (HINSTANCE hInstance);
#define WasCancelled(hDlg) \
   (!IsWindowEnabled(GetDlgItem(hDlg, IDCANCEL)))

HINSTANCE PrepareSetup (LPCSTR szSrcDir);

char _szAppName[] = "Setup";
char _szSrcDir[MAXDIR] = "x:\\"; // Where SETUP.EXE was run from.
char _szDstDir[MAXDIR];


HINSTANCE _hInstVer = NULL;   // Handle of VER.DLL library

UINT (WINAPI *_VerFindFile)(UINT, LPCSTR, LPCSTR, LPCSTR,
   LPSTR, UINT FAR*, LPSTR, UINT FAR*);

DWORD (WINAPI* _VerInstallFile)(UINT, LPCSTR, LPCSTR, LPCSTR,
   LPCSTR, LPCSTR, LPSTR, UINT FAR*);

DWORD (WINAPI* _GetFileVersionInfoSize)(LPCSTR, DWORD FAR *);

BOOL  (WINAPI* _GetFileVersionInfo)
   (LPCSTR, DWORD, DWORD, void FAR*);

UINT  (WINAPI* _VerLanguageName)(UINT, LPSTR, UINT);
BOOL  (WINAPI* _VerQueryValue)
   (const void FAR*, LPCSTR, void FAR* FAR*, UINT FAR*);


#pragma argsused
extern "C" int WinMain (HINSTANCE hInstance,
   HINSTANCE hPrevInstance, LPSTR lpszCmdLine, int nCmdShow) {
   int nResult;
   HWND hDlgStatus, hWndSignOn;
   DWORD dwDiskSpaceNeeded, dwFreeDiskSpace;
   struct dfree DiskFreeSpace;
   char szBuf[100];
   HINSTANCE hInstMeter = NULL;
   SETUPINFO::ISIERROR ISIError;

   // Don't let another instance of this application execute.
   if (hPrevInstance != NULL) return(0);

   // Prepare the DDE Client window class.
   if (!RegisterDDEClient(hInstance))
      goto Cleanup;

   // Initialize the default source path so that it uses the same
   // drive that the SETUP.EXE application was executed from.
   GetModuleFileName(hInstance, _szSrcDir, sizeof(_szSrcDir));
   *(_fstrrchr(_szSrcDir, '\\') + 1) = 0;

   // Sign-on window while initializing
   hWndSignOn = CreateDialog(hInstance,
      MAKEINTRESOURCE(DLG_INITIALIZE),
      NULL, (DLGPROC) InitDlgProc);

   hInstMeter = LoadLibrary("Meter.DLL");
   if (hInstMeter < HINSTANCE_ERROR)
      goto Cleanup;

   _hInstVer = PrepareSetup(_szSrcDir);
   if (_hInstVer < HINSTANCE_ERROR) {
      MsgBox(hInstance, NULL, IDS_CANNOTINIT, _szAppName,
         MB_OK | MB_ICONINFORMATION);
      goto Cleanup;
   }


   // Read the SETUP.INF file into memory.
   wsprintf(szBuf, "%s%sSETUP.INF", _szSrcDir,
      ((*(_fstrrchr(_szSrcDir, '\\') + 1) == 0) ? "" : "\\"));

   ISIError = _SetupInfo.InitSetupInfo(szBuf);

   // Remove sign-on window after initialization
   if (IsWindow(hWndSignOn))
      DestroyWindow(hWndSignOn);

   if (ISIError != SETUPINFO::ISI_NOERROR) {
      MsgBox(hInstance, NULL,
         (ISIError == SETUPINFO::ISI_NOMEM)
         ? IDS_NOMEMORY : IDS_NOSETUPINFOFILE, _szAppName,
            MB_ICONINFORMATION | MB_OK | MB_TASKMODAL, szBuf);
      goto Cleanup;
   }

   // Get the amount of memory (in K) needed for the installation
   dwDiskSpaceNeeded = _SetupInfo.GetDstDiskSpaceNeeded();

   // Create the Status dialog box.
   hDlgStatus = CreateDialog(hInstance,
      MAKEINTRESOURCE(DLG_STATUS), NULL, (DLGPROC) StatusDlgProc);

   do {
      // Welcome user and prompt for destination directory.
      nResult = DialogBox(hInstance,
         MAKEINTRESOURCE(DLG_WELCOME), NULL,
         (DLGPROC) WelcomeDlgProc);
      if (nResult == IDCANCEL) break;

      // Check if there is sufficient disk space on 
      // the destination drive.
      getdfree(_szDstDir[0] - 'A' + 1, &DiskFreeSpace);
      dwFreeDiskSpace = ((DWORD) DiskFreeSpace.df_avail *
                         (DWORD) DiskFreeSpace.df_sclus *
                         (DWORD) DiskFreeSpace.df_bsec) / 1024UL;

      if (dwFreeDiskSpace < dwDiskSpaceNeeded) {
         MsgBox(hInstance, NULL, IDS_NODISKSPACE, _szAppName,
            MB_OK | MB_ICONINFORMATION | MB_TASKMODAL,
            _szDstDir[0], dwFreeDiskSpace, dwDiskSpaceNeeded);
         continue;
      }

      // Try to create the destination directory tree.
      nResult = CreateDstDirTree(hDlgStatus);

      if (nResult == FALSE) {
         // If the directory tree cannot be created, 
         // force loop to repeat.
         dwFreeDiskSpace = 0;
      }

   } while (dwFreeDiskSpace < dwDiskSpaceNeeded);

   if (nResult == IDCANCEL) {
      DestroyWindow(hDlgStatus);
      goto Cleanup;
   }

   // Make the destination directory the current directory.
   _chdrive(_szDstDir[0] - 'A' + 1);
   chdir(_szDstDir);

   // Try to copy the files.
   ShowWindow(hDlgStatus, SW_SHOW);
   UpdateWindow(hDlgStatus);
   nResult = CopyAllFiles(hDlgStatus);
   ShowWindow(hDlgStatus, SW_HIDE);

   // Cleanup the things that we no longer need.
   DestroyWindow(hDlgStatus);

   if (nResult == FALSE) {
      // Installation not complete.
      MsgBox(hInstance, NULL, IDS_SETUPNOGOOD, _szAppName,
         MB_OK | MB_ICONINFORMATION | MB_TASKMODAL);
      goto Cleanup;
   }

   MsgBox(hInstance, NULL,
      CreatePMInfo(hInstance) ? IDS_PMADDOK : IDS_PMADDNOGOOD,
      _szAppName, MB_OK | MB_ICONINFORMATION | MB_TASKMODAL);

   Cleanup:
   if (hInstMeter >= HINSTANCE_ERROR)
      FreeLibrary(hInstMeter);
   if (_hInstVer >= HINSTANCE_ERROR)
      FreeLibrary(_hInstVer);
   return(0);
}



// Returns the hInstance for VER.DLL on hard disk
HINSTANCE PrepareSetup (LPCSTR szSrcDir) {
   HINSTANCE hInstLib;
   char szOurLZExpand[] = "LZExpand.DLx";
   char szBuf[MAXPATH], szDstDir[MAXPATH];


   // Check for the existence of the VER.DLL and LZEXPAND.DLL 
   // files on the user's system.  If they are on the user's 
   // hard disk, lets use them.  If not, let's copy them from 
   // the floppy to the hard disk.
   UINT uPrevErrMode = SetErrorMode(SEM_NOOPENFILEERRORBOX);
   hInstLib = LoadLibrary("VER.DLL");
   SetErrorMode(uPrevErrMode);
   if (hInstLib >= HINSTANCE_ERROR)
      goto GetVerFuncAddresses;

   // User does not have VER.DLL installed on their hard disk
   // The user doesn't have a VER.DLL file or a LZEXPAND.DLL
   // file since VER.DLL requires LZEXPAND.DLL.

   // Load our LZEXPAND.DLL file from our floppy
   hInstLib = LoadLibrary(szOurLZExpand);
   if (hInstLib < HINSTANCE_ERROR) return(hInstLib);


   int   (LZAPI* LZStart)(void);
   void  (LZAPI* LZDone)(void);
   HFILE (LZAPI* LZOpenFile)(LPCSTR, OFSTRUCT FAR *, UINT);
   LONG  (LZAPI* CopyLZFile)(HFILE, HFILE);
   void  (LZAPI* LZClose)(HFILE);

   * ((FARPROC *) &LZStart)    =
      GetProcAddress(hInstLib, "LZStart");
   * ((FARPROC *) &LZDone)     =
      GetProcAddress(hInstLib, "LZDone");
   * ((FARPROC *) &LZOpenFile) =
      GetProcAddress(hInstLib, "LZOpenFile");
   * ((FARPROC *) &CopyLZFile) =
      GetProcAddress(hInstLib, "CopyLZFile");
   * ((FARPROC *) &LZClose)    =
      GetProcAddress(hInstLib, "LZClose");

   GetSystemDirectory(szDstDir, sizeof(szDstDir));
   if (GetDriveType(szDstDir[0] - 'A') == DRIVE_REMOTE) {
      // User is running a shared copy of Windows
      GetWindowsDirectory(szDstDir, sizeof(szDstDir));
   }

   OFSTRUCT ofSrc, ofDst; HFILE hFileSrc, hFileDst;

   LZStart();  // Initialize the LZExpand library
   wsprintf(szBuf, "%s%s", szSrcDir, szOurLZExpand);
   hFileSrc = LZOpenFile(szBuf, &ofSrc, OF_READ);

   wsprintf(szBuf, "%s\\LZExpand.DLL", szDstDir);
   hFileDst = LZOpenFile(szBuf, &ofDst, OF_CREATE);


   // Error check
   CopyLZFile(hFileSrc, hFileDst);
   LZClose(hFileSrc); LZClose(hFileDst);


   wsprintf(szBuf, "%sVER.DL_", szSrcDir);
   hFileSrc = LZOpenFile(szBuf, &ofSrc, OF_READ);

   wsprintf(szBuf, "%s\\Ver.DLL", szDstDir);
   hFileDst = LZOpenFile(szBuf, &ofDst, OF_CREATE);
   // Error check
   CopyLZFile(hFileSrc, hFileDst);
   LZClose(hFileSrc); LZClose(hFileDst);

   LZDone();
   FreeLibrary(hInstLib);  // Free our LZEXPAND.DLL

   // Load the copied VER.DLL from the hard disk
   hInstLib = LoadLibrary("VER.DLL");
   if (hInstLib < HINSTANCE_ERROR) return(hInstLib);

GetVerFuncAddresses:
   // Get the address of all VER.DLL functions
   *((FARPROC *) &_VerFindFile) =
      GetProcAddress(hInstLib, "VerFindFile");
   *((FARPROC *) &_VerInstallFile) =
      GetProcAddress(hInstLib, "VerInstallFile");
   *((FARPROC *) &_GetFileVersionInfoSize) =
      GetProcAddress(hInstLib, "GetFileVersionInfoSize");
   *((FARPROC *) &_GetFileVersionInfo) =
      GetProcAddress(hInstLib, "GetFileVersionInfo");
   *((FARPROC *) &_VerLanguageName) =
      GetProcAddress(hInstLib, "VerLanguageName");
   *((FARPROC *) &_VerQueryValue) =
      GetProcAddress(hInstLib, "VerQueryValue");
   return(hInstLib);
}



// **** Functions for Creating the destination directory tree ****
BOOL WINAPI CreateDstDirTree (HWND hDlgStatus) {
   int nResult, nMaxDirs, nDirNum;
   char szBuf[MAXDIR]; MSG Msg;

   ShowWindow(hDlgStatus,
      _SetupInfo.ShowSubDirCreateStats() ? SW_SHOW : SW_HIDE);
   UpdateWindow(hDlgStatus);
   SetDlgItemText(hDlgStatus, ID_STATLINE1,
      "Creating destination directory tree...");
   nMaxDirs = _SetupInfo.GetNumDstDirs();
   SendDlgItemMessage(hDlgStatus, ID_METER,
      MM_SETPARTSCOMPLETE, 0, 0);
   SendDlgItemMessage(hDlgStatus, ID_METER,
      MM_SETPARTSINJOB, nMaxDirs + 1, 0);
   SetDlgItemText(hDlgStatus, ID_STATLINE2, _szDstDir);

   // Create the destination directory.
   nResult = chdir(_szDstDir);
   if (nResult != 0) {
      nResult = mkdir(_szDstDir);
      if (nResult != 0) {
         MsgBox(_hInstance, hDlgStatus, IDS_CANTMAKEDIR,
            _szAppName, MB_ICONINFORMATION | MB_OK, _szDstDir);
         ShowWindow(hDlgStatus, SW_HIDE);
         return(FALSE);
      } else chdir(_szDstDir);
   }
   SendDlgItemMessage(hDlgStatus, ID_METER,
      MM_SETPARTSCOMPLETE, 1, 0);

   // Create any subdirectories under the destination directory.
   for (nDirNum = 0; nDirNum < nMaxDirs; nDirNum++) {
      // Let some other applications execute.
      while (PeekMessage(&Msg, NULL, NULL, NULL, PM_REMOVE)) {
         TranslateMessage(&Msg);
         DispatchMessage(&Msg);
      }

      if (WasCancelled(hDlgStatus)) {
         nResult = IDCANCEL;
         break;
      }

      wsprintf(szBuf, "%s%s", _szDstDir,
         ((*(_fstrrchr(_szDstDir, '\\') + 1) == 0) ? "" : "\\"));
      _SetupInfo.GetDstDir(nDirNum, _fstrchr(szBuf, 0));
      SetDlgItemText(hDlgStatus, ID_STATLINE2, szBuf);

      nResult = chdir(szBuf);
      if (nResult != 0) {
         nResult = mkdir(szBuf);
         if (nResult != 0) {
            MsgBox(_hInstance, hDlgStatus, IDS_CANTMAKEDIR,
               _szAppName, MB_ICONINFORMATION | MB_OK, szBuf);
            nResult = IDCANCEL;
            break;
         } else chdir(szBuf);
      }
      nResult = IDOK;
      SendDlgItemMessage(hDlgStatus, ID_METER,
         MM_SETPARTSCOMPLETE, nDirNum + 2, 0);
   }
   ShowWindow(hDlgStatus, SW_HIDE);
   return(nResult != IDCANCEL);
}


// **************** Functions for Copying Files ******************

#define PTIF_INSTALLTHISFILE        0
#define PTIF_ABORTSETUP             1
#define PTIF_SKIPTHISFILE           2
#define PTIF_REPLACEEXISTINGFILE    3
#define PTIF_DONTERASEEXISTINGFILE  4
#define PTIF_INTERNALERROR          5

UINT WINAPI PrepareToInstallFile (HWND hWnd, UINT uFlags,
   LPCSTR szDstFileName, LPCSTR szAppDir, LPSTR szCurDir,
   UINT FAR* lpuCurDirLen, LPSTR szDestDir,
   UINT FAR* lpuDestDirLen, LPCSTR szSrcDir,
   LPCSTR szSrcFileName);

#define IF_INSTALLEDOK              0
#define IF_ABORTSETUP               1
#define IF_SKIPTHISFILE             2
#define IF_CHANGEDISK               3
UINT WINAPI InstallFile (HWND hWnd, UINT uFlags,
   LPCSTR szSrcFileName, LPCSTR szDestFileName, LPCSTR szSrcDir,
   LPCSTR szDestDir, LPCSTR szCurDir, LPSTR szTmpFile,
   UINT FAR* lpuTmpFileLen);


void LockResources (void) {
   HRSRC hResDlg;
   HGLOBAL hGlblRes;
   int nDlg, nIcon;

   for (nDlg = DLG_FIRST; nDlg <= DLG_LAST; nDlg++) {
      // Get the resource handle of the dlg box from 
      // the executable file.
      hResDlg = FindResource(_hInstance,
         MAKEINTRESOURCE(nDlg), RT_DIALOG);

      // Get the memory handle of the dialog box in memory.
      // The block is already in memory because the dialog 
      // box is marked as PRELOAD FIXED.
      hGlblRes = LoadResource(_hInstance, hResDlg);

      // Force the memory block to be locked down.  This 
      // prohibits Windows from discarding the dialog box 
      // template from memory.
      LockResource(hGlblRes);
   }

   for (nIcon = ICN_FIRST; nIcon <= ICN_LAST; nIcon++) {
      LoadIcon(_hInstance, MAKEINTRESOURCE(nIcon));
   }
}


// Returns FALSE is Setup must be terminated.
BOOL WINAPI CopyAllFiles (HWND hDlgStatus) {
   int nMaxFiles, nFileNum;
   char szSrcDir[MAXPATH];
   char szFileDesc[MAXFILEDESC], szSrcDiskDesc[MAXDISKDESC];
   char szSrcFilePath[MAXPATH], szSrcFileName[MAXPATH];
   char szDstFilePath[MAXPATH], szDstFileName[MAXPATH];
   char szCurDir[MAXPATH], szDstDir[MAXPATH], szTmpFile[MAXPATH];
   UINT uCurDirLen, uDstDirLen, uTmpFileLen;
   DWORD x;
   MSG Msg;
   BOOL fSharable;

   LockResources();

   SetDlgItemText(hDlgStatus, ID_STATLINE1, "Copying files...");
   nMaxFiles = _SetupInfo.GetNumFilestoInstall();
   SendDlgItemMessage(hDlgStatus, ID_METER,
      MM_SETPARTSCOMPLETE, 0, 0);
   SendDlgItemMessage(hDlgStatus, ID_METER,
      MM_SETPARTSINJOB, nMaxFiles, 0);

   for (nFileNum = 0; nFileNum < nMaxFiles; nFileNum++) {
Retry:
      // Let other applications execute.
      while (PeekMessage(&Msg, NULL, NULL, NULL, PM_REMOVE)) {
         TranslateMessage(&Msg); DispatchMessage(&Msg);
      }
      if (WasCancelled(hDlgStatus)) // Terminate Setup
         return(FALSE);

      _SetupInfo.GetFileInstallInfo(nFileNum,
         szSrcDiskDesc, szFileDesc,
         szSrcFilePath, szSrcFileName,
         szDstFilePath, szDstFileName, &fSharable);
      SetDlgItemText(hDlgStatus, ID_STATLINE2, szFileDesc);

      if (!lstrcmp(szSrcFilePath, "."))
         lstrcpy(szSrcDir, _szSrcDir);
      else
         wsprintf(szSrcDir, "%s%s%s", _szSrcDir,
            ((*(_fstrrchr(_szSrcDir, '\\') + 1) == 0)
            ? "" : "\\"), szSrcFilePath);

      if (!lstrcmp(szDstFilePath, "."))
         lstrcpy(szDstDir, _szDstDir);
      else
         wsprintf(szDstDir, "%s%s%s", _szDstDir,
            ((*(_fstrrchr(_szDstDir, '\\') + 1) == 0)
            ? "" : "\\"), szDstFilePath);

      uCurDirLen  = sizeof(szCurDir);
      uDstDirLen = sizeof(szDstDir);
      uTmpFileLen = sizeof(szTmpFile);

//      x = PrepareToInstallFile(hDlgStatus,
//         fSharable ? VFFF_ISSHAREDFILE : 0,
//         szDstFileName, szDstDir, szCurDir, &uCurDirLen,
//         szDstDir, &uDstDirLen, szSrcDir, szSrcFileName);
//      if (x == PTIF_ABORTSETUP)    return(FALSE);  // Abort Setup
//      if (x == PTIF_INTERNALERROR) return(FALSE);  // Abort Setup
//      if (x == PTIF_SKIPTHISFILE)  continue;

      UINT uVIFFlags = 0;

      if (x == PTIF_DONTERASEEXISTINGFILE)
         uVIFFlags |= VIFF_DONTDELETEOLD;

      if (x == PTIF_REPLACEEXISTINGFILE)
         lstrcpy(szDstDir, szCurDir);

      // Attempt to install the file 
      x = InstallFile(hDlgStatus, uVIFFlags, szSrcFileName,
         szDstFileName, szSrcDir, szDstDir, szCurDir,
         szTmpFile, &uTmpFileLen);
      if (x == IF_ABORTSETUP)   return(FALSE);
      if (x == IF_SKIPTHISFILE) continue;
      if (x == IF_INSTALLEDOK)
         _SetupInfo.SetFileDstDir(nFileNum, szDstDir);

      if (x == IF_CHANGEDISK) {
         // Normally, Windows would have discarded the dialog box
         // template from memory after the dialog box had been 
         // created.  By forcing the memory block to be locked by
         // the call to LockResource() above, the template will 
         // NOT be discarded.  If the template were discarded, 
         // the next time this dialog box needed to be created,
         // Windows would have to load the template from the 
         // executable file.  However, the SETUP.EXE file is 
         // probably not on the diskette that is currently in the
         // drive.  This would cause the program to crash.
         x = DialogBoxParam(_hInstance,
            MAKEINTRESOURCE(DLG_INSERTDISK), hDlgStatus,
            (DLGPROC) InsertDiskDlgProc,
            (LONG) (LPSTR) szSrcDiskDesc);
         if (x == IDCANCEL) return(FALSE);
         if (x == IDOK) {
            // The dialog box already copied the new directory
            // into the _szSrcDir buffer.  Attempt to install
            // the file again.
            goto Retry;
         }
      }

      SendDlgItemMessage(hDlgStatus, ID_METER,
         MM_SETPARTSCOMPLETE, nFileNum + 1, 0);
   }
   // We'll let Windows worry about unlocking the resources
   // when Setup terminates.
   return(TRUE);  // Setup went to completion.
}

// ***************** Miscellaneous Function **********************
int _cdecl MsgBox (HINSTANCE hInstance, HWND hWnd, WORD wID,
   LPCSTR szCaption, WORD wType, ...) {
   char szResString[200], szText[200];
   void FAR *VarArgList = (WORD FAR *) &wType + 1;
   LoadString(hInstance, wID, szResString,
      sizeof(szResString) - 1);
   wvsprintf(szText, szResString, (LPSTR) VarArgList);
   return(MessageBox(hWnd, szText, szCaption, wType));
}


#define CFTS_CANTOPENFILE1    0
#define CFTS_CANTOPENFILE2    1
#define CFTS_FILE1ISNEWER     2
#define CFTS_FILE1ISOLDER     3
#define CFTS_FILESARESAME     4

int CompareFileTimeStamps (LPCSTR szPath1, LPCSTR szPath2) {
   struct ftime ftime1, ftime2;
   OFSTRUCT of;

   HFILE hFile = OpenFile(szPath1, &of, OF_READ);
   if (hFile == -1) return(CFTS_CANTOPENFILE1);
   getftime(hFile, &ftime1);
   _lclose(hFile);

   hFile = OpenFile(szPath2, &of, OF_READ);
   if (hFile == -1) return(CFTS_CANTOPENFILE2);
   getftime(hFile, &ftime2);
   _lclose(hFile);

   int x = ftime1.ft_year - ftime2.ft_year;
   if (x != 0)
      return((x > 0) ? CFTS_FILE1ISNEWER : CFTS_FILE1ISOLDER);

   x = ftime1.ft_month - ftime2.ft_month;
   if (x != 0)
      return((x > 0) ? CFTS_FILE1ISNEWER : CFTS_FILE1ISOLDER);

   x = ftime1.ft_day - ftime2.ft_day;
   if (x != 0)
      return((x > 0) ? CFTS_FILE1ISNEWER : CFTS_FILE1ISOLDER);

   x = ftime1.ft_hour - ftime2.ft_hour;
   if (x != 0)
      return((x > 0) ? CFTS_FILE1ISNEWER : CFTS_FILE1ISOLDER);

   x = ftime1.ft_min - ftime2.ft_min;
   if (x != 0)
      return((x > 0) ? CFTS_FILE1ISNEWER : CFTS_FILE1ISOLDER);

   x = ftime1.ft_tsec - ftime2.ft_tsec;
   if (x != 0)
      return((x > 0) ? CFTS_FILE1ISNEWER : CFTS_FILE1ISOLDER);

   return(CFTS_FILESARESAME);
}


UINT WINAPI PrepareToInstallFile (HWND hWnd, UINT uFlags,
   LPCSTR szDstFileName, LPCSTR szAppDir, LPSTR szCurDir,
   UINT FAR* lpuCurDirLen, LPSTR szDestDir,
   UINT FAR* lpuDestDirLen, LPCSTR szSrcDir,
   LPCSTR szSrcFileName) {

   char szBuf[100];
   UINT x;

Retry:
   UINT uVFFResult = _VerFindFile(uFlags, szDstFileName,
      NULL, szAppDir, szCurDir, lpuCurDirLen, szDestDir,
      lpuDestDirLen);

   if (uVFFResult & VFF_FILEINUSE) {
      char szSrcPath[MAXPATH];

      wsprintf(szSrcPath, "%s%s%s", szSrcDir,
         ((szSrcDir[lstrlen(szSrcDir) - 1] == '\\') ? "" : "\\"),
         szSrcFileName);

      wsprintf(szBuf, "%s%s%s", szCurDir,
         ((szCurDir[lstrlen(szCurDir) - 1] == '\\') ? "" : "\\"),
         szDstFileName);

      x = CompareFileTimeStamps(szSrcPath, szBuf);
      if (CFTS_FILESARESAME != x) {
         x = DialogBoxParam(_hInstance,
            MAKEINTRESOURCE(DLG_FILEINUSE), hWnd,
            (DLGPROC) FileInUseDlgProc, (LPARAM) szBuf);
         if (x == IDABORT) return(PTIF_ABORTSETUP);
         if (x == IDRETRY) goto Retry;
      }
      return(PTIF_SKIPTHISFILE); // x == IDIGNORE
   }

   if ((uVFFResult & VFF_CURNEDEST) && (szCurDir[0] != 0)) {
      CURNEDESTSTRUCT CurNEDestStruct;
      lstrcpy(CurNEDestStruct.szDstFileName, szDstFileName);
      lstrcpy(CurNEDestStruct.szDstDir, szDestDir);
      lstrcpy(CurNEDestStruct.szCurDir, szCurDir);
      x = DialogBoxParam(_hInstance,
         MAKEINTRESOURCE(DLG_CURNEDEST), hWnd,
         (DLGPROC) CurNEDestDlgProc, (LPARAM) &CurNEDestStruct);
      if (x == ID_INSTALLANDDELETE)
         return(PTIF_INSTALLTHISFILE);
      if (x == ID_INSTALLANDKEEP)
         return(PTIF_DONTERASEEXISTINGFILE);
      if (x == ID_REPLACEEXISTING)
         return(PTIF_REPLACEEXISTINGFILE);
      if (x == ID_SKIPINSTALL)
         return(PTIF_SKIPTHISFILE);
      if (x == ID_ABORTINSTALL)
         return(PTIF_ABORTSETUP);
   }

   if (uVFFResult & VFF_BUFFTOOSMALL)
      return(PTIF_INTERNALERROR);

   // Everything was sucessful
   return(PTIF_INSTALLTHISFILE);
}



UINT WINAPI InstallFile (HWND hWnd, UINT uFlags,
   LPCSTR szSrcFileName, LPCSTR szDestFileName, LPCSTR szSrcDir,
   LPCSTR szDestDir, LPCSTR szCurDir, LPSTR szTmpFile,
   UINT FAR* lpuTmpFileLen) {

   char szBuf[100];
   UINT x;
   OFSTRUCT of;

Retry:
   DWORD dwVIFResult = _VerInstallFile(uFlags, szSrcFileName,
      szDestFileName, szSrcDir, szDestDir, szCurDir, szTmpFile,
      lpuTmpFileLen);
   wsprintf(szBuf, "%s%s%s", szCurDir,
      ((szCurDir[lstrlen(szCurDir) - 1] == '\\') ? "" : "\\"),
      szDestFileName);


   // Check to see if an unrecoverable error occured
   // Errors are listed in least critical to most critical order
   x = 0;
   if (dwVIFResult & VIF_CANNOTCREATE)
      x = IDS_CANNOTCREATE;

   if (dwVIFResult & VIF_CANNOTDELETE)
      x = IDS_CANNOTDELETE;

   if (dwVIFResult & VIF_CANNOTRENAME)
      x = IDS_CANNOTRENAME;

   if (dwVIFResult & VIF_OUTOFSPACE)
      x = IDS_OUTOFSPACE;

   if (dwVIFResult & VIF_ACCESSVIOLATION)
      x = IDS_ACCESSVIOLATION;

   if (dwVIFResult & VIF_SHARINGVIOLATION)
      x = IDS_SHARINGVIOLATION;

   if (dwVIFResult & VIF_FILEINUSE)
      x = IDS_FILEINUSE;

   if (dwVIFResult & VIF_OUTOFMEMORY)
      x = IDS_OUTOFMEMORY;

   if (dwVIFResult & VIF_CANNOTREADDST)
      x = IDS_CANNOTREADDST;

   if (dwVIFResult & VIF_WRITEPROT)
      x = IDS_WRITEPROT;

   if (x != 0) {
      x = MsgBox(_hInstance, hWnd, x, _szAppName,
         MB_ICONQUESTION | MB_ABORTRETRYIGNORE, szBuf);

      // If the temporary file was created, delete it
      if (dwVIFResult & VIF_TEMPFILE) {
         wsprintf(szBuf, "%s%s%s", szDestDir,
            ((szDestDir[lstrlen(szDestDir) - 1] == '\\')
            ? "" : "\\"), szDestFileName);
         OpenFile(szBuf, &of, OF_DELETE);
      }

      if (x == IDRETRY)
         goto Retry;

      if (x == IDIGNORE)
         return(IF_SKIPTHISFILE);

      if (x == IDABORT)
         return(IF_ABORTSETUP);
   }

   if (dwVIFResult & VIF_CANNOTREADSRC)
      return(IF_CHANGEDISK);

   // An unrecoverable error did NOT occur
   if (dwVIFResult & (VIF_MISMATCH | VIF_SRCOLD | VIF_DIFFLANG |
      VIF_DIFFCODEPG | VIF_DIFFTYPE)) {

      MISMATCHSTRUCT MismatchStruct;

      wsprintf(MismatchStruct.szExistingPath, "%s%s%s", szCurDir,
         ((szCurDir[lstrlen(szCurDir) - 1] == '\\') ? "" : "\\"),
         szDestFileName);

      wsprintf(MismatchStruct.szTmpPath, "%s%s%s", szDestDir,
         ((szDestDir[lstrlen(szDestDir) - 1] == '\\')
         ? "" : "\\"), szTmpFile);

      wsprintf(MismatchStruct.szSrcPath, "%s%s%s", szSrcDir,
         ((szSrcDir[lstrlen(szSrcDir) - 1] == '\\')
         ? "" : "\\"), szSrcFileName);

      MismatchStruct.dwVIFResult = dwVIFResult;

      x = DialogBoxParam(_hInstance,
         MAKEINTRESOURCE(DLG_MISMATCH), hWnd,
         (DLGPROC) MismatchDlgProc, (LPARAM) &MismatchStruct);
      if (x == IDYES) {
         // Force the installation
         uFlags |= VIFF_FORCEINSTALL;
         goto Retry;
      }

      // We are not installing this file, delete the temp. file
      if (dwVIFResult & VIF_TEMPFILE) {
         wsprintf(szBuf, "%s\\%s", szDestDir, szTmpFile);
         OpenFile(szBuf, &of, OF_DELETE);
      }

      if (x == IDNO)
         return(IF_SKIPTHISFILE);

      if (x == IDABORT)
         return(IF_ABORTSETUP);
   }
   return(IF_INSTALLEDOK);
}
