// ObjectWindows - (C) Copyright 1992 by Borland International

/* --------------------------------------------------------
  FILEWND.CPP
  Defines type TxFileWindow, a text editor which can read
  and write to a file.
  -------------------------------------------------------- */

#include <string.h>
#include <stdio.h>
#include <alloc.h>
#include "xfilewnd.h"

#define Min(a, b)     ((a) < (b) ? (a) : (b))

/* Constructor for a TxFileWindow.  Initializes its data members using
  passed parameters and default values. */
TxFileWindow::TxFileWindow(PTWindowsObject AParent,LPSTR ATitle,
                         LPSTR AFileName, PTModule AModule)
             :   TEditWindow(AParent, ATitle, AModule)
{
  IsNewFile = TRUE;
  FileName = _fstrdup(AFileName ? AFileName : "");
}

/* Dispose of the file name. */
TxFileWindow::~TxFileWindow()
{
  if ( FileName )
    farfree(FileName);
}

/* Performs setup for a TxFileWindow, appending 'Untitled' to its caption */
void TxFileWindow::SetupWindow()
{
  TEditWindow::SetupWindow();
  SetFileName(FileName);
  if ( FileName[0] != '\0')
    if ( !Read() )
      SetFileName("");
}

/* Sets the file name of the window and updates the caption. */
void TxFileWindow::SetFileName(LPSTR AFileName)
{
  char NewCaption[81];
  LPSTR P[2];

  if ( FileName != AFileName )
  {
    farfree(FileName);
    FileName = _fstrdup(AFileName ? AFileName : "");
  }
  P[0] = Title;
  if ( FileName[0] == '\0' )
    P[1] = "(Untitled)";
  else
    P[1] = AFileName;
  if ( Title == NULL || Title[0] == '\0' )
    SetWindowText(HWindow, P[1]);
  else
  {
    wvsprintf(NewCaption, "%s - %s", (LPSTR)P);
    SetWindowText(HWindow, NewCaption);
  }
}

/* Begins the edit of a new file, after determining that it is Ok to
  clear the TEdit's text. */
void TxFileWindow::NewFile()
{
  if ( CanClear() )
  {
    Editor->Clear();
    InvalidateRect(Editor->HWindow, NULL, FALSE);
    Editor->ClearModify();
    IsNewFile = TRUE;
    SetFileName(NULL);
  }
}

/* Replaces the current file with the given file. */
void TxFileWindow::ReplaceWith(LPSTR AFileName)
{
  char OldName[MAXPATH];

  _fstrcpy(OldName, FileName);
  SetFileName(AFileName);
  if ( Read() )
    InvalidateRect(Editor->HWindow, NULL, FALSE);
  else
    SetFileName(OldName);
}

/* Brings up a dialog allowing the user to open a file into this
  window.  Same as selecting File|Open from the menus. */
void TxFileWindow::Open()
{
  char TmpName[MAXPATH];

  if ( CanClear() && (GetModule()->ExecDialog(
         new TFileDialog(this, SD_FILEOPEN,
                         _fstrcpy(TmpName, "*.*"), GetModule())) == IDOK) )
    ReplaceWith(TmpName);
}

static HANDLE LocalReAllocDS(HANDLE hMem, WORD wBytes, WORD wFlags,
                                                                 WORD TheDS)
{
  WORD SavedDS;
  HANDLE ReturnValue;

  SavedDS = _DS;
  _DS = TheDS;
  ReturnValue = LocalReAlloc(hMem, wBytes, wFlags);
  _DS = SavedDS;
  return ReturnValue;
}

static LPSTR LocalLockDS(HANDLE hMem, WORD TheDS)
{
  WORD SavedDS;
  LPSTR ReturnValue;

  SavedDS = _DS;
  _DS = TheDS;
  ReturnValue = (LPSTR)LocalLock(hMem);
  _DS = SavedDS;
  return ReturnValue;
}

static BOOL LocalUnlockDS(HANDLE hMem, WORD TheDS)
{
  WORD SavedDS;
  BOOL ReturnValue;

  SavedDS = _DS;
  _DS = TheDS;
  ReturnValue = LocalUnlock(hMem);
  _DS = SavedDS;
  return ReturnValue;
}

/* Reads the contents of a previously-specified file into the TEdit
  child control. */
BOOL TxFileWindow::Read()
{
  long CharsToRead;
  UINT HEditorBuffer;
  LPSTR EditorBuffer;
  WORD EditorsDS;
  char S[MAXPATH + 33];
  int AFile;
  BOOL Success = FALSE;

  AFile = _lopen(FileName, OF_READ);
  if ( AFile != -1 )
  {
    CharsToRead = _llseek(AFile, 0L, 2);
    _llseek(AFile, 0L, 0);
    if ( CharsToRead < MaxInt && CharsToRead > 0 )
    {
      Editor->Clear();

      // attempt to reallocate Editor's buffer to the size of the file
      HEditorBuffer = SendMessage(Editor->HWindow, EM_GETHANDLE, 0, 0L);
      EditorsDS = FP_SEG(GlobalLock((Editor->GetModule())->hInstance));
      if ( LocalReAllocDS((HANDLE)HEditorBuffer, (WORD)(CharsToRead+1),
                          LHND, EditorsDS) != NULL )
      {
        // read the file into EditorBuffer
	EditorBuffer = (LPSTR)LocalLockDS((HANDLE)HEditorBuffer, EditorsDS);
        if ( _lread(AFile, EditorBuffer, (WORD)CharsToRead) == CharsToRead )
        {
          // NULL terminate Editor's buffer
          EditorBuffer[(WORD)CharsToRead] = '\0';
	  LocalUnlockDS((HANDLE)HEditorBuffer, EditorsDS);

	  SendMessage(Editor->HWindow, EM_SETHANDLE, HEditorBuffer, 0L);
          Success = TRUE;

          IsNewFile = FALSE;
          Editor->ClearModify();
          Editor->SetSelection(0,0);
        }
      }
      GlobalUnlock((Editor->GetModule())->hInstance);
    }
    _lclose(AFile);
  }
  if ( !Success )
  {
    wsprintf(S, "Unable to read file \"%s\" from disk", FileName);
    MessageBox(HWindow, S, GetModule()->Name, MB_ICONEXCLAMATION | MB_OK);
  }
  return Success;
}

/* Saves the contents of the TEdit child control into the file currently
  being editted.  Returns true if the file was saved or
  Editor->IsModified returns FALSE (contents already saved). */
BOOL TxFileWindow::Save()
{
  if ( Editor->IsModified() )
  {
    if ( IsNewFile )
      return SaveAs();
    else
    {
      if ( Write() )
        return TRUE;
      else
        return FALSE;
    }
  }
  else       // Editor's contents haven't been changed. No need to write.
    return TRUE;
}

/* Saves the contents of the TEdit child control into a file whose name
  is retrieved from the user, through execution of a "Save" file
  dialog. Returns true if the file was saved. */
BOOL TxFileWindow::SaveAs()
{
  char TmpName[MAXPATH];
  char OldName[MAXPATH];
  OFSTRUCT TmpOfStruct;
  char S[MAXPATH+20];

  _fstrcpy(OldName, FileName);
  if ( FileName )
    _fstrcpy(TmpName, FileName);
  else
    TmpName[0] = '\0';
  if ( GetModule()->ExecDialog( new TFileDialog
            (this, SD_FILESAVE, TmpName, GetModule())) == IDOK )
  {
    if ( OpenFile(TmpName, &TmpOfStruct, OF_EXIST) != -1 )
    {
      wsprintf(S, "Replace Current \"%s\"?", (LPSTR)TmpName);
      if ( MessageBox(HWindow, S, "File Changed",
                          MB_YESNO | MB_ICONQUESTION) == IDNO )
      {
        SetFileName(OldName);
        return FALSE;
      }
    }
    SetFileName(TmpName);
    if ( Write() )
      return TRUE;
    else
    {
      SetFileName(OldName);
      return FALSE;
    }
  }
  return FALSE;
}

/* Writes the contents of the TEdit child control to a
previously-specified file. */
BOOL TxFileWindow::Write()
   {
   int AFile;
   HANDLE hEditBuffer;                       /* handle to editing buffer      */
   LPSTR pEditBuffer;                        /* address of the edit buffer      */
   char S[MAXPATH + 33];
   WORD EditorsDS;
   HANDLE hinst;
   
   AFile = _lcreat(FileName, 0);
   if ( AFile == -1 )
      {
      wsprintf(S, "Unable to write file \"%s\" to disk", FileName);
      MessageBox(HWindow, S, GetModule()->Name, MB_ICONEXCLAMATION | MB_OK);
      return FALSE;
      }
   else
      {
      
      hEditBuffer = (HANDLE)SendMessage(Editor->HWindow, EM_GETHANDLE, 0, 0L);
      EditorsDS = FP_SEG(GlobalLock((Editor->GetModule())->hInstance));
      pEditBuffer = (LPSTR)LocalLockDS(hEditBuffer, EditorsDS);
      
      if (_lwrite(AFile, pEditBuffer, strlen(pEditBuffer)) == (WORD)-1)
         {
         LocalUnlockDS(hEditBuffer, EditorsDS);
         GlobalUnlock((Editor->GetModule())->hInstance);
         _lclose(AFile);
         return FALSE;
         }
      IsNewFile = FALSE;
      Editor->ClearModify();
      LocalUnlockDS(hEditBuffer, EditorsDS);
      GlobalUnlock((Editor->GetModule())->hInstance);
      _lclose(AFile);
      }
   
   return TRUE;
   }

/* Returns a BOOL value indicating whether or not it is Ok to clear
  the TEdit's text.  Returns TRUE if the text has not been changed, or
  if the user Oks the clearing of the text. */
BOOL TxFileWindow::CanClear()
{
  char S[MAXPATH+28];
  int Rslt;

  if ( Editor->IsModified() )
  {
    if ( !FileName || (FileName[0]=='\0'))
      _fstrcpy(S, "Untitled file has changed.  Save?");
    else
      wsprintf(S, "File \"%s\" has changed.  Save?", FileName);
    Rslt = MessageBox(HWindow, S, "File Changed",
                          MB_YESNOCANCEL | MB_ICONQUESTION);
    if ( Rslt == IDYES )
      return Save();
    else
      return Rslt != IDCANCEL;
  }
  return TRUE;
}

/* Returns a BOOL value indicating whether or not it is Ok to close
  the TEdit's text.  Returns the result of a call to CanClear. */
BOOL TxFileWindow::CanClose()
{
  return CanClear();
}

/* Responds to an incoming "New" command (with a CM_FILENEW command
  identifier) by calling NewFile. */
void TxFileWindow::CMFileNew(TMessage&)
{
  NewFile();
}
	
/* Responds to an incoming "Open" command (with a CM_FILEOPEN command
  identifier) by calling Open(). */
void TxFileWindow::CMFileOpen(TMessage&)
{
  Open();
}

/* Responds to an incoming "Save" command (with a CM_FILESAVE
   command identifier) by calling Save. */
void TxFileWindow::CMFileSave(TMessage&)
{
  Save();
}
/* Responds to an incoming "SaveAs" command (with a CM_FILESAVEAS
   command identifier) by calling SaveAs. */
void TxFileWindow::CMFileSaveAs(TMessage&)
{
  SaveAs();
}

/* Reads an instance of TxFileWindow from the passed ipstream. */
void *TxFileWindow::read(ipstream& is)
{
  TEditWindow::read(is);
  FileName = is.freadString();
  IsNewFile = ( FileName[0] == '\0' );
  return this;
}

/* Writes the TxFileWindow to the passed opstream. */
void TxFileWindow::write(opstream& os)
{
  TEditWindow::write(os);
  os.fwriteString(FileName);
}

TStreamable *TxFileWindow::build()
{
  return new TxFileWindow(streamableInit);
}

TStreamableClass RegFileWindow("TxFileWindow", TxFileWindow::build,
					    __DELTA(TxFileWindow));
