PROGRAM toshiba;

{ Power supply control functions on Toshiba T1000SE. }
{ Largely the same on other Toshiba laptops. }

PROCEDURE ps_command(b: byte);
  { Sends a command to the Toshiba power supply }
VAR r: byte;
    i: integer;
BEGIN
  port[$0e8] := $5a;                            { Send the key }
  port[$0e9] := b;                          { Send the command }
  i := 0;
  REPEAT
    i := i + 1;
    r := port[$0e9]
  UNTIL ((r AND $40) <> 0) OR (i > 30000);
                                     { Wait until cmd accepted }
  IF i>30000 THEN
    write('<WARNING: Command timed out>')
  ELSE
    r := port[$0e8]               { Accept the acknowledgement }
END;

FUNCTION ps_response: byte;
  { Obtains data sent by power supply to CPU }
VAR r: byte;
    i: integer;
BEGIN
  i := 0;
  REPEAT
    i := i + 1;
    r := port[$0e9]
  UNTIL ((r AND $40) <> 0) OR (i > 30000);
                                       { Wait until data ready }
  IF i>30000 THEN
    BEGIN
      write('<WARNING: Data timed out>');
      ps_response := port[$0e8]
    END
  ELSE
    BEGIN
      ps_response := port[$0e8];            { Get the response }
      port[$0e8] := $5a;              { Unlock the output port }
      port[$0e9] := $ff;         { Send acknowledgement signal }
    END
END;

FUNCTION battery_level: byte;
  { Range 0 to 7.  If user has not set it, then set it to 7. }
  { That's reasonable, because if weak, the battery will }
  { almost immediately drop to a lower level. }
VAR r: byte;
BEGIN
  ps_command($c0);
  r := ps_response AND $0f;
  IF r < 8 THEN
    BEGIN
      ps_command($d7);  { set to level 7 }
      ps_command($c0);  { inquire again }
      r := ps_response;
    END;
  battery_level := r AND $07
END;

PROCEDURE set_battery_level(b:byte);
  { Sets the battery level. Range is 0 to 7. }
BEGIN
  ps_command($d0 + (b AND $07))
END;

PROCEDURE screen_off;
  { Turns fluorescent screen off if battery powered. }
  { If using AC power, screen stays on. }
BEGIN
  ps_command($b0)
END;

PROCEDURE screen_on;
  { Turns fluorescent screen on. }
BEGIN
  ps_command($bf)
END;


BEGIN  { Demonstration of the above }
  writeln('TOSHIBA CONTROL DEMONSTRATION');
  writeln('Switching screen illumination off (if battery powered).');
  writeln('Press Enter when ready, the Enter again...');
  readln;

  screen_off;
  readln;

  screen_on;

  writeln('Current battery level is ',battery_level);
  writeln('Press Enter...');
  readln;

  writeln('Setting battery level to 7...');
  set_battery_level(7);

  writeln('Current battery level is ',battery_level);
  writeln('Press Enter...');
  readln;

  writeln('That''s all, folks!')
END.
