// scroll.cpp -- Inefficient scrolling example by Tom Swan

#include <owl.h>

#define NUMROWS   48    // Number of rows (and colors)
#define NUMCOLS    3    // Number of columns
#define ROWSEP    10    // Separation between rows
#define COLSEP    32    // Separation between columns 
#define RHEIGHT   40    // Rectangle height
#define RWIDTH    72    // Rectangle width

COLORREF colors[NUMROWS] = {
  0xff8080L, 0xff0000L, 0x804040L, 0x800000L,
  0x400000L, 0x000000L, 0xffffe8L, 0xffff80L,
  0xffff00L, 0xff8000L, 0x804000L, 0x808000L,
  0x80ff80L, 0x80ff00L, 0x00ff00L, 0x008000L,
  0x004000L, 0x808040L, 0x00ff80L, 0x00ff40L,
  0x008080L, 0x008040L, 0x004040L, 0x808080L,
  0x80ffffL, 0x00ffffL, 0x004080L, 0x0000ffL,
  0x000080L, 0x408080L, 0x0080ffL, 0x0080c0L,
  0x8080ffL, 0x0000a0L, 0x000040L, 0xc0c0c0L,
  0xff80c0L, 0x8080c0L, 0x800040L, 0x800080L,
  0x400040L, 0x828282L, 0xff80ffL, 0xff00ffL,
  0xff0080L, 0x8000ffL, 0x400080L, 0xffffffL
};

class TScrollApp: public TApplication {
public:
  TScrollApp(LPSTR aName, HANDLE hInstance, HANDLE hPrevInstance,
    LPSTR lpCmd, int nCmdShow) : TApplication(aName, hInstance,
    hPrevInstance, lpCmd, nCmdShow) { };
  virtual void InitMainWindow();
};

_CLASSDEF(TScrollWin)
class TScrollWin: public TWindow {
public:
  HBRUSH rowBrushes[NUMROWS];  // One brush for each row
  HBRUSH colBrushes[NUMCOLS];  // One brush for each column
  TScrollWin(PTWindowsObject AParent, LPSTR ATitle);
  ~TScrollWin();
  void GetCoords(int row, int col,
    int &x1, int &y1, int &x2, int &y2);
  virtual void Paint(HDC PaintDC, PAINTSTRUCT _FAR &ps);
};

void TScrollApp::InitMainWindow()
{
  MainWindow = new TScrollWin(NULL, "Slow Scroll Bars");
}

TScrollWin::TScrollWin(PTWindowsObject AParent, LPSTR ATitle)
  : TWindow(AParent, ATitle)
{
  int i;

  Attr.Style = Attr.Style | (WS_VSCROLL | WS_HSCROLL);
  for (i = 0; i < NUMROWS; i++) {
    rowBrushes[i] = CreateSolidBrush(colors[i]);
    if (rowBrushes[i] == 0) {
      Status = -1;  // Possibly out of memory
      return;
    }
  }
  colBrushes[0] = CreateSolidBrush(RGB(0, 0, 255));
  colBrushes[1] = CreateSolidBrush(RGB(0, 0, 0));
  colBrushes[2] = CreateSolidBrush(RGB(255, 255, 255));
  for (i = 0; i < NUMCOLS; i++)
    if (colBrushes[i] == 0) {
      Status = -2;  // Possibly out of memory
      return;
    }
  Scroller = new TScroller(this, 1, 1, 2000, 4000);
}

TScrollWin::~TScrollWin()
{
  int i;

  for (i = 0; i < NUMROWS; i++)
    DeleteObject(rowBrushes[i]);
  for (i = 0; i < NUMCOLS; i++)
    DeleteObject(colBrushes[i]);
}

void TScrollWin::GetCoords(int row, int col,
  int &x1, int &y1, int &x2, int &y2)
{
  x1 = COLSEP + (col * (COLSEP + RWIDTH));
  y1 = ROWSEP + (row * (ROWSEP + RHEIGHT));
  x2 = x1 + RWIDTH;
  y2 = y1 + RHEIGHT;
}

void TScrollWin::Paint(HDC PaintDC, PAINTSTRUCT _FAR &/*ps*/)
{
  HBRUSH oldBrush;         // Saved DC brush
  int x1, y1, x2, y2;      // Ellipse coordinates
  int row, col;            // Row and column numbers

  for (row = 0; row < NUMROWS; row++) {
    for (col = 0; col < NUMCOLS; col++) {
      GetCoords(row, col, x1, y1, x2, y2);
      oldBrush = SelectObject(PaintDC, colBrushes[col]);
      Rectangle(PaintDC, x1, y1, x2, y2);
      SelectObject(PaintDC, rowBrushes[row]);
      Ellipse(PaintDC, x1 + 4, y1 + 4, x2 - 4, y2 - 4);
      SelectObject(PaintDC, oldBrush);
    }
  }
}

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance,
  LPSTR lpCmd, int nCmdShow)
{
  TScrollApp ScrollApp("ScrollApp", hInstance, hPrevInstance,
    lpCmd, nCmdShow);
  ScrollApp.Run();
  return ScrollApp.Status;
}
