{
 
                                                                          
         TITLE :      DGCMDLN.TPU                                         
       PURPOSE :      Parameter and Command Line handling.                
        AUTHOR :      David Gerrold, CompuServe ID:  70307,544            
  ______________________________________________________________________  
                                                                          
    Written in Turbo Pascal, Version 5.5,                                 
    with routines from TurboPower, Object Professional.                   
                                                                          
    Turbo Pascal is a product of Borland International.                   
    Object Professional is a product of TurboPower Software.              
  ______________________________________________________________________  
                                                                          
    Copyright 1991, by David Gerrold.                                     
    Permission is hereby granted for reuse.                               
                                                                          
         The Brass Cannon Corporation                                     
         9420 Reseda Blvd., #804                                          
         Northridge, CA  91324-2932.                                      
                                                                          
 
                                                                            }
{ Compiler Directives ===================================================== }

{$R-}    {Range checking off}
{$B-}    {Boolean complete evaluation off}
{$S-}    {Stack checking off}
{$I-}    {I/O checking off}
{$V-}    {Variable range checking off}

{ Name ==================================================================== }

UNIT DgCmdLn;
{
	The purpose of CmdLn is to provide routines for parsing and handling
	command line parameters and parameter strings passed to a program.
}

{ Interface =============================================================== }

INTERFACE

USES
{ Object Professional Units }
	OpString;                                      { for trim function }

{ Declarations ============================================================ }
{ Constants --------------------------------------------------------------- }

TYPE
	CmdLnOb = Object                               
		ParameterCount    : byte;                    { number of params }
		ParameterStr      : string;                  { store params }
		ParameterStrArray : array [1 .. 25] of byte; { index to params }

		SwitchCount       : byte;                    { number of switches }
		SwitchStr         : string;                  { store switches }
		SwitchStrArray    : array [1 .. 25] of byte; { index to switches }

		Procedure Init;                              { set up ParameterStr }
		Function Parameter (Index : byte) : string;  { return string }
		Function PosParameter (S : string) : byte;   { index pos of param }
		Procedure DeleteIndex (Index : byte);        { delete param by index }
		Procedure DeleteParameter (S : string);      { zap all occurrences of S }

		Function Switch (Index : byte) : string;     { return switch }
		Function ExistSwitch (S : string ) : boolean;{ does Switch exist }
		Function ExistSwitchCombined (S : string) : boolean;
		end;

VAR
	CmdLn : CmdLnOb;


{ ========================================================================= }
{ IMPLEMENTATION ========================================================== }

IMPLEMENTATION

{ ========================================================================= }
{ CmdLnOb.Init ============================================================ }

PROCEDURE CmdLnOb.Init;
VAR
	Loop  : byte;
	Index : byte;
	S     : string;

BEGIN
{
	First null out the ParameterStr variables and the SwitchStr variables.
}
	ParameterCount := 0;                           { null out counter }
	ParameterStr   := '';                          { null out string }
	fillchar
		(ParameterStrArray, 0, sizeof (ParameterStrArray));  { null out array }
	SwitchCount := 0;                              { null out counter }
	SwitchStr   := '';                             { null out string }
	fillChar
		(SwitchStrArray, 0, sizeof (SwitchStrArray));{ null out array }

{
	Now, loop through command line parameters.
}
	for Index := 1 to ParamCount do begin          { loop through params }
		S := trim (ParamStr (Index));                { get S }
		if                                           { if }
			(S [1] <> '/') and                         { S doesn't lead with '/' }
			(S [1] <> '-')                             { or '-' }
		then begin                                   { then it's a parameter }
{
	If the ParamStr doesn't lead with a switch char,
	then it's a parameter, not a switch.  Stash it.

	But if it's got a '/' embedded in it, then it has
	an appended switch or set of switches.
}
			inc (ParameterCount);                      { inc parameter counter }
			ParameterStrArray [ParameterCount] :=
				succ (length (ParameterStr));            { store parameter index }

			Loop := pos ('/', S);                      { get pos of '/' }
			if Loop = 0 then                           { if not found then }
				ParameterStr := ParameterStr + S + ' '   { stash param }
			else begin                                 { else there's a switch }
				dec (Loop);                              { dec index }
				ParameterStr := ParameterStr +
					copy (S, 1, Loop) + ' ';               { stash param, leave switch }
				delete (S, 1, Loop);                     { delete param from S }
				end;
			end;  { if S [1] <> '/' }

{
	Now, stash -s-e or other switches/p/d/q.
}
		While                                        { while }
			(S [1] = '/') or                           { S leads with '/' }
			(S [1] = '-')                              { or '-' }
		do begin                                     { do: }
			delete (S, 1, 1);                          { delete '/' lead }
			if S = '' then S := '@';                   { trap nul string }
			inc (SwitchCount);                         { add to count }
			SwitchStrArray [SwitchCount] :=
				succ (length (SwitchStr));               { store index }

			Loop := pos ('/', S);                      { get pos of mid '/' }
			if Loop = 0 then                           { if no pos then }
				Loop := pos ('-', S);                    { get pos of mid '-' }
			if Loop = 0 then
				SwitchStr := SwitchStr + S               { stash switch }
			else begin
				dec (Loop);                              { dec index }
				SwitchStr := SwitchStr +
					copy (S, 1, Loop);                     { stash 1st switch only }
				delete (S, 1, Loop);                     { delete 1st switch }
				end;
			end;  { While S[1] = '/', etc. }
		end;  { for Index := 1 to ParamCount }

	ParameterStrArray [succ (ParameterCount)] :=
		succ (length (ParameterStr));                { store param index }
	SwitchStrArray [succ (SwitchCount)] :=
		succ (length (SwitchStr));                   { store switch index }
	SwitchStr := StUpCase (SwitchStr);             { upcase SwitchStr }
END;

{ CmdLnOb.Parameter ======================================================= }

FUNCTION CmdLnOb.Parameter (Index : byte) : string;
{ returns parameter string }

BEGIN
	Parameter := copy (ParameterStr,
										 ParameterStrArray [Index],
										 ParameterStrArray [succ (Index)] -
											 succ (ParameterStrArray [Index]));
END;

{ CmdLnOb.PosParameter ==================================================== }

FUNCTION CmdLnOb.PosParameter (S : string) : byte;   { does param exist? }
VAR
	Flag : boolean;
	Loop : byte;

BEGIN
	PosParameter := 0;
	Flag := false;
	Loop := 0;

	repeat
		inc (Loop);
		Flag := (CompUCString (S, Parameter (Loop)) = Equal);
	until
		Flag or (Loop = ParameterCount);

	if Flag then
		PosParameter := Loop;
END;

{ CmdLnOb.DeleteIndex ===================================================== }

PROCEDURE CmdLnOb.DeleteIndex (Index : byte);
VAR
	Loop,
	Len   : byte;
BEGIN
	if Index > ParameterCount then exit;

	Len := ParameterStrArray [succ (Index)] -
					 ParameterStrArray [Index];                 { get length of word }
	delete
		(ParameterStr, ParameterStrArray [Index], Len);   { delete word }
	for Loop := Index to succ (ParameterCount) do
		ParameterStrArray [Loop] :=
			ParameterStrArray [succ (Loop)] - Len;     { adjust array }
	dec (ParameterCount);                          { dec param count }
END;

{ CmdLnOb.DeleteParameter ================================================= }

PROCEDURE CmdLnOb.DeleteParameter (S : string);
VAR
	Loop : byte;
BEGIN
	repeat
		Loop := PosParameter (S);
		if Loop > 0 then
			DeleteIndex (Loop);
	until
		Loop = 0;
END;

{ CmdLnOb.Switch ========================================================== }

FUNCTION CmdLnOb.Switch (Index : byte) : string;
{ returns switch }

BEGIN
	Switch := copy (SwitchStr,
									SwitchStrArray [Index],
									SwitchStrArray [succ (Index)] -
									SwitchStrArray [Index]);
END;

{ CmdLnOb.ExistSwitch ===================================================== }

FUNCTION CmdLnOb.ExistSwitch (S : string) : boolean;
{ returns true if switch exists, then deletes it from the switch array }

VAR
	Len,
	Loop : byte;
	Flag : boolean;
BEGIN
	Flag := false;                                 { nul out flag }
	S := StUpCase (S);
	Loop := 1;

	{ is switch in array? }
	repeat                                         { loop through array }
		inc (Loop);
		Flag := (S = Switch (Loop));                 { looking for switch }
	until                                          { until }
		Flag or                                      { found }
		(Loop = SwitchCount);                        { or end of array }
	ExistSwitch := Flag;
	if not Flag then exit;                         { no switch, return }

	{ delete switch from array }
	Len := SwitchStrArray [succ (Loop)] -
					 SwitchStrArray [Loop];                { get length of switch }
	delete (SwitchStr, SwitchStrArray [Loop], Len);{ delete switch }
	for Loop := Loop to succ (SwitchCount) do
		SwitchStrArray [Loop] :=
			SwitchStrArray [succ (Loop)] - Len;        { adjust array }
	dec (SwitchCount);                             { dec param count }
END;

{ CmdLnOb.ExistSwitchCombined ============================================= }

FUNCTION CmdLnOb.ExistSwitchCombined (S : string) : boolean;
{ returns true if switch exists, then deletes it from the switch array }

VAR
	Flag : boolean;
	Loop : byte;
	I    : byte;
	Len  : byte absolute S;

BEGIN
	Flag := ExistSwitch (S);                       { single switch? }
	if Flag then begin                             { if yes then }
		ExistSwitchCombined := Flag;                 { return value }
		exit;                                        { and exit }
		end;

	S := StUpCase (S);                             { upcase S }
	Loop := Pos (S, SwitchStr);                    { get pos }
	Flag := Loop > 0;                              { flag = switch exists }
	ExistSwitchCombined := Flag;                   { return value }

	if Flag then                                   { if switch exists then }
		repeat                                       { delete all occurrences }
			if Loop > 0 then begin                     { if switch exists }
				delete (SwitchStr, Loop, Len);           { delete from SwitchStr }
				for I := 1 to SwitchCount do             { loop through array }
					if SwitchStrArray [I] > Loop then      { if array member > Loop }
						dec (SwitchStrArray [I]);            { then dec array member }
				end;
			Loop := pos (S, SwitchStr);                { does switch still exist? }
		until
			Loop = 0;                                  { until no more occurrences }
END;

{ ========================================================================= }
{ Initialization ========================================================== }

BEGIN
	CmdLn.Init;                                    { initialize CmdLn object }
END.

{ ========================================================================= }
{ DgCmdLn History ========================================================= }

VERSION HISTORY:
	9104.25
		CmdLnOb (Command Line Object) allows sophisticated access to
		parameter strings.  This will allow programs to parse for
		specific switches no matter where they occur on the command line.

{ ========================================================================= }
{ ========================================================================= }

{
	Compile and run test program to demonstrate operation of CmdLn object.
}

PROGRAM TestCmd;

USES
	DgCmdLn;

BEGIN
	with CmdLn do begin

		if ExistSwitch ('x') then
			writeln ('X switch set')
		else
			writeln ('X switch not set');

		if ExistSwitchCombined ('r') then
			writeln ('R switch set')
		else
			writeln ('R switch not set');

		end;
END.

{ ========================================================================= }
{ ========================================================================= }
