//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE.CPP - IBM PC Diskette Services Class Module
//
//  Purpose:   To provide a diskette services class.
//
//  Version:    0.9     *** NOT FOR GNERAL RELEASE ***
//
//  History:    91/07/31 - Created.
//
//  Compiler:   Microsoft C/C++ V7.0
//
//  Author:     Ian Ashdown, P.Eng.
//              byHeart Software Ltd.
//              620 Ballantree Road
//              West Vancouver, B.C.
//              Canada V7S 1W3
//              Tel. (604) 922-6148
//              Fax. (604) 987-7621
//
//  Copyright:  Public Domain
//
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
//
//  PORTABILITY NOTES
//
//  While this program is written in draft ANSI C++, it uses a number of
//  function calls that are specific to the Microsoft C/C++ V7.0 library.
//  These are documented as follows for the purposes of porting this
//  software to other MS-DOS C++ compilers:
//
//    _fcalloc      - allocate initialized far memory (cf. calloc)
//    _ffree        - release far memory (cf. free)
//    _fmemcpy      - copy bytes between far memory buffers (cf. memcpy)
//    FP_OFF        - get or set 16-bit far pointer offset
//    FP_SEG        - get or set 16-bit far pointer segment
//    int86         - execute 80x86 interrupt routine
//    int86x        - execute 80x86 interrupt routine (far data)
//    intdosx       - execute DOS system call (far data)
//    _osmajor      - MS-DOS major version number (global variable)
//    _osminor      - MS-DOS minor version number (global variable)
//
//////////////////////////////////////////////////////////////////////////

// .... INCLUDE FILES

#include "diskette.hpp"
#include <stdlib.h>
#include <malloc.h>
#include <memory.h>
#include <dos.h>

// .... PUBLIC FUNCTIONS

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::DISKETTE - Diskette Services Class Constructor
//
//  Purpose:    To open an IBM PC diskette drive.
//
//  Setup:      Diskette::Diskette
//              (
//                int drive
//              )
//
//  Where:      drive is the drive number (0 or 1).
//
//  Result:     The object status flag "obj_flag" is set to TRUE if the
//              drive was successfully opened; otherwise FALSE.
//
//////////////////////////////////////////////////////////////////////////

Diskette::Diskette
(
  int drive
)
{
  union REGS regs;      // 80x86 register values
  struct SREGS sregs;   // 80x86 segment register values

  obj_flag = TRUE;      // Assume successful initialization of object

  // Initialize the class members

  st_bufp = NULL;

  prev_dptp = NULL;

  if (drive != 0 && drive != 1)         // Validate drive number
  {
    obj_flag = FALSE;

    return;
  }

  // Get the Diskette Parameters Table (interrupt vector 0x1e) pointer

  regs.h.ah = 0x35;     // Specify "Get Interrupt Vector" function
  regs.h.al = 0x1e;     // Interrupt vector number

  // Call the MS-DOS interrupt function

  intdosx(&regs, &regs, &sregs);

  FP_OFF(curr_dptp) = regs.x.bx;
  FP_SEG(curr_dptp) = sregs.es;

  drive_num = (byte) drive;
  drive_error = FD_OK;

  bps_code = FD_BPS_512;        // MS-DOS default

  // Check System Identification Byte at ROM BIOS location F000:FFFE

  switch ((int) *((unsigned char _far *) 0xf000fffeL))
  {
    case FD_PC:         // PC
    case FD_XE:         // XT (early models)
    case FD_JR:         // PC Junior
    case FD_XL:         // XT (later models)

      system_type = FD_XT_CLASS;

      // XT-class systems only support 360K diskette drives

      cl_support = FD_NOCHANGE;
      drive_type = FD_DRV_360;

      max_track = 39;
      max_sector = 9;

      return;

    default:            // Assume AT-class

      system_type = FD_AT_CLASS;

      cl_support = get_change_support();

      if (cl_support == FD_NODRIVE)
      {
        obj_flag = FALSE;       // No drive installed

        return;
      }

      if (read_drive_parms() == FALSE)
      {
        obj_flag = FALSE;       // Drive type not in CMOS RAM
      }

      break;
  }
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::~DISKETTE - Diskette Services Class Destructor
//
//  Purpose:    To close a diskette drive.
//
//  Setup:      Diskette::~Diskette()
//
//////////////////////////////////////////////////////////////////////////

Diskette::~Diskette()
{
  // Remove any user-defined Diskette Parameters Table

  remove_table();
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::INSTALL_TABLE - Install Custom Diskette Parameters Table
//
//  Purpose:    To install a user-defined Diskette Parameters Table for a
//              diskette format not supported by the ROM BIOS.
//
//  Setup:      boolean Diskette::install_table
//              (
//                fd_dprm _far *new_dptp
//              )
//
//  Where:      new_dptp is a far pointer to a user-defined Diskette
//                Parameters Table.
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The global Diskette Parameters Table pointer is modified
//              to point to the user-defined table.
//
//  Note:       The ROM BIOS accesses the Diskette Parameters Table each
//              time it is called to perform a diskette function using
//              software interrupt vector 0x13 through a pointer to the
//              table stored in interrupt vector 0x1e (location
//              0000:0078).  The BIOS can be redirected to a user-defined
//              Diskette Parameters Table by modifying the table pointer.
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::install_table
(
  fd_dprm _far *new_dptp
)
{
  union REGS regs;      // 80x86 register values
  struct SREGS sregs;   // 80x86 segment register values

  if (prev_dptp != NULL)        // Check for existing user-defined table
    return (FALSE);

  // Save the current Diskette Parameters Table

  prev_dptp = curr_dptp;

  // Redirect the Diskette Parameters Table pointer

  regs.h.ah = 0x25;     // Specify "Set Interrupt Vector" function
  regs.h.al = 0x1e;     // Interrupt vector number

  regs.x.dx = FP_OFF(new_dptp);
  sregs.ds = FP_SEG(new_dptp);

  // Call the MS-DOS interrupt function

  intdosx(&regs, &regs, &sregs);

  // Update the current Diskette Parameters Table pointer

  curr_dptp = new_dptp;

  max_sector = curr_dptp->sect_trk;
  bps_code = curr_dptp->bps_code;

  // Check for 512 bytes per sector

  if (new_dptp->bps_code != FD_BPS_512)
  {
    // Allocate a sector transfer buffer

    if (allocate_buffer() == FALSE)
    {
      remove_table();   // Remove the user-defined table

      return (FALSE);
    }
  }

  return (TRUE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::REMOVE_TABLE - Remove Custom Diskette Parameters Table
//
//  Purpose:    To remove a user-defined Diskette Parameters Table.
//
//  Setup:      void Diskette::remove_table()
//
//  Result:     The global Diskette Parameters Table pointer is modified
//              to point to the previous table.
//
//              The class members "prev_dptp" and "curr_dptp" are updated.
//
//////////////////////////////////////////////////////////////////////////

void Diskette::remove_table()
{
  union REGS regs;      // 80x86 register values
  struct SREGS sregs;   // 80x86 segment register values

  if (prev_dptp != NULL)        // Check for existing table
  {
    // Restore the previous Diskette Parameters Table pointer

    regs.h.ah = 0x25;   // Specify "Set Interrupt Vector" function
    regs.h.al = 0x1e;   // Interrupt vector number

    regs.x.dx = FP_OFF(prev_dptp);
    sregs.ds = FP_SEG(prev_dptp);

    // Call the MS-DOS interrupt function

    intdosx(&regs, &regs, &sregs);

    curr_dptp = prev_dptp;

    prev_dptp = NULL;

    if (st_bufp != NULL)
    {
      _ffree(st_bufp);          // Release the sector transfer buffer

      st_bufp = NULL;
    }

    // Update the class members

    max_sector = curr_dptp->sect_trk;
    bps_code = curr_dptp->bps_code;
  }
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::RESET_DRIVE - Reset Diskette Drive
//
//  Purpose:    To reset a diskette drive.
//
//  Setup:      boolean Diskette::reset_drive()
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The class member "drive_error" is updated.
//
//  Note:       The diskette drive will not be recalibrated immediately.
//              The ROM BIOS sets the appropriate bit in the diskette
//              drive recalibrate status byte at 0040:003e to indicate
//              to itself that the drive is to be recalibrated when the
//              next I/O request is issued to the drive.
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::reset_drive()
{
  union REGS regs;      // 80x86 register values

  regs.h.ah = 0x00;             // Specify "Reset Diskette System" fcn
  regs.h.dl = drive_num;        // Specify drive

  // Call the Diskette Services interrupt function

  int86(FD_INTERRUPT, &regs, &regs);

  drive_error = regs.h.ah;      // Update drive error status

  if (regs.x.cflag == 0)        // Check error return
    return (TRUE);
  else
    return (FALSE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::GET_STATUS - Get Drive Error Code Status
//
//  Purpose:    To read the drive error code status.
//
//  Setup:      int Diskette::get_status()
//
//  Return:     Value of class member "drive_err".  It may be one of:
//
//                FD_OK    - No error
//                FD_IFR   - Invalid function request
//                FD_NAM   - Address mark not found
//                FD_WPD   - Write-protected disk
//                FD_SNF   - Sector not found
//                FD_REM   - Diskette removed
//                FD_DOR   - DMA overrun
//                FD_DBE   - DMA boundary error
//                FD_UMT   - Media type not available
//                FD_CRC   - Bad cyclic redundancy check
//                FD_DCF   - Diskette controller failed
//                FD_BSK   - Seek failed
//                FD_TMO   - Time-out
//
//  Note:       The ROM BIOS Diskette Services function "Read Diskette
//              Status" accepts a drive number, but accesses the diskette
//              error code status byte at 0040:0041.  This global variable
//              is set to the error code status of the last Diskette
//              Services function to be called, regardless of which drive
//              was accessed.
//
//////////////////////////////////////////////////////////////////////////

int Diskette::get_status()
{
  return ((int) drive_error);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::READ_SECTORS - Read Diskette Sectors
//
//  Purpose:    To read sectors from the diskette.
//
//  Setup:      boolean Diskette::read_sectors
//              (
//                int num_sectors,
//                int track_num,
//                int sector_num,
//                int head_num,
//                char _far *bufp
//              )
//
//  Where:      num_sectors is the number of sectors to be read (1 to
//                "max_sector" - "sector_num" + 1).
//              track_num is the track number (0 to "max_track").
//              sector_num is the sector number (1 to "max_sector").
//              head_num is the head number (0 or 1).
//              bufp is a far pointer to a buffer where the diskette data
//                is to be transferred to.
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The class member "drive_error" is updated.
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::read_sectors
(
  int num_sectors,
  int track_num,
  int sector_num,
  int head_num,
  char _far *bufp
)
{
  int i;                // Scratch counter
  char _far *tbp;       // Transfer buffer pointer
  union REGS regs;      // 80x86 register values
  struct SREGS sregs;   // 80x86 segment register values

  // Set transfer buffer pointer

  tbp = (bps_code == FD_BPS_512) ? bufp : st_bufp;

  for (i = 0; i < FD_MAXTRY; i++)  
  {
    regs.h.ah = 0x02;           // Specify "Read Diskette Sectors" fcn
    regs.h.dl = drive_num;      // Specify drive

    regs.h.al = (byte) num_sectors;     // Specify number of sectors
    regs.h.ch = (byte) track_num;       // Specify track number
    regs.h.cl = (byte) sector_num,      // Specify sector number
    regs.h.dh = (byte) head_num;        // Specify head number

    // Specify transfer buffer

    regs.x.bx = FP_OFF(tbp);
    sregs.es = FP_SEG(tbp);

    // Call the Diskette Services interrupt function

    int86x(FD_INTERRUPT, &regs, &regs, &sregs);

    drive_error = regs.h.ah;    // Update drive error status

    if (regs.x.cflag == 0)      // Successful read ?
      break;

    if (i < (FD_MAXTRY - 1))
      (void) reset_drive();     // Reset the diskette drive
  }

  // Copy data to user-supplied buffer if necessary

  if (bps_code != FD_BPS_512)
    _fmemcpy(bufp, tbp, (size_t) (num_sectors * (128 << bps_code)));

  if (regs.x.cflag == 0)        // Check error return
    return (TRUE);
  else
    return (FALSE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::WRITE_SECTORS - Write Diskette Sectors
//
//  Purpose:    To write sectors to the diskette.
//
//  Setup:      boolean Diskette::write_sectors
//              (
//                int num_sectors,
//                int track_num,
//                int sector_num,
//                int head_num,
//                char _far *bufp
//              )
//
//  Where:      num_sectors is the number of sectors to be written (1 to
//                "max_sector" - "sect_num" + 1).
//              track_num is the track number (0 to "max_track").
//              sector_num is the sector number (1 to "max_sector").
//              head_num is the head number (0 or 1).
//              bufp is a far pointer to a buffer where the diskette data
//                is to be transferred from.
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The class member "drive_error" is updated.
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::write_sectors
(
  int num_sectors,
  int track_num,
  int sector_num,
  int head_num,
  char _far *bufp
)
{
  int i;                // Scratch counter
  char _far *tbp;       // Transfer buffer pointer
  union REGS regs;      // 80x86 register values
  struct SREGS sregs;   // 80x86 segment register values

  if (bps_code != FD_BPS_512)   // Default MS-DOS sector size ?
  {
    // Copy data to sector transfer buffer

    _fmemcpy(st_bufp, bufp, (size_t) (num_sectors * (128 <<
        bps_code)));

    tbp = st_bufp;      // Point to sector transfer buffer
  }
  else
    tbp = bufp;         // Point to user-supplied buffer

  for (i = 0; i < FD_MAXTRY; i++)
  {
    regs.h.ah = 0x03;           // Specify "Write Diskette Sectors" fcn
    regs.h.dl = drive_num;      // Specify drive

    regs.h.al = (byte) num_sectors;     // Specify number of sectors
    regs.h.ch = (byte) track_num;       // Specify track number
    regs.h.cl = (byte) sector_num,      // Specify sector number
    regs.h.dh = (byte) head_num;        // Specify head number

    // Specify transfer buffer

    regs.x.bx = FP_OFF(tbp);
    sregs.es = FP_SEG(tbp);

    // Call the Diskette Services interrupt function

    int86x(FD_INTERRUPT, &regs, &regs, &sregs);

    drive_error = regs.h.ah;    // Update drive error status

    if (regs.x.cflag == 0)      // Successful write ?
      break;

    if (i < (FD_MAXTRY - 1))
      (void) reset_drive();     // Reset the diskette drive
  }

  if (regs.x.cflag == 0)        // Check error return
    return (TRUE);
  else
    return (FALSE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::VERIFY_SECTORS - Verify Diskette Sectors
//
//  Purpose:    To verify the address fields of the specified sectors by
//              determining if the sectors can be found, read and pass a
//              cyclic redundancy check (CRC).
//
//  Setup:      boolean Diskette::verify_sectors
//              (
//                int num_sectors,
//                int track_num,
//                int sector_num,
//                int head_num,
//                fd_afld _far *af_tblp
//              )
//
//  Where:      num_sectors is the number of sectors to be verified (1 to
//                "max_sector" - "sect_num" + 1).
//              track_num is the track number (0 to "max_track").
//              sector_num is the sector number (1 to "max_sector").
//              head_num is the head number (0 or 1).
//              af_tblp is a far pointer to an address field table.  There
//                must be one table element for each sector on the track
//                to be verified.
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The class member "drive_error" is updated.
//
//  Note:       This function does not transfer data from the diskette or
//              compare diskette data to data in memory.
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::verify_sectors
(
  int num_sectors,
  int track_num,
  int sector_num,
  int head_num,
  fd_afld _far *af_tblp
)
{
  int i;                // Scratch counter
  union REGS regs;      // 80x86 register values
  struct SREGS sregs;   // 80x86 segment register values

  for (i = 0; i < FD_MAXTRY; i++)
  {
    regs.h.ah = 0x04;           // Specify "Verify Diskette Sectors" fcn
    regs.h.dl = drive_num;      // Specify drive

    regs.h.al = (byte) num_sectors;     // Specify number of sectors
    regs.h.ch = (byte) track_num;       // Specify track number
    regs.h.cl = (byte) sector_num,      // Specify sector number
    regs.h.dh = (byte) head_num;        // Specify head number

    // Specify address field table

    regs.x.bx = FP_OFF(af_tblp);
    sregs.es = FP_SEG(af_tblp);

    // Call the Diskette Services interrupt function

    int86x(FD_INTERRUPT, &regs, &regs, &sregs);

    drive_error = regs.h.ah;    // Update drive error status

    if (regs.x.cflag == 0)      // Successful verify ?
      break;

    if (i < (FD_MAXTRY - 1))
      (void) reset_drive();     // Reset the diskette drive
  }

  if (regs.x.cflag == 0)        // Check error return
    return (TRUE);
  else
    return (FALSE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::FORMAT_TRACK - Format Diskette Track
//
//  Purpose:    To format a diskette track.
//
//  Setup:      boolean Diskette::format_track
//              (
//                int track_num,
//                int head_num,
//                fd_afld _far *af_tblp
//              )
//
//  Where:      track_num is the track number (0 to "max_track").
//              head_num is the head number (0 or 1).
//              af_tblp is a far pointer to an address field table.  There
//                must be one table element for each sector on the track
//                to be formatted.
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The diskette sector and track address field data is
//              written to the specified track.
//
//              The class member "drive_error" is updated.
//
//  Note:       If the diskette drive supports more than one media type,
//              the function "Diskette::set_media_type" must be called
//              before calling this function UNLESS a custom Diskette
//              Parameters Table has been installed by calling
//              "Diskette::install_table".
//
//              Each sector of the track is verified after formatting.
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::format_track
(
  int track_num,
  int head_num,
  fd_afld _far *af_tblp
)
{
  int i;                // Scratch counter
  union REGS regs;      // 80x86 register values
  struct SREGS sregs;   // 80x86 segment register values

  for (i = 0; i < FD_MAXTRY; i++)
  {
    regs.h.ah = 0x05;           // Specify "Format Diskette Track" fcn

    // Most ROM BIOS documentation states that register AL must be set
    // to the "number of sectors to be formatted".  However, this
    // input parameter is ignored.  The number of sectors per track is
    // taken from the current Diskette Parameters Table.  (In order to
    // accomodate any ROM BIOS which may actually behave according to
    // the documentation, register AL is set according to the Diskette
    // Parameters Table value).

    regs.h.al = max_sector;             // Specify number of sectors

    regs.h.dl = drive_num;              // Specify drive
    regs.h.ch = (byte) track_num;       // Specify track number
    regs.h.dh = (byte) head_num;        // Specify head number

    // Specify address field table

    regs.x.bx = FP_OFF(af_tblp);
    sregs.es = FP_SEG(af_tblp);

    // Call the Diskette Services interrupt function

    int86x(FD_INTERRUPT, &regs, &regs, &sregs);

    drive_error = regs.h.ah;    // Update drive error status

    if (regs.x.cflag == 0)      // Successful format ?
      break;

    if (i < (FD_MAXTRY - 1))
      (void) reset_drive();     // Reset the diskette drive
  }

  if (regs.x.cflag == 0)        // Check error return
  {
    // The diskette controller will accept any value for the number of
    // sectors per track to be formatted.  However, it will not return
    // an error if the diskette could not be formatted properly (for
    // example, attempting to format a 360K diskette with 15 sectors
    // per track).  To check for this condition and other possible
    // errors, the track must be verified after formatting.

    return (verify_sectors((int) max_sector, track_num, 1, head_num,
        af_tblp));
  }
  else
    return (FALSE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::DETECT_CHANGE - Detect Media Change
//
//  Purpose:    To determine the diskette drive change line status
//              (AT-class systems only).
//
//  Setup:      boolean Diskette::detect_change()
//
//  Return:     TRUE if media change or error detected; otherwise FALSE.
//
//  Result:     The class member "drive_error" is set to one of the
//              following:
//
//                FD_OK  - no error
//                FD_DOR - change line active
//                FD_TMO - timeout (no diskette in drive)
//
//  Note:       If the change line signal is not supported, the return
//              value is FALSE.
//
//              The change line signal is set if the diskette is removed
//              from the drive and returned or replaced with another
//              diskette.  (Some drives set the change line signal if the
//              drive door is opened and closed without removing the
//              diskette.)
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::detect_change()
{
  union REGS regs;      // 80x86 register values

  if (system_type == FD_XT_CLASS)       // Check for AT-class system
    return (FALSE);

  if (cl_support != FD_CHANGE)  // Change line signal supported ?
    return (FALSE);

  regs.h.ah = 0x16;             // Specify "Detect Change Media" function
  regs.h.dl = drive_num;        // Specify drive

  // Call the Diskette Services interrupt function

  int86(FD_INTERRUPT, &regs, &regs);

  drive_error = regs.h.ah;      // Update drive error status

  if (regs.h.ah == 0x00)
  {
    return (FALSE);     // Change line signal clear
  }
  else
  {
    return (TRUE);      // Change line signal set (or an error occurred)
  }
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::SET_MEDIA_TYPE - Set Media Type
//
//  Purpose:    To set the media type for formatting (AT-class systems
//              only).
//
//  Setup:      boolean Diskette::set_media_type
//              (
//                int disk_type
//              )
//
//  Where:      disk_type is the diskette type.  It may be one of:
//
//                FD_DSK_320 -  5.25", 320K
//                FD_DSK_360 -  5.25", 360K
//                FD_DSK_120 -  5.25", 1.2M
//                FD_DSK_720 -  3.5", 720K
//                FD_DSK_144 -  3.5", 1.44M
//                FD_DSK_288 -  3.5", 2.88M
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The class members "max_track", "max_sector" and
//              "curr_dptp" are updated.
//
//              The class member "drive_error" is set to one of the
//              following:
//
//                FD_OK  - no error
//                FD_UMT - unknown media type (the CMOS RAM is invalid,
//                         the drive is not configured in CMOS RAM, or
//                         the diskette is not described in the Diskette
//                         Parameters Tables stored in the BIOS ROM)
//
//  Note:       This function returns FALSE if a user-defined Diskette
//              Parameters Table is installed or the drive type does not
//              support the diskette type.
//
//              FD_UMT will be returned in "drive_error" only for MS-DOS
//              3.2 and greater.
//
////////////////////////////////////////////////////////////////////////// 

boolean Diskette::set_media_type
(
  int disk_type
)
{
  int num_tracks;               // Number of tracks
  int num_sectors;              // Number of sectors
  union REGS regs;              // 80x86 register values
  struct SREGS sregs;           // 80x86 segment register values
  fd_dprm _far *temp_dptp;      // Temporary Diskette Parameters Table ptr

  drive_error = FD_IFR;         // Initialize drive error status

  if (system_type == FD_XT_CLASS)       // Check for AT-class system
    return (FALSE);

  // Check for user-defined Diskette Parameters Table

  if (prev_dptp != NULL)
    return (FALSE);

  switch (disk_type)    // Set number of sectors and tracks
  {
    case FD_DSK_320:    // 5.25", 320K

      if (drive_type != FD_DRV_360 && drive_type != FD_DRV_120)
        return (FALSE);

      num_sectors = 8;
      num_tracks = 40;

      break;

    case FD_DSK_360:    // 5.25", 360K

      if (drive_type != FD_DRV_360 && drive_type != FD_DRV_120)
        return (FALSE);

      num_sectors = 9;
      num_tracks = 40;

      break;

    case FD_DSK_120:    // 5.25", 1.2M

      if (drive_type != FD_DRV_120)
        return (FALSE);

      num_sectors = 15;
      num_tracks = 80;

      break;

    case FD_DSK_720:    // 3.5", 720K

      if (drive_type != FD_DRV_720 && drive_type != FD_DRV_144 &&
          drive_type != FD_DRV_288)
        return (FALSE);

      num_sectors = 9;
      num_tracks = 80;

      break;

    case FD_DSK_144:    // 3.5", 1.44M

      if (drive_type != FD_DRV_144 && drive_type != FD_DRV_288)
        return (FALSE);

      num_sectors = 18;
      num_tracks = 80;

      break;

    case FD_DSK_288:    // 3.5", 2.88M

      if (drive_type != FD_DRV_288)
        return (FALSE);

      num_sectors = 36;
      num_tracks = 80;

      break;

    default:            // Unknown diskette type

      return (FALSE);
  }

  if (_osmajor == 3 && _osminor < 2)    // Check MS-DOS version
  {
    // MS-DOS 3.0 and 3.1 do not support the ROM BIOS "Set Media Type"
    // function

    switch (disk_type)
    {
      case FD_DSK_360:  // 5.25", 360K

        switch (drive_type)
        {
          case FD_DRV_360:      // 360K drive

            return (set_diskette_type(0x01));

          case FD_DRV_120:      // 1.2M drive

            return (set_diskette_type(0x02));

          default:

            return (FALSE);
        }

      case FD_DSK_120:  // 5.25", 1.2M

        return (set_diskette_type(0x03));

      case FD_DSK_720:  // 3.5", 720K

        return (set_diskette_type(0x04));

      default:          //  Diskette type not supported

        return (FALSE);
    }      
  }

  // Save the previous Diskette Parameter Table pointer

  temp_dptp = prev_dptp;
  prev_dptp = curr_dptp;

  regs.h.ah = 0x18;             // Specify "Set Media Type" function
  regs.h.dl = drive_num;        // Specify drive

  // Most ROM BIOS documentation states that register CH must be set to
  // the "maximum number of tracks".  This is incorrect - it must be set
  // to the maximum track number (e.g. - 79 instead of 80).

  regs.h.ch = (byte) (num_tracks - 1);  // Specify maximum track number
  regs.h.cl = (byte) num_sectors;       // Specify number of sectors

  // Call the Diskette Services interrupt function

  int86x(FD_INTERRUPT, &regs, &regs, &sregs);

  drive_error = regs.h.ah;      // Update drive error status

  if (regs.x.cflag == 0)        // Check error return
  {
    // Redirect the Diskette Parameters Table pointer

    regs.h.ah = 0x25;   // Specify "Set Interrupt Vector" function
    regs.h.al = 0x1e;   // Interrupt vector number

    regs.x.dx = regs.x.di;
    sregs.ds = sregs.es;

    // Call the MS-DOS interrupt function

    intdosx(&regs, &regs, &sregs);

    // Update the current Diskette Parameters Table pointer

    FP_OFF(curr_dptp) = regs.x.di;
    FP_SEG(curr_dptp) = sregs.es;

    // Update the class members

    max_track = (byte) (num_tracks - 1);
    max_sector = (byte) num_sectors;

    return (TRUE);
  }
  else
  {
    // Restore the previous Diskette Parameters Table pointer

    prev_dptp = temp_dptp;

    return (FALSE);
  }
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::SUPPORT_CHANGE - Report Change Line Support
//
//  Purpose:    To indicate whether the diskette drive supports a change
//              line (AT-class systems only).
//
//  Setup:      boolean Diskette::support_change()
//
//  Return:     TRUE if change line supported; otherwise FALSE.
//
////////////////////////////////////////////////////////////////////////// 

boolean Diskette::support_change()
{
  return ((cl_support == FD_CHANGE) ? TRUE : FALSE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::TRACKS - Report Number Of Tracks
//
//  Purpose:    To report the number of tracks on the diskette.
//
//  Setup:      int Diskette::tracks()
//
//  Return:     Number of tracks on the diskette.
//
//////////////////////////////////////////////////////////////////////////

int Diskette::tracks()
{
  return ((int) max_track + 1);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::SECTORS - Report Number Of Sectors Per Track
//
//  Purpose:    To report the number of sectors per track on the diskette.
//
//  Setup:      int Diskette::sectors()
//
//  Return:     Number of sectors per track.
//
//////////////////////////////////////////////////////////////////////////

int Diskette::sectors()
{
  return ((int) max_sector);
}

// .... OTHER FUNCTIONS

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::ALLOCATE_BUFFER - Allocate Sector Transfer Buffer
//
//  Purpose:    To allocate an intermediate sector transfer buffer.
//
//  Setup:      boolean Diskette::allocate_buffer()
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Note:       ALL implementations of the ROM BIOS functions "Read
//              Diskette Sectors" and "Write Diskette Sectors" contain a
//              serious bug that causes the functions to quietly fail if:
//
//                1.  The sector size is other than (the default MS-DOS)
//                    512 bytes; and
//
//                2.  The user-supplied buffer crosses a physical 64K page
//                    boundary.
//
//              It occurs because the data must be transferred between the
//              user-supplied buffer and the NEC 765 (or equivalent)
//              diskette controller IC in DMA mode using the Intel 8273
//              (or equivalent) DMA controller.  The DMA controller can
//              only address 64K of memory; it cannot perform a DMA 
//              transfer across a physical 64K page boundary.
//
//              The original IBM PC design used a 74LS670 4x4 register
//              file IC to store the four upper address bits for each of
//              the four DMA channels to extend the address range to 1 MB.
//              The DMA controller can still only transfer memory to or
//              from one 64K page at a time; the register file must be
//              reprogrammed at each page boundary.  Due to the need for
//              downward compatibility, this hardware kludge is still
//              part of every IBM PC and clone design, either as
//              individual ICs or (most commonly) embedded in an IC chip
//              set firmware.
//
//              This poses a problem for the ROM BIOS manufacturer.  The
//              diskette controller knows nothing about the internals of
//              the DMA controller addressing circuitry.  It can only be
//              requested to perform a DMA transfer after reading or
//              before writing one or more diskette sectors.
//
//              The only solution is for the ROM BIOS to perform the
//              following for a sector read request (a sector write
//              request is performed in a similar manner):
//
//                1.  Program the register file to address the 64K
//                    physical page containing the start of the user-
//                    supplied data buffer.
//
//                2.  Determine the number of sectors to be read into
//                    a user-supplied buffer.
//
//                3.  Determine if the data will cross a physical 64K page
//                    boundary when read into the buffer.  If it doesn't,
//                    read the requested number of sectors and return.
//
//                4.  Read as many sectors as will fit into the buffer
//                    without crossing the page boundary.
//
//                5.  Read the next sector into a scratch buffer that
//                    does not cross a 64K page boundary.
//
//                6.  Transfer the contents of the scratch buffer to the
//                    user-supplied buffer (in non-DMA mode).
//
//                7.  Reprogram the register file to address the next
//                    64K page.
//
//                8.  Read the remainder of the sectors into the user-
//                    supplied buffer.
//
//              This solution requires a suitably-sized scratch buffer.
//              The ROM BIOS has no access to RAM that it can use for a
//              buffer.  However, MS-DOS allocates at least one single-
//              sector (512 byte) file buffer during system
//              initialization.  The ROM BIOS function can therefore call
//              the appropriate MS-DOS floppy disk device driver to
//              perform the single sector transfer.  The driver reads the
//              sector into the file buffer, then copies the data to the
//              user-supplied buffer before returning control to the ROM
//              BIOS function. 
//
//              Unfortunately, the standard MS-DOS floppy disk device
//              drivers only know about 512-byte sectors.  Thus,
//              regardless of the number of bytes actually read into its
//              file buffer, the driver consistently transfers 512 bytes
//              to the user-supplied buffer. 
//
//              Now, consider these two situations:
//
//                1.  A 1024-byte sector diskette must be read.
//
//                2.  A single 128-byte or 256-byte sector must be read
//                    into a similarly-sized user-supplied buffer.
//
//                3.  A 128-byte or 256-byte sector diskette must be read
//                    into a user buffer that crosses a 64K page boundary. 
//
//              In the first situation, the diskette controller reads
//              1024 bytes into the file buffer.  This buffer will
//              overflow, with the usual interesting results.  In the
//              second situation, the data copied by MS-DOS will overflow
//              the user buffer, again with interesting results. 
//
//              The third situation is one most likely to occur in practice.  
//              The data may or may not overflow the user buffer, depending 
//              on the number of sectors read.  If the data crosses the page 
//              boundary, however, there will be 384 or 128 extraneous bytes 
//              in the buffer.  Unless the application program validates the 
//              data after reading it from the diskette, this data 
//              corruption may go unnoticed for months or years. 
//
//              There is no solution to these problems available to the
//              ROM BIOS manufacturer.  In other words, the ROM BIOS
//              "Read Diskette Sectors" and "Write Diskette Sectors"
//              function simply do not and cannot work unless the sector
//              size is 512 bytes.
//
//              These problems are particularly insidious in that they
//              only occur when the user-supplied buffer crosses a 64K
//              page boundary.  They may occur on some machines but not on
//              others, or may occur at random times, depending on where
//              the program happens to locate the static or dynamically-
//              allocated user buffer in memory.  Running the program with
//              a software debugger typically causes the problems to
//              disappear, since the buffers are then relocated in memory.
//
//              Worse still is that the ROM BIOS function will indicate
//              a successful transfer (i.e. - it returns with register AH
//              set to 0x00 and the Carry Flag cleared), so that the 
//              corrupted data or buffer overflow may not be immediately 
//              recognized.  Even if it is, the nature of the bug is such 
//              that it is outside the scope of most software debuggers.  
//              Trapping the actual error requires an in-circuit emulator 
//              (ICE).  It also takes some careful sleuthing, since the
//              actual data transfer is done in DMA mode, which bypasses
//              the CPU and hence the ICE.
//
//              Fortunately, there is a solution to all this.  It requires
//              allocating an intermediate sector transfer buffer that is
//              guaranteed not to cross a page boundary (if it is known
//              that the sector size is not 512 bytes).  This method works
//              because the ROM BIOS functions then never have to contend 
//              with a 64K page boundary.  It is somewhat inefficient in 
//              that data read into the buffer must be transferred to the 
//              user-supplied buffer, but the CPU time required to do so is 
//              infinitesimal compared to the overall diskette transfer 
//              rate. 
//
//              The sector transfer buffer MUST be dynamically allocated
//              from the far heap to ensure that a suitably aligned buffer
//              can be found.  (A maximum of FD_MAXBUFTRY attempts are
//              made.  If a suitable buffer cannot be found, the heap is
//              likely extremely fragmented, indicating a possibly
//              inefficient program design.)
//
////////////////////////////////////////////////////////////////////////// 

boolean Diskette::allocate_buffer()
{
  int i;                                // Scratch counter
  int j;                                // Scratch counter
  int buff_size;                        // Buffer size
  boolean status = FALSE;               // Status flag
  char _far *sp;                        // Buffer start pointer
  char _far *ep;                        // Buffer end pointer
  char _far *bufp[FD_MAXBUFTRY];        // Candidate buffer pointers
  dword s_segment;                      // Buffer start segment
  dword e_segment;                      // Buffer end segment

  // Calculate the required buffer size for an entire track

  buff_size = max_sector * (128 << bps_code);

  for (i = 0; i < FD_MAXBUFTRY; )
  {
    // Allocate a candidate buffer

    if ((bufp[i] = (char _far *) _fcalloc(buff_size, sizeof(char))) ==
        NULL)
    {
      i++;      // Increment count to free buffer

      break;
    }

    // Initialize the buffer start and end pointers

    sp = bufp[i];
    ep = bufp[i] + buff_size - 1;

    // Determine the buffer normalized start and end addresses

    s_segment = ((((dword) FP_SEG(sp)) << 4) + (dword) FP_OFF(sp)) >> 16;
    e_segment = ((((dword) FP_SEG(ep)) << 4) + (dword) FP_OFF(ep)) >> 16;

    // Check to see whether the buffer crosses a 64K page boundary

    if (s_segment == e_segment)
    {
      status = TRUE;    // Indicate success

      break;
    }

    i++;
  }

  for (j = i - 1; j >= 0; j--)  // Free the other allocated buffers
    _ffree(bufp[j]);

  if (status == TRUE)
  {
    st_bufp = bufp[i];  // Set the sector transfer buffer pointer

    return (TRUE);
  }
  else
    return (FALSE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::GET_CHANGE_SUPPORT - Determine Change Line Support
//
//  Purpose:    To determine the change line support (AT-class systems
//              only).
//
//  Setup:      int Diskette::get_change_support()
//
//  Return:     The change line support.  It may be one of:
//
//                FD_NODRIVE  - no drive installed.
//                FD_NOCHANGE - diskette drive with no change line.
//                FD_CHANGE   - diskette drive with change line.
//
//  Note:       This function is not supported by XT-class systems.
//
//////////////////////////////////////////////////////////////////////////

int Diskette::get_change_support()
{
  union REGS regs;      // 80x86 register values

  if (system_type == FD_XT_CLASS)       // Check for AT-class system
    return (FALSE);

  regs.h.ah = 0x15;             // Specify "Read Drive Type" function
  regs.h.dl = drive_num;        // Specify drive

  // Call the Diskette Services interrupt function

  int86(FD_INTERRUPT, &regs, &regs);
  
  return (regs.h.ah);   // Return drive type
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::READ_DRIVE_PARMS - Read Drive Parameters
//
//  Purpose:    To determine the drive parameters (AT-class systems only).
//
//  Setup:      boolean Diskette::read_drive_parms()
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The class members "drive_type", "max_track" and
//              "max_sector" are updated.
//
//              The "drive type" may be one of the following:
//
//                FD_DRV_360   - 5.25", 360K
//                FD_DRV_120   - 5.25", 1.2M
//                FD_DRV_720   - 3.5", 720K
//                FD_DRV_144   - 3.5", 1.44M
//                FD_DRV_288   - 3.5", 2.88M
//
//  Note:       The drive type information stored in 0x10 of CMOS RAM
//              determines the parameters returned by this function.  This
//              function returns FALSE if the drive type is not stored in
//              CMOS RAM.
//
//              This function returns FALSE if a user-defined Diskette
//              Parameters Table is installed.
//
//              This function is not supported by XT-class systems.
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::read_drive_parms()
{
  union REGS regs;      // 80x86 register values

  if (system_type == FD_XT_CLASS)       // Check for AT-class system
    return (FALSE);

  // Check for user-defined Diskette Parameters Table

  if (prev_dptp != NULL)
    return (FALSE);

  regs.h.ah = 0x08;             // Specify "Read Drive Parameters" fcn
  regs.h.dl = drive_num;        // Specify drive

  // Call the Diskette Services interrupt function

  int86(FD_INTERRUPT, &regs, &regs);

  drive_error = regs.h.ah;      // Update drive error status

  if (regs.x.cflag == 0)        // Check error return
  {
    drive_type = (int) regs.h.bl;       // Get drive type

    max_track = regs.h.ch;      // Get maximum usable track number
    max_sector = regs.h.cl;     // Get maximum usable sector number

    return (TRUE);
  }
  else
    return (FALSE);
}

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::SET_DISKETTE_TYPE - Set Diskette Type
//
//  Purpose:    To specify the drive transfer rate for formatting
//              (AT-class systems only).
//
//  Setup:      boolean Diskette::set_diskette_type
//              (
//                int type
//              )
//
//  Where:      type indicates the diskette/drive combination.  It must
//                be one of:
//
//                  0x01 - 360K diskette in 360K drive
//                  0x02 - 360K diskette in 1.2M drive
//                  0x03 - 1.2M diskette in 1.2M drive
//                  0x04 - 720K diskette in 720K drive
//
//  Return:     TRUE if successful; otherwise FALSE.
//
//  Result:     The diskette controller media transfer rate is set
//              according to the "type" parameter as follows:
//
//                0x01 - 250 Kbs
//                0x02 - 300 Kbs
//                0x03 - 500 Kbs
//                0x04 - 250 Kbs
//
//              Since a 1.2M drive supports 80 tracks, it is programmed
//              to double step the heads when it accesses a 360K diskette,
//              which has 40 tracks.
//
//              The class members "max_track", "max_sector" and
//              "curr_dptp" are updated.
//
//  Note:       This function is provided only for MS-DOS 3.0 and 3.1
//              compatibility.  It does not support 1.44M or 2.88M
//              drives.
//
//              If the change line is set, it is reset.
//
//              This function returns FALSE if a diskette/drive
//              combination that is incompatible with the drive type was
//              specified.
//
//              The ROM BIOS drive media type byte at either 0040:0090
//              (drive 0) or 0040:0091 (drive 1) is updated.
//
//              This function is not supported by XT-class systems.
//
//////////////////////////////////////////////////////////////////////////

boolean Diskette::set_diskette_type
(
  int type
)
{
  union REGS regs;              // 80x86 register values
  struct SREGS sregs;           // 80x86 segment register values
  boolean status = TRUE;        // Status flag
  fd_dprm _far *temp_dptp;      // Temporary Diskette Parameters Table ptr

  drive_error = FD_IFR;         // Initialize drive error status

  // Save the previous Diskette Parameter Table pointer

  temp_dptp = prev_dptp;
  prev_dptp = curr_dptp;

  // Validate requested drive type

  switch (type)         // Validate drive type
  {
    case 0x01:          // 360K diskette in 360K drive

      if (drive_type != FD_DRV_360)
        status = FALSE;

      break;

    case 0x02:          // 360K diskette in 1.2M drive
    case 0x03:          // 1.2M diskette in 1.2M drive

      if (drive_type != FD_DRV_120)
        status = FALSE;

      break;

    case 0x04:          // 720K diskette in 720K drive

      if (drive_type != FD_DRV_720)
        status = FALSE;

      break;

    default:            // Illegal parameter

      status = FALSE;

      break;
  }

  if (status == FALSE)          // Check for error
  {
    // Restore the previous Diskette Parameters Table pointer

    prev_dptp = temp_dptp;

    return (FALSE);
  }

  // Set the diskette type

  regs.h.ah = 0x17;             // Specify "Set Diskette Type" function
  regs.h.dl = drive_num;        // Specify drive

  regs.h.al = (byte) type;      // Specify diskette/drive type

  // Call the Diskette Services interrupt function

  int86(FD_INTERRUPT, &regs, &regs);

  drive_error = regs.h.ah;      // Update drive error status

  if (regs.x.cflag == 0)        // Check error return
  {
    // Update the Diskette Parameters Table pointer

    regs.h.ah = 0x35;   // Specify "Get Interrupt Vector" function
    regs.h.al = 0x1e;   // Interrupt vector number

    // Call the MS-DOS interrupt function

    intdosx(&regs, &regs, &sregs);

    FP_OFF(curr_dptp) = regs.x.bx;
    FP_SEG(curr_dptp) = sregs.es;

    // Update the class members

    max_track = (byte) ((type == 0x01) ? 39 : 79);
    max_sector = curr_dptp->sect_trk;

    return (TRUE);
  }
  else
  {
    // Restore the previous Diskette Parameters Table pointer

    prev_dptp = temp_dptp;

    return (FALSE);
  }
}
