///////////////////////////////////////////////////////////////
//  WINDOWS ANIMATION CLASS
//      wanimate.cxx
//
//      This class defines a set of bitmaps as an animation.
//
//      Copyright 1992 by Scott Robert Ladd. 
//      All Rights Reserved. 
///////////////////////////////////////////////////////////////

#include "wanimate.h"
#include "mmsystem.h"

const DWORD BitmapAnimation::DefaultDelay = 150;

// constructor
BitmapAnimation::BitmapAnimation
    (
    size_t max, 
    int    celx, 
    int    cely)
    {
    Film  = NULL;

    // jump out if parameters are bad
    if ((max == 0) || (celx == 0) || (cely == 0))
        return;

    // get a dc for the display
    HDC dc = GetDC(NULL);

    if (dc == NULL)
        return;

    // create a memory dc
    HDC memDC = CreateCompatibleDC(dc);

    if (memDC == NULL)
        return;

    // release the display dc
    ReleaseDC(NULL,dc);

    // create the celstrip
    Film = CreateDiscardableBitmap(dc,celx,cely * max);

    if (Film == NULL)
        return;

    // select the celstrip into the memory dc
    HBITMAP oldBmp = (HBITMAP)SelectObject(memDC,Film);

    // fill the strip with white
    PatBlt(memDC,0,0,celx,cely,WHITENESS);

    // remove celstrip by selecting prev. bitmap
    SelectObject(memDC,oldBmp);

    // delete memory dc
    DeleteDC(memDC);

    // store values
    MaxCels    =  max;
    NoOfCels   =    0;
    LastCel    =    0;
    CurrentCel =    0;
    CelHeight  = cely;
    CelWidth   = celx;
    DelayTime  =  DefaultDelay;
    }

BitmapAnimation::BitmapAnimation
    (
    const BitmapAnimation & anim
    )
    {
    CopyFilm(anim);
    }

// destructor
BitmapAnimation::~BitmapAnimation()
    {
    if (Film != NULL)
        DeleteObject(Film);
    }

// assignment operator
void BitmapAnimation::operator = 
    (
    const BitmapAnimation & anim
    )
    {
    // create the celstrip
    if (Film != NULL)
        DeleteObject(Film);

    CopyFilm(anim);
    }

void BitmapAnimation::CopyFilm
    (
    const BitmapAnimation & anim
    )
    {
    Film = NULL;

    // jump out if parameters are bad
    if (anim.Film == NULL)
        return;

    // get a dc for the display
    HDC dc = GetDC(NULL);

    if (dc == NULL)
        return;

    Film = CreateDiscardableBitmap(dc,anim.CelWidth,
                            anim.CelHeight * anim.NoOfCels);

    if (Film == NULL)
        return;

    // create memory DCs for source and destination bitmaps
    HDC newDC = CreateCompatibleDC(dc);
    HDC srcDC = CreateCompatibleDC(dc);

    if ((newDC == NULL) || (srcDC == NULL))
        return;

    // release the display dc
    ReleaseDC(NULL,dc);

    // select bitmaps into contexts
    HBITMAP bmp1 = (HBITMAP)SelectObject(newDC,Film);
    HBITMAP bmp2 = (HBITMAP)SelectObject(srcDC,Film);

    // copy source celstrip to destination
    BitBlt(newDC,0,0,anim.CelWidth,anim.CelHeight,srcDC,0,0,SRCCOPY);

    // release celstrips
    SelectObject(newDC,bmp1);
    SelectObject(srcDC,bmp2);

    // delete memory DCs
    DeleteDC(newDC);
    DeleteDC(srcDC);

    // store values
    MaxCels    = anim.MaxCels;
    NoOfCels   = anim.NoOfCels;
    LastCel    = anim.LastCel;
    CurrentCel = anim.CurrentCel;
    CelHeight  = anim.CelHeight;
    CelWidth   = anim.CelWidth;
    DelayTime  = anim.DelayTime;
    }

// add bitmaps
BOOL BitmapAnimation::Append
    (
    HBITMAP cel
    )
    {
    if ((Film == NULL) 
    ||  (cel  == NULL)
    ||  (NoOfCels == MaxCels))
        return FALSE;

    // get DC for display
    HDC dc = GetDC(NULL);

    if (dc == NULL)
        return FALSE;

    // create memory DCs
    HDC memDC = CreateCompatibleDC(dc);

    if (memDC == NULL)
        return FALSE;

    HDC celDC = CreateCompatibleDC(dc);

    if (celDC == NULL)
        return FALSE;

    // release DC for display
    ReleaseDC(NULL,dc);

    // select celstrip into memory DC
    HBITMAP bmp1 = (HBITMAP)SelectObject(memDC,Film);
    HBITMAP bmp2 = (HBITMAP)SelectObject(celDC,cel);
    
    ++NoOfCels;

    LastCel = NoOfCels - 1;

    // copy cel into celstrip
    BitBlt(memDC,0,LastCel * CelHeight,
           CelWidth,CelHeight,
           celDC,0,0,SRCCOPY);

    // select original bitmaps
    SelectObject(memDC,bmp1);
    SelectObject(celDC,bmp2);

    // delete memory DC
    DeleteDC(memDC);
    DeleteDC(celDC);

    return TRUE;
    }

// run bitmap from current position to end
void BitmapAnimation::Play
    (
    HDC dc, 
    int x, 
    int y
    )
    {
    if ((Film == NULL)
    ||  (dc == NULL)
    ||  (NoOfCels == 0) 
    ||  (CurrentCel == LastCel))
        return;

    // create memory DC
    HDC memDC = CreateCompatibleDC(dc);

    if (memDC == NULL)
        return;

    // select celstrip into memory DC
    HBITMAP oldBmp = (HBITMAP)SelectObject(memDC,Film);

    // loop until last cel is displayed
    int pos = CurrentCel * CelHeight;

    for (;;)
        {
        BitBlt(dc,x,y,CelWidth,CelHeight,memDC,0,pos,SRCCOPY);

        if (CurrentCel == LastCel)
            break;

        ++CurrentCel;
        pos += CelHeight;

        Sleep();
        }

    // remove memory DC
    SelectObject(memDC,oldBmp);
    DeleteDC(memDC);
    }

// go to next cell
BOOL BitmapAnimation::Step
    (
    HDC dc, 
    int x, 
    int y
    )
    {
    if ((Film == NULL) 
    ||  (dc   == NULL)
    ||  (CurrentCel == LastCel))
        return FALSE;

    ++CurrentCel;

    ShowCel(dc,x,y);

    return TRUE;
    }

// display current cell
BOOL BitmapAnimation::ShowCel
    (
    HDC dc, 
    int x, 
    int y
    )
    {
    if ((Film == NULL) 
    ||  (dc   == NULL)
    ||  (CurrentCel == LastCel))
        return FALSE;

    // create memory DC
    HDC memDC = CreateCompatibleDC(dc);

    if (memDC == NULL)
        return FALSE;

    // select celstrip into memory DC
    HBITMAP oldBmp = (HBITMAP)SelectObject(memDC,Film);

    // display cel 
    int pos = CurrentCel * CelHeight;

    BitBlt(dc,x,y,CelWidth,CelHeight,memDC,0,pos,SRCCOPY);

    // remove memory DC
    SelectObject(memDC,oldBmp);
    DeleteDC(memDC);

    return TRUE;
    }

void BitmapAnimation::Sleep()
    {
    DWORD end = timeGetTime() + DelayTime;

    for (;;)
        {
        if (timeGetTime() > end)
            break;
        }
    }
