{===============================================================
  GrWorld - TP Unit for containing WCoordPlane class definition.

  Copyright (C) 1992 by Raymond Konopka, Jr.
 ===============================================================}

unit GrWorld;

interface  {=== GrWorld Unit ===}

uses
    Graph;

const
    MAX_POINTS = 5000;          { Max of 5000 points in Polygon }

type                            { Types needed for WCP Polygons }
    RealPointType = record
        x, y : Real;
    end;
    PPointArray = ^PointArray; { PointType defined in GRAPH.TPU }
    PointArray = Array[ 1..MAX_POINTS ] of PointType;
    PRealPointArray = ^RealPointArray;
    RealPointArray = Array[ 1..MAX_POINTS ] of RealPointType;

type
    PWCoordPlane = ^WCoordPlane;
    WCoordPlane = object         { WCoordPlane class definition }
        {= Data =}
        wxUL, wyUL, wxLR, wyLR : Real;        { Limits of World }
        vpMaxX, vpMaxY : Integer;      { Max x & y values in VP }
        flipX, flipY : Boolean;   { T if Axis diff dir than ACP }
        xScale, yScale : Real;            { Scale of x & y axes }
        currentPtrPos : PointType;   { Current Pointer Position }
        viewPort : ViewPortType;    { Location of WCP on Screen }
        textSettings : TextSettingsType;
        lineSettings : LineSettingsType;
        fillSettings : FillSettingsType;
        fillPattern : FillPatternType;
        writeMode : Integer;
        currentColor : Word;

        {= Methods =}
        constructor Init( wULx, wULy, wLRx, wLRy : Real;
                          vULx, vULy, vLRx, vLRy : Integer );
        {= Internal Methods - Not Inforced =}
        procedure Convert( wx, wy : Real; var x, y : Integer );
        procedure ConvertRel( wdx, wdy: Real; var dx, dy: Integer );
        procedure ConvertRadii( wxRadius, wyRadius : Real;
                                var xRadius, yRadius : Word );
        procedure ActivateViewPort; virtual;
        procedure LocateCurrentPtr; virtual;
        {= Drawing Methods =}
        procedure Arc( wx, wy : Real; stAngle, endAngle : Word;
                       wRadius : Real ); virtual;
        procedure Bar( wx1, wy1, wx2, wy2 : Real ); virtual;
        procedure Bar3D( wx1, wy1, wx2, wy2 : Real; depth : Word;
                         top : Boolean ); virtual;
        procedure Circle( wx, wy, wRadius : Real ); virtual;
        procedure ClearWorld; virtual;
        procedure DrawPoly( numPoints : Word;
                            var polyPoints ); virtual;
        procedure Ellipse( wx, wy : Real; stAngle, endAngle : Word;
                           wxRadius, wyRadius : Real ); virtual;
        procedure FillEllipse( wx, wy,
                               wxRadius, wyRadius : Real ); virtual;
        procedure FillPoly( numPoints : Word;
                            var polyPoints ); virtual;
        procedure FloodFill( wx, wy : Real; border: Word ); virtual;
        procedure GetImage( wx1, wy1, wx2, wy2 : Real;
                            var bitMap ); virtual;
        function GetPixel( wx, wy : Real ): Word; virtual;
        function ImageSize( wx1, wy1, wx2, wy2:Real ):Word; virtual;
        procedure Line( wx1, wy1, wx2, wy2 : Real ); virtual;
        procedure LineRel( wdx, wdy : Real ); virtual;
        procedure LineTo( wx, wy : Real ); virtual;
        procedure MoveRel( wdx, wdy : Real ); virtual;
        procedure MoveTo( wx, wy : Real ); virtual;
        procedure OutTextXY( wx, wy : Real;
                             message : String ); virtual;
        procedure PieSlice( wx, wy : Real;
                            stAngle, endAngle : Word;
                            wRadius : Real ); virtual;
        procedure PutImage( wx, wy : Real; var bitMap;
                            bitBlt : Word ); virtual;
        procedure PutPixel( wx, wy : Real; color : Word ); virtual;
        procedure Rectangle( wx1, wy1, wx2, wy2 : Real ); virtual;
        procedure Sector( wx, wy : Real; stAngle, endAngle : Word;
                          wxRadius, wyRadius : Real ); virtual;
        procedure SetColor( color : Word ); virtual;
        procedure SetFillPattern( pattern : FillPatternType;
                                  color : Word ); virtual;
        procedure SetFillStyle( pattern, color : Word ); virtual;
        procedure SetLineStyle( lineStyle, pattern,
                                thickness : Word ); virtual;
        procedure SetTextJustify( horiz, vert : Word ); virtual;
        procedure SetTextStyle( font, direction,
                                charSize : Word ); virtual;
        procedure SetWriteMode( mode : Integer ); virtual;
        procedure Axes( wxOrigin, wyOrigin, xDelta, yDelta : Real;
                        xAxis, yAxis, precision : Integer;
                        arrows : Boolean;
                        color : Word ); virtual;
    end; {= WCoordPlane Class Definition =}

implementation {=== GrWorld Unit ===}

{== WCoordPlane Methods ==}

{= Init - This constructor is used to create a WCP.           =}
{=        wULx, wULy, wLRx, wLRy - specify limits of WCP      =}
{=        vULx, vULy, vLRx, vLRy - specify location on screen =}
constructor WCoordPlane.Init( wULx, wULy, wLRx, wLRy : Real;
                              vULx, vULy, vLRx, vLRy : Integer );
var
    temp : Real;
begin
    Graph.SetViewPort( vULx, vULy, vLRx, vLRy, ClipOn );
    Graph.GetViewSettings( viewPort );
    Graph.GetFillSettings( fillSettings );
    Graph.GetLineSettings( lineSettings );
    Graph.GetFillPattern( fillPattern );
    Graph.GetTextSettings( textSettings );

    with viewPort do
    begin
        vpMaxX := x2 - x1;        { Calculate Max X value in VP }
        vpMaxY := y2 - y1;        { Calculate Max Y value in VP }
    end;

    if wULx > wLRx then
    begin                            { Does x axis inc to left? }
        flipX := TRUE;                 { If so then flip values }
        temp := wULx;
        wULx := wLRx;
        wLRx := temp;
    end
    else
        flipX := FALSE;

    if wULy > wLRy then
    begin                         { Does y axis inc going down? }
        flipY := TRUE;              { & y values in upper right }
        temp := wULy;                
        wULy := wLRy;
        wLRy := temp;
    end
    else
        flipY := FALSE;

    { Calculate logical distance between adjacent pixels }
    xScale := vpMaxX / Abs( wLRx - wULx );
    yScale := vpMaxY / Abs( wLRy - wULy );

    wxUL := wULx;              { Save the limits of the WCP for }
    wyUL := wULy;              { use in Convert and Axes methods}
    wxLR := wLRx;           
    wyLR := wLRy;

    currentPtrPos.x := 0;        { CP starts in UL corner of VP }
    currentPtrPos.y := 0;
    writeMode := CopyPut;                { Default Writing Mode }
    currentColor := GetColor;      { Set color to Current Value }
end; {= Init =}

{= Convert - Translates WCP point to ACP pixel =}
procedure WCoordPlane.Convert( wx, wy : Real; var x, y: Integer );
begin
    if flipX then  { If flip then base trans on LR point of WCP }
        x := Trunc( ( wxLR - wx ) * xScale )
    else
        x := Trunc( ( wx - wxUL ) * xScale );

    if flipY then  { If flip then base trans on LR point of WCP }
        y := Trunc( ( wyLR - wy ) * yScale )
    else
        y := Trunc( ( wy - wyUL ) * yScale );
end; {= Convert =}

{= ConvertRel - Translate delta in WCP to delta in ACP. =}
procedure WCoordPlane.ConvertRel( wdx, wdy : Real;
                                  var dx, dy : Integer );
var
    xNeg, yNeg : Boolean;
    xOrigin, yOrigin : Integer;
begin
    Convert( 0, 0, xOrigin, yOrigin );{ Get VP coords of origin }
    Convert( wdx, wdy, dx, dy );  { Treat (wdx, wdy) as a point }
    dx := dx - xOrigin;                   { Translate to Origin }
    dy := dy - yOrigin;                   { Translate to Origin }

    if flipX and ( wdx >= 0 ) then
        xNeg := TRUE
    else
        xNeg := FALSE;

    if not flipY and ( wdy >= 0 ) then
        yNeg := TRUE
    else
        yNeg := FALSE;

    if xNeg then
        dx := -dx;               { Adjust for correct direction }
    if yNeg then
        dy := -dy;
end; {= ConvertRel =}

{= ConvertRadii - Translate radius in WCP to radius in VCP.    =}
procedure WCoordPlane.ConvertRadii( wxRadius, wyRadius : Real;
                                    var xRadius, yRadius: Word );
var
    xOrigin, yOrigin, xRad, yRad : Integer;
begin
    Convert( 0, 0, xOrigin, yOrigin );{ Get VP coords of origin }
    Convert( wxRadius, wyRadius, xRad, yRad );
    xRadius := Abs( xRad - xOrigin );     { Translate to Origin }
    yRadius := Abs( yRad - yOrigin );     { Translate to Origin }
end; {= ConvertRadii =}

{= ActivateViewPort - Reestablish saved settings of VP.  Req'd =}
{=                    when multiple WCPs are being used.       =}
procedure WCoordPlane.ActivateViewPort;
begin
    with viewport do
        SetViewPort( x1, y1, x2, y2, clip );
    Graph.MoveTo( currentPtrPos.x, currentPtrPos.y );

    with fillSettings do
    begin
        Graph.SetFillStyle( pattern, color );
        if pattern = UserFill then
            Graph.SetFillPattern( fillPattern, color );
    end;
    with lineSettings do
        Graph.SetLineStyle( lineStyle, pattern, thickness );
    with textSettings do
    begin
        Graph.SetTextJustify( horiz, vert );
        Graph.SetTextStyle( font, direction, charSize );
    end;
    Graph.SetWriteMode( writeMode );
    Graph.SetColor( currentColor );
end;

{= LocateCurrentPtr - Records the location of current pointer =}
procedure WCoordPlane.LocateCurrentPtr;
begin
    currentPtrPos.x := GetX;
    currentPtrPos.y := GetY;
end;

{= Arc - WCP equivalent to BGI Arc procedure =}
procedure WCoordPlane.Arc( wx, wy : Real; stAngle, endAngle: Word;
                           wRadius : Real );
var
    x, y : Integer;
begin
    Ellipse( wx, wy, stAngle, endAngle, wRadius, wRadius );
end; {= Arc =}

{= Bar - WCP equivalent to BGI Bar procedure =}
procedure WCoordPlane.Bar( wx1, wy1, wx2, wy2 : Real );
var 
    x1, y1, x2, y2 : Integer;
begin
    ActivateViewPort;
    Convert( wx1, wy1, x1, y1 );
    Convert( wx2, wy2, x2, y2 );
    Graph.Bar( x1, y1, x2, y2 );
end; {= Bar =}

{= Bar3D - WCP equivalent to BGI Bar3D procedure =}
procedure WCoordPlane.Bar3D( wx1, wy1, wx2, wy2 : Real;
                             depth : Word; top : Boolean );
var
    x1, y1, x2, y2 : Integer;
begin
    ActivateViewPort;
    Convert( wx1, wy1, x1, y1 );
    Convert( wx2, wy2, x2, y2 );
    Graph.Bar3D( x1, y1, x2, y2, depth, top );
end; {= Bar3D =}

{= Circle - WCP equivalent to BGI Circle procedure =}
procedure WCoordPlane.Circle( wx, wy, wRadius : Real );
begin
    Ellipse( wx, wy, 0, 360, wRadius, wRadius );
end; {= Circle =}

{= ClearWorld - WCP equivalent to BGI ClearViewPort procedure =}
procedure WCoordPlane.ClearWorld;
begin
    ActivateViewPort;
    ClearViewPort;
    LocateCurrentPtr;
end; {= Clear =}

{= DrawPoly - WCP equivalent to BGI DrawPoly procedure =}
procedure WCoordPlane.DrawPoly( numPoints: Word; var polyPoints );
var
    points : PPointArray;
    i : Integer;
begin
    GetMem( points, numPoints * SizeOf( PointType ) );
    ActivateViewPort;
    for i := 1 to numPoints do
    begin             { Typecast polyPoints to a RealPointArray }
        Convert( RealPointArray( polyPoints )[ i ].x,
                 RealPointArray( polyPoints )[ i ].y,
                 points^[ i ].x,
                 points^[ i ].y );
    end;
    Graph.DrawPoly( numPoints, points^ );
end; {= DrawPoly =}

{= Ellipse - WCP equivalent to BGI Ellipse procedure =}
procedure WCoordPlane.Ellipse( wx, wy : Real;
                               stAngle, endAngle : Word;
                               wxRadius, wyRadius : Real );
var
    x, y : Integer;
    xRadius, yRadius : Word;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    ConvertRadii( wxRadius, wyRadius, xRadius, yRadius );
    Graph.Ellipse( x, y, stAngle, endAngle, xRadius, yRadius );
end; {= Ellipse =}

{= FillEllipse - WCP equivalent to BGI FillEllipse procedure =}
procedure WCoordPlane.FillEllipse( wx, wy,
                                   wxRadius, wyRadius : Real );
var
    x, y : Integer;
    xRadius, yRadius : Word;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    ConvertRadii( wxRadius, wyRadius, xRadius, yRadius );
    Graph.FillEllipse( x, y, xRadius, yRadius );
end; {= FillEllipse =}

{= FillPoly - WCP equivalent to BGI FillPoly procedure =}
procedure WCoordPlane.FillPoly( numPoints: Word; var polyPoints );
var
    points : PPointArray;
    i : Integer;
begin
    GetMem( points, numPoints * SizeOf( PointType ) );
    ActivateViewPort;
    for i := 1 to numPoints do
    begin               { Typecase polyPoints to RealPointArray }
        Convert( RealPointArray( polyPoints )[ i ].x,
                 RealPointArray( polyPoints )[ i ].y,
                 points^[ i ].x,
                 points^[ i ].y );
    end;
    Graph.FillPoly( numPoints, points^ );
end; {= FillPoly =}

{= FloodFill - WCP equivalent to BGI FloodFill procedure =}
procedure WCoordPlane.FloodFill( wx, wy : Real; border : Word );
var
    x, y : Integer;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    Graph.FloodFill( x, y, border );
end; {= FloodFill =}

{= GetImage - WCP equivalent to BGI GetImage procedure =}
procedure WCoordPlane.GetImage( wx1, wy1, wx2, wy2 : Real;
                                var bitMap );
var
    x1, y1, x2, y2 : Integer;
begin
    ActivateViewPort;
    Convert( wx1, wy1, x1, y1 );
    Convert( wx2, wy2, x2, y2 );
    Graph.GetImage( x1, y1, x2, y2, bitMap );
end; {= GetImage =}

{= GetPixel - WCP equivalent to BGI GetPixel function =}
function WCoordPlane.GetPixel( wx, wy : Real ): Word;
var
    x, y : Integer;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    GetPixel := Graph.GetPixel( x, y );
end; {= GetPixel =}

{= ImageSize - WCP equivalent to BGI ImageSize function =}
function WCoordPlane.ImageSize( wx1, wy1, wx2, wy2 : Real ): Word;
var
    x1, y1, x2, y2 : Integer;
begin
    ActivateViewPort;
    Convert( wx1, wy1, x1, y1 );
    Convert( wx2, wy2, x2, y2 );
    ImageSize := Graph.ImageSize( x1, y1, x2, y2 );
end; {= ImageSize =}

{= Line - WCP equivalent to BGI Line procedure =}
procedure WCoordPlane.Line( wx1, wy1, wx2, wy2 : Real );
var
    x1, x2, y1, y2 : Integer;
begin
    ActivateViewPort;
    Convert( wx1, wy1, x1, y1 );
    Convert( wx2, wy2, x2, y2 );
    Graph.Line( x1, y1, x2, y2 );
end; {= Line =}

{= LineRel - WCP equivalent to BGI LineRel procedure =}
procedure WCoordPlane.LineRel( wdx, wdy : Real );
var
    x, y : Integer;
begin
    ActivateViewPort;
    ConvertRel( wdx, wdy, x, y );
    Graph.LineRel( x, y );
    LocateCurrentPtr;
end; {= LineRel =}

{= LineTo - WCP equivalent to BGI LineTo procedure =}
procedure WCoordPlane.LineTo( wx, wy : Real );
var
    x, y : Integer;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    Graph.LineTo( x, y );
    LocateCurrentPtr;
end; {= LineTo =}

{= MoveRel - WCP equivalent to BGI MoveRel procedure =}
procedure WCoordPlane.MoveRel( wdx, wdy : Real );
var
    x, y : Integer;
begin
    ActivateViewPort;
    ConvertRel( wdx, wdy, x, y );
    Graph.MoveRel( x, y );
    LocateCurrentPtr;
end; {= MoveRel =}

{= MoveTo - WCP equivalent to BGI MoveTo procedure =}
procedure WCoordPlane.MoveTo( wx, wy : Real );
var
    x, y : Integer;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    Graph.MoveTo( x, y );
    LocateCurrentPtr;
end; {= MoveTo =}

{= OutTextXY - WCP equivalent to BGI OutTextXY procedure =}
procedure WCoordPlane.OutTextXY( wx, wy : Real; message: String );
var
    x, y : Integer;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    Graph.OutTextXY( x, y, message );
end; {= OutTextXY =}

{= PieSlice - WCP equivalent to BGI PieSlice procedure =}
procedure WCoordPlane.PieSlice( wx, wy : Real;
                                stAngle, endAngle : Word;
                                wRadius : Real );
begin
    Sector( wx, wy, stAngle, endAngle, wRadius, wRadius );
end;

{= PutImage - WCP equivalent to BGI PutImage procedure =}
procedure WCoordPlane.PutImage( wx, wy : Real; var bitMap;
                                bitBlt : Word );
var
    x, y : Integer;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    Graph.PutImage( x, y, bitMap, bitBlt );
end;

{= PutPixel - WCP equivalent to BGI PutPixel procedure =}
procedure WCoordPlane.PutPixel( wx, wy : Real; color : Word );
var
    x, y : Integer;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    Graph.PutPixel( x, y, color );
end; {= PutPixel =}

{= Rectangle - WCP equivalent to BGI Rectangle procedure =}
procedure WCoordPlane.Rectangle( wx1, wy1, wx2, wy2 : Real );
var
    x1, x2, y1, y2 : Integer;
begin
    ActivateViewPort;
    Convert( wx1, wy1, x1, y1 );
    Convert( wx2, wy2, x2, y2 );
    Graph.Rectangle( x1, y1, x2, y2 );
end; {= Rectangle =}

{= Sector - WCP equivalent to BGI Sector procedure =}
procedure WCoordPlane.Sector( wx, wy : Real;
                              stAngle, endAngle : Word;
                              wxRadius, wyRadius : Real );
var
    x, y : Integer;
    xRadius, yRadius : Word;
begin
    ActivateViewPort;
    Convert( wx, wy, x, y );
    ConvertRadii( wxRadius, wyRadius, xRadius, yRadius );
    Graph.Sector( x, y, stAngle, endAngle, xRadius, yRadius );
end; { WSector }

{= SetColor - WCP equivalent to BGI SetColor procedure =}
procedure WCoordPlane.SetColor( color : Word );
begin
    Graph.SetColor( color );
    currentColor := color;
end;

{= SetFillPattern - WCP equivalent to BGI SetFillPattern procedure=}
procedure WCoordPlane.SetFillPattern( pattern : FillPatternType;
                                      color : Word );
begin
    Graph.SetFillPattern( pattern, color );
    fillPattern := pattern;
    fillSettings.color := color;
end; {= SetFillPattern =}

{= SetFillStyle - WCP equivalent to BGI SetFillStyle procedure =}
procedure WCoordPlane.SetFillStyle( pattern : Word; color: Word );
begin
    Graph.SetFillStyle( pattern, color );
    fillSettings.pattern := pattern;
    fillSettings.color := color;
end; {= SetFillStyle =}

{= SetLineStyle - WCP equivalent to BGI SetLineStyle procedure =}
procedure WCoordPlane.SetLineStyle( lineStyle, pattern,
                                    thickness : Word );
begin
    Graph.SetLineStyle( lineStyle, pattern, thickness );
    lineSettings.lineStyle := lineStyle;
    lineSettings.pattern := pattern;
    lineSettings.thickness := thickness;
end; {= SetLineStyle =}

{= SetTextJustify - WCP equivalent to BGI SetTextJustify procedure=}
procedure WCoordPlane.SetTextJustify( horiz, vert : Word );
begin
    Graph.SetTextJustify( horiz, vert );
    textSettings.horiz := horiz;
    textSettings.vert := vert;
end; {= SetTextJustify =}

{= SetTextStyle - WCP equivalent to BGI SetTextStyle procedure =}
procedure WCoordPlane.SetTextStyle( font, direction,
                                    charSize : Word );
begin
    Graph.SetTextStyle( font, direction, charSize );
    textSettings.font := font;
    textSettings.direction := direction;
    textSettings.charSize := charSize;
end; {= SetTextStyle =}

{= SetWriteMode - WCP equivalent to BGI SetWriteMode procedure =}
procedure WCoordPlane.SetWriteMode( mode : Integer );
begin
    Graph.SetWriteMode( mode );
    writeMode := mode;
end; {= SetWriteMode =}


{= Axes - Draws a Set of Coordinate Axes in a WCP. See article =}
{=        for a description of parameters.                     =}
procedure WCoordPlane.Axes( wxOrigin,wyOrigin,xDelta,yDelta: Real;
                            xAxis, yAxis, precision : Integer;
                            arrows : Boolean;
                            color : Word );
var
    xOrigin, yOrigin, x, y, center : Integer;
    xi, yi : Real;            { Indexes for tick mark locations }
    xStr, yStr : String;                     { Temp string vars }
    oldTextSettings : TextSettingsType;
    oldColor : Word;
    oldLineSettings : LineSettingsType;

{ DrawXTick - Draws tick mark on x axis }
procedure DrawXTick( xi : Real );
begin
    Convert( xi, 0, x, y );
    if ( Abs( x - xOrigin ) > 10 ) or ( yAxis < 0 ) then
    begin                 { Display tick mark and label }
        Graph.Line( x, yOrigin + 3, x, yOrigin - 3 );
        Str( xi:1:precision, xStr );
        Graph.OutTextXY( x, yOrigin + 6, xStr );
    end;
end;

{ DrawYTick - Draws tick mark on y axis }
procedure DrawYTick( yi : Real );
begin
    Convert( 0, yi, x, y );
    if ( Abs( y - yOrigin ) > 10 ) or ( xAxis < 0 ) then
    begin                 { Display tick mark and label }
        Graph.Line( xOrigin - 4, y, xOrigin + 4, y );
        Str( yi:1:precision, yStr );
        center := ( Length( yStr ) * 8 ) div 2;
        Graph.OutTextXY( xOrigin - center - 5, y, yStr );
    end;
end;

begin {= Axes =}
    ActivateViewPort;
    Convert( wxOrigin, wyOrigin, xOrigin, yOrigin );
    oldColor := GetColor;
    SetColor( color );
    Graph.GetLineSettings( oldLineSettings );
    if xAxis >= 0 then
    begin                             { If visible, draw x-axis }
        SetLineStyle( Abs( xAxis ), 0, NormWidth );
        Graph.Line( 0, yOrigin, vpMaxX, yOrigin );
    end;

    if yAxis >= 0 then
    begin                             { If visible, draw y-axis }
        SetLineStyle( Abs( yAxis ), 0, NormWidth );
        Graph.Line( xOrigin, 0, xOrigin, vpMaxY );
    end;

    if ( ( xDelta < 0 ) and ( yDelta > 0 ) ) or
       ( ( xDelta > 0 ) and ( yDelta < 0 ) ) then
    begin                    { If opposite signs, draw a border }
        xDelta := Abs( xDelta );
        yDelta := Abs( yDelta );
        Graph.Rectangle( 0, 0, vpMaxX, vpMaxY );
    end;

    if arrows then
    begin                       { Draw arrows on ends of axes.. }
        if yAxis >= 0 then
        begin                     { But only if axis is visible }
                                              { Top YAxis arrow }
            Graph.Line( xOrigin + 4, 4, xOrigin, 0 );
            Graph.Line( xOrigin, 0, xOrigin - 4, 4 );
                                           { Bottom YAxis arrow }
            Graph.Line( xOrigin + 4, vpMaxY - 4, xOrigin, vpMaxY );
            Graph.Line( xOrigin, vpMaxY, xOrigin - 4, vpMaxY - 4 );
        end;
        
        if xAxis >= 0 then
        begin                     { But only if axis is visible }
                                             { Left XAxis arrow }
            Graph.Line( 4, yOrigin + 4, 0, yOrigin );
            Graph.Line( 0, yOrigin, 4, yOrigin - 4 );
                                            { Right XAxis arrow }
            Graph.Line( vpMaxX - 4, yOrigin + 4, vpMaxX, yOrigin );
            Graph.Line( vpMaxX, yOrigin, vpMaxX - 4, yOrigin - 4 );
        end;
    end;
    GetTextSettings( oldTextSettings );
    SetTextStyle( SMALLFONT, HORIZDIR, 4 );
    SetTextJustify( CENTERTEXT, CENTERTEXT );

    if xAxis >= 0 then
    begin                       { Display ticks if axis visible }
        xi := wxOrigin + xDelta;              { Start at Origin }
        while Abs( wxLR ) - Abs( xi ) > 0.000001 do
        begin                        { While still have room... }
            DrawXTick( xi );               { Draw the tick mark }
            xi := xi + xDelta;           { Go to next tick mark }
        end;
                                         { Other side of x axis }
        xi := wxOrigin - xDelta;               { Back to Origin }
        while Abs( wxUL ) - Abs( xi ) > 0.000001 do
        begin                        { While still have room... }
            DrawXTick( xi );               { Draw the tick mark }
            xi := xi - xDelta;           { Go to next tick mark }
        end;
    end; { if xAxis }

    if yAxis >= 0 then
    begin                       { Display ticks if axis visible }
        yi := wyOrigin + yDelta;              { Start at Origin }
        while Abs( wyLR ) - Abs( yi ) > 0.000001 do
        begin                        { While still have room... }
            DrawYTick( yi );               { Draw the tick mark }
            yi := yi + yDelta;           { Go to next tick mark }
        end;
                                         { Other side of y axis }
        yi := wyOrigin - yDelta;               { Back to Origin }
        while Abs( wyLR ) - Abs( yi ) > 0.000001 do
        begin                        { While still have room... }
            DrawYTick( yi );               { Draw the tick mark }
            yi := yi - yDelta;           { Go to next tick mark }
        end;
    end; { if yAxis }

    with oldTextSettings do              { Restore old settings }
    begin
        SetTextStyle( Font, Direction, CharSize );
        SetTextJustify( Horiz, Vert );
    end;
    with oldLineSettings do
        SetLineStyle( lineStyle, pattern, thickness );
    SetColor( oldColor );
end; {= Axes =}

end. {=== GraphWld Unit ===}

