{ FSPOOLER.PAS }
{$B-}
{
Description:  Printer spooling object for Turbo Vision programs.
Author:       Don Taylor
Date:         20 January 1993
Last revised: 23 July 1993 12:15
Application:  IBM PC and compatibles; BP7.0; TV2.0
}
unit FSpooler;

{ ///// } INTERFACE { ///// }
uses
 Dos, Objects, Views,
 Drivers, Dialogs;

const
 { Command Constants }
 cmSpoolNoCommand      = 5010;
 cmSpoolCancelPrint    = 5011;
 cmSpoolPrintCancelled = 5012;
 cmSpoolJobComplete    = 5013;
 cmSpoolPrintComplete  = 5014;

 { Error Return Values }
 SpoolOK         = 0;
 SpoolOpenError  = 1;
 SpoolPrtrError  = 2;

type
 FileSpoolJobRec = record
  FName          : PathStr;  { Name of text file to print from }
  PNum           : Word;     { Number of the printer (0 = LPT1) }
  EraseOnCancel  : Boolean;  { Erase file if job cancelled }
  EraseWhenDone  : Boolean;  { Erase file when job complete }
 end; { record }

 PFileSpooler = ^TFileSpooler;
 TFileSpooler = object(TDialog)
  Printing       : Boolean;           { Read only }
  Paused         : Boolean;           { Read only }
  PData          : FileSpoolJobRec;   { Read only }
  PFile          : Text;              { Read only }
  PFMode         : Byte;              { Read only }
  PFAttr         : Word;              { Read only }
  PrintingString : Boolean;           { Read only }
  PrintStr       : String;            { Read only }
  ChrIdx         : Word;              { Read only }
  AllJobsDone    : Boolean;           { Read only }
  { Note: AllJobsDone is included to support
    a future queueing capability }

  constructor Init(var Bounds : TRect; ATitle : TTitleStr);
  constructor Load(var S : TStream);
  procedure Store(var S : TStream); virtual;
  function  Start(PD : FileSpoolJobRec) : Word; virtual;
  function Update : Word; virtual;
  procedure Pause; virtual;
  procedure Resume; virtual;
  procedure Cancel; virtual;
  function InputFileOpen : Boolean; virtual;
  procedure PrintString; virtual;
  procedure HandleEvent(var Event : TEvent); virtual;
 end; { object }

const
 RFileSpooler : TStreamRec = (
  ObjType : 1501;
  VmtLink : Ofs(TypeOf(TFileSpooler)^);
  Load    : @TFileSpooler.Load;
  Store   : @TFileSpooler.Store
 );

{ ///// } IMPLEMENTATION { ///// }
const
 FormFeed = #12;       { ASCII formfeed character }
 CRLF     = #13#10;    { ASCII CR/LF charactr pair }
 PrtrInt  = $17;       { BIOS printer interrupt }

{ ///// Global Routines ///// }
{--------------------
  PrinterSelected returns a Boolean which is True if the
  selected bit of the printer status word is set.
---------------------}
function PrinterSelected(LptNum : Word) : Boolean;
var
 Regs : Registers;
begin
 Regs.AH := 2;
 Regs.DX := LptNum;
 Intr(PrtrInt, Regs);
 PrinterSelected := Regs.AH and $10 = $10;
end; { PrinterSelected }

{--------------------
  PrinterReady returns True if the specified printer is selected
  and not busy.
---------------------}
function PrinterReady(LptNum : Word) : Boolean;
var
 Regs : Registers;
begin
 Regs.AH := 2;
 Regs.DX := LptNum;
 Intr(PrtrInt, Regs);
 PrinterReady := Regs.AH and $90 = $90;
end; { PrinterReady }

{-------------------
  PrintChar prints the specified character to the
  specified printer, using a call to BIOS.
--------------------}
procedure PrintChar(LptNum : Word; PrintCh : Char);
var
 Regs : Registers;
begin
 Regs.AH := 0;
 Regs.AL := Ord(PrintCh);
 Regs.DX := LptNum;
 Intr(PrtrInt, Regs);
end; { PrintChar }

{--------------------
  RegisterSpoolers registers the TFileSpooler object
  for use with Streams.
---------------------}
procedure RegisterSpoolers;
begin
 RegisterType(RFileSpooler);
end; { RegisterSpoolers }


{ ///// TFileSpooler Methods ///// }
{--------------------
  Init constructs a Spooler object. Override: Sometimes.
---------------------}
constructor TFileSpooler.Init(var Bounds : TRect;
                              ATitle : TTitleStr);
begin
 inherited Init(Bounds, ATitle);
 Flags          := Flags and not wfClose;
 Options        := Options or ofSelectable or ofTopSelect;
 State          := State and not sfVisible;
 Printing       := False;
 Paused         := False;
 PrintingString := False;
 AllJobsDone    := True; { Set False if jobs are in a queue }
 ChrIdx         := 0;
 PrintStr       := '';
end; { TFileSpooler.Init }

{--------------------
  The Start function provides the data for the job to be printed
  and initiates the printing process. Returns a status value.
  Override: Sometimes.
---------------------}
function TFileSpooler.Start(PD : FileSpoolJobRec) : Word;
begin
 PData := PD;
 if not PrinterReady(PData.PNum)
  then begin { Printer not on line so reject the job }
        Start := SpoolPrtrError;
        Exit;
       end
  else begin { Try to open the input file }
        PFMode := FileMode;
        FileMode := 0; { Open for read only }
        Assign(PFile, PData.FName);
        GetFAttr(PFile, PFAttr);
        SetFAttr(PFile, PFAttr and not Hidden and not SysFile);
        if DosError <> 0
         then begin
              Start := SpoolOpenError;
              FileMode := PFMode;
              Exit;
              end;
       {$I-} Reset(PFile); {$I+}
       if IOResult <> 0
        then begin { Error, so return False }
              Start := SpoolOpenError;
              FileMode := PFMode;
              Exit;
             end;
       end;

 ChrIdx         := 0;
 Printing       := True;
 Paused         := False;
 PrintingString := False;
 Start          := SpoolOK;
end; { TFileSpooler.Start }

{--------------------
  The Update function checks if a line is formatted and ready
  for the printer. If so, it prints it. Otherwise, it attempts to
  get a line for printing in the next pass. Returns a value that
  indicates when a non-modal job is done. Override: Sometimes.
---------------------}
function TFileSpooler.Update : Word;
var
 MsgVal : Word;
begin
 MsgVal := cmSpoolNoCommand;
 if Printing
  then
    begin
      if not Paused then
        begin { Printing is active }
          if PrintingString then
            begin { Print next char or string }
              PrintString;
            end
          else
            begin { Get next string }
              if InputFileOpen and not Eof(PFile) then
                begin  { Get next string }
                  Readln(PFile, PrintStr);
                  PrintStr := PrintStr + CRLF;
                  PrintingString := True;
                  ChrIdx := 0;
                end
              else
                begin  { Try to terminate process }
                  if InputFileOpen then
                    begin { Close file and erase or restore attr:}
                      {$I-}
                      System.Close(PFile);
                      if PData.EraseWhenDone
                      then System.Erase(PFile)
                      else SetFAttr(PFile, PFAttr);
                      {$I+}
                      FileMode := PFMode;
                    end;

                  if (State and sfModal <> 0) and AllJobsDone then
                    begin { Modal: end state and finish }
                      EndModal(cmSpoolPrintComplete);
                      SetState(sfSelected, False);
                      Printing := False;
                    end
                  else
                    begin { Non-modal: end if not being dragged }
                      if Owner^.Current^.State and sfDragging = 0
                      then begin
                             if AllJobsDone
                             then MsgVal := cmSpoolPrintComplete
                             else MsgVal := cmSpoolJobComplete;
                             SetState(sfSelected, False);
                             Printing := False;
                           end;
                    end;
                end;
            end;
        end;
    end;
 Update := MsgVal;
end; { TFileSpooler.Update }

{--------------------
  The Pause procedure halts the spooler operation until a Resume
  is executed. Override: Seldom.
---------------------}
procedure TFileSpooler.Pause;
begin
 if Printing and not Paused
  then
    begin
      Paused := True;
      if State and sfVisible <> 0 then SetState(sfSelected, False);
    end;
end; { TFileSpooler.Pause }

{--------------------
  The Resume procedure resumes spooler operation following a
  Pause. Override: Seldom.
---------------------}
procedure TFileSpooler.Resume;
begin
 if Printing and Paused
  then
    begin
      Paused := False;
      if (State and sfVisible <> 0) and (State and sfFocused <> 0)
       then SetState(sfSelected, True);
    end;
end; { TFileSpooler.Resume }

{--------------------
  The Cancel procedure enables the Spooler's owner to
  force it to cancel the job currently in process.
  Override: Seldom.
---------------------}
procedure TFileSpooler.Cancel;
begin
 if Printing
  then begin { Close the file, if open }
        if InputFileOpen
         then begin { Close file and erase }
               {$I-}
               System.Close(PFile);
               if PData.EraseOnCancel
                then System.Erase(PFile)
                else SetFAttr(PFile, PFAttr);
               {$I+}
               FileMode := PFMode;
              end;
        SetState(sfSelected, False);
       end;
 Printing := False;
 Paused   := False;
end; { TFileSpooler.Cancel }

{--------------------
  PrintString processes the currently formatted string,
  sending the next character to the designated printer.
  On completion of printing the current string, PrintString
  sets PrintingString False. Override: Seldom.
---------------------}
procedure TFileSpooler.PrintString;
begin
 if PrinterSelected(PData.PNum)
  then begin { No device errors detected }
        if PrinterReady(PData.PNum)
         then begin { Print character }
               Inc(ChrIdx);
               if ChrIdx > Ord(PrintStr[0])
                then begin
                      PrintStr   := '';
                      PrintingString := False;
                     end
                else begin
                      PrintChar(PData.PNum, PrintStr[ChrIdx]);
                     end;
              end;
        end
   else begin { Printer is off line }
         { Error Hook: Add error handling here if desired }
        end;
end; { TFileSpooler.PrintString }

{--------------------
  InputFileOpen returns True if the spooler file
  is currently open and ready to be read from.
  Override : Seldom.
---------------------}
function TFileSpooler.InputFileOpen : Boolean;
begin
 InputFileOpen := TextRec(PFile).Mode = fmInput;
end; { TFileSpooler.InputFileOpen }

{--------------------
  HandleEvent overrides TDialog's HandleEvent to provide special
  handling for Esc and Enter, and to provide the capability to
  handle a cancel print command. Override: Seldom.
---------------------}
procedure TFileSpooler.HandleEvent(var Event : TEvent);
begin
 case Event.What of
  evCommand :
   case Event.Command of
    cmSpoolCancelPrint : begin
                          Cancel;
                          EndModal(cmSpoolPrintCancelled);
                          ClearEvent(Event);
                         end;
   end; { case }
  evKeyDown:
   case Event.KeyCode of
    kbEsc  : begin
              Event.What    := evCommand;
              Event.Command := cmSpoolCancelPrint;
              Event.InfoPtr := nil;
              PutEvent(Event);
              ClearEvent(Event);
             end;

    kbEnter: begin
              Event.What    := evCommand;
              Event.Command := cmSpoolCancelPrint;
              Event.InfoPtr := nil;
              PutEvent(Event);
              ClearEvent(Event);
             end;
   end; { case }
 end; { case }
 inherited HandleEvent(Event);
end; { TFileSpooler.HandleEvent }

{--------------------
  Load constructs a TFileSpooler object by reading it in from
  a stream. Override: Seldom. NOTE: Never store a spooler
  while it is active!
---------------------}
constructor TFileSpooler.Load(var S: TStream);
begin
 TDialog.Load(S);
 S.Read(Printing, SizeOf(Printing));
 S.Read(Paused, SizeOf(Paused));
 S.Read(PData, SizeOf(PData));
 S.Read(PFile, SizeOf(PFile));
 S.Read(PFMode, SizeOf(PFMode));
 S.Read(PFAttr, SizeOf(PFAttr));
 S.Read(PrintingString, SizeOf(PrintingString));
 S.Read(PrintStr, SizeOf(PrintStr));
 S.Read(ChrIdx, SizeOf(ChrIdx));
end; { TFileSpooler.Load }

{--------------------
  Store sends a TFileSpooler object to a stream. Override: Seldom.
  NOTE: Never store a spooler while it is active!
---------------------}
procedure TFileSpooler.Store(var S : TStream);
begin
 TDialog.Store(S);
 S.Write(Printing, SizeOf(Printing));
 S.Write(Paused, SizeOf(Paused));
 S.Write(PData, SizeOf(PData));
 S.Write(PFile, SizeOf(PFile));
 S.Write(PFMode, SizeOf(PFMode));
 S.Write(PFAttr, SizeOf(PFAttr));
 S.Write(PrintingString, SizeOf(PrintingString));
 S.Write(PrintStr, SizeOf(PrintStr));
 S.Write(ChrIdx, SizeOf(ChrIdx));
end; { TFileSpooler.Store }
end. { unit FSpool }