
/*
    pclio.c

    This module attaches PCL escape sequences to a raster row and
    prints it to the printer or a disk file. 

    Why not use fprintf(stdprn,...) for the printer? DOS isn't 
    reliable when sending large amounts of data to the parallel
    port. The computer can send data a lot faster than the printer
    can process it. When the printer gets backlogged it sends a busy
    signal to DOS. DOS, in turn, gives you an 'Abort, Retry, Ignore'
    message. The send routine here usios biosprint(2,...) to wait
    till the printer is ready before sending data to it.

    Copyright (C) 1993  M. K. Fleming  All Rights Reserved.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bios.h>
#include "prntpcx.h"

extern DEVPARM dvParms;
static int dstRl, PrinterPort;
static FILE *fo=0;

/* PCL escape sequences */
#define LPLEN 8
static unsigned char LinePrefix[]="\033*b0000W";

#define TLEN 6
static unsigned char JobTail[]="\033*rB\033E";

#define PAPEROUT  0x20
#define SELECTED  0x10
#define ENDLINE 'V'
#define ENDPLANE 'W'

/* ---------------------------------------------------------------- */
             void Send (unsigned char *b, int len)
/* ---------------------------------------------------------------- */
{
unsigned status;

    if (fo) fwrite(b,1,len,fo);
    else while (len-->0)
    {
        /* wait forever while prtr busy */
        do
        {
            status=biosprint(2,0,PrinterPort);
            if ((status & PAPEROUT) != 0) PgmAbort("Out of Paper");
            if ((status & SELECTED) == 0) PgmAbort("Printer not online");
        } while ((status & 0x80)==0);

        /* then print character */
        status=biosprint(0,*b++,PrinterPort);
    }
}

/* ---------------------------------------------------------------- */
           void PrintInit(char *filename, int outWidth)
/* Reset printer, and send raster graphics initialization stuff     */
/* ---------------------------------------------------------------- */
{
unsigned char JobHead[80];

    fo=0;
    if (strnicmp(filename,"LPT1",4)==0) PrinterPort=0;
    else if (strnicmp(filename,"LPT2",4)==0) PrinterPort=1;
    else
    {
        fo=fopen(filename,"wb");
        if (!fo) PgmAbort("Can't open output file");
    }

    /* set up line length for PrintLine */
    dstRl = (outWidth+7)/8;
    sprintf(LinePrefix+3,"%04d", dstRl );

    /* initialization */
    sprintf(JobHead,"\033E\033*t%dR\033*r%du%ds0A",
        xres, -dvPlanes, 8*dstRl );
    Send (JobHead, strlen(JobHead) );
}

/* ---------------------------------------------------------------- */
                    void PrintClose(void)
/* Send End-Of-Raster-Graphics and Printer-Reset (to eject)         */
/* ---------------------------------------------------------------- */
{
    Send (JobTail,TLEN);
    if (fo) fclose(fo);
}

/* ---------------------------------------------------------------- */
                void PrintLine(unsigned char *b)
/* Send all planes of the next scan line                            */
/* ---------------------------------------------------------------- */
{
int i;

    LinePrefix[LPLEN-1] = ENDLINE;
    for (i=0; i<dvPlanes-1; i++, b+=dstRl)
    {
        Send (LinePrefix, LPLEN);
        Send (b,dstRl);
    }
    LinePrefix[LPLEN-1] = ENDPLANE;
    Send (LinePrefix, LPLEN);
    Send (b,dstRl);
}
