
/*
    scale.c

    This module rescales the images. Inputs (to ScaleInit) are the 
    width and height of the original picture, and the width and 
    height to output.

    Scale gets called once for each line of input. It returns both
    an output line and a replication count that tells you how many
    times to copy the line into the output stream. The replication
    count may be zero (if we are reducing the picture by selecting
    every n-th line), one, or greater than one (if we are enlarging
    the picture).

    Copyright (C) 1993  M. K. Fleming  All Rights Reserved.
*/

#include <stdio.h>
#include "prntpcx.h"

static int wsrc, hsrc, wdst, hdst, ysrc, ydst, syctr, yscur;
static int dstSize, dstRl, srcRl, planes;

/* --------------------------------------------------------------- */
    int ScaleInit(int ws, int hs, int wd, int hd, int dpl)
/*  Scaling Initialization                                         */
/*  dpl = number of planes                                         */
/*  ws,hs  = source width and height in pixels                     */
/*  wd,hd  = dest width & height in pixels                         */
/*  returns size of dst buffer (bytes)                             */
/* --------------------------------------------------------------- */
{
    wsrc=ws;  hsrc=hs;  wdst=wd;  hdst=hd;
    planes=(dpl==1) ? 1 : 3;
    srcRl = (wsrc +7) & ~7;
    dstRl = (wdst +7) & ~7;
    yscur=-1; ysrc=0; ydst=-1; syctr=-hsrc;
    return (dstSize=dstRl*planes);
}

/* --------------------------------------------------------------- */
    int Scale(unsigned char *src, unsigned char *dst)
/* Scale the next line                                             */
/* Input can be 24-bit color, stored by color plane, 256-level     */
/*      grayscale, or 256-color palette index                      */
/* Returns the number of times to replicate dst (may be zero)      */
/* --------------------------------------------------------------- */
{
int xsrc, xdst, sxctr, i, count=0;
unsigned char *p, *q;

    ydst++; syctr += hsrc;
    while (syctr>=hdst) { syctr-=hdst; ysrc++; }
    if (++yscur < ysrc) { ydst--; syctr-=hsrc; return (0); }

    for (i=0; i<planes; i++, src+=srcRl, dst+=dstRl)
    {
        p=src; q=dst;
        xsrc=0; xdst=-1; sxctr=-wsrc;
        while (++xdst<wdst)
        {
            sxctr+=wsrc;
            while (sxctr>=wdst) { sxctr-=wdst; xsrc++; p++; }
            *q++=*p;
        }
    }
    count++;

    for (syctr+=hsrc; syctr<hdst; ydst++, syctr+=hsrc) count++;
    syctr -= hsrc;
    return(count);
}
