
/*
    sort.c

    This module re-arranges the image data from its natural order
    into a form more convenient for printing. If the target device
    is a color printer, sort produces an array of Red, Green, and
    Blue levels, with values 0-255. It organazies the data by
    plane, red plane first:

        RRRRRRR
        GGGGGGG
        BBBBBBB

    Planes are padded to an even multiple of 8 pixels.

    For a monochrome printer, sort creates a single gray level
    plane instead of 3 RGB planes. An RGB triplet is converted to
    gray (Intensity) from

        I = 0.30*R + 0.59*G + 0.11*B

    Copyright (C) 1993  M. K. Fleming  All Rights Reserved.
*/

#include <stdio.h>
#include <stdlib.h>
#include "prntpcx.h"

static int inFmt, srcRl, dstRl, scanLength;
static unsigned char *pal;

#define RGB             1
#define RGBP            2
#define C256            3
#define RGBtoMONO      11
#define RGBPtoMONO     12
#define C256toMONO     13


/* ------------------------------------------------------------ */
            int SortInit(IMHEAD *pcx, int outPlanes)
/* Formats supported are:                                       */
/*         1 plane, 8 bits/pixel (256-color)         (C256)     */
/*         3 planes, 8 bits/pixel (24 bit by plane)  (RGBP)     */
/*         1 plane, 24 bits/pixel (24-bit)           (RGB)      */
/* ------------------------------------------------------------ */
{
char *errmsg="This format is not supported";

    inFmt=0;  /* initialize to invalid */
    if (pcx->hd.Nplanes==1)
    {
        if (pcx->hd.BitsPerPix == 24) inFmt=RGB;
        else if (pcx->hd.BitsPerPix==8) inFmt=C256;
    }
    else if (pcx->hd.Nplanes==3)
    {
        if (pcx->hd.BitsPerPix == 8) inFmt=RGBP;
    }
    if (inFmt==0) PgmAbort(errmsg);

    /* is output monochrome? */
    if (outPlanes==1) inFmt += 10; else outPlanes=3;

    srcRl = pcx->hd.BytesPerLine;
    scanLength = pcx->hd.xmax - pcx->hd.xmin + 1;
    dstRl = (scanLength+7) & ~7;  /* even multiple of 8 */
    pal = pcx->RGBpal;
    return (outPlanes*dstRl);
}


/* ------------------------------------------------------------ */
 void SortPlanes(unsigned char *src, unsigned char *dst)
/* ------------------------------------------------------------ */
{
unsigned char *R, *G, *B, *Ri, *Gi, *Bi;
unsigned vr, vg, vb;
int i,j,k;

    switch (inFmt)
    {

    /* Monochrome output cases: output Intensity */
    case RGBtoMONO:
        for (i=0; i<scanLength; i++)
        {
            vr=*src++;  vg=*src++;  vb = *src++;
            *dst++ = (30*vr + 59*vg + 11*vb + 50)/100;
        }
        break;
    case RGBPtoMONO:
        Ri=src; Gi=Ri+srcRl; Bi=Gi+srcRl;
        for (i=0; i<scanLength; i++)
        {
            vr=*Ri++;  vg=*Gi++;  vb = *Bi++;
            *dst++ = (30*vr + 59*vg + 11*vb + 50)/100;
        }
        break;
    case C256toMONO:
        for (i=0; i<scanLength; i++)
        {
            j = *src++;  j *= 3;
            vr = pal[j];
            vg = pal[j+1];
            vb = pal[j+2];
            *dst++ = (30*vr + 59*vg + 11*vb + 50)/100;
        }
        break;

    /* Color output cases: output RRRRR....GGGGG....BBBBB.... */
    case RGB:
        R=dst; G=R+dstRl; B=G+dstRl;
        for (i=0; i<scanLength; i++)
        {
            *R++ = *src++;
            *G++ = *src++;
            *B++ = *src++;
        }
        break;
    case RGBP:
        R=dst; G=R+dstRl; B=G+dstRl;
        Ri=src; Gi=Ri+srcRl; Bi=Gi+srcRl;
        for (i=0; i<scanLength; i++)
        {
            *R++ = *Ri++;
            *G++ = *Gi++;
            *B++ = *Bi++;
        }
        break;
    case C256:
        R=dst; G=R+dstRl; B=G+dstRl;
        for (i=0; i<scanLength; i++)
        {
            j = *src++;  j *= 3;
            *R++ = pal[j];
            *G++ = pal[j+1];
            *B++ = pal[j+2];
        }
        break;
    }
}
