;-----------------------------------------------------------------
; SAMPLE.ASM
;
; Author: Brian Hook
;
; This file is a sample driver file used to illustrate dynamically
; loading executable code.  This particular example shows a very
; simple VGA 320x200x256 video driver.
;-----------------------------------------------------------------
                   MODEL TINY

                   CODESEG
                   ALIGN 2
                   ORG 0

;-----------------------------------------------------------------
; Required function jumptable
;-----------------------------------------------------------------
VDRV_HEADER:

   dw InitDriver
   dw EndDriver
   dw StartGraphics
   dw EndGraphics

   dw Point

;-----------------------------------------------------------------
; Miscellaneous information
;-----------------------------------------------------------------
VDRV_INFO:
   magic_cookie  dd 32607H
ModeArray:
   dw 0, 320, 200, 8, 1   ; mode, width, height, bits-pixel, pages

VIDEO_SEG = 0A000H

;-----------------------------------------------------------------
; EXECUTABLE CODE
;
; This is where the exectuable code actually begins.  To make
; things very simple, no PROCs are declared.  Instead, all
; procedures are simply given labels, and the addresses of these
; labels are stored in the driver's header for use by the loader.
; Since these functions will be called via constructed far
; pointers, they should have far returns ( RETF ).
;-----------------------------------------------------------------

;-----------------------------------------------------------------
; InitDriver
;
; When loading code that is responsible for interacting with some
; type of installed device, be it software or hardware, it is
; generally a good idea to provide some method for the driver to
; ascertain that the device actually exists, and to also
; initialize internal variables if necessary.  This could be
; handled by an InitDriver() function of some type.
;
; Returns:
;    int       error_code ( 0 == no error )
;-----------------------------------------------------------------
InitDriver:
   mov ax, 0                      ; 0 == VINIT_OK
   retf

;-----------------------------------------------------------------
; EndDriver
;
; As with initializing a driver, there should be some standard
; method for telling a driver that its work is done, and that it
; should free any resources allocated.
;
; Since this VGA driver doesn't allocate any resources, this is a
; simply a stub.
;-----------------------------------------------------------------
EndDriver:

   retf

;-----------------------------------------------------------------
; StartGraphics
;
; A routine should be specified to tell the driver that it is
; expected to be functioning.  This is applicable to most types of
; drivers.  In this example, StartGraphics() is used to tell the
; VGA driver to initialize the graphics system and clear the frame
; buffer.
;-----------------------------------------------------------------
StartGraphics:

   push    si             ; Save C register variables
   push    di             ;

   mov     ax, 13h        ; Set VGA mode 13h (320x200x256)
   int     10h            ; using BIOS int 10h

                          ; Clear video memory to black
   mov     ax, VIDEO_SEG  ; Load ES:DI to point to video memory
   mov     es, ax         ;
   xor     di, di         ;
   cld                    ; CLD so that DI increments
   xor     ax, ax         ; Set AX to 0 (black)
   mov     cx, 32000      ; Number of words to clear (320x200/2)

   rep     stosw          ; Clear out the memory

   pop     di             ; Restore C register variables
   pop     si             ;

   retf

;-----------------------------------------------------------------
; EndGraphics
;
; There should also be some way to tell it is expected to shutdown
; whatever system it is managing or interfacing wtih.  In this
; case, EndGraphics() is responsible for restoring text mode.
;-----------------------------------------------------------------
EndGraphics:
   mov ax, 03
   int 10h
   retf

;-----------------------------------------------------------------
; Point( int x, int y, COLOR color )
;
; Draws a point on the currently active display page.
;-----------------------------------------------------------------
   X_POS   equ      6[bp]           ; word
   Y_POS   equ      8[bp]           ; word
   COLOR   equ     10[bp]           ; byte
Point:
   push    bp
   mov     bp,sp

   mov     ax, VIDEO_SEG
   mov     es, ax           ; put video segment in ES

   mov     bx, Y_POS        ; Compute address of pixel ( y*320+x )
   mov     ax, 320
   mul     bx
   add     ax, X_POS
   mov     bx, ax

   mov     al,byte ptr COLOR  ; Set AL to color
   mov     es:[bx],al         ; Set the byte to the desired color

   pop     bp
   retf

END
