/* =========================================================== *\
**  vbxdemo.cpp -- Demonstrate OWL's VBX Control interface     **
** =========================================================== **
**  Uses the SPIN.VBX custom control supplied with Visual      **
**  BASIC 1.0. SPIN.VBX must be in the current directory or    **
**  it must be installed in WINDOWS\SYSTEM.                    **
** =========================================================== **
**     Copyright (c) 1993 by Tom Swan. All rights reserved.    **
\* =========================================================== */

#include <owl\applicat.h>
#include <owl\framewin.h>
#include <owl\dialog.h>
#include <owl\edit.h>
#include <owl\vbxctl.h>
#include "vbxdemo.h"

#define MINVALUE 0    // Minimum SpinButton value
#define MAXVALUE 99   // Maximum SpinButton value
#define EDITLEN 4     // Buffer size for editing value as text

// ===========================================================
// TVbxDialog class
// ===========================================================

class TVbxDialog: public TDialog, public TVbxEventHandler {
public:
  TVbxDialog(TWindow *parent, TResId resId, int &v);
protected:
  void SetupWindow();
  BOOL CanClose();
  void EvSpinUp(VBXEVENT far *event);
  void EvSpinDown(VBXEVENT far *event);
  void LimitValue();
  void GetValue();
  void SetValue();
  void AdjustValue(int k);
private:
  int &value;           // Reference to caller's value v
  TVbxControl *Spin1;   // Pointer to VBX SpinButton Control
  TEdit *Edit1;         // Pointer to text edit control
DECLARE_RESPONSE_TABLE(TVbxDialog);
};

DEFINE_RESPONSE_TABLE2(TVbxDialog, TDialog, TVbxEventHandler)
  EV_VBXEVENTNAME(IDC_SPIN1,"SpinUp",EvSpinUp),
  EV_VBXEVENTNAME(IDC_SPIN1,"SpinDown",EvSpinDown),
END_RESPONSE_TABLE;

// Constructor
TVbxDialog::TVbxDialog(TWindow *parent, TResId resId, int &v)
  : TDialog(parent, resId),  // Call TDialog constructor
    TWindow(parent),         // Call TWindow constructor
    value(v)                 // Set value to caller's int v
{
  Spin1 = new TVbxControl(this, IDC_SPIN1);
  Edit1 = new TEdit(this, IDC_VALUE, EDITLEN);
}

// Extract current value; return ancestor CanClose
BOOL
TVbxDialog::CanClose()
{
  // Optional: Test edit control for errors here
  GetValue();
  return TDialog::CanClose();
}

// Initialize dialog window
void
TVbxDialog::SetupWindow()
{
  TDialog::SetupWindow();
  SetValue();  // Make edit control and int value agree
}

// Respond to SpinButton SpinUp event
void
TVbxDialog::EvSpinUp(VBXEVENT far * /*event*/)
{
  AdjustValue(1);         // Increment text edit control
  Edit1->UpdateWindow();  // Show new edit control value
  Spin1->Refresh();       // Animate VBX SpinButton Control
}

// Respond to SpinButton SpinDown event
void
TVbxDialog::EvSpinDown(VBXEVENT far * /*event*/)
{
  AdjustValue(-1);        // Decrement text edit control
  Edit1->UpdateWindow();  // Show new edit control value
  Spin1->Refresh();       // Animate VBX SpinButton Control
}

// Limit value to allowed range MINVALUE ... MAXVALUE
void
TVbxDialog::LimitValue()
{
  if (value < MINVALUE)
    value = MAXVALUE;
  else if (value > MAXVALUE)
    value = MINVALUE;
}

// Extract current Text Edit control value
void
TVbxDialog::GetValue()
{
  char buffer[EDITLEN];
  Edit1->GetText(buffer, EDITLEN);
  value = atoi(buffer);
  LimitValue();
}

// Set Text Edit control value
void
TVbxDialog::SetValue()
{
  char buffer[EDITLEN];
  LimitValue();
  wsprintf(buffer, "%d", value);
  Edit1->SetText(buffer);
}

// Add k to edit control value
void
TVbxDialog::AdjustValue(int k)
{
  GetValue();
  value += k;
  SetValue();
}

// ===========================================================
// The application's main window
// ===========================================================

class TVbxDemoWin: public TFrameWindow {
public:
  TVbxDemoWin(TWindow* parent, const char far* title);
protected:
  void CmDemoTest();
DECLARE_RESPONSE_TABLE(TVbxDemoWin);
};

DEFINE_RESPONSE_TABLE1(TVbxDemoWin, TFrameWindow)
  EV_COMMAND(CM_DEMO_TEST, CmDemoTest),
END_RESPONSE_TABLE;

// Constructor
TVbxDemoWin::TVbxDemoWin(TWindow* parent, const char far* title)
  : TFrameWindow(parent, title),
    TWindow(parent, title)
{
  AssignMenu(ID_MENU);
}

// Respond to Demo|Test command; show final value
void
TVbxDemoWin::CmDemoTest()
{
  int value = 50;  // Dialog directly modifies this object
  TVbxDialog *dialog = new TVbxDialog(this, ID_DIALOG, value);
  dialog->Execute();
  delete dialog;
  char buffer[EDITLEN];
  wsprintf(buffer, "%d", value);
  MessageBox(buffer, "Final Value", MB_OK);
}

// ===========================================================
// The application class
// ===========================================================

class TVbxDemoApp: public TApplication {
public:
  TVbxDemoApp(const char far* name)
    : TApplication(name) {};
  void InitMainWindow();
};

// Initialize the program's main window
void
TVbxDemoApp::InitMainWindow()
{
  MainWindow = new TVbxDemoWin(0, "VBX Control Demonstration");
}

#pragma argsused    // argc and argv are unused

// Main program
int
OwlMain(int argc, char* argv[])
{
  VBXInit(_hInstance);          // Enable VBX controls
  TVbxDemoApp app("VbxDemo");   // Construct application object
  int runResult = app.Run();    // Run app and save result
  VBXTerm();                    // Terminate VBX control handler
  return runResult;             // Return app run result
}

