/*
 * uu2pm.c
 *
 * Converts incoming mail files from UUPC format into PMAIL format.
 * Written for the SNUUPM package.
 * Based on ideas from Paul Nash <paul@frcs.alt.za>.
 *
 * Author: Mike Lawrie <mlawrie@apies.frd.ac.za>
 * Date:   14 October 1993
 *
 * Version 2.02, 29 November 1993
 *	Checks for precisely 20 ^A characters as message separators
 *	in the UUPC file.
 *
 * Version 2.03, 22 December 1993.
 *	Incorporated ideas for chktmp() from 
 *	Jiri Kuchta <kuchta@dcse.fee.vutbr.cz>
 *	6.12.1993
 *	Added TURBOC compatibility
 *	add .CNM extension for better PMAIL operation
 *
 * Verson 2.04, 1 March 1995
 *	Fixed a bug that caused a stray character to be written
 *	after the EOF. Thanks to John McCombs.
 */

/*
 * A UUPC incoming mail file may have more that one message
 * inside it. The messages are separated by a series of ^A characters.
 * The PMAIL mail files have one message in each file. The name
 * of the PMAIL mail file is not significant.
 */

/* Algorithm
 * ---------
 *
 * Read name of source file and the name of the output directory.
 * clear the "got message" and "EOF_flag" flag, zeroise counter
 * open the input file
 * While true
 *  do
 *    read a character
 *    if EOF, set "EOF_flag" and break
 *    if character is ^A, count
 *  while character is ^A
 *  if counter is 20 and character is newline
 *    if "got message"
 *      end off message
 *    set "got message" flag
 *    open an output file
 *    zeroise the counter
 *  elseif counter is not zero
 *    write ^A for 'counter' times, followed by the character
 *    zeroise counter
 *  if "EOF_flag" set
 *    close
 * end of while loop
 */

/* Usage:
 *
 * uu2pm [-d] <source_file> <destination_directory>
 *
 * The -d flag causes the <source_file> to be deleted at the end of the run.
 *
 * exit(0) => OK
 * exit(1) => wrong number of parameters on command line
 * exit(2) => source file not found
 * exit(4) => cannot create an output filename
 * exit(5) => error in opening an output file
 * exit(6) => could not delete the input file
 */

#define VERSION 2.04

#include <stdio.h>
#include <direct.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <conio.h>
#define writech(xxx) putch(xxx); putch('\010')

#define YES 1
#define NO 0
#define CTLA 1
#define SPLIT 20

void usage( path ) char* path;
{
    fprintf( stderr, "Usage: %s [-d] <source_file> <destination_directory>\n", path );
}

char *chktmp(dir0) char *dir0;
{
    FILE *ff;

    if (( ff=fopen(dir0,"r")) == NULL )
    {
        return( dir0 );
    } else {
        fclose( ff );
        return( NULL );
    }
}

main( argc, argv ) int argc; char* argv[];
{

    FILE    *src, *dst;
    int    intchar, gotmsg, EOF_flag, delinp;
    char   outpath[MAXPATH], tmpdrive[MAXDRIVE], tmpdir[MAXDIR];
    char   tmppath[MAXPATH];
    char   tmpfile[MAXFILE], tmpext[MAXEXT];
    char   jm[9];
    int    outflags;
    int    hctr;
    char   rotate[]=">X<X";
    int    irot=0, lrot;

    fprintf( stdout, "uu2pm version %5.2f ", VERSION );

/* Parse the command line */
    if( argc < 2 || argc > 4 || (argc==4 && (strcmp(argv[1], "-d") != 0) ))
    {
        usage( argv[0] );
        exit( 1 );
    }
    if( argc == 4 && (strcmp(argv[1], "-d") == 0) )
    {
        delinp = YES;
        argv[1] = argv[2];
        argv[2] = argv[3];
    }
    else
        delinp = NO;

/* Open the input file */

    src=fopen( argv[1], "r" );
    if( src == NULL )
    {
        fprintf( stderr, "Source file \"%s\" not found\n", argv[1] );
        exit( 2 );
    }

    gotmsg = EOF_flag = NO;
    hctr = 0;


    outpath[0] = NULL;
    strcat( outpath, argv[2] );
    if( outpath[strlen(outpath) - 1] != '\\' )
        strcat( outpath, "\\" );

    lrot = strlen( rotate );

/*
 * Main loop ------------------------------------------------------
 */

    while( 1 )
    {

/*
 * Keep eyes peeled for 20 ^As followed by \n
 */
        do
        {
            intchar=fgetc( src );
            if( intchar == EOF )
            {
                EOF_flag = YES;
                break;
            }
            if( intchar == CTLA )
                hctr++;
        } while( intchar == CTLA );

/*
 * Have we got a new message?
 */

        if( hctr == SPLIT && intchar == '\n' )
        {

/*
 * Catherine wheel entertainment when a new message starts.
 */
            writech( rotate[irot++] );
            if( irot >= lrot )
                irot = 0;

            if( gotmsg == YES )
            {
                fclose( dst );
            }
            gotmsg = YES;
            hctr = 0;
/*
 * Get the time of day in hex seconds since 1970 to use as a "random"
 * number for this file. The result is 8 hex digits plus NULL, we will
 * use the last 5 digits.
 */

            strcpy( tmppath, outpath );
            strcat( tmppath, "AAA" );

            sprintf(jm,"%lX\0",time(NULL));
            strcat( tmppath, &jm[3] );
            strcat( tmppath, ".CNM" );

            fnsplit( tmppath, tmpdrive, tmpdir, tmpfile, tmpext );
            while( chktmp(tmppath) == NULL )
            {
                tmpfile[2]++;
                if( tmpfile[2] > 'Z' )
                {
                    tmpfile[2] = 'A';
                    tmpfile[1]++;
                    if( tmpfile[1] > 'Z' )
                    {
                        tmpfile[1] = 'A';
                        tmpfile[0]++;
                        if( tmpfile[0] > 'Z' )
                        {
                            fprintf( stderr, "Cannot create an output filename\n" );
                            exit( 4 );
                        }
                    }
                }
                fnmerge( tmppath, tmpdrive, tmpdir, tmpfile, tmpext );
            }
            dst = fopen( tmppath, "w" );
            if( dst == NULL )
            {
                fprintf( stderr, "Error in opening an output file\n" );
                exit( 5 );
            }

/*
 * .. or have we got one or more ^As (plus the current character) that
 * are not the signal for the start of a new message?
 */

        } else if( hctr != 0 ) {
            for( ; hctr!=0; hctr-- )
                fputc( '\001', dst );
            fputc( intchar, dst );

/*
 * .. or is this simply an ordinary character?
 */

        } else {
            if( EOF_flag != YES )
                fputc( intchar, dst );
        }

/*
 * Everything that has been read so far, be it a single character
 * or a string of ^As, has been written to the output file.
 * So, check to see if we are at the end of the input file.
 */

        if( EOF_flag )
        {
            if( delinp == YES )
            {
                if( unlink(argv[1]) != 0 )
                {
                    fprintf( stderr, "Could not delete input file %s\n", argv[1] );
                    exit(6);
                }
            }
            fprintf( stderr, "\n" );
            exit( 0 );
        }
    }
}
