/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:     $RCSfile: list.hpp $
* Version:      $Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for a class to link objects together into a
*				singly linked list.
*
* $Id: list.hpp 1.1 1994/03/09 12:17:52 kjb release $
*
****************************************************************************/

#ifndef	__LIST_HPP
#define	__LIST_HPP

#ifndef	__DEBUG_H
#include "debug.h"
#endif

/*--------------------------- Class Definition ----------------------------*/

//---------------------------------------------------------------------------
// The ListNode class is a simple class used to link the objects in the list
// together. To put anything useful into the list, you must derive the
// object placed into the list from ListNode.
//---------------------------------------------------------------------------

class ListNode {
protected:
	ListNode	*next;

	friend class GenList;
	friend class GenListIterator;
	friend class SimpleGenList;
	friend class SimpleGenListIterator;
public:
			// Constructor to statisfy some compilers :-)
			ListNode()	{};

			// Virtual destructor to delete a list node
	virtual	~ListNode();
	};

//---------------------------------------------------------------------------
// The GenList class is designed to manipulate a list of ListNode objects.
// In the simple form, ListNode objects contain nothing special. To add
// an arbitrary class to the list, you must derive the class from ListNode
// (either through single or multiple inheritance).
//---------------------------------------------------------------------------

class GenList {
protected:
	ulong		count;		// Number of objects in list
	ListNode	*head;		// Pointer to first node in list
	ListNode	*z;			// Pointer to last node in list
	ListNode	hz[2];		// Space for head and z nodes

	static	int (*cmp)(ListNode*,ListNode*);

			// Protected member to merge two lists together
			ListNode* merge(ListNode* a,ListNode* b, ListNode*& end);

public:
			// Constructor
			GenList();

			// Destructor
			~GenList();

			// Method to examine the first node in the List
			ListNode* peekHead() const;

			// Method to return the next node in the list
			ListNode* next(ListNode* node) const;

			// Method to add a node to the head of the list
			void addToHead(ListNode* node);

			// Method to add a node after another node in the list
			void addAfter(ListNode* node,ListNode* after);

			// Method to detach a specified ListNode from the list.
			ListNode* removeNext(ListNode* prev);

			// Method to detach the first node from the list.
			ListNode* removeFromHead();

			// Sort the linked list of objects
			void sort(int (*cmp)(ListNode*,ListNode*));

			// Empties the entire list by destroying all nodes
			void empty();

			// Returns the number of items in the list
			ulong numberOfItems() const	{ return count; };

			// Returns true if the list is empty
			bool isEmpty() const	{ return count == 0; };

private:
	friend class GenListIterator;
	};

//---------------------------------------------------------------------------
// The SimpleGenList class is designed to manipulate a list of ListNode
// objects. The SimpleGenList class only maintains a single pointer to the
// head of the list rather than using the dummy node system, so is useful
// when you need to maintain an array of linked lists, and the list type
// itself needs to be as small as possible for memory efficiency.
//---------------------------------------------------------------------------

class SimpleGenList {
protected:
	ListNode	*head;		// Pointer to first node in list

public:
			// Constructor
			SimpleGenList()	{ head = NULL; };

			// Destructor
			~SimpleGenList();

			// Method to examine the first node in the List
			ListNode* peekHead() const
				{ return head; };

			// Method to return the next node in the list
			ListNode* next(ListNode* node) const
				{ return node->next; };

			// Method to add a node to the head of the list
			void addToHead(ListNode* node);

			// Method to add a node after another node in the list
			void addAfter(ListNode* node,ListNode* after);

			// Method to detach a specified ListNode from the list.
			ListNode* removeNext(ListNode* prev);

			// Method to detach the first node from the list.
			ListNode* removeFromHead();

			// Empties the entire list by destroying all nodes
			void empty();

			// Returns the number of items in the list
			ulong numberOfItems() const;

			// Returns true if the list is empty
			bool isEmpty() const
				{ return head == NULL; };

private:
	friend class SimpleGenListIterator;
	};

//---------------------------------------------------------------------------
// The GenListIterator is the class of iterator that is used to step through
// the elements in the list.
//---------------------------------------------------------------------------

class GenListIterator {
protected:
	ListNode		*cursor;
	const GenList	*beingIterated;

public:
			// Constructor
			GenListIterator();

			// Constructor given a list reference
			GenListIterator(const GenList& l);

			// Intialise a list iterator from a list
			void operator = (const GenList& l);

			// assignment operator between two listIterators
			void operator = (const GenListIterator& i);

			// Overloaded cast to an integer
			operator int ();

			// Convert the iterator to the corresponding node
			ListNode* node();

			// Pre-increment operator for the iterator
			ListNode* operator ++ ();

			// Post-increment operator for the iterator
			ListNode* operator ++ (int);

			// Method to restart the iterator
			void restart();
	};

//---------------------------------------------------------------------------
// The SimpleGenListIterator is the class of iterator that is used to step
// through the elements in the list.
//---------------------------------------------------------------------------

class SimpleGenListIterator {
protected:
	ListNode			*cursor;
	const SimpleGenList	*beingIterated;

public:
			// Constructor
			SimpleGenListIterator()
				{ cursor = NULL; beingIterated = NULL; };

			// Constructor given a list reference
			SimpleGenListIterator(const SimpleGenList& l)
				{ beingIterated = &l; cursor = l.head; };

			// Intialise a list iterator from a list
			void operator = (const SimpleGenList& l)
				{ beingIterated = &l; cursor = l.head; };

			// assignment operator between two listIterators
			void operator = (const SimpleGenListIterator& i)
				{ beingIterated = i.beingIterated; cursor = i.cursor; };

			// Overloaded cast to an integer
			operator int ()
				{ return cursor != NULL; };

			// Convert the iterator to the corresponding node
			ListNode* node()
				{ return cursor; };

			// Pre-increment operator for the iterator
			ListNode* operator ++ ();

			// Post-increment operator for the iterator
			ListNode* operator ++ (int);

			// Method to restart the iterator
			void restart()
				{ cursor = beingIterated->head; };
	};

//---------------------------------------------------------------------------
// Set of template wrapper classes for declaring Type Safe linked lists.
// Note that the elements of the linked list must still be derived from
// ListNode.
//---------------------------------------------------------------------------

typedef	int (*_GenListCmp)(ListNode*,ListNode*);

template <class T> class List : public GenList {
public:
			T* peekHead() const
				{ return (T*)GenList::peekHead(); };
			T* next(T* node) const
				{ return (T*)GenList::next(node); };
			T* removeNext(T* prev)
				{ return (T*)GenList::removeNext(prev); };
			T* removeFromHead()
				{ return (T*)GenList::removeFromHead(); };
			void sort(int (*cmp)(T*,T*))
				{ GenList::sort((_GenListCmp)cmp); };
	};

template <class T> class ListIterator : public GenListIterator {
public:
			ListIterator()
				: GenListIterator() {};
			ListIterator(const List<T>& l)
				: GenListIterator(l) {};
			void operator = (const List<T>& l)
				{ GenListIterator::operator=(l); };
			void operator = (const ListIterator<T>& i)
				{ GenListIterator::operator=(i); };
			T* node()
				{ return (T*)GenListIterator::node(); };
			T* operator ++ ()
				{ return (T*)GenListIterator::operator++(); };
			T* operator ++ (int)
				{ return (T*)GenListIterator::operator++(1); };
	};

template <class T> class SimpleList : public SimpleGenList {
public:
			T* peekHead() const
				{ return (T*)SimpleGenList::peekHead(); };
			T* next(T* node) const
				{ return (T*)SimpleGenList::next(node); };
			T* removeNext(T* prev)
				{ return (T*)SimpleGenList::removeNext(prev); };
			T* removeFromHead()
				{ return (T*)SimpleGenList::removeFromHead(); };
	};

template <class T> class SimpleListIterator : public SimpleGenListIterator {
public:
			SimpleListIterator()
				: SimpleGenListIterator() {};
			SimpleListIterator(const SimpleList<T>& l)
				: SimpleGenListIterator(l) {};
			void operator = (const SimpleList<T>& l)
				{ SimpleGenListIterator::operator=(l); };
			void operator = (const SimpleListIterator<T>& i)
				{ SimpleGenListIterator::operator=(i); };
			T* node()
				{ return (T*)SimpleGenListIterator::node(); };
			T* operator ++ ()
				{ return (T*)SimpleGenListIterator::operator++(); };
			T* operator ++ (int)
				{ return (T*)SimpleGenListIterator::operator++(1); };
	};

/*------------------------ Inline member functions ------------------------*/

inline ListNode* GenList::peekHead() const
/****************************************************************************
*
* Function:		GenList::peekHead
* Parameters:
* Returns:		Returns a pointer to the head node on the list, or NULL if
*				the list is empty.
*
****************************************************************************/
{
	return (head->next == z ? NULL : head->next);
}

inline ListNode* GenList::next(ListNode *node) const
/****************************************************************************
*
* Function:		GenList::next
* Parameters:	node	- Node to obtain next from
* Returns:		Pointer to the next node in the list, NULL if none.
*
****************************************************************************/
{
	return (node->next == z ? NULL : node->next);
}

inline void GenList::addAfter(ListNode* node,ListNode* after)
/****************************************************************************
*
* Function:		GenList::addAfter
* Parameters:	node	- Node to attach new node after in list
*				after	- New node to attach to list
*
* Description:	Attaches a new node after a specified node in the list.
*				The list must contain at least one node, and after may
*				be the tail node of the list.
*
****************************************************************************/
{
	node->next = after->next;
	after->next = node;
	count++;
}

inline void GenList::addToHead(ListNode* node)
/****************************************************************************
*
* Function:		GenList::addToHead
* Parameters:	node	- Node to add to list
*
* Description:	Attaches the node to the head of the list, maintaining the
*				head and tail pointers.
*
****************************************************************************/
{
	addAfter(node,head);
}

inline ListNode* GenList::removeNext(ListNode* prev)
/****************************************************************************
*
* Function:		GenList::removeNext
* Parameters:	node	- Pointer to node remove from the list
*				prev	- Pointer to the previous node in the list
* Returns:		Node removed from list, or NULL if prev is the last node in
*				the list.
*
* Description:	Attempts to remove the specified node from the list. 'prev'
*				should point to the previous node in the list.
*
****************************************************************************/
{
	ListNode	*node;

	if ((node = prev->next) != z) {
		prev->next = prev->next->next;
		count--;
		return node;
		}
	else
		return NULL;
}

inline ListNode* GenList::removeFromHead()
/****************************************************************************
*
* Function:		GenList::removeFromHead
* Returns:		Pointer to the node removed from the head of the list,
*				or NULL if the list is empty.
*
****************************************************************************/
{
	return removeNext(head);
}

inline GenListIterator::GenListIterator()
/****************************************************************************
*
* Function:		GenListIterator::GenListIterator
*
* Description:	Default constructor for a list iterator.
*
****************************************************************************/
{
	cursor = NULL;
	beingIterated = NULL;
}

inline GenListIterator::GenListIterator(const GenList& l)
/****************************************************************************
*
* Function:		GenListIterator::GenListIterator
* Parameters:	l	- List to construct iterator from
*
* Description:	Constructor for a ListIterator given a reference to a list
*				to iterate.
*
****************************************************************************/
{
	beingIterated = (GenList*)&l;
	cursor = l.head->next;
}

inline void GenListIterator::operator = (const GenList& l)
/****************************************************************************
*
* Function:		GenListIterator::operator =
* Parameters:	l	- List to assign to iterator
*
* Description:	Assignment operator for a ListIterator given a reference to
*				a list to iterate.
*
****************************************************************************/
{
	beingIterated = &l;
	cursor = l.head->next;
}

inline void GenListIterator::operator = (const GenListIterator& i)
/****************************************************************************
*
* Function:		GenListIterator::operator =
* Parameters:	i	- Iterator to assign from
*
* Description:	Assignment operator for a ListIterator given a reference to
*				another ListIterator.
*
****************************************************************************/
{
	beingIterated = i.beingIterated;
	cursor = i.cursor;
}

inline GenListIterator::operator int()
/****************************************************************************
*
* Function:		GenListIterator::operator int
*
* Description:	Overloaded cast to integer for the list iterator. Evaluates
*				to 0 when the end of the list is reached.
*
****************************************************************************/
{
	return (cursor != beingIterated->z);
}

inline ListNode* GenListIterator::node()
/****************************************************************************
*
* Function:		GenListIterator::node
* Returns:		Returns a reference to the node in the list.
*
****************************************************************************/
{
	return (cursor == beingIterated->z ? NULL : cursor);
}

inline ListNode* GenListIterator::operator ++ ()
/****************************************************************************
*
* Function:		GenListIterator::operator ++
* Returns:		Pointer to node after incrementing
*
* Description:	Increments the iterator by moving it to the next object
*				in the list. We return a pointer to the node pointed to
*				after the increment takes place.
*
****************************************************************************/
{
	cursor = cursor->next;
	return (cursor == beingIterated->z ? NULL : cursor);
}

inline ListNode* GenListIterator::operator ++ (int)
/****************************************************************************
*
* Function:		GenListIterator::operator ++ (int)
* Returns:		Pointer to node before incrementing
*
* Description:	Increments the iterator by moving it to the next object
*				in the list. We return a pointer to the node pointed to
*				before the increment takes place.
*
****************************************************************************/
{
	ListNode	*prev = cursor;

	cursor = cursor->next;
	return (prev == beingIterated->z ? NULL : prev);
}

inline void GenListIterator::restart()
/****************************************************************************
*
* Function:		GenListIterator::restart
*
* Description:	Restart the iterator at the beginning of the list.
*
****************************************************************************/
{
	cursor = beingIterated->head->next;
}

inline void SimpleGenList::addToHead(ListNode* node)
/****************************************************************************
*
* Function:		SimpleGenList::addToHead
* Parameters:	node	- Node to add to SimpleList
*
* Description:	Attaches the node to the head of the SimpleList.
*
****************************************************************************/
{
	node->next = head;
	head = node;
}

inline void SimpleGenList::addAfter(ListNode* node,ListNode* after)
/****************************************************************************
*
* Function:		SimpleGenList::addAfter
* Parameters:	node	- Node to attach new node after in SimpleList
*				after	- New node to attach to SimpleList
*
* Description:	Attaches a new node after a specified node in the SimpleList.
*				The SimpleList must contain at least one node, and after may
*				be the tail node of the SimpleList.
*
****************************************************************************/
{
	node->next = after->next;
	after->next = node;
}

inline ListNode* SimpleGenList::removeNext(ListNode* prev)
/****************************************************************************
*
* Function:		SimpleGenList::removeNext
* Parameters:	node	- Pointer to node remove from the SimpleList
*				prev	- Pointer to the previous node in the SimpleList
* Returns:		Node removed from SimpleList, or NULL if prev is the last
*				node in the SimpleList.
*
* Description:	Attempts to remove the specified node from the SimpleList.
*				'prev' should point to the previous node in the SimpleList.
*
****************************************************************************/
{
	ListNode	*node;

	if ((node = prev->next) != NULL) {
		prev->next = prev->next->next;
		return node;
		}
	else
		return NULL;
}

inline ListNode* SimpleGenList::removeFromHead()
/****************************************************************************
*
* Function:		SimpleGenList::removeFromHead
* Returns:		Pointer to the node removed from the head of the SimpleList,
*				or NULL if the SimpleList is empty.
*
****************************************************************************/
{
	ListNode	*node = head;

	if (head)
		head = head->next;
	return node;
}

inline ListNode* SimpleGenListIterator::operator ++ ()
/****************************************************************************
*
* Function:		SimpleGenListIterator::operator ++
* Returns:		Pointer to node after incrementing
*
* Description:	Increments the iterator by moving it to the next object
*				in the SimpleList. We return a pointer to the node pointed to
*				after the increment takes place.
*
****************************************************************************/
{
	return cursor ? (cursor = cursor->next) : NULL;
}

inline ListNode* SimpleGenListIterator::operator ++ (int)
/****************************************************************************
*
* Function:		SimpleGenListIterator::operator ++ (int)
* Returns:		Pointer to node before incrementing
*
* Description:	Increments the iterator by moving it to the next object
*				in the SimpleList. We return a pointer to the node pointed to
*				before the increment takes place.
*
****************************************************************************/
{
	ListNode	*prev = cursor;

	if (cursor)
		cursor = cursor->next;
	return prev;
}

#endif	// __LIST_HPP
