/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: point.hpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for the integer point class. We provide
*				methods to convert between integer points and real points
*				(Vec2d's).
*
* $Id: point.hpp 1.1 1994/03/10 11:50:47 kjb release $
*
****************************************************************************/

#ifndef	__POINT_HPP
#define	__POINT_HPP

#ifndef	__DEBUG_H
#include "debug.h"
#endif

#ifndef	__IOSTREAM_H
#include "iostream.h"
#endif

#if	defined(FLOAT) || defined(DOUBLE) || defined(LONGDOUBLE) ||	defined(FIXEDPOINT)
#define	__REAL_CONVERSION
#include "vec2d.hpp"
#endif

/*--------------------------- Class Definition ----------------------------*/

//---------------------------------------------------------------------------
// The following defines a 2d integer point type, and defines a number of
// useful inline operations on these points. Use the defined methods to
// convert between 2d real points and 2d integer points. The methods provided
// will ensure that the results obtained are consistent.
//---------------------------------------------------------------------------

class Point {
public:
	int		x,y;			// Coordinates of the point

			// Default constructor
			Point()	{x = y = 0; };

			// Constructor given 2 integers
			Point(int x1,int y1)	{ x = x1; y = y1; };

			// Constructor given a point
			Point(const Point& p)	{ x = p.x; y = p.y; };

			// Assignment operator given a point
			Point& operator = (const Point& p)
			{
				x = p.x; y = p.y; return *this;
			};

#ifdef	__REAL_CONVERSION

			// Set of conversion routines between 2d real vectors/points
			// and integer points.
			Point(real x1,real y1) { x = realToInt(x1); y = realToInt(y1); };
			Point(const Point2d& p)	{ x = realToInt(p.x); y = realToInt(p.y); };
			Point& operator = (const Point2d& p)
			{
				x = realToInt(p.x); y = realToInt(p.y); return *this;
			};

			// Cast an integer point to a real point
			operator Point2d ()
			{
				return Point2d(intToReal(x + REAL(0.5)),
					   intToReal(y + REAL(0.5)));
			};

#endif	// __REAL_CONVERSION

			// Standard arithmetic operators for 2d integer points
	friend	Point operator + (const Point& v1,const Point& v2);
	friend	Point operator - (const Point& v1,const Point& v2);
	friend	Point operator * (const Point& v1,const Point& v2);
	friend	Point operator * (const Point& v,int s);
	friend	Point operator * (int s,const Point& v);
	friend	Point operator / (const Point& v,int s);

			// Faster methods to add and multiply
			Point& operator += (const Point& v);
			Point& operator -= (const Point& v);
			Point& operator *= (const Point& v);
			Point& operator *= (int s);
			Point& operator /= (int s);

			// Methods to negate a vector
			Point operator - () const;
			Point& negate();			// Faster

			// Method to determine if a point is zero
			bool isZero() const;

			// Method to determine if two points are equal
			bool operator == (const Point& p) const;

			// Method to determine if two points are not equal
			bool operator != (const Point& p) const;

			// Friend to display the contents of a 2d point
	friend	ostream& operator << (ostream& s,const Point& p);
	};

/*------------------------- Inline member functions -----------------------*/

//---------------------------------------------------------------------------
// Standard arithmetic operators for integer points.
//---------------------------------------------------------------------------

inline Point operator + (const Point& p1,const Point& p2)
{
	return Point(p1.x + p2.x, p1.y + p2.y);
}

inline Point operator - (const Point& p1,const Point& p2)
{
	return Point(p1.x - p2.x, p1.y - p2.y);
}

inline Point operator * (const Point& p1,const Point& p2)
{
	return Point(p1.x * p2.x, p1.y * p2.y);
}

inline Point operator * (const Point& p1,int s)
{
	return Point(p1.x * s, p1.y * s);
}

inline Point operator * (int s,const Point& p1)
{
	return Point(p1.x * s, p1.y * s);
}

inline Point operator / (const Point& p1,int s)
{
	return Point(p1.x / s, p1.y / s);
}

//---------------------------------------------------------------------------
// Faster methods to add and multiply integer points.
//---------------------------------------------------------------------------

inline Point& Point::operator += (const Point& p)
{
	x += p.x;
	y += p.y;
	return *this;
}

inline Point& Point::operator -= (const Point& p)
{
	x -= p.x;
	y -= p.y;
	return *this;
}

inline Point& Point::operator *= (const Point& p)
{
	x *= p.x;
	y *= p.y;
	return *this;
}

inline Point& Point::operator *= (int s)
{
	x *= s;
	y *= s;
	return *this;
}

inline Point& Point::operator /= (int s)
{
	x /= s;
	y /= s;
	return *this;
}

//---------------------------------------------------------------------------
// Methods to negate points.
//---------------------------------------------------------------------------

inline Point Point::operator - () const
{
	return Point(-x,-y);
}

inline Point& Point::negate()
{
	x = -x;
	y = -y;
	return *this;
}

//---------------------------------------------------------------------------
// Miscellaneous operations.
//---------------------------------------------------------------------------

inline bool Point::isZero() const
{
	return (x == 0 && y == 0);
}

inline bool Point::operator == (const Point& p) const
{
	return (x == p.x && y == p.y);
}

inline bool Point::operator != (const Point& p) const
{
	return (x != p.x || y != p.y);
}

#endif	// __POINT_HPP
