/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: rect.hpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for an integer rectangle class.
*
* $Id: rect.hpp 1.1 1994/03/10 11:50:47 kjb release $
*
****************************************************************************/

#ifndef	__RECT_HPP
#define	__RECT_HPP

#ifndef	__POINT_HPP
#include "point.hpp"
#endif

/*--------------------------- Class Definition ----------------------------*/

//---------------------------------------------------------------------------
// The following defines an integer class of rectangles. Note that we do
// not overload the = operator, nor do we provide a copy constructor for
// the class. Rectangles are simple classes, and the compiler does a better
// job of copying them for us if we don't provide these member functions.
// Copy constructors and overloaded = operators should only really be used
// when the class contains pointers.
//
// A point to note about the definition of rectangles used here. The bottom
// right point is _NOT_ included in the rectangle's definition. This means
// that a rectangle (10,10) - (20,20) includes all the points from 10,10
// thru to 19,19 inclusive. This is consistent with the way Quickdraw on
// the Mac defines rectangles and makes processing a little easier in some
// cases.
//---------------------------------------------------------------------------

class Rect {
public:
	Point	topLeft;		// Top left point of rectangle
	Point	botRight;		// Bottom right point of rectangle

			// Default constructor
			Rect();

			// Constructor given a set of integers
			Rect(int left,int top,int right,int bottom) :
				topLeft(left,top),botRight(right,bottom) {};

#ifdef	__REAL_CONVERSION
			Rect(real left,real top,real right,real bottom) :
				topLeft(left,top),botRight(right,bottom) {};
#endif

			// Constructor given two points
			Rect(const Point& topLeft,const Point& botRight) :
				topLeft(topLeft),botRight(botRight) {};

			// Methods to access the left, top, right and bottom coords.
			int& left(void)					{ return topLeft.x; };
			int& top(void)					{ return topLeft.y; };
			int& right(void)				{ return botRight.x; };
			int& bottom(void)				{ return botRight.y; };
			const int& left(void) const		{ return topLeft.x; };
			const int& top(void) const		{ return topLeft.y; };
			const int& right(void) const	{ return botRight.x; };
			const int& bottom(void) const	{ return botRight.y; };

			// Method to move a rectangle to a specified position
			Rect& moveTo(int x,int y);
			Rect& moveTo(Point p)	{ return moveTo(p.x,p.y); };

			// Method to offset a rectangle by a specified amount
			Rect& offset(int dx,int dy);
			Rect& offset(const Point& d)	{ return offset(d.x,d.y); };

			// Method to inset a rectangle by a specified amount
			Rect& inset(int dx,int dy);
			Rect& inset(const Point& d)		{ return inset(d.x,d.y); };

			// Method's to determine if a point is within the rectangle
			bool includes(int x,int y) const;
			bool includes(const Point& p) const;

			// Method to find the union of two rectangles
			Rect operator + (const Rect& p) const;

			// Method to find the union of two rectangles (faster)
			Rect& operator += (const Rect& p);

			// Methods to expand a rectangle to include a specified point
			Rect operator + (const Point& p) const;
			Rect& operator += (const Point& p);

			// Method to find the intersection of two rectangles
			Rect operator & (const Rect& p) const;

			// Method to find the intersection of two rectangles
			Rect& operator &= (const Rect& p);

			// Method to determine if two rectangles intersect
			bool intersect(const Rect& r) const;

			// Method to create an empty rectangle
			Rect& empty(void);

			// Method to determine if a rectangle is empty
			bool isEmpty(void) const;

			// Method to determine if two rectangles are equal
			bool operator == (const Rect& r) const;

			// Method to determine if two rectangles are not equal
			bool operator != (const Rect& r) const;

			// Friend to display the rectangle on a stream
	friend	ostream& operator << (ostream& o,const Rect& r);
	};

/*------------------------ Inline member functions ------------------------*/

inline Rect& Rect::empty(void)
{
	topLeft.x = topLeft.y = botRight.x = botRight.y = 0;
	return *this;
}

inline Rect::Rect()
{
	empty();
}

inline bool Rect::isEmpty(void) const
{
	return (botRight.y <= topLeft.y || botRight.x <= topLeft.x);
}

inline Rect& Rect::moveTo(int x,int y)
{
	return offset(x-left(),y-top());
}

inline Rect& Rect::offset(int dx,int dy)
{
	topLeft.x += dx;	botRight.x += dx;
	topLeft.y += dy;	botRight.y += dy;
	return *this;
}

inline Rect& Rect::inset(int dx,int dy)
{
	topLeft.x += dx;	botRight.x -= dx;
	topLeft.y += dy;	botRight.y -= dy;
	if (isEmpty())
		empty();
	return *this;
}

inline bool Rect::includes(int x,int y) const
{
	return (x >= topLeft.x && x < botRight.x &&
			y >= topLeft.y && y < botRight.y);
}

inline bool Rect::includes(const Point& p) const
{
	return (p.x >= topLeft.x && p.x < botRight.x &&
			p.y >= topLeft.y && p.y < botRight.y);
}

inline bool Rect::operator == (const Rect& r) const
{
	return (topLeft == r.topLeft && botRight == r.botRight);
}

inline bool Rect::operator != (const Rect& r) const
{
	return (topLeft != r.topLeft || botRight != r.botRight);
}

#endif	// __RECT_HPP
