/*
  @Project: The cross-platform plug-in toolkit (XPIN).
  @Project-Version: 1.2
  @Project-Date: December 1992.
  @Filename: WCALLER.C

  @Author: Ramin Firoozye - rp&A Inc.
  @Copyright: (c) 1992, rp&A Inc. - San Francisco, CA.
  @License.
  The portable plug-in toolkit source and library code is
  presented without any warranty or assumption of responsibility
  from rp&A Inc. and the author. You can not sell this toolkit in
  source or library form as a standalone product without express
  permission of rp&A Inc. You are, however, encouraged to use this
  code in your personal or commercial application without any
  royalties. You can modify the sources as long as the license
  and credits are included with the source.

  For bugs, updates, thanks, or suggestions you can contact the
  author at:
	CompuServe: 70751,252
	Internet: rpa@netcom.com

  @End-License.

  @Description.
  This is a very simple caller program that is a variation on the
  simplest window program... It uses the XPIN facility to add items
  to its menu bar at runtime, but only if there are PIN files located
  out there in the specified directory.

  To make a program plug-in capable, a few small changes have to be
  made:

  a) The program should call XPINInit early on in the program to
     find all the plug-ins out there on the disk. The on-disk search
     happens only when you call XPINInit, so any plug-ins copied into
     the directory after a call to XPINInit will be ignored. XPINInit
     should be matched by an XPINDone when the program is finished
     with plug-ins.

  b) Assuming the users are to know of the existence of the PIN's out
     there, the program also needs to somehow tell the user what
     extra functions are available. This program adds a new menu item
     to the main menu at runtime, called "Plug-ins", and adds the
     "label" for each plug-in to the menu. There are two function
     calls for this: PINCount returns the number of PIN's found out
     there. PINLabel returns the label assigned to each PIN. The
     label is returned by the PIN itself and has no relation to the
     filename.

  c) When the user invokes an item in the menu, the program has to
     call the proper plug-in function. This requires the program
     to somehow associate menu ID's with a plug-in ID (which is an
     index from 0 to PINCount()-1). Invoking the plug-in requires
     a single call to PINCall.

  d) Before quitting, the internal PIN stuff has to be cleared
     out. So right before quitting, the program has to call PINDone.

  That's it. Can't get much easier.

  The toughest task for the developer is deciding how and what the user
  sees at runtime (i.e. a menu, a popup list, etc...) and coming up with
  a way to associate the command identifier with the plug-in ID.
  In this prorgam we just use subtraction/addition to get from menu-ID's
  to plug-in ID's. Each menu item we create at runtime happens to be
  have 1000 added to it. At runtime, if we get a command with ID higher
  than 1000, we know it came from the plug-in menu. We subtract 1000
  and call the plug-in with that number. You may want to create your
  own scheme (i.e. lookup tables, hashed lists, etc...)

  Each pin also has a "Description" field associated with it. You can
  use that in any way you want. A nice technique might be to have
  an ABOUT box for each PIN, then call XPINDesc to get the description
  for each plug-in, and show it in a dialog box (MessageBox, Alert,
  or what have you...). At some point in the future, I'll add ICON
  support to the plug-in so each plug-in can have an attached ICON...
  There's enough here now to get things going.

  One other thing: You can call XPINInit once for each file type
  you want to load up. For example, once for files of type XTS,
  another for PIN, another for FIL, etc...
  Just remember that the calls to XPINInit and XPINDone have to be
  matched. Another thing to remember is that if you call XPINInit
  before you start the event loop, make sure you also call XPINDone
  after the event loop is finished. If you put an XPINInit in with
  the code that runs once for each window, then you want to make sure
  there's a corresponding PINDone in the code that destroys each window.
  It's OK to call XPINDone more than once, but it's probably not too
  cool to leave the program without calling XPINDone at all...
  DLL's and locked code resources have a habit of sticking around...

  NOTE: If you plan on having multiple instances of the
  application active, keep in mind that plug-ins are DLL's, and
  rules on internal storage and access to the stack apply to them.
  Read up on DLL rules for DDE_SHARED memory.

  NOTE 2: This code is just a quick and dirty example of using XPIN's.
  I deliberately kept the amount of error checking down to help keep
  things understandable. Please make your programs have proper error
  checking.
  @End-Description.

 */

#include <windows.h>
#include "WCALLER.h"		// menu ID's for this program
#include "XPINSHR.h"		// Shared portion of XPIN defs here
#include "XPINCALL.h"		// Caller portion of XPIN defs here

/*
 * In this example, we add the plug-in labels at runtime as menu items.
 * Menu items require unique control ID's. In order to avoid conflicts
 * with any existing menu item, we start the runtime menu ID's at some
 * large value (i.e. 1000). When a command comes in, we need to match
 * the control ID with the PIN index so we can call the right function.
 * In this simple case, if we see a command ID higher than XPINBASE,
 * we subtract XPINBASE from it, and use the value to call the proper PIN.
 */
#define    XPINBASE  1000	// base control ID number for the PIN
XPIN	   xpin;		// PIN structure, allocated in heap.
HMENU	   theXPINmenu;		// Handle to the plug-in menu (if any)
void	   AddXPINMenu(HWND hWnd, XPIN *xpin);

/*
 * Normal minimal app declarations...
 */
long FAR PASCAL MainFunction (HWND, WORD, WORD, LONG);
char applicationName[] = "SampleCaller";

/*
 * Main Application Entry point
 */
int	PASCAL	WinMain(HANDLE hInstance, HANDLE hPrevInstance,
			LPSTR lpszCmdLine, int nCmdShow)
{
HWND		hWindow;
WNDCLASS	wClass;
MSG		message;
XPINErr		status;
long		length;

	if (!hPrevInstance) {
		wClass.style		= CS_HREDRAW | CS_VREDRAW;
		wClass.lpfnWndProc	= MainFunction;
		wClass.cbClsExtra	= 0;
		wClass.cbWndExtra	= 0;
		wClass.hInstance	= hInstance;
		wClass.hIcon		= LoadIcon(NULL, IDI_APPLICATION);
		wClass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wClass.hbrBackground	= GetStockObject(WHITE_BRUSH);
		wClass.lpszMenuName	= applicationName;
		wClass.lpszClassName	= applicationName;

		RegisterClass(&wClass);
	}

	hWindow = CreateWindow(applicationName, "Sample Caller",
					WS_OVERLAPPEDWINDOW,
					CW_USEDEFAULT, CW_USEDEFAULT,
					CW_USEDEFAULT, CW_USEDEFAULT,
					NULL, NULL, hInstance, NULL);

	// Initialize the XPIN structure. Look into the $HOME directory
	// for plug-ins. Look for files ending with PIN.
	// NOTE: If you want to specify paths, remember that C eats
	// backslashes inside quoted texts. So to look inside the
	// PIN directory of the application's home directory, you
	// would use "$HOME\\PIN"

	status = XPINInit(&xpin, "$HOME", "PIN");
    if (status != XPINOK) {
	  MessageBox(NULL, "XPINInit Failed.",
		"Error", MB_ICONSTOP | MB_OK);
	  return(XPINInit);
    }

/*
   Before we do a ShowWindow, we add some items to the menu
   (if we've found any plug-in files... of course). This could also
   be something like "Add the name for each plug-in to the listbox"
   or something like that...
   NOTE: Unless you want to call XPINInit once for each window, you
   can put XPINInit right before you start the message loop and
   XPINDone right after the message loop ends. If you somehow
   need this to happen inside the message loop, you can shoot
   for WM_INITWINDOW and WM_DESTROY (but remember, each window
   gets one of these).
 */
	if (XPINCount(&xpin) > 0)
		AddXPINMenu(hWindow, &xpin);


/* Back to the regularly scheduled program... */

	ShowWindow(hWindow, nCmdShow);
	UpdateWindow(hWindow);

	while(GetMessage(&message, NULL, 0, 0)) {
		TranslateMessage(&message);
		DispatchMessage(&message);
		}

 /*
  * Since we call XPINInit before the main event loop, we'll make sure
  * the matching XPINDone is called after the even loop is done...
  * We wipe out the menu we added at runtime (if any) then the internal
  * XPIN structure...
  */
	if (XPINCount(&xpin) > 0) { // make sure we created one before.
		DestroyMenu(theXPINmenu);
		XPINDone(&xpin);
		}

	return(message.wParam);
}

/*
 * The Main even loop...
 */
long FAR PASCAL MainFunction(HWND hWindow, WORD message, WORD wParam, LONG lParam)
{
long	xpinIndex;
XPINErr status;
XBlock	xb = { 0 };

	switch(message) {
		case WM_COMMAND:

		// First let's see if the menuID is the one for the menu items
		// we dynamically added (i.e. it's a PIN)...
			if (wParam >= XPINBASE) {
				xpinIndex = (long) (wParam - XPINBASE);
				status = XPINCall(&xpin, xpinIndex, &xb);
// NOTE: You should check the status to make sure the call worked.
				return 0;
			}

        // Otherwise, it's a normal static menu request
			switch(wParam) {
				case msgEXIT:
					SendMessage(hWindow, WM_CLOSE, 0, 0L);
					return 0;

				case msgHELP:
					MessageBox(hWindow, "Windows Help.",
						applicationName,
						MB_ICONINFORMATION | MB_OK);
					return 0;

				case msgABOUT:
				MessageBox(hWindow,
				"(c) 1992, rp&A Inc.\n\nThis is a sample plug-in caller.",
				applicationName,
				MB_ICONINFORMATION | MB_OK);
				return 0;
			}
			break;

		case WM_DESTROY:
			PostQuitMessage(0);
			return 0;
		}
	return(DefWindowProc(hWindow, message, wParam, lParam));
}

/*
 * This function is called only if we've found an XPIN out there
 * in the specified directory... In our case, we add a new menu
 * to the menu bar and add the labels for each XPIN to the menu bar.
 * The menu ID for each label starts at XPINBASE and goes up (see
 * above for a description of why).
 */
void	AddXPINMenu(HWND theWindow, XPIN *xpin)
{
HMENU	theMenu;
int	index,	menuID;
char	*label;

	theXPINmenu = CreatePopupMenu();	// make an empty menu

	for (index = 0; index < XPINCount(xpin); index++) {
		label = XPINLabel(xpin, index);
		menuID = XPINBASE + index;
		AppendMenu(theXPINmenu,
			   MF_ENABLED | MF_STRING,
			   (UINT) menuID, label);
	}

/* Now we Add the pop-up menu to the main menu. */

	theMenu = GetMenu(theWindow);
	InsertMenu(theMenu, (UINT) 2,
		   MF_BYPOSITION | MF_ENABLED | MF_POPUP,
		   (UINT) theXPINmenu, "&Plug-in");
}
