/***************************************************************************\
*                                                                           *
* clparam.cpp -- generic parser for command line parameters.                *
*                                                                           *
* 1993-1994, Mark Betz, released to the Public Domain                       *
* version:  2.0                                                             *
* date:     January 23, 1993                                                *
* compiler: Borland C++ 3.1, C++ source and keywords                        *
*                                                                           *
* changes:  1.0a - corrected logic error in parser during tag scan          *
*           2.0  - made the parser into a class                             *
\***************************************************************************/

#include <stdlib.h>
#include "clparam.h"

// the constructor assigns the value of the global argc and argv variables
// to members fargc, and fargv for later referral.

ClParam::ClParam( int argc, char* argv[] )
{
   fargc = argc;           // set the member data to point to the...
   fargv = argv;           // command line count and args
}

// scan() does the work of scanning the arguments. Parameter delim is
// a pointer to a null-termed character string representing the delimiter/
// switch sequence to search for, and buf is a pointer to the character
// buffer where arguments associated with that delimiter will be copied.
// The arglen parameter specifies the argument length restriction, and
// argtype is the argument type restriction.  See CLPARAM.H, and the doc
// file CLPARAM.DOC for more information.

int ClParam::scan( const char* delim, char* buf, const int arglen,
                   const int argtype ) const
{
   int i, j, k, result;
   int delim_len = 0;
   char c;

   while(delim[delim_len]) delim_len++;
   result = NO_DELIM;
   // scan all the argv[] strings
   for (i = 1; i < fargc; i++)
   {
      j = 0;
      k = 0;

      // scan string i for the delimiter character
      while ((c = fargv[i][j++]) && (c != delim[k]));
      k++;

      // if c nonzero the delimeter character was found
      if (c)
      {
         // scan for the tags. k tracks number of matches.
         while ((k < delim_len) && ((c = fargv[i][j++]) == delim[k]))
            k++;

         // if k == the delimiter length all tags were found
         if (k == delim_len)
         {
            k = 0;

            // copy the argument string until a null is found, or
            // the delimiter character is repeated
            while ((c = fargv[i][j++]) && (c != delim[0]))
            {
               if ((arglen) && (k == arglen))
               {
                  result = BAD_ARGLEN;
                  break;
               }
               else buf[k++] = c;
            }
            buf[k] = 0;

            // if we have a valid string, then type-check it
            if (result != BAD_ARGLEN)
            {
               if (typeCheck( argtype, buf ))
                  result = SCAN_OK;
               else
                  result = BAD_ARGTYPE;
            }
         }
      }
   }
   return result;
}

// the getProgPath function returns the full directory path and program
// name of the program executing this function. For example, if the
// program test.exe were run from the directory C:\UTIL then this function
// will copy the string "C:\\UTIL\\TEST.EXE" to the passed buffer. No range
// checking is performed on the buffer, but the function will not copy
// more than 80 characters.

void ClParam::getProgPath( char* buf ) const
{
   int j = 0;
   // watch the order of evaluation here!
   while ((buf[j++] = fargv[0][j]) && (j < 80));
}

// The typeCheck function is used internally by the ClParam class to
// check the type of arguments.

int ClParam::typeCheck( const int argtype, const char* buf ) const
{
   int llim, hlim;         // ascii upper and lower limits of type
   int i = 0;              // counter used to walk the char buffer
   char c;                 // current character
   int result = 1;         // return code

   switch(argtype)         // set the ascci limits based on the type
   {
      case ANY    : llim = 0; hlim = 255; break;
      case ALPHA  : llim = 33; hlim = 126; break;
      case NUMBER : llim = 48; hlim = 57; break;
      default     : llim = 0; hlim = 0;
   }
   while (c = buf[i++])    // walk the buffer and test each char
   {
      if ((c < llim) || (c > hlim))
      {
         result = 0;       // if any chars outside range return false
         break;
      }
   }
   return result;
}

