/***************************************************************************\
*                                                                           *
* pmtest.c -- testing the color matching algorithm by comparing it's        *
* results to a brute-force sequential search.                               *
*                                                                           *
* Useage:  PMTEST -f[find-file] -s[source-file] -t[value]                   *
* Example: pmtest -ftest1.pal -stest2.pal -t10                              *
*                                                                           *
*          [source-file] and [find-file] are any two valid DOS files,       *
*          assumed to contain a linear array of palette data in the         *
*          form of triplets (char[?][3]). The length of the palette is      *
*          controlled by the PAL_SIZE define in RGBTREE.H, which is         *
*          set at 768 bytes by default.                                     *
*                                                                           *
* The program tests the speed and accuracy of the 3-D binary tree rgb       *
* search algorithm against the brute-force sequential search method,        *
* which serves as a standard. The accuracy is tested by searching for       *
* the source-file for matches to the colors in the find-file. Speed is      *
* tested in a loop which performs multiple sequential and tree lookup.      *
*                                                                           *
* Copyright (c) 1992, Mark Betz, Betz Associates Inc.                       *
\***************************************************************************/

#include <conio.h>
#include <stdlib.h>
#include <io.h>
#include <stdio.h>
#include <dos.h>
#include <fcntl.h>
#include <sys\stat.h>
#include "rgbtree.h"
#include "clparam.h"

typedef char FPATH[81];                    // filepath type

const THOLD_DEF = 5;                       // default search threshold

unsigned char pal1[PAL_SIZE];              // source file colors
unsigned char pal2[PAL_SIZE];              // find file colors
unsigned char arrymat[DAC_SIZE];           // results of seq. search
unsigned char treemat[DAC_SIZE];           // results of tree lookup
int visitCnts[DAC_SIZE];                   // number of nodes visited

RGBBinTree RGBTree;                        // the RGB BSP tree object
int thold;                                 // tree search threshold
FPATH tpal1;                               // source file path
FPATH tpal2;                               // find file path
int index_miss = 0;                        // index mismatch counter
int maxd_index;                            // position of max. inacc.
unsigned nodesAvg = 0;                     // average number of nodes visited
float max_diverg = 0;                      // maximum inaccuracy %

unsigned start = 0;                        // starting time
unsigned stop = 0;                         // ending time
float a_elap = 0;                          // seq. search elapsed ms
float t_elap = 0;                          // tree search elapsed ms

// The get_clock() function calls the BIOS service 0x1a, subfunction
// 00, to get the current value of the bios time counter

unsigned get_clock()
{
   union REGS r;
   r.h.ah = 0;
   int86(0x1a, &r, &r);
   return _DX;
}

// The do_clp() function parses the command line parameters, and
// replaces the default value with the passed value for any valid
// parameter found. It returns 1 if valid or no parameters are found,
// and 0 if an invalid parameter is found.

// command-line switches:   -s[filename], source palette filename
//                          -f[filename], find palette filename
//                          -t[value], threshold value

int do_clp( int argc, char* argv[] )
{
   int result = 0;
   char temp[10];
   ClParam clp( argc, argv );

   if (clp.scan( "-s", tpal1, 80, ALPHA ) == SCAN_OK)
   {
      if (clp.scan( "-f", tpal2, 80, ALPHA ) == SCAN_OK)
      {
         result = 1;
         if (clp.scan( "-t", temp, 9, NUMBER ) == SCAN_OK)
         {
            thold = atoi(temp);
            if ((thold < 0) || (thold > 63))
               thold = THOLD_DEF;
         } else thold = THOLD_DEF;
      }
   }
   return result;
}

/* find a match for the given rgb triplet in tpal1, using a sequential
   search of the palette array, and a 3-D point distance formula for
   comparisons. Should be the most accurate method, so it is used as a
   standard for this test.                                            */

int seq_search(char r, char g, char b)
{
   int dx, min_dx, color;
   int rp, gp, bp, j;

   min_dx = 3*63*63;
   for (j = 0; j < PAL_SIZE-1; j+=3)
   {
      rp = pal1[j];
      gp = pal1[j+1];
      bp = pal1[j+2];

      /* distance between 3 points in space, squared */

      dx = (rp-r)*(rp-r) + (gp-g)*(gp-g) + (bp-b)*(bp-b);

      /* looking for minimum distance */

      if (dx < min_dx)
      {
         min_dx = dx;
         color = (char)(j/3);
      }
   }
   return color;
}

/* The init_pals() function opens the test palette files and reads
   them into the palette arrays pal1 and pal2. Handle any i/o errors.
   Returns 1 on success, 0 on failure.                                */

int init_pals()
{
   int result = 1;
   int htp1, htp2;

   printf("->Loading %s ...", tpal1);
   if ((htp1 = open(tpal1, O_BINARY | O_RDONLY)) == -1)
   {
      printf("error opening file %s!\r\n",tpal1);
      result = 0;
   }
   else if (read(htp1, pal1, PAL_SIZE) != PAL_SIZE)
   {
      printf("error reading %s data!\r\n",tpal1);
      result = 0;
   }
   else if (close(htp1) == -1)
   {
      printf("bad file number closing %s!\r\n",tpal1);
      result = 0;
   }
   else printf("done.\r\n");
   if (result)
   {
      printf("        ->Loading %s ...", tpal2);
      if ((htp2 = open(tpal2, O_BINARY | O_RDONLY)) == -1)
      {
         printf("error opening file %s!\r\n", tpal2);
         result = 0;
      }
      else if (read(htp2, pal2, PAL_SIZE) != PAL_SIZE)
      {
         printf("error reading %s data!\r\n", tpal2);
         result = 0;
      }
      else if (close(htp1) == -1)
      {
         printf("bad file number closing %s!\r\n",tpal2);
         result = 0;
      }
      else printf("done.\r\n");
   }
   return result;
}

/* The analyse_results() function compares the results returned by
   both methods. For each color index stored in tree_mat[] and
   arry_mat[] the point distance is computed and compared, and the
   percentage of inaccuracy for each value in tree_mat[] is calc-
   ulated. The function sets the file-scope variables max_diverg,
   and maxd_index to the maximum inaccuracy and the index where it
   was found, respectively.                                           */

void analyse_results()
{
   int adx=0, tdx=0;
   float div;
   int rx, gx, bx;
   int r, g, b;
   int i,j;

   for (i = 0; i < PAL_SIZE/3; i++)
   {
      if (arrymat[i] != treemat[i])
      {
         index_miss++;
         j = i*3;
         r = pal2[j];
         g = pal2[j+1];
         b = pal2[j+2];
         j = arrymat[i]*3;
         rx = pal1[j];
         gx = pal1[j+1];
         bx = pal1[j+2];
         adx = (rx-r)*(rx-r) + (gx-g)*(gx-g) + (bx-b)*(bx-b);
         j = treemat[i]*3;
         rx = pal1[j];
         gx = pal1[j+1];
         bx = pal1[j+2];
         tdx = (rx-r)*(rx-r) + (gx-g)*(gx-g) + (bx-b)*(bx-b);
         tdx -= adx;
         div = (tdx/(float)(63*63*3))*100;
         if (div > max_diverg)
         {
            max_diverg = div;
            maxd_index = i;
         }
      }
   }
   for (i = 0; i < DAC_SIZE; i++)
      nodesAvg += visitCnts[i];
   nodesAvg /= DAC_SIZE;
   return;
}

int main(int argc, char* argv[])
{
   int i, j;

   clrscr();
   if (do_clp(argc, argv))
   {
      printf("*************** RGB color matching test 01 ***************\r\n");
      printf("\r\n");
      printf("task:   search method comparison, sequential vs. treesort\r\n");
      printf("\r\n");
      printf("test01: ");
      if( init_pals())
      {
         printf("        ->Building RGB sort tree ...");
         RGBTree.build( pal1 );
         printf("done.\r\n\r\n");
         printf("        ->Matching find-file colors to source-file colors:\r\n");
         printf("           -by linear sequential search ...");
         for (i = 0; i < PAL_SIZE; i+=3)
            arrymat[i/3] = seq_search(pal2[i], pal2[i+1], pal2[i+2]);
         printf("done.\r\n");
         printf("           -by threshold %i tree search ...",thold);
         for (i = 0; i < PAL_SIZE; i+=3)
         {
            treemat[i/3] = RGBTree.rgbMatch(pal2[i], pal2[i+1], pal2[i+2],
                                            thold);
            visitCnts[i/3] = RGBTree.nodesVisited();
         }
         printf("done.\r\n");
         printf("        ->Performing timing tests ...");

         start = get_clock();
         for (j = 0; j < 5; j++)
         {
            for (i = 0; i < PAL_SIZE; i+=3)
               seq_search(pal2[i], pal2[i+1], pal2[i+2]);
         }
         stop = get_clock();
         a_elap = (float)((stop-start) * 54.9)/5;
         start = get_clock();
         for (j = 0; j < 5; j++)
         {
            for (i = 0; i < PAL_SIZE; i+=3)
               RGBTree.rgbMatch(pal2[i], pal2[i+1], pal2[i+2], thold);
         }
         stop = get_clock();
         t_elap = (float)((stop-start) * 54.9)/5;
         printf("done.\r\n");
         printf("        ->Analysing results ...");
         analyse_results();
         printf("done.\r\n");
         printf("\r\n");
         printf("Results: Index mismatch in %i out of %i cases\r\n",
                 index_miss, PAL_SIZE/3);
         printf("         Maximum RGB innacuracy %.2f percent at index %i\r\n",
                 max_diverg, maxd_index);
         printf("         Sequential search elapsed time %.2f milliseconds\r\n",
                 a_elap);
         printf("         Tree range search elapsed time %.2f milliseconds\r\n",
                 t_elap);
         printf("         Average number of nodes visited %u per search\r\n",
                 nodesAvg);
         printf("\r\n");
         printf("******************* test 01 completed ********************\r\n");
      }
   }
   else
   {
      printf("Useage       : PMTEST [-ffilename] [-sfilename] <-tvalue>\r\n");
      printf("example      : pmtest -ftest1.pal -stest2.pal -t15\r\n");
      printf("\r\n");
      printf("-ffilename   : Specifies the file of palette values to search for.\r\n");
      printf("               filename is any valid file path and name.\r\n");
      printf("-sfilename   : Specifies the file of palette values which will be\r\n");
      printf("               searched.\r\n");
      printf("-t[value]    : Specifies an optional value for the search threshold.\r\n");
      printf("               Expects an integer in the range [0..63].\r\n");
   }
   exit(0);
}

