/**************************************************************************\
*                                                                          *
* remap.c -- remaps the colors in a target pcx to use the palette in       *
*            a source pcx.                                                 *
*                                                                          *
* Useage:  REMAP -s[source-file] -r[target-file] -t[value]                 *
* Example: remap -stest1.pcx -rtest2.pcx -t5                               *
*                                                                          *
*                                                                          *
* Copyright (c) 1992, Mark Betz, Betz Associates Inc.                      *
\**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fastgraf.h>
#include <dos.h>
#include <conio.h>
#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include "clparam.h"
#include "rgbtree.h"

typedef char FPATH[81];                      // filepath type

static unsigned char* rmarray;               // destination pixel buffer
static unsigned char* sarray;                // source pixel buffer
static RGBBinTree RGBTree;                   // RGB BSP tree object
static FPATH rgbsrc;                         // source palette filename
static FPATH rmpcx;                          // target image filname
static FPATH wrpcx = "output.pcx";           // default write filename
static DACTBL sdacs;                         // palette values from source
static DACTBL rmdacs;                        // palette values from target
static unsigned char icache[DAC_SIZE][2];    // cache for palette indices
static int thold;                            // search threshold argument
static int vmode;                            // start-up video mode
static int exit_loc;                         // program location flag
static int saved_file;                       // true if file saved
static int using_pal;                        // true if using palette file as
                                             // source

const PIX_SIZE = 64000;                      // picture specifics
const WIDTH = 320;
const DEPTH = 200;
const TOP = 0;
const LEFT = 0;
const BOTTOM = 199;
const RIGHT = 319;

const THOLD_DEF = 5;                         // search threshold default

const STARTUP  = 0;                          // exit conditions for...
const CLPARAM  = 1;                          // controlled termination
const MEMSETUP = 2;
const SETMODE  = 3;
const GETSRGB  = 4;
const GETTPCX  = 5;
const MAPRGB   = 6;
const FILESAVE = 7;
const COMPLETE = 8;

// generates a tone for a successful save

void save_tone()
{
   sound(300);
   delay(75);
   sound(350);
   delay(75);
   sound(300);
   delay(75);
   nosound();
}

// generates a tone for an unsuccessful save

void nosave_tone()
{
   sound(100);
   delay(75);
   sound(50);
   delay(150);
   nosound();
}

// generates a tone for completed remapping

void done_tone()
{
   sound(100);
   delay(75);
   sound(150);
   delay(75);
   sound(100);
   delay(75);
   sound(175);
   delay(75);
   nosound();
   return;
}

// Uses the command line parser object in CLPARAM.C to see what options were
// entered on the command line

// command-line switches:   -s[filename], source pcx filename
//                          -p[filename], source palette filename
//                          -r[filename], remap pcx filename
//                          -o[filename], output filename
//                          -t[value], threshold value

int do_clp( int argc, char* argv[] )
{
   int result = 0;
   char temp[10];
   ClParam clp( argc, argv );

   exit_loc = CLPARAM;
   if ((clp.scan( "-s", rgbsrc, 80, ALPHA ) == SCAN_OK) ||
      ((clp.scan( "-p", rgbsrc, 80, ALPHA ) == SCAN_OK) &&
       (using_pal = 1)))
   {
      if (clp.scan( "-r", rmpcx, 80, ALPHA ) == SCAN_OK)
      {
         result = 1;
         if (clp.scan( "-t", temp, 9, NUMBER ) == SCAN_OK)
         {
            thold = atoi(temp);
            if ((thold < 0) || (thold > 32))
               result = 0;
         } else thold = THOLD_DEF;
         clp.scan( "-o", wrpcx, 80, ALPHA );
      }
   }
   return result;
}

// Allocates the bitmap arrays, then zeros the destination pixel array and
// the color cache.

int mem_setup()
{
   int result = 1;

   exit_loc = MEMSETUP;
   if (((sarray = new unsigned char[PIX_SIZE]) == NULL) ||
       ((rmarray = new unsigned char[PIX_SIZE]) == NULL))
   {
      result = 0;
      if (sarray) delete [] sarray;
   }
   else
   {
      memset(rmarray, 0, PIX_SIZE);
      memset(icache, 0, DAC_SIZE*2);
   }
   return result;
}

// This function is called when the source for the RGB control values is a
// palette file instead of a pcx. It opens the file, and reads the palette
// values into the sdacs array. It returns 1 on success, 0 on failure.

int get_pal()
{
   int result = 0;
   int htp1;

   if ((htp1 = open(rgbsrc, O_BINARY | O_RDONLY)) != -1)
   {
      if (read(htp1, sdacs, PAL_SIZE) == PAL_SIZE)
      {
         if (close(htp1) != -1)
            result = 1;
      }
   }
   return result;
}

// Initializes the video mode, loads the image to be remapped, captures the
// dacs, and loads the image into the source pixel buffer, then loads the
// rgb source image or palette file, and captures the dacs.

int vid_setup()
{
   int result = 0;

   exit_loc = SETMODE;
   if (fg_testmode(19,1) != 0)
   {
      fg_setmode(19);
      fg_move(0,0);
      exit_loc = GETTPCX;


      if (!fg_disppcx(rmpcx,0))
      {
         fg_getdacs(0,256,rmdacs);
         fg_move(0,199);
         fg_getimage(sarray, WIDTH, DEPTH);
         fg_move(0,0);
         exit_loc = GETSRGB;
         if (using_pal && get_pal())
            result = 1;
         else if (!using_pal && !fg_disppcx(rgbsrc,0))
         {
            fg_getdacs(0,256,sdacs);
            result = 1;
         }
      }
   }
   return result;
}

// Dumps the program help to screen. Called whenever an incomplete or
// incorrect command line is passed

void dump_useage()
{
   printf("useage    : REMAP [-ssource|-ppalette] [-rtarget] [-ooutput] <-tvalue>\r\n");
   printf("example   : remap -rceltic.pcx -pmain.pal -ocelt01.pcx -t6\r\n");
   printf("          : remap -rtest1.pcx -stest2.pcx -ooutput.pcx -t5\r\n");
   printf("\r\n");
   printf("-ssource  : specifies a pcx file source for the control palette\r\n");
   printf("-ppalette : specifies a file of RGB values to use as a control\r\n");
   printf("-rtarget  : specifies the pcx file to be remapped\r\n");
   printf("-ooutput  : specifies the desired output filename\r\n");
   printf("-tvalue   : is 1/2 the threshold value for the tree search\r\n");
   printf("            (max. value of 32 yields a search range of 64)\r\n");
   printf("\r\n");
   printf("Press 's' or 'S' to save image after remapping, using the\r\n");
   printf("-o output filename, or press 'esc' to exit without saving.\r\n");
}

// handles controlled program termination based on the value of exit_loc

int terminate()
{
   int exit_code = 0;

   fg_setmode(vmode);
   if (exit_loc > MEMSETUP)
   {
      delete [] sarray;
      delete [] rmarray;
   }
   switch(exit_loc)
   {
      case STARTUP  : break;
      case CLPARAM  : dump_useage(); break;
      case MEMSETUP : printf("-not enough memory to run remap.exe\r\n"); break;
      case SETMODE  : printf("-vga card not detected\r\n"); break;
      case GETSRGB  : printf("-error loading source rgb file: %s\r\n", rgbsrc);
                      break;
      case GETTPCX  : printf("-error loading target pcx file: %s\r\n", rmpcx);
                      break;
      case MAPRGB   : printf("-unknown error during color mapping\r\n"); break;
      case FILESAVE : printf("-error writing mapped pcx file: %s\r\n", wrpcx);
                      break;
      case COMPLETE : if(saved_file) printf("-file %s written to disk\r\n",wrpcx);
                      exit_code = 1;
                      break;
   }
   return exit_code;
}

int main( int argc, char* argv[] )
{
   unsigned i, j;
   unsigned char spix, rmpix;
   unsigned char r, g, b;

   clrscr();
   vmode = fg_getmode();
   saved_file = 0;
   using_pal = 0;
   exit_loc = STARTUP;

   if (do_clp(argc, argv))
   {
      if (mem_setup())
      {
         if (vid_setup())
         {
            exit_loc = MAPRGB;
            RGBTree.build( sdacs );
            // this block does the actual remapping of the pixels
            for(i = 0; i < PIX_SIZE; i++)
            {
               rmpix = sarray[i];
               if (icache[rmpix][0])
                  spix = icache[rmpix][1];
               else
               {
                  j = rmpix*3;
                  r = rmdacs[j++];
                  g = rmdacs[j++];
                  b = rmdacs[j];
                  spix = RGBTree.rgbMatch(r, g, b, thold);
                  icache[rmpix][0] = 1;
                  icache[rmpix][1] = spix;
               }
               rmarray[i] = spix;
            }
            fg_setdacs(0,256,sdacs);
            _fmemset(MK_FP(0xa000,0), 0, PIX_SIZE);
            fg_move(0,199);
            fg_drwimage(rmarray, WIDTH, DEPTH);
            done_tone();

            // wait for key, and save new image if 's' or 'S' pressed
            while(!kbhit());
            i = getch();
            if ((i == 's') || (i == 'S'))
            {
               exit_loc = FILESAVE;
               if (!fg_makepcx(LEFT, RIGHT, TOP, BOTTOM, wrpcx))
               {
                  exit_loc = COMPLETE;
                  save_tone();
                  saved_file = 1;
               } else nosave_tone();
            }
            else exit_loc = COMPLETE;
         } // vid_setup
      } // mem_setup
   } // do_clp
   return (terminate());
}











