/**********************************************************************\
*                                                                      *
* rgbtree.cpp -- RGBBinTree member function definitions                *
*                                                                      *
*    *** LISTING 2 ***                                                 *
\**********************************************************************/

// Mark Betz
// 120 N. Shore Rd.
// Derry, NH 03038
// 603-898-8214

// Copyright (c) 1993, Mark Betz.

#include <stdlib.h>
#include <mem.h>
#include "rgbtree.h"

// definition of static members

int RGBBinTree::n_stack[];
int RGBBinTree::n_sp = 0;

// default constructor does nothing but initialize member data

RGBBinTree::RGBBinTree() : treeBuilt(0), visitCnt(0) {}

// The build() function is called to construct the search tree for
// a particular palette. The address of the palette is passed in the
// dacs parameter. The tree is constructed by walking the palette
// array, and inserting the rgb values into nodes in the tree. In
// order to minimize the chance of an ordered set causing a worst-
// case search performance, the function first calls getMid() to
// find the starting triplet, which it inserts as the root node.
// From this point the function moves forward and backward in the
// array, inserting the rest of the nodes. Note that this function
// may be called repeatedly to sort new rgb tables into the tree,
// but it must be called at least once before calling rgbMatch().

void RGBBinTree::build( const DACTBL dacs )
{
   struct pnode newn;                    // local node for insertions
   int i, j, di;                         // color index, generic counters
   int node = 1;                         // counts tree array elements

   pn[HEAD].index = ZIDX;                // initialize the head and tail...
   pn[HEAD].left = TAIL;                 //  nodes in the array
   pn[HEAD].right = TAIL;                // indices = -1, child nodes set...
   pn[TAIL].index = ZIDX;                //  to point to TAIL
   pn[TAIL].left = TAIL;
   pn[TAIL].right = TAIL;
   newn.left = 0;                        // don't use these, so zero them
   newn.right = 0;

   i = getMid(dacs);                     // find the color nearest the center
   j = i+1;                              // point j to the next color
   for (;i > ZIDX; i--)                  // insert the lower half of the dacs
   {
      di = i*3;                          // color index = dac index * 3
      newn.c[R] = dacs[di];              // load the primaries into the node
      newn.c[G] = dacs[di+1];
      newn.c[B] = dacs[di+2];
      newn.index = i;                    // load the dac index into the node
      insertNode( newn, node );          // insert it into the tree
      node++;                            // increment to next node
   }
   for (;j < DAC_SIZE; j++)              // insert the upper half of the dacs
   {
      di = j*3;                          // same procedure as above
      newn.c[R] = dacs[di];
      newn.c[G] = dacs[di+1];
      newn.c[B] = dacs[di+2];
      newn.index = j;
      insertNode( newn, node );
      node++;
   }
   treeBuilt = 1;                        // set treeBuilt to true
}

// The insertnode() function is used internally by the build()
// function. It inserts a node by traversing the tree, doing a key
// comparison on the red, green, and blue values. The node is
// inserted at the first leaf node (pointer to TAIL).

void RGBBinTree::insertNode( const struct pnode& newn, int elem )
{
   int par = HEAD;                       // HEAD is the first parent node
   int cur = pn[HEAD].right;             // current node is HEAD's right chi.
   int colorf = -1;                      // color flag will be 0 on entry
   int dirflag = 1;                      // tracks direction of traversal

   pn[elem].c[R] = newn.c[R];            // load the data from the new node...
   pn[elem].c[G] = newn.c[G];            //  into the array at elem
   pn[elem].c[B] = newn.c[B];
   pn[elem].index = newn.index;
   pn[elem].left = TAIL;
   pn[elem].right = TAIL;

   while (cur != TAIL)                   // looking for first leaf node
   {
      par = cur;                         // the current node is new parent
      colorf == 2 ? colorf = 0 : colorf++;      // roll the color flag
      dirflag = 0;                              // clear the direction flag
      if (newn.c[colorf] >= pn[cur].c[colorf])  // if search color greater...
      {                                         // or equal...
         cur = pn[cur].right;            // go right, and set the dirflag
         dirflag++;
      } else cur = pn[cur].left;         // else go left, leave dirflag = 0
   }
   if (!dirflag)                         // based on dirflag, insert the...
      pn[par].left = elem;               //  new node as either the right...
   else                                  //  or left child of the parent
      pn[par].right = elem;
   return;
}

// The rgbMatch() function takes as parameters three byte-wide values
// representing the red, green, and blue components of the desired
// color, and a threshold value which is used to determine the compare
// window. Only the lower six bits of each value is significant, for a
// range of 0..63. It returns the best match from the current sorted
// palette by traversing the search tree, using the red, green, and
// blue values as search keys.

// At each node, if the key byte lies outside the search range to the
// right (less than) the algorithm moves left down the tree. If the
// key byte lies outside the range to the left (greater than), it
// moves right down the tree. If the key byte lies within the range
// it saves the node, checks the left sub-tree, and then returns to
// check the right sub-tree, because nodes in both subtrees could fall
// within the search range. If the disparities between the color being
// matched, and the colors in the palette, is high enough, and the
// search range is narrow enough, the algorithm may find no matches,
// in which case it returns -1.

int RGBBinTree::rgbMatch( unsigned char r, unsigned char g,
                          unsigned char b, int thold )
{
   int cur;                              // current node being visited
   register int dx = 3*63*63;            // best distance between colors
   int best = TAIL;                      // saves the current best match
   int t, rx, gx, bx;                    // used in distance calculation
   int colorf = -1;                      // color key compare flag
   int rng[3][2];                        // search range for each primary
   int key;                              // used in range comparisons

   if (!treeBuilt) return -1;            // bail if no tree built

   visitCnt = 0;                         // track nodes visited
   rng[0][0] = g-thold;                  // set up the comparison windows...
   rng[0][1] = g+thold;                  //  for the three primaries
   rng[1][0] = r-thold;
   rng[1][1] = r+thold;
   rng[2][0] = b-thold;
   rng[2][1] = b+thold;

   n_stack[n_sp++] = colorf;             // push the starting node and the...
   n_stack[n_sp++] = HEAD;               //  color compare flag

   while ((n_sp) && (dx))                // subtree loop runs until stack...
   {                                     //  empty or distance (dx) = 0
      cur = pn[n_stack[--n_sp]].right;   // pop the next node and compare flag
      colorf = n_stack[--n_sp];

      while ((cur != TAIL) && (dx))      // nodewalk loop runs till TAIL is...
      {                                  // hit or distance (dx) = 0
         visitCnt++;                     // increment nodes visited counter
         colorf == 2 ? colorf = 0 : colorf++;  // roll the compare flag

         key = pn[cur].c[colorf];        // get the current compare color
         if (key > rng[colorf][0])       // test against lower range
         {
            if (key <= rng[colorf][1])   // now test against upper range
            {
               n_stack[n_sp++] = colorf; // it's in the window, so push the...
               n_stack[n_sp++] = cur;    //  node and color flag

               rx = pn[cur].c[R];        // get the primaries from the node
               gx = pn[cur].c[G];
               bx = pn[cur].c[B];

               t = (rx-r)*(rx-r) +       // calculate the point distance
                   (gx-g)*(gx-g) +
                   (bx-b)*(bx-b);
               if (t < dx)               // if it's smaller than the...
               {                         //  current best distance, save it
                  dx = t;
                  best = cur;
               }
            }
            cur = pn[cur].left;          // node outside range to right...
         }                               //  or inside range
         else cur = pn[cur].right;       // node outside range to left
      }
   }
   return pn[best].index;                // return the best match
}

// The get_mid() function is used internally by the build()
// function. It finds a triplet near the middle of the rgb cube, and
// as close to the center of the palette array as possible. This
// triplet is then inserted into the search tree as the root node,
// causing the tree to be more balanced. Since this function is only
// performed once whenever a new tree is built, it uses a brute-force
// sequential search in order to find the middle triplet.

int RGBBinTree::getMid( const DACTBL dacs ) const
{
   int i, rx, gx, bx;                    // counter, distance calc vars
   int t, best = 0, dx = 512;            // counter, best match, distance

   for (i = 0; i < PAL_SIZE; i+=3)       // look through the dac palette
   {
      rx = dacs[i];                      // get the currect set of primaries
      gx = dacs[i+1];
      bx = dacs[i+2];
      t = (rx-31)*(rx-31)+               // how far are they from center?
          (gx-31)*(gx-31)+
          (bx-31)*(bx-31);
      if (t < dx)                        // if less than the current best...
      {                                  //  distance, save the distance...
         dx = t;                         //  and color index
         best = i/3;
      }
   }
   return best;                          // return the best match
}

