/*----------------------------------------------------------------------------*\
 *       Mci.h:  C++ class hierarchy for Windows multimedia objects using MCI.
 *  
 *   $Revision:   1.3  $
 *     $Author:   John Musser  $
 *       $Date:   05 Mar 1993 15:04:02  $
\*----------------------------------------------------------------------------*/

#ifndef __MCI_H__
#define __MCI_H__

#include <windows.h>
#include <mmsystem.h>
#include <digitalv.h>
#include <memory.h>


// Standard MCI device names
//
#define DEVICE_NAME_AVI  "avivideo"
#define DEVICE_NAME_CDA  "cdaudio"
#define DEVICE_NAME_MIDI "sequencer"
#define DEVICE_NAME_WAVE "waveaudio"

#define MCI_WARN_ALREADY_OPEN  0x001L
#define MCI_ERR_NOT_OPEN       0x002L
#define MCI_ERR_NO_PARENT      0x004L
#define MCI_ERR_NO_DEVICENAME  0x008L


class MCIDevice 
{
public:
                   MCIDevice(LPSTR lpszDevice = NULL);
   virtual         ~MCIDevice();
   
   //------------------------------------------- Play/Position functions
   virtual DWORD   Open(LPSTR lpszDevice = NULL);
   virtual DWORD   Close();
   virtual DWORD   Play(LONG lFrom = current, LONG lTo = end);
   virtual DWORD   Stop();
   virtual DWORD   Seek(LONG lTo);
   virtual DWORD   Pause();
   virtual DWORD   Resume();

   //------------------------------------------- Device capability functions
   inline  BOOL    CompoundDevice();
   inline  DWORD   DeviceType();
   inline  BOOL    CanRecord();
   inline  BOOL    HasAudio();
   inline  BOOL    HasVideo();
   inline  BOOL    UsesFiles();
   inline  LPSTR   InfoProduct();

   //------------------------------------------- Device Information functions
   inline  BOOL    IsOpen();
   inline  BOOL    IsPlaying();
   inline  DWORD   Mode();
   inline  BOOL    Ready();
   inline  DWORD   Length();
   inline  DWORD   Position();
   inline  DWORD   TimeFormat();
   inline  DWORD   NumberOfTracks();
   inline  DWORD   CurrentTrack();
   inline  DWORD   TrackPosition(DWORD dwTrack);
   inline  DWORD   TrackLength(DWORD dwTrack);

   //------------------------------------------- Control functions
   inline  DWORD   SetTimeFormatMilliseconds();

   //------------------------------------------- Miscellaneous functions
           void    ErrorMessageBox();

           void    SetParent(HWND hWnd) { hWndParent = hWnd; }
           HWND    GetParent(void)      { return hWndParent; }
           UINT    GetDeviceID()        { return wDeviceID; }
           DWORD   GetErrState(void)    { return dwErrState; }

   static  DWORD   StopAll();         // Stop all mci devices (for this app)

   enum { start = 0, current = -1, end = -2 };   // Args for Play()/Seek() 

   //------------------------------------------- Protected data & functions
protected:
   virtual LPSTR   Info(DWORD dFlags);
   virtual DWORD   GetDevCaps(DWORD dwItem);
   virtual DWORD   Set(DWORD dwFlags, DWORD dwExtra = 0L);
   virtual DWORD   Status(DWORD dwFlags, DWORD dwItem, DWORD dwExtra = 0L);

   HWND            hWndParent;            // Parent window (for Notify)
   DWORD           dwErrState;            // Return code from last mci call
   UINT            wDeviceID;             // The MCI device ID
   LPSTR           lpszDeviceType;        // String value for device type

   //------------------------------------------- Private utility functions
private:
   void            SetDeviceType(LPSTR lpszDevice);
};


class CDAudio : public MCIDevice
{
public:
                   CDAudio() : MCIDevice(DEVICE_NAME_CDA) {}

   inline  DWORD   Eject();               // Eject the disc
   inline  DWORD   SetTimeFormatMSF();    // Minutes/Seconds/Frames
   inline  DWORD   SetTimeFormatTMSF();   // Tracks/Minutes/Seconds/Frames
};


class CompoundDevice : public MCIDevice
{
public:
                   CompoundDevice(LPSTR lpszFile=NULL, LPSTR lpszDevice=NULL);
                   ~CompoundDevice();

   virtual DWORD   Open(LPSTR lpszFile);
   virtual DWORD   Close();

   inline  LPSTR   InfoFile();

protected:
   LPSTR           lpszFileName;       // Element name
};


class Wave : public CompoundDevice
{
public:
      Wave(LPSTR lpszFile = NULL) :
           CompoundDevice(lpszFile, DEVICE_NAME_WAVE) {}

   inline DWORD    Channels();                     // 1 = mono, 2 = stereo
   inline DWORD    BitsPerSample();                // 8 or 16 bit
   inline DWORD    SamplesPerSecond();             // 11, 22, or 44 kHz
};


class AVI : public CompoundDevice
{
public:
                   AVI(LPSTR lpszFile = NULL) :
                       CompoundDevice(lpszFile, DEVICE_NAME_AVI) {}

   virtual DWORD   Step(DWORD dwFrames = 1, BOOL bReverse = FALSE);
   virtual DWORD   PutDestination(RECT& rect);     // place at rect in window
   virtual DWORD   Update(HDC hdc);                // redraw current frame
           DWORD   Cue(DWORD dwTo = 0);            // pre-load
           DWORD   Configure();                    // pop-up configure dialog
           DWORD   Window(HWND hwnd);              // set display window
           DWORD   SetSpeed(DWORD dwSpeed);        // playback rate (0->10,000)
           DWORD   Signal(DWORD dwPosition);       // notify parent at pos
           DWORD   SetAudioVolume(DWORD dwVolume); // volume (0->1,000)
   inline  DWORD   SetTimeFormatFrames();          // time format with frames
};


//---------------------------------------------------------------
// Inline definitions
//
BOOL  MCIDevice::IsOpen()
   { return (BOOL)wDeviceID; }
DWORD MCIDevice::DeviceType()
   { return GetDevCaps(MCI_GETDEVCAPS_DEVICE_TYPE); }
DWORD MCIDevice::Position()
   { return Status(MCI_STATUS_ITEM, MCI_STATUS_POSITION); }
DWORD MCIDevice::Length()
   { return Status(MCI_STATUS_ITEM, MCI_STATUS_LENGTH); }
DWORD MCIDevice::NumberOfTracks()
   { return Status(MCI_STATUS_ITEM, MCI_STATUS_NUMBER_OF_TRACKS); }
DWORD MCIDevice::CurrentTrack()
   { return Status(MCI_STATUS_ITEM, MCI_STATUS_CURRENT_TRACK); }
DWORD MCIDevice::SetTimeFormatMilliseconds()
   { return Set(MCI_SET_TIME_FORMAT, MCI_FORMAT_MILLISECONDS); }
BOOL  MCIDevice::IsPlaying()
   { return (BOOL)(Status(MCI_STATUS_ITEM, MCI_STATUS_MODE) == MCI_MODE_PLAY); }
BOOL  MCIDevice::CompoundDevice()
   { return (BOOL)GetDevCaps(MCI_GETDEVCAPS_COMPOUND_DEVICE); }
DWORD MCIDevice::Mode()
   { return Status(MCI_STATUS_ITEM, MCI_STATUS_MODE); }
BOOL  MCIDevice::Ready()
   { return (BOOL)Status(MCI_STATUS_ITEM, MCI_STATUS_READY); }
DWORD MCIDevice::TimeFormat()
   { return Status(MCI_STATUS_ITEM, MCI_STATUS_TIME_FORMAT); }
LPSTR MCIDevice::InfoProduct()
   { return Info(MCI_INFO_PRODUCT); }
BOOL  MCIDevice::CanRecord()
   { return (BOOL)GetDevCaps(MCI_GETDEVCAPS_CAN_RECORD); }
BOOL  MCIDevice::HasAudio()
   { return (BOOL)GetDevCaps(MCI_GETDEVCAPS_HAS_AUDIO); }
BOOL  MCIDevice::HasVideo()
   { return (BOOL)GetDevCaps(MCI_GETDEVCAPS_HAS_VIDEO); }
BOOL  MCIDevice::UsesFiles()
   { return (BOOL)GetDevCaps(MCI_GETDEVCAPS_USES_FILES); }
DWORD MCIDevice::TrackLength(DWORD dwTrack)
   { return Status(MCI_STATUS_ITEM|MCI_TRACK, MCI_STATUS_LENGTH, dwTrack); }
DWORD MCIDevice::TrackPosition(DWORD dwTrack)
   { return Status(MCI_STATUS_ITEM|MCI_TRACK, MCI_STATUS_POSITION, dwTrack); }

DWORD CDAudio::Eject()
   { return Set(MCI_SET_DOOR_OPEN); }
DWORD CDAudio::SetTimeFormatMSF()
   { return Set(MCI_SET_TIME_FORMAT, MCI_FORMAT_MSF); }
DWORD CDAudio::SetTimeFormatTMSF()
   { return Set(MCI_SET_TIME_FORMAT, MCI_FORMAT_TMSF); }

LPSTR CompoundDevice::InfoFile()
   { return Info(MCI_INFO_FILE); }

DWORD Wave::Channels()
   { return Status(MCI_STATUS_ITEM, MCI_WAVE_STATUS_CHANNELS); }
DWORD Wave::BitsPerSample()
   { return Status(MCI_STATUS_ITEM, MCI_WAVE_STATUS_BITSPERSAMPLE); }
DWORD Wave::SamplesPerSecond()
   { return Status(MCI_STATUS_ITEM, MCI_WAVE_STATUS_SAMPLESPERSEC); }

DWORD AVI::SetTimeFormatFrames()
   { return Set(MCI_SET_TIME_FORMAT, MCI_FORMAT_FRAMES); }

//---------------------------------------------------------------


#endif  /* !__MCI_H__ */
