/* ----------------------------------------------------------- *\
**  tpack.cpp -- Test packing uncompressed bitmap data         **
** ----------------------------------------------------------- **
**                                                             **
**  Data file format (in ASCII hex):                           **
**                                                             **
**  np ns                   # pixels per line; # scan lines    **
**  pp pp pp pp pp ... pp   pixel byte values in hex           **
**  pp pp pp pp pp ... pp      "        "        "             **
**  pp pp pp pp pp ... pp      "        "        "             **
**                                                             **
** ----------------------------------------------------------- **
**     Copyright (c) 1993 by Tom Swan. All rights reserved.    **
\* ----------------------------------------------------------- */

#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include <stdlib.h>

typedef unsigned char Byte;

#define FALSE 0
#define TRUE 1

// State-machine definitions
#define READING 0    // General reading mode
#define ENCODING 1   // Encoding same-color pixel runs
#define ABSMODE 2    // Encoding different-color pixel runs
#define SINGLE 3     // Encoding short absolute-mode runs
#define ENDOFLINE 4  // End of scan line detected

void Error(const char *msg);
void CompressFile(const char *fname);
int Odd(int v);
Byte *NextScanLine(ifstream &ifs, int np);
void PutByte(Byte b);
void PackRLE8(int np, const Byte *sl);

int main(int argc, char *argv[])
{
  if (argc <= 1)
    Error("filename missing");
  CompressFile(argv[1]);
  return 0;
}

// Display error message and halt
void Error(const char *msg)
{
  cerr << endl << "Error: " << msg << endl;
  exit(1);
}

// Compress test bitmap file fname
// Write compressed results to stdout
void CompressFile(const char *fname)
{
  int np, ns;  // Number of pixels per line, number of scan lines
  Byte *sl;    // Pointer to scan line

  ifstream ifs(fname, ios::in);
  if (!ifs)
    Error("unable to open file");
  ifs >> hex >> np >> ns;
  if ((np <= 0) || (ns <= 0))
    Error("bad file format");
  while (ns-- > 0) {
    sl = NextScanLine(ifs, np);
    PackRLE8(np, sl);
    delete sl;
  }
  PutByte(0);  // Mark end of bitmap
  PutByte(1);
  cout << endl;
}

// Return true if v is odd
int Odd(int v)
{
  return v & 0x01;
}

// Read next scan line of np pixels from file ifs
Byte *NextScanLine(ifstream &ifs, int np)
{
  if (np <= 0) Error("np zero or less in NextScanLine()");
  Byte *sl = new Byte[np];   // Allocate scan line
  if (!sl) Error("out of memory");
  int j = 0;
  int k;
  while (np-- > 0) {
    ifs >> hex >> k;
    sl[j++] = k;
  }
  return sl;
}

// Write byte b in hex in 2 columns with leading 0
// plus one blank to cout
void PutByte(Byte b)
{
  cout << setiosflags(ios::uppercase);
  cout << setw(2) << setfill('0') << hex << (int)b << ' ';
  cout << setfill(' ') << dec;
  cout << resetiosflags(ios::uppercase);
}

// Compress np pixels in sl
// Write compressed line to stdout
void PackRLE8(int np, const Byte *sl)
{
  int slx = 0;           // Scan line index
  int state = READING;   // State machine control variable
  int pixel, count;      // Used by various states
  int done = FALSE;      // Ends while loop when true
  int oldcount, oldslx;  // Copies of count and slx

  while (!done) {

    switch (state) {

      case READING:
      // Input:
      // np == number of pixels in scan line
      // sl == scan line
      // sl[slx] == next pixel to process

        if (slx > np) Error("READING in PackRLE8()");

        if (slx >= np)                      // No pixels left
          state = ENDOFLINE;
        else if (slx == np - 1) {           // One pixel left
          count = 1;
          state = SINGLE;
        } else if (sl[slx] == sl[slx + 1])  // Next 2 pixels equal
          state = ENCODING;
        else                                // Next 2 pixels differ
          state = ABSMODE;
        break;

      case ENCODING:
      // Input: 
      // slx <= np - 2 (at least 2 pixels in run)
      // sl[slx] == first pixel of run
      // sl[slx] == sl[slx + 1]

        count = 2;
        pixel = sl[slx];
        slx += 2;
        while ((slx < np) && (pixel == sl[slx]) && (count < 255)) {
          count++;
          slx++;
        }
        PutByte(count);  // Output run-length-encoded unit
        PutByte(pixel);
        state = READING;
        break;

      case ABSMODE:
      // Input:
      // slx <= np - 2 (at least 2 pixels in run)
      // sl[slx] == first pixel of run
      // sl[slx] != sl[slx + 1]

        oldslx = slx;
        count = 2;
        slx += 2;
        // Compute number of bytes in run
        while ((slx < np) && (sl[slx] != sl[slx - 1]) && (count < 255)) {
          count++;
          slx++;
        }
        // If same-color run found, back up one byte
        if ((slx < np) && (sl[slx] == sl[slx - 1]))
          if (count > 1) count--;
        slx = oldslx;  // Restore scan-line index
        // Output short absolute runs of less than 3 pixels
        if (count < 3 )
          state = SINGLE;
        else {
          // Output absolute-mode run
          PutByte(0);
          PutByte(count);
          oldcount = count;
          while (count > 0) {
            PutByte(sl[slx]);
            slx++;
            count--;
          }
          if (Odd(oldcount))
            PutByte(0);  // End run on word boundary
          state = READING;
        }
        break;

      case SINGLE:
      // Input:
      // count == number of pixels to output
      // slx < np
      // sl[slx] == first pixel of run
      // sl[slx] != sl[slx + 1]

      while (count > 0) {
        PutByte(01);
        PutByte(sl[slx]);
        slx++;
        count--;
      }
      state = READING;
      break;

      case ENDOFLINE:
        PutByte(0);
        PutByte(0);
        done = TRUE;
        break;

      default:
        Error("unknown state in PackRLE8()");
        break;
    }
  }
}


// --------------------------------------------------------------
// Copyright (c) 1993 by Tom Swan. All rights reserved.
// Revision 1.00    Date: 04/27/1993   Time: 09:07 am
