/* ----------------------------------------------------------- *\
**  tunpack.cpp -- Test unpacking compressed bitmap data       **
** ----------------------------------------------------------- **
**                                                             **
**  Note: Delta escape-code unpacking is not implemented.      **
**                                                             **
** ----------------------------------------------------------- **
**     Copyright (c) 1993 by Tom Swan. All rights reserved.    **
\* ----------------------------------------------------------- */

#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include <stdlib.h>

typedef unsigned char Byte;

#define FALSE 0
#define TRUE 1

// Compression escape codes
#define ESCAPE 0
#define ENDOFLINE 0
#define ENDOFBITMAP 1
#define DELTA 2

void Error(const char *msg);
void DecompressFile(const char *fname);
int Odd(int v);
int GetByte(ifstream &ifs);
void PutByte(unsigned char b);

int main(int argc, char *argv[])
{
  if (argc <= 1)
    Error("filename missing");
  DecompressFile(argv[1]);
  return 0;
}

// Display error message and halt
void Error(const char *msg)
{
  cerr << endl << "Error: " << msg << endl;
  exit(1);
}

// Decompress fake bitmap file fname
// Write decompressed results to stdout
void DecompressFile(const char *fname)
{
  int endOfBitmap;  // True if 00 01 found at end of input
  int count;        // Used at various locations in function
  int oldCount;     // Copy of count for absolute-mode units
  unsigned byte;    // Input bytes

  ifstream ifs(fname, ios::in);
  if (!ifs)
    Error("unable to open file");
  endOfBitmap = FALSE;
  while (!endOfBitmap && !ifs.eof()) {
    byte = GetByte(ifs);
    if (byte == ESCAPE) {
    // Unpack escape code unit
      byte = GetByte(ifs);
      switch (byte) {
        case ENDOFLINE:
          cout << endl;
          break;
        case ENDOFBITMAP:
          endOfBitmap = TRUE;
          cout << endl;
          break;
        case DELTA:
          Error("Delta escape codes not implemented");
          break;
        default:  // Absolute-mode run
          count = byte;
          oldCount = count;
          while (count > 0) {
            byte = GetByte(ifs);
            PutByte(byte);
            count--;
          }
          if (Odd(oldCount))
            byte = GetByte(ifs);  // Read word-boundary padding byte
          break;
      }
    } else {
    // Unpack run-length-encoded unit
      count = byte;
      byte = GetByte(ifs);
      while (count > 0) {
        PutByte(byte);
        count--;
      }
    }
  }
  if (!endOfBitmap) 
    Error("End of bitmap marker (00 01) not found");
}

// Return next byte from input file ifs
// Count number of bytes read
int GetByte(ifstream &ifs)
{
  int b;
  ifs >> hex >> b;
  return b;
}

// Return true if v is odd
int Odd(int v)
{
  return v & 0x01;
}

// Write byte b in hex in 2 columns with leading 0
// plus one blank to cout
void PutByte(unsigned char b)
{
  cout << setiosflags(ios::uppercase);
  cout << setw(2) << setfill('0') << hex << (int)b << ' ';
  cout << setfill(' ') << dec;
  cout << resetiosflags(ios::uppercase);
}


// --------------------------------------------------------------
// Copyright (c) 1993 by Tom Swan. All rights reserved.
// Revision 1.00    Date: 04/27/1993   Time: 08:55 am
