/* ----------------------------------------------------------- *\
**  bpack.cpp -- Pack (compress) a Windows bitmap file         **
** ----------------------------------------------------------- **
**                                                             **
** ----------------------------------------------------------- **
**     Copyright (c) 1993 by Tom Swan. All rights reserved.    **
\* ----------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <alloc.h>
#include <windows.h>

// Miscellaneous definitions
#define FALSE 0
#define TRUE 1

// State-machine definitions
#define READING 0         // General reading mode
#define ENCODING 1        // Encoding same-color pixel runs
#define ABSMODE 2         // Absolute-mode encoding
#define SINGLE 3          // Encoding short absolute-mode runs
#define ENDOFLINE 4       // End of scan line detected

// Type declarations
typedef unsigned char Byte;

struct BitmapStruct {
  BITMAPFILEHEADER bfh;   // Bitmap file header
  BITMAPINFOHEADER bih;   // Bitmap info header
  RGBQUAD *bmiColors;     // Pointer to color table (variable size)
  int clrSize;            // Number of colors in table
};

// Function prototypes
void Instruct();
void Error(char *msg);
int Odd(int v);
void ReadBitmapHeader(FILE *inpf, BitmapStruct &rbs);
int IsCompressible(BitmapStruct &rbs);
void WriteBitmapHeader(FILE *outf, BitmapStruct &rbs);
void WriteBitmapBits(FILE *inpf, FILE *outf, const BitmapStruct &rbs);
void PackRLE8(FILE *outf, int np, const Byte *sl);
void PutByte(FILE *outf, Byte b);

// Global variable
long imageSize;

int main(int argc, char *argv[])
{
  BitmapStruct bs;

  puts("\nBitmap file compressor by Tom Swan");
  if (argc <= 2) Instruct();
  FILE *inpf = fopen(argv[1], "rb");
  if (!inpf) Error("Can't open input file");
  ReadBitmapHeader(inpf, bs);
  if (!IsCompressible(bs)) Error("Cannot compress this file");
  FILE *outf = fopen(argv[2], "wb");
  if (!outf) Error("Can't open output file");
  printf("Compressing file...");
  WriteBitmapHeader(outf, bs);      // Write dummy header
  WriteBitmapBits(inpf, outf, bs);  // Compress bitmap image
  bs.bih.biCompression = BI_RLE8;   // Mark bitmap as compressed
  bs.bih.biSizeImage = imageSize;   // Modify image size value
  fseek(outf, 0, SEEK_END);         // Seek to eof for next statement
  bs.bfh.bfSize = ftell(outf);      // Modify file size in bytes
  WriteBitmapHeader(outf, bs);      // Write real header
  fclose(inpf);
  fclose(outf);
  printf("\n%s --> %s\n", argv[1], argv[2]);
  delete bs.bmiColors;
  return 0;
}

// Display instructions and exit program
void Instruct()
{
  puts("\nSyntax: BPACK infile outfile");
  puts("\nEnter the name of a bitmap (infile) to compress.");
  puts("The program packs the bitmap if possible, and");
  printf("stores the results in a new file (outfile).\n");
  puts("The original bitmap file is not changed in any way.");
  puts("This version is limited to 8-bit (256-color) files");
  puts("that are not already compressed.");
  exit(0);
}

// Display error message and exit program
void Error(char *msg)
{
  printf("\nERROR: %s\n", msg);
  exit(1);
}

// Return true if v is odd
int Odd(int v)
{
  return v & 0x01;
}

// Read bitmap headers and color table into rbs
void ReadBitmapHeader(FILE *inpf, BitmapStruct &rbs)
{
  if (fread(&rbs.bfh, sizeof(rbs.bfh), 1, inpf) != 1)
    Error("Cannot read bitmap file header");
  if (fread(&rbs.bih, sizeof(rbs.bih), 1, inpf) != 1)
    Error("Cannot read bitmap info header");
  if (rbs.bih.biClrUsed != 0) {
    rbs.clrSize = (int)rbs.bih.biClrUsed;
  } else switch (rbs.bih.biBitCount) {
    case 1: rbs.clrSize = 2; break;
    case 4: rbs.clrSize = 16; break;
    case 8: rbs.clrSize = 256; break;
    case 24: rbs.clrSize = 0; break;
    default: Error("biBitCount not 1, 4, 8, or 24");
  }
  if (rbs.clrSize == 0) Error("clrSize == 0");
  rbs.bmiColors = new RGBQUAD[rbs.clrSize];
  if (!rbs.bmiColors) 
    Error("bmiColors is null. Out of memory.");
  if (fread(rbs.bmiColors, sizeof(RGBQUAD), 
    rbs.clrSize, inpf) != rbs.clrSize)
    Error("Cannot read color table");
}

// Returns true if bitmap header rbs is compressible
// Required format: MS DIB, uncompressed, 8-bit (256-color)
int IsCompressible(BitmapStruct &rbs)
{
  if (rbs.bfh.bfType != 0x4d42) return FALSE;
  if (rbs.bih.biSize != sizeof(BITMAPINFOHEADER)) return FALSE;
  if (rbs.bih.biBitCount != 8) return FALSE;
  if (rbs.bih.biCompression != BI_RGB) return FALSE;
  return TRUE;
}

// Write bitmap headers and color table in bs to outf
void WriteBitmapHeader(FILE *outf, BitmapStruct &rbs)
{
  rewind(outf);
  if (fwrite(&rbs.bfh, sizeof(rbs.bfh), 1, outf) != 1)
    Error("writing bitmap file header");
  if (fwrite(&rbs.bih, sizeof(rbs.bih), 1, outf) != 1)
    Error("writing bitmap info header");
  if (fwrite(rbs.bmiColors, sizeof(RGBQUAD), 
    rbs.clrSize, outf) != rbs.clrSize)
    Error("writing color table");
}

// Read pixel data from inf, compress and write to outf
void WriteBitmapBits(FILE *inpf, FILE *outf, const BitmapStruct &rbs)
{
  int np;      // Number of pixels per scan line
  int ns;      // Number of scan lines
  int slSize;  // Size of one scan line in bytes
  Byte *sl;    // Pointer to scan line

  // Assign miscellaneous sizes
  np = (int)rbs.bih.biWidth;
  ns = (int)rbs.bih.biHeight;

  // Allocate scan line buffer
  if (Odd(np)) 
    slSize = np + 1;  // Must have an even number of bytes
  else
    slSize = np;
  if (slSize <= 0) Error("slSize <= 0");
  sl = new Byte[slSize];
  if (!sl) Error("out of memory");

  // Read and compress scan lines
  while (ns-- > 0) {
    if (fread(sl, 1, slSize, inpf) != slSize)
      Error("reading pixel scan line");
    PackRLE8(outf, np, sl);
  }
  delete sl;
  PutByte(outf, 0);  // Mark end of bitmap
  PutByte(outf, 1);
}

// Compress and write np pixels in sl to output file outf
void PackRLE8(FILE *outf, int np, const Byte *sl)
{
  int slx = 0;           // Scan line index
  int state = READING;   // State machine control variable
  int count;             // Used by various states
  Byte pixel;            // Holds single pixels from sl
  int done = FALSE;      // Ends while loop when true
  int oldcount, oldslx;  // Copies of count and slx

  while (!done) {

    switch (state) {

      case READING:
      // Input:
      // np == number of pixels in scan line
      // sl == scan line
      // sl[slx] == next pixel to process

        if (slx >= np)                      // No pixels left
          state = ENDOFLINE;
        else if (slx == np - 1) {           // One pixel left
          count = 1;
          state = SINGLE;
        } else if (sl[slx] == sl[slx + 1])  // Next 2 pixels equal
          state = ENCODING;
        else                                // Next 2 pixels differ
          state = ABSMODE;
        break;

      case ENCODING:
      // Input: 
      // slx <= np - 2 (at least 2 pixels in run)
      // sl[slx] == first pixel of run
      // sl[slx] == sl[slx + 1]

        count = 2;
        pixel = sl[slx];
        slx += 2;
        while ((slx < np) && (pixel == sl[slx]) && (count < 255)) {
          count++;
          slx++;
        }
        PutByte(outf, (Byte)count);  // Output run-length-encoded unit
        PutByte(outf, pixel);
        state = READING;
        break;

      case ABSMODE:
      // Input:
      // slx <= np - 2 (at least 2 pixels in run)
      // sl[slx] == first pixel of run
      // sl[slx] != sl[slx + 1]

        oldslx = slx;
        count = 2;
        slx += 2;
        // Compute number of bytes in run
        while ((slx < np) && (sl[slx] != sl[slx - 1]) && (count < 255)) {
          count++;
          slx++;
        }
        // If same-color run found, back up one byte
        if ((slx < np) && (sl[slx] == sl[slx - 1]))
          if (count > 1) 
            count--;
        slx = oldslx;  // Restore scan-line index
        // Output short absolute runs of less than 3 pixels
        if (count < 3 )
          state = SINGLE;
        else {
          // Output absolute-mode run
          PutByte(outf, 0);
          PutByte(outf, (Byte)count);
          oldcount = count;
          while (count > 0) {
            PutByte(outf, sl[slx]);
            slx++;
            count--;
          }
          if (Odd(oldcount))
            PutByte(outf, 0);  // End run on word boundary
          state = READING;
        }
        break;

      case SINGLE:
      // Input:
      // count == number of pixels to output
      // slx < np
      // sl[slx] == first pixel of run
      // sl[slx] != sl[slx + 1]

      while (count > 0) {
        PutByte(outf, 01);
        PutByte(outf, sl[slx]);
        slx++;
        count--;
      }
      state = READING;
      break;

      case ENDOFLINE:
        PutByte(outf, 0);
        PutByte(outf, 0);
        done = TRUE;
        break;

      default:
        Error("unknown state in PackRLE8()");
        break;
    }
  }
}

// Write byte b to output file outf
// Increments global imageSize variable
void PutByte(FILE *outf, Byte b)
{
  if (fwrite(&b, 1, 1, outf) != 1)
    Error("writing byte to output file");
  imageSize++;
}


// --------------------------------------------------------------
// Copyright (c) 1993 by Tom Swan. All rights reserved.
// Revision 1.00    Date: 04/21/1993   Time: 08:02 pm
