/* Extended IEEE Compatible Floating Point Arithmetic Library
**
** Version 1.1
** Copyright (C) 1990, 1992 by Fred Motteler
** All Rights Reserved
**
** This is collection of numeric conversion routines.  These functions
** support arbitrary precision integer to float, float to integer
** and float to float conversions.
**
** Each routine that returns a floating point result returns a condition
** code value:
**
**	0 0 0 0 N Z I n
**	Where N = negative (FFNEG)
**	      Z = zero (FFSIGN)
**	      I = infinite (FFINF)
**	      n = not a number (FFNAN)
**
** Each routine that returns a integer result returns a condition code
** value:
**
**	0 0 0 0 Z V S C
**	Where Z = zero (ZERO)
**	      V = overflow (OVERFLOW)
**	      S = sign (SIGN)
**	      C = carry (CARRY)
**
** Note that these routines recognize zero and infinite input values.  Not a
** number and denormalized input parameters are not recognized.  The routines
** blindly assume that the arguments are valid floating point numbers.
**
** These routines will return valid zero, infinite, and not a number values.
*/
#include <stdio.h>
#ifndef MWC
#include <stdlib.h>
#endif
#include "imlib.h"
#include "ffmlib.h"
#include "fmlib.h"

#ifdef TESTM
#define SINGLEXP 8
#define SINGLEFRAC 23
#define DOUBLEXP 11
#define DOUBLEFRAC 52
#define EXTENDEXP 15
#define EXTENDFRAC 63
unsigned char intval1AB[4] = { 0x0, 0x0, 0x12, 0x34};
unsigned char intval2AB[4] = { 0x0, 0x0, 0x12, 0x34};
unsigned char fltvalAB[4];
unsigned char dblvalAB[8];
unsigned char intval1B = 10;
unsigned char intval2B = 1;

extern unsigned char intoflt();
extern unsigned char fltoint();
extern unsigned char fltoflt();

void
main()
{
    printf("LONG TO SINGLE PRECISION:\n");
    intoflt( intval1AB, 4, fltvalAB, SINGLEFRAC, SINGLEXP );
    printf("\nLONG TO DOUBLE PRECISION:\n");
    intoflt( intval2AB, 4, dblvalAB, DOUBLEFRAC, DOUBLEXP );

    printf("\nSINGLE PRECISION TO LONG:\n");
    fltoint( fltvalAB, SINGLEFRAC, SINGLEXP, intval1AB, 4 );
    printf("\nDOUBLE PRECISION TO LONG:\n");
    fltoint( dblvalAB, DOUBLEFRAC, DOUBLEXP, intval2AB, 4 );

    printf("\nSINGLE PRECISION TO DOUBLE PRECISION:\n");
    fltoflt( fltvalAB, SINGLEFRAC, SINGLEXP, dblvalAB, DOUBLEFRAC, DOUBLEXP );
    printf("\nDOUBLE PRECISION TO SINGLE PRECISION:\n");
    fltoflt( dblvalAB, DOUBLEFRAC, DOUBLEXP, fltvalAB, SINGLEFRAC, SINGLEXP );

    printf("\nBYTE TO SINGLE PRECISION:\n");
    intoflt( &intval1B, 1, fltvalAB, SINGLEFRAC, SINGLEXP );
    printf("\nBYTE TO DOUBLE PRECISION:\n");
    intoflt( &intval2B, 1, dblvalAB, DOUBLEFRAC, DOUBLEXP );
}
#endif

/* Function:	intoflt(unsigned char *intvalBP, int intlenN,
**			unsigned char *fltvalBP, int fracbitN,
**			int expbitN)
**
** This function converts the integer value pointed to by intvalBP to a
** floating point value pointed to by fltvalBP.  intlenN gives the length
** of the integer value in bytes (1 = byte, 2 = word, 4 = long, etc...).
** fracbitN gives the length of the floating point mantissa field in bits.
** expbitN gives the length of the floating point exponent field in bits.
**
** This routine routine supports conversion of any size integer to any
** size floating point number.
*/
unsigned char
#ifdef PROTOTYPES
intoflt(unsigned char *intvalBP, int intlenN, unsigned char *fltvalBP,
	int fracbitN, int expbitN)
#else
intoflt(intvalBP, intlenN, fltvalBP, fracbitN, expbitN)
unsigned char *intvalBP;
int intlenN;
unsigned char *fltvalBP;
int fracbitN;
int expbitN;
#endif
{
    unsigned char condcodeB;
    int totalenN, explenN;
    int i, trialbitN;
    unsigned char *fracbufBP;
    unsigned char *expbufBP, *expbiasBP;

    /* Initialize the condition code byte to zero */
    condcodeB = 0;

    /* Determine the total byte length of the floating point number */
    totalenN = fftotlen(expbitN, fracbitN);

    /* Initialize the result float value to zero */
    for (i = 0; i < totalenN; i++)
	*(fltvalBP + i) = 0;

    /* Check if the input integer is zero, if so then just return with zero
     * condition code set */
    if (ucheckm(intvalBP, intlenN) == 0)
    {
	return(FFZERO);
    }

    /* Input integer is not zero, convert it to a float.  First determine the
     * sign of the input value and change the sign of negative values */
    if (((*intvalBP) & NSIGN) != 0)
    {
	inegm(intvalBP, intlenN);
	ffbitset(fltvalBP, totalenN, (expbitN + fracbitN));
	condcodeB |= FFNEG;
    }

    /* Determine how much of the integer to extract and where in the float
     * to write it out to.  This is done by shifting the integer left
     * until the first non-zero bit is encountered.  Start off with the
     * maximum fraction precision as the integer bit length minus one.
     * The one comes from ms bit being the sign bit. */
    trialbitN = (intlenN << 3) - 1;
    while (((*intvalBP) & NSIGN) == 0)
    {
	ushftlm(intvalBP, intlenN);
	trialbitN--;
    }
    ushftlm(intvalBP, intlenN);		/* Shift off the ms 1, is implied */

    /* The precision of the integer (in bits) is given by trialbitN.  The
     * maximum precision of the floating point number is given by fracbitN.
     * Use the smaller length as the number of bits to extract from the
     * integer and to insert into the floating point mantissa.  First
     * allocate space for a temporary buffer for the integer value converter
     * to mantissa format */
    fracbufBP = (unsigned char *) FCALLOC(intlenN, 1, "INTOFLT1");

    if (trialbitN > fracbitN)
    {
	/* The precision of the integer is greater than the precision of
	 * the floating point mantissa.  The least significant bits of
	 * the integer must be truncated.  Recall that the mantissa fraction
	 * is left hand justified and that the left most fracbitN bits are
	 * to be copied. */
	ffbitext(intvalBP, intlenN, ((intlenN << 3) - fracbitN),
		 fracbitN, fracbufBP, intlenN);
	ffbitins(fracbufBP, intlenN, 0, fracbitN, fltvalBP, totalenN);
    }
    else
    {
	/* The precision of the integer is less than the precision of
	 * the floating point mantissa.  The least significant bits of
	 * the floating point mantissa must be padded with zeros.  As
	 * before, the mantissa fraction is left hand justified and that
	 * the left most trialbitN bits are to be copied. */
	ffbitext(intvalBP, intlenN, ((intlenN << 3) - trialbitN),
		 trialbitN, fracbufBP, intlenN);
	ffbitins(fracbufBP, intlenN, (fracbitN - trialbitN), trialbitN,
		 fltvalBP, totalenN);
    }
    FFREE(fracbufBP);

    /* The value in trialbitN is the value of the exponent.  With normal
     * integer values (byte, word, long, and double long), it is not
     * possible to overflow even a single precision floating point value
     * exponent.  However, if for some reason, a really long integer value
     * is converted to a floating point value with a really small exponent
     * range, then an overflow will occur. */
    if (((long)(1L << (expbitN - 1))) < trialbitN)
    {
	return((unsigned char)(FFINF | condcodeB));
    }

    /* Determine the total byte length of the exponent, and allocate
     * buffer space for it. */
    explenN = ((expbitN % 8) == 0) ? 1 : 2;
    explenN += (expbitN >> 3);
    expbufBP = (unsigned char *) FMALLOC(explenN, "INTOFLT2");
    expbiasBP = (unsigned char *) FCALLOC(explenN, 1, "INTOFLT3");

    /* Write out the exponent value to temporary buffer in exponent format. */
    for (i = (explenN - 1); i >= 0; i--)
    {
	*(expbufBP + i) = (unsigned char) trialbitN;
	trialbitN >>= 8;
    }

    /* Add the exponent bias to the exponent value.  This requires that the
     * bias value be calculated first. */
    ffgenbias(expbiasBP, explenN, expbitN);

    /* Add on the exponent bias */
    iaddm(expbufBP, expbiasBP, explenN);

    /* Insert the exponent value into the proper place in the floating
     * point number */
    ffbitins(expbufBP, explenN, fracbitN, expbitN, fltvalBP, totalenN);
    FFREE(expbufBP);
    FFREE(expbiasBP);
    return(condcodeB);
}

/* Function:	fltoint(unsigned char *fltvalBP, int fracbitN, int expbitN,
**			unsigned char *intvalBP, int intlenN)
**
** This function converts the floating point value pointed to by fltvalBP
** to an integer value pointed to by intvalBP.  intlenN gives the length
** of the integer value in bytes (1 = byte, 2 = word, 4 = long, etc...).
** fracbitN gives the length of the floating point mantissa field in bits.
** expbitN gives the length of the floating point exponent field in bits.
**
** This routine routine supports conversion of any size floating point
** number to any size integer number.
*/
unsigned char
#ifdef PROTOTYPES
fltoint(unsigned char *fltvalBP, int fracbitN, int expbitN,
	unsigned char *intvalBP, int intlenN)
#else
fltoint(fltvalBP, fracbitN, expbitN, intvalBP, intlenN)
unsigned char *fltvalBP;
int fracbitN;
int expbitN;
unsigned char *intvalBP;
int intlenN;
#endif
{
    int expbyteN, fracbyteN;
    unsigned char *valexpBP, *valfracBP;
    unsigned char *expbiasPB, *exponePB;
    long expvalL;
    unsigned char valsignB;
    int totalenN;
    int intbitsN;
    int i;

    /* Initialize the integer result area to zero */
    for (i = 0; i < intlenN; i++)
	*(intvalBP + i) = 0;

    /* Determine the total byte length of the floating point number */
    totalenN = fftotlen(expbitN, fracbitN);

    /* Check if the float is zero.  If so, then just return zero. */
    if (ffchkzero(fltvalBP, totalenN) == 0)
    {
	return(ZERO);
    }

    /* Determine the number of bytes required to hold the mantissa and
     * exponent.  Allocate space for each... */
    expbyteN = ffexplen(expbitN);
    fracbyteN = ffraclen(fracbitN);

    valfracBP = (unsigned char *) FCALLOC(fracbyteN, 1, "FLTOINT1");
    valexpBP = (unsigned char *) FMALLOC(expbyteN, "FLTOINT2");
    expbiasPB = (unsigned char *) FCALLOC(expbyteN, 1, "FLTOINT3");
    exponePB = (unsigned char *) FCALLOC(expbyteN, 1, "FLTOINT4");

    /* Isolate the mantissas, exponents, and signs.  Calling ffextall()
     * also sets the implied ms bit of the mantissa to 1. */
    ffextall(fltvalBP, totalenN, fracbitN, fracbyteN, expbitN, expbyteN,
	     valfracBP, valexpBP, &valsignB);

    *(exponePB + (expbyteN - 1)) = 1;

    /* Calculate the exponent bias and subtract it first. */
    ffgenbias(expbiasPB, expbyteN, expbitN);

    /* Subtract off exponent bias, note that if the result is negative, the
     * sign is properly extended.  This is important since it allows
     * exponent overflow and underflow to be detected much more easily. */
    if ((isubm(valexpBP, expbiasPB, expbyteN) & SIGN) != 0)
    {
	/* Return zero value if the exponent is zero. */
	FFREE(valfracBP);
	FFREE(valexpBP);
	FFREE(expbiasPB);
	FFREE(exponePB);
	return(ZERO);
    }

    /* Convert the exponent value to an integer */
    expvalL = 0L;
    for (i = 0; i < expbyteN; i++)
    {
	expvalL <<= 8;
	expvalL += (unsigned long) (*(valexpBP + i));
    }

    if (expvalL == 0L)
    {
	if (valsignB == 0)
	{
	    *(intvalBP + intlenN - 1) = (unsigned char) 1;
	    FFREE(valfracBP);
	    FFREE(valexpBP);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return(0);
	}
	else
	{
	    for (i = 0; i < intlenN; i++)
		*(intvalBP + i) = (unsigned char) 0xff;
	    FFREE(valfracBP);
	    FFREE(valexpBP);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return(SIGN);
	}
    }
    /* Check if the exponent is too big for the float to fit into the
     * integer value. */
    intbitsN = (intlenN << 3) - 2;	/* Maximum exponent for no overflow. */
    if (expvalL > ((long) intbitsN))
    {
	/* Overflow */
	if (valsignB == 0)
	{
	    for (i = 1; i < intlenN; i++)
		*(intvalBP + i) = (unsigned char) 0xff;
	    *intvalBP = (unsigned char)0x7f;
	    FFREE(valfracBP);
	    FFREE(valexpBP);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return(OVERFLOW);
	}
	else
	{
	    *intvalBP = (unsigned char) 0x80;
	    FFREE(valfracBP);
	    FFREE(valexpBP);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return(OVERFLOW | SIGN);
	}
    }
    /* Extract the appropriate integer value from the mantissa.  The number of
     * bits to extract should be given by expvalL + 2.  Note however, that the
     * number of bits in the mantissa (fracbitN + 1) may be less than the
     * number of bits to extract.  In this case the resulting integer value
     * must be padded with zero bits. */
    if (fracbitN < (((int)expvalL) + 1))
    {
	/* The mantissa is too short, grab the whole mantissa and shift it
	 * left a few times to zero pad the resulting integer value. */
	ffbitext( valfracBP, fracbyteN, 0, (fracbitN + 1), intvalBP, intlenN );
	for (i = 0; i < (((int) expvalL) - fracbitN); i++)
	    ushftlm(intvalBP, intlenN);
    }
    else
    {
	/* Grab expvalL + 1 bits from the mantissa */
	ffbitext(valfracBP, fracbyteN, (fracbitN - ((int) expvalL)),
		  (int) (expvalL + 1), intvalBP, intlenN);
    }

    if (valsignB != 0)
    {
	inegm(intvalBP, intlenN);
	FFREE(valfracBP);
	FFREE(valexpBP);
	FFREE(expbiasPB);
	FFREE(exponePB);
	return(SIGN);
    }
    FFREE(valfracBP);
    FFREE(valexpBP);
    FFREE(expbiasPB);
    FFREE(exponePB);
    return(0);
}

/* Function:	fltoflt(unsigned char *fltinBP, int mantinN, int expinN,
**			unsigned *fltoutBP, int mantoutN, int expoutN)
**
** This function converts the floating point value pointed to by fltinBP to
** a different precision floating point value pointed to by fltoutBP.
** mantinN and expinN give the input floating point value mantissa length
** and exponent lengths in bits.  mantoutN and expoutN give the output
** floating point value mantissa length and exponent lengths in bits.
*/
unsigned char
#ifdef PROTOTYPES
fltoflt(unsigned char *fltinBP, int mantinN, int expinN,
	unsigned char *fltoutBP, int mantoutN, int expoutN)
#else
fltoflt(fltinBP, mantinN, expinN, fltoutBP, mantoutN, expoutN)
unsigned char *fltinBP;		/* Pointer to input float value */
int mantinN;			/* Length of input mantissa in bits */
int expinN;			/* Length of input exponent in bits */
unsigned char *fltoutBP;	/* Pointer to output float value */
int mantoutN;			/* Length of output mantissa in bits */
int expoutN;			/* Length of output mantissa in bits */
#endif
{
    int expbinN, mantbinN;	/* Lengths in bytes */
    int expboutN, mantboutN; /* Lengths in bytes */
    unsigned char *expinBP, *mantiBP, *expoBP;
    unsigned char *biasinPB, *biasoutPB;
    unsigned char condcodeB;
    long expvalL;
    long maxoutL;
    long minoutL;
    unsigned char valsignB, expsignB;
    int totalinN, totaloutN;
    int i;
#ifdef MWC
    unsigned char tempB;
    unsigned char tempL;
#endif

    /* Determine the total byte length of the floating point numbers */
    totalinN = fftotlen(expinN, mantinN);
    totaloutN = fftotlen(expoutN, mantoutN);

   /* Initialize the float result area to zero */
    for (i = 0; i < totaloutN; i++)
	*(fltoutBP + i) = 0;
    condcodeB = 0;

    /* Check if the float is zero.  If so, then just return zero. */
    if (ffchkzero(fltinBP, totalinN) == 0)
    {
	return(FFZERO);
    }

    /* Determine the number of bytes required to hold the mantissas and
     * exponents.  Allocate space for each... */
    expbinN = ffexplen(expinN) + 1;
    mantbinN = ffraclen(mantinN);

    expboutN = ffexplen(expoutN) + 1;
    mantboutN = ffraclen(mantoutN);

    mantiBP = (unsigned char *) FCALLOC(mantbinN, 1, "FLTOFLT1");
    expinBP = (unsigned char *) FMALLOC(expbinN, "FLTOFLT2");
    biasinPB = (unsigned char *) FCALLOC(expbinN, 1, "FLTOFLT3");
    expoBP = (unsigned char *) FMALLOC(expboutN, "FLTOFLT4");
    biasoutPB = (unsigned char *) FCALLOC(expboutN, 1, "FLTOFLT5");

    /* Isolate the mantissas, exponents, and signs.  Calling ffextall()
     * also sets the implied ms bit of the mantissa to 1. */
    ffextall(fltinBP, totalinN, mantinN, mantbinN, expinN, expbinN,
	     mantiBP, expinBP, &valsignB);

    /* Calculate the input float's exponent bias and subtract it first. */
    *(biasinPB + (expbinN - 1)) = 1;	/* Set the ls bit to 1 */
    for (i = 0; i < (expinN - 2); i++)
    {
	ushftlm(biasinPB, expbinN);
	*(biasinPB + (expbinN - 1)) |= 1;	/* Set next ls bit to 1 */
    }

    /* Subtract off exponent bias, note that if the result is negative, the
     * sign is properly extended.  This is important since it allows
     * exponent overflow and underflow to be detected much more easily. */
    expsignB = isubm(expinBP, biasinPB, expbinN);

    /* Convert the exponent value to an long.  Note that this restricts
     * the maximum exponent size to be 31 bits. */
    expvalL = 0;
    /* Make sure that the sign of the value is positive. */
    if ((expsignB & SIGN) != 0)
	inegm( expinBP, expbinN );
    for (i = 0; i < expbinN; i++)
    {
	expvalL <<= 8;
#ifdef MWC
	tempB = *(expinBP + i);
	tempL = tempB;
	expvalL += tempL;
#else
	expvalL += (unsigned char) (*(expinBP + i));
#endif
    }
    /* Fix the sign if necessary. */
    if ((expsignB & SIGN) != 0)
	expvalL = (-expvalL);

    /* Calculate the output float's exponent bias, in both byte array format
     * and also in long format. */
    *(biasoutPB + (expboutN - 1)) = 1;	/* Set the ls bit to 1 */
    maxoutL = 1;
    for (i = 0; i < (expoutN - 2); i++)
    {
	ushftlm(biasoutPB, expboutN);
	*(biasoutPB + (expboutN - 1)) |= 1;	/* Set next ls bit to 1 */
	maxoutL <<= 1;
	maxoutL |= 1;
    }
    minoutL = 1 - maxoutL;

    /* Check if the input float's exponent is too big (or too small) to fit
     * into the output float's exponent. */
    if (expvalL > maxoutL)
    {
	/* Overflow, set the result to infinity.  This is done by shifting
	 * the output exponent bias left once and setting the least
	 * significant bit to 1.  The result is then written out the
	 * result's exponent field. */
	ushftlm(biasoutPB, expboutN);
	*(biasoutPB + (expboutN - 1)) |= ((unsigned char) 1);
	ffbitins(biasoutPB, expboutN, mantoutN, expoutN, fltoutBP,
		 totaloutN);
	/* Set the sign bit of the result. */
	if (valsignB != 0)
	{
	    ffbitset(fltoutBP, totaloutN, (expoutN + mantoutN));
	    condcodeB |= FFNEG;
	}

	FFREE(mantiBP);
	FFREE(expinBP);
	FFREE(biasinPB);
	FFREE(expoBP);
	FFREE(biasoutPB);
	return((unsigned char) (condcodeB | FFINF));
    }
    if (expvalL < minoutL)
    {
	/* Set the sign bit of the result. */
	if (valsignB != 0)
	{
	    ffbitset(fltoutBP, totaloutN, (expoutN + mantoutN));
	    condcodeB |= FFNEG;
	}
	FFREE(mantiBP);
	FFREE(expinBP);
	FFREE(biasinPB);
	FFREE(expoBP);
	FFREE(biasoutPB);
	return((unsigned char) (condcodeB | FFZERO));
    }
    /* Write out the exponent value to temporary buffer in exponent format. */
    for (i = (expboutN - 1); i >= 0; i--)
    {
	*(expoBP + i) = (unsigned char) expvalL;
	expvalL >>= 8;
    }

    /* Add the exponent bias to the output exponent value. */
    iaddm(expoBP, biasoutPB, expboutN);

    /* Extract input mantissa.  If the output mantissa has more bits of
     * significance than the input mantissa, then the extra output mantissa
     * bits are padded with zeros.  If the output mantissa has fewer bits
     * of significance than the input mantissa, then the extra input
     * mantissa bits are truncated. */
    if (mantinN <= mantoutN)
    {
	/* The input mantissa is shorter or equal in length to the output
	 * mantissa, grab the whole input mantissa and insert it into the
	 * proper location in the output mantissa. */
	ffbitins(mantiBP, mantbinN, (mantoutN - mantinN), mantinN,
		 fltoutBP, totaloutN);
    }
    else
    {
	/* The input mantissa is longer in length than the output
	 * mantissa, extract the most significant part of the input mantissa
	 * and copy it to the output mantissa. */
	ffbitext(mantiBP, mantbinN, (mantinN - mantoutN),
		 mantoutN, fltoutBP, totaloutN);
    }

    /* Insert the exponent value into the proper place in the floating
     * point number */
    ffbitins(expoBP, expboutN, mantoutN, expoutN, fltoutBP, totaloutN);

    /* Set the sign bit of the result. */
    if (valsignB != 0)
    {
	ffbitset(fltoutBP, totaloutN, (expoutN + mantoutN));
	condcodeB |= FFNEG;
    }

    FFREE(mantiBP);
    FFREE(expinBP);
    FFREE(biasinPB);
    FFREE(expoBP);
    FFREE(biasoutPB);
    return(condcodeB);
}
