/* Extended IEEE Compatible Floating Point Arithmetic Library
**
** Version 1.1
** Copyright (C) 1990, 1992 by Fred Motteler
** All Rights Reserved
**
** This is a simple extended floating point arithmetic package.
** It must be used with the extended integer arithmetic package.
**
** Format:
**
** |S| n bit signed exponent | m bit fraction |
**
** The format is compatible with IEEE Standard for Binary Floating
** when:
**		n = 8,   m = 23		single precision
**		n = 11,  m = 52		double precision
**
** If n = 15 and m = 64, this format is almost compatible with IEEE
** extended precision format.  The main difference are
**	1) IEEE extended precision format is a 12 byte format with 10 bytes of
**	   information.  Bits 64 thru 80 are zero.
**	2) The most significant bit of the mantissa (implied in single and
**	   double precision IEEE formats) is explicitly included.
**
** Each routine returns a condition code value:
**
**	0 0 0 0 N Z I n
**	Where N = negative
**	      Z = zero
**	      I = infinite
**	      n = not a number
**
** Note that these routines recognize zero and infinite input values. Not a
** number and denormalized input values are not handled properly.  The
** routines blindly assume that the arguments are valid floating point numbers.
**
** These routines will return valid zero, infinite, and not a number values.
*/
#include <stdio.h>
#ifndef MWC
#include <stdlib.h>
#endif
#include "imlib.h"
#include "ffmlib.h"
#include "fmlib.h"

/* Lookup table for bit masks to clear extra high order bits */
unsigned char ffclearmaskAB[] =
    { 0xff, 0x1, 0x3, 0x7, 0xf, 0x1f, 0x3f, 0x7f };

/* Lookup table for bit masks to set/clear/test specific bits */
unsigned char ffsetmaskAB[] =
    { 0x1, 0x2, 0x4, 0x8, 0x10, 0x20, 0x40, 0x80 };
unsigned char ffclrmaskAB[] =
    { 0xfe, 0xfd, 0xfb, 0xf7, 0xef, 0xdf, 0xbf, 0x7f };

#ifdef TEST
#define SINGLEXP 8
#define SINGLEFRAC 23
#define DOUBLEXP 11
#define DOUBLEFRAC 52
#define EXTENDEXP 15
#define EXTENDFRAC 63

/* Single precision test operands */
unsigned char soper1AB[] =
    { 0x41, 0x7f, 0x12, 0x34 };
unsigned char soper2AB[] =
    { 0xb0, 0x84, 0xef, 0xcd };
unsigned char soper3AB[] =
    { 0x41, 0x12, 0x34, 0x56 };

/* Double precision test operands */
unsigned char doper1AB[] =
    { 0x20, 0x17, 0xf1, 0x23, 0x45, 0x67, 0x89, 0xab };
unsigned char doper2AB[] =
    { 0xde, 0x88, 0x4e, 0xfc, 0xda, 0xb8, 0x96, 0x75 };
unsigned char doper3AB[] =
    { 0x20, 0x17, 0xf0, 0x12, 0x34, 0x56, 0x78, 0x9a };

int
main()
{
    printf("SINGLE PRECISION MULTIPLY TEST:\n");
    fmultm(&(soper1AB[0]), &(soper2AB[0]), SINGLEXP, SINGLEFRAC);
    printf("\nSINGLE PRECISION DIVIDE TEST:\n");
    fdivm(&(soper1AB[0]), &(soper2AB[0]), SINGLEXP, SINGLEFRAC);
    printf("\nSINGLE PRECISION ADDITION TEST:\n");
    faddm(&(soper1AB[0]), &(soper3AB[0]), SINGLEXP, SINGLEFRAC);
    printf("\n\nDOUBLE PRECISION MULTIPLY TEST:\n");
    fmultm(&(doper1AB[0]), &(doper2AB[0]), DOUBLEXP, DOUBLEFRAC);
    printf("\nDOUBLE PRECISION DIVIDE TEST:\n");
    fdivm(&(doper1AB[0]), &(doper2AB[0]), DOUBLEXP, DOUBLEFRAC);
    printf("\nDOUBLE PRECISION ADDITION TEST:\n");
    faddm(&(doper1AB[0]), &(doper3AB[0]), DOUBLEXP, DOUBLEFRAC);
    return;
}
#endif

/*
** Function:	unsigned char fmultm(unsigned char *prodPB,
**				     unsigned char *termPB,
**				     int expbitN, int fracbitN)
**
** This function multiplies the floating point number pointed to by
** prodPB by the floating point number pointed to by termPB.  The product
** is written out to prodPB.  The value of expbitN determines the number
** of bits in the signed exponent value.  The value of fracbitN determines
** the number of bits in the mantissa (ms bit implied.)
*/
unsigned char
#ifdef PROTOTYPES
fmultm(unsigned char *prodPB, unsigned char *termPB, int expbitN, int fracbitN)
#else
fmultm(prodPB, termPB, expbitN, fracbitN)
unsigned char *prodPB;
unsigned char *termPB;
int expbitN;
int fracbitN;
#endif
{
    int expbyteN, fracbyteN;
    unsigned char *expbiasPB;
    unsigned char *exponePB;
    unsigned char *prodfracPB, *termfracPB, *prodexpPB, *termexpPB;
    unsigned char prodsignB, termsignB;
    unsigned char statusB, condcodeB, expccB;
    int i, totalenN;

    /* Initialize the condition code byte to zero */
    condcodeB = 0;

    /* Determine the total byte length of the floating point number */
    totalenN = fftotlen(expbitN, fracbitN);

    /* Check if either operand is zero.  If so, then just return zero. */
    if ((ffchkzero(prodPB, totalenN) == 0) ||
	(ffchkzero(termPB, totalenN) == 0))
    {
	for (i = 0; i < totalenN; i++)
	    *prodPB++ = 0;
	return(FFZERO);
    }

    /* Determine the number of bytes required to hold the mantissa and
     * exponent.  Allocate space for each... */
    expbyteN = ffexplen( expbitN );
    fracbyteN = ffraclen( fracbitN );

    /* Note that twice as much room as required is allocated for the
     * mantissas - this is to allow a "fracbyteN << 1" length product
     * mantissa to be generated. */
    fracbyteN <<= 1;
    prodfracPB = (unsigned char *) FCALLOC(fracbyteN, 1, "FMULTM1");
    termfracPB = (unsigned char *) FCALLOC(fracbyteN, 1, "FMULTM2");
    prodexpPB = (unsigned char *) FMALLOC(expbyteN, "FMULTM3");
    termexpPB = (unsigned char *) FMALLOC(expbyteN, "FMULTM4");
    expbiasPB = (unsigned char *) FCALLOC(expbyteN, 1, "FMULTM5");
    exponePB = (unsigned char *) FCALLOC(expbyteN, 1, "FMULTM6");

    /* Isolate the mantissas, exponents, and signs.  Calling ffextall()
     * also sets the implied ms bit of the mantissa to 1. */
    ffextall(prodPB, totalenN, fracbitN, fracbyteN, expbitN, expbyteN,
	     prodfracPB, prodexpPB, &prodsignB);
    ffextall(termPB, totalenN, fracbitN, fracbyteN, expbitN, expbyteN,
	     termfracPB, termexpPB, &termsignB);

    *(exponePB + (expbyteN - 1)) = 1;

    /* Add the exponents together, this requires that bias be calculated and
     * subtracted off first. */
    ffgenbias(expbiasPB, expbyteN, expbitN);

    /* Subtract off exponent bias, note that if the result is negative, the
     * sign is properly extended.  This is important since it allows
     * exponent overflow and underflow to be detected much more easily. */
    isubm(prodexpPB, expbiasPB, expbyteN);
    isubm(termexpPB, expbiasPB, expbyteN);

    /* Add exponents together, save the condition codes to check for
    * overflow or underflow */
    expccB = iaddm(prodexpPB, termexpPB, expbyteN);
    /* check if the exponent is in a valid range */
    if ((expccB & OVERFLOW) == 0)
    {
	expccB = ffexpchk(prodexpPB, expbyteN, expbiasPB, expbitN, expccB);
    }

    if ((expccB & OVERFLOW) != 0)
    {
	condcodeB =  ffexpover(prodPB, totalenN, expccB, expbiasPB, expbyteN,
	    exponePB, condcodeB, fracbitN, expbitN);
	condcodeB = ffsetsign(termsignB, prodsignB, prodPB, totalenN,
	    (fracbitN + expbitN), condcodeB);
	FFREE(prodfracPB);
	FFREE(termfracPB);
	FFREE(prodexpPB);
	FFREE(termexpPB);
	FFREE(expbiasPB);
	FFREE(exponePB);
	return(condcodeB);
    }

    /* Multiply the mantissas together */
    statusB = umultm(prodfracPB, termfracPB, fracbyteN);

    /* Check if bit ((2 * fracbitN) + 1) of the result is 1.  This is the
     * most significant possible bit of the product.  If the bit is one, then
     * extract fracbitN bits starting with the (fracbitN + 1) bit and add 1 to
     * the product exponent value.  If the bit is not one, then extract
     * fracbitN bits starting with the fracbitN bit.  This effectively
     * writes out the product mantissa. */

    if (fftstbit(prodfracPB, fracbyteN, ((2 * fracbitN) + 1)) == 0)
    {
	ffbitext(prodfracPB, fracbyteN, fracbitN, fracbitN, prodPB, totalenN);
    }
    else
    {
	ffbitext(prodfracPB, fracbyteN, (fracbitN + 1), fracbitN, prodPB,
	    totalenN);
	expccB = iaddm(prodexpPB, exponePB, expbyteN);
	/* check if the exponent is in a valid range */
	if ((expccB & OVERFLOW) == 0)
	{
	    expccB = ffexpchk(prodexpPB, expbyteN, expbiasPB, expbitN, expccB);
	}

	/* Check if the exponent overflowed */
	if ((expccB & OVERFLOW) != 0)
	{
	    condcodeB = ffexpover(prodPB, totalenN, expccB, expbiasPB,
	   	expbyteN, exponePB, condcodeB, fracbitN, expbitN);
	    condcodeB = ffsetsign(termsignB, prodsignB, prodPB, totalenN,
		(fracbitN + expbitN), condcodeB);
	    FFREE(prodfracPB);
	    FFREE(termfracPB);
	    FFREE(prodexpPB);
	    FFREE(termexpPB);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return(condcodeB);
	}
    }

    /* Add exp bias back on */
    expccB = iaddm(prodexpPB, expbiasPB, expbyteN);

    /* Check if the exponent is zero.  This is a case of underflow that
     * is not caught earlier. */
    if ((expccB & ZERO) != 0)
    {
	/* Underflow, return zero. */
	for (i = 0; i < totalenN; i++)
	    *(prodPB + i) = 0;
	condcodeB |= FFZERO;
    }
    else
    {
	/* Stuff the product exponent into the result field. */
	ffbitins(prodexpPB, expbyteN, fracbitN, expbitN, prodPB, totalenN);
    }

    /* Now determine the sign of the result */
    condcodeB = ffsetsign(termsignB, prodsignB, prodPB, totalenN,
	(fracbitN + expbitN), condcodeB);

    /* Free allocated memory */
    FFREE(prodfracPB);
    FFREE(termfracPB);
    FFREE(prodexpPB);
    FFREE(termexpPB);
    FFREE(expbiasPB);
    FFREE(exponePB);
    /* Return condition codes */
    return(condcodeB);
}

/*
** Function:	unsigned char fdivm(unsigned char *dividPB,
**				    unsigned char *termPB,
**				    int expbitN, int fracbitN)
**
** This function divides the floating point number pointed to by
** dividPB by the floating point number pointed to by termPB.  The quotient
** is written out to dividPB.  The value of expbitN determines the number
** of bits in the signed exponent value.  The value of fracbitN determines
** the number of bits in the mantissa (ms bit implied.)
*/
unsigned char
#ifdef PROTOTYPES
fdivm(unsigned char *dividPB, unsigned char *termPB, int expbitN, int fracbitN)
#else
fdivm(dividPB, termPB, expbitN, fracbitN)
unsigned char *dividPB;
unsigned char *termPB;
int expbitN;
int fracbitN;
#endif
{
    int expbyteN, fracbyteN;
    unsigned char *expbiasPB;
    unsigned char *exponePB;
    unsigned char *dividfracPB, *termfracPB, *dividexpPB, *termexpPB;
    unsigned char dividsignB, termsignB;
    unsigned char statusB, condcodeB, expccB;
    unsigned char dividchkB, tempchkB;
    int i, totalenN;

    /* Initialize the condition code byte to zero */
    condcodeB = 0;

    /* Determine the total byte length of the floating point number */
    totalenN = fftotlen(expbitN, fracbitN);

    /* Determine the number of bytes required to hold the mantissa and
     * exponent.  Allocate space for each... */
    expbyteN = ffexplen(expbitN);
    fracbyteN = ffraclen(fracbitN);

    /* Calculate exponent bias and value of "1" early - these are useful
     * to write out floating values of 1 and infinity. */
    expbiasPB = (unsigned char *) FCALLOC(expbyteN, 1, "FDIVM1");
    ffgenbias(expbiasPB, expbyteN, expbitN);
    exponePB = (unsigned char *) FCALLOC(expbyteN, 1, "FDIVM2");
    *(exponePB + (expbyteN - 1)) = 1;

    /* Check if the divisor and dividend are zero.  If both are zero, then
     * set the result to 1 and return with the Not-A-Number condition
     * code set. */
    dividchkB = ffchkzero(dividPB, totalenN);
    tempchkB = ffchkzero(termPB, totalenN);
    if ((dividchkB == 0) && (tempchkB == 0))
    {
	/* Initialize the result to zero. */
	for (i = 0; i < totalenN; i++)
	    *(dividPB + i) = 0;

	/* Stuff the exponent bias into the result field.  This generates the
	 * floating representation of 1. */
	ffbitins(expbiasPB, expbyteN, fracbitN, expbitN, dividPB, totalenN);
	FFREE(expbiasPB);
	FFREE(exponePB);
	return(FFNAN);
    }

    /* Check if either divisor or dividend are zero.  If the divisor is zero,
     * then return Infinity, and set the FFINF and FFNAN bits.  If the
     * dividend is zero, then return zero and set the FFZERO bit. */
    if ((dividchkB == 0) || (tempchkB == 0))
    {
	/* Initialize the result to zero. */
	for (i = 0; i < totalenN; i++)
	    *(dividPB + i) = 0;

	if (tempchkB == 0)
	{
	    /* Divisor is zero, set the result to infinity.  This is done
	     * by shifting the exponent bias left once and setting the least
	     * significant bit to 1.  The result is then written out the
	     * result's exponent field. */
	    ushftlm(expbiasPB, expbyteN);
	    *(expbiasPB + expbyteN - 1) |= 1;
	    ffbitins(expbiasPB, expbyteN, fracbitN, expbitN, dividPB,
		     totalenN);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return((unsigned char)(FFINF | FFNAN));
	}
	else
	{
	    /* Dividend is zero, return zero result. */
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return(FFZERO);
	}
    }

    /* Note that twice as much room as required is allocated for the
     * mantissas - this is to allow a fracbyteN length quotient
     * mantissa to be generated. */
    fracbyteN <<= 1;
    dividfracPB = (unsigned char *) FCALLOC(fracbyteN, 1, "FDIVM3");
    termfracPB = (unsigned char *) FCALLOC(fracbyteN, 1, "FDIVM4");
    dividexpPB = (unsigned char *) FMALLOC(expbyteN, "FDIVM5");
    termexpPB = (unsigned char *) FMALLOC(expbyteN, "FDIVM6");

    /* Isolate the mantissas, exponents, and signs.  Calling ffextall()
     * also sets the implied ms bit of the mantissa to 1. */
    ffextall(dividPB, totalenN, fracbitN, fracbyteN, expbitN, expbyteN,
	     dividfracPB, dividexpPB, &dividsignB);
    ffextall(termPB, totalenN, fracbitN, fracbyteN, expbitN, expbyteN,
	     termfracPB, termexpPB, &termsignB);

    /* Subtract off exponent bias, note that if the result is negative, the
     * sign is properly extended.  This is important since it allows
     * exponent overflow and underflow to be detected much more easily. */
    isubm(dividexpPB, expbiasPB, expbyteN);
    isubm(termexpPB, expbiasPB, expbyteN);

    /* Subtract divisor exponent, save the condition codes to check
     * for overflow or underflow */
    expccB = isubm(dividexpPB, termexpPB, expbyteN);
    /* check if the exponent is in a valid range */
    if ((expccB & OVERFLOW) == 0)
    {
	expccB = ffexpchk(dividexpPB, expbyteN, expbiasPB, expbitN, expccB);
    }

    if ((expccB & OVERFLOW) != 0)
    {
	condcodeB =  ffexpover(dividPB, totalenN, expccB, expbiasPB, expbyteN,
	    exponePB, condcodeB, fracbitN, expbitN);
	condcodeB = ffsetsign(termsignB, dividsignB, dividPB, totalenN,
	    (fracbitN + expbitN), condcodeB);
	FFREE(dividfracPB);
	FFREE(termfracPB);
	FFREE(dividexpPB);
	FFREE(termexpPB);
	FFREE(expbiasPB);
	FFREE(exponePB);
	return(condcodeB);
    }

    /* Divide the dividend by the divisor.  In order to maximize the number
     * of significant bits in the quotient, the dividend must be shifted left
     * fracbitN bits.  Note that the dividend field is at least
     * two times fracbitN bits long... */
     for (i = 0; i < fracbitN; i++)
	ushftlm(dividfracPB, fracbyteN);
 
    /* Note that the quotient goes to termfracPB and the remainder goes
     * goes to dividfracPB. */
    statusB = udivm(dividfracPB, termfracPB, fracbyteN);

    /* Check if bit fracbitN of the result is 1.  This is the
     * most significant possible bit of the quotient.  If the bit is one,
     * then the result mantissa is in the right position.  If the bit
     * is zero, then shift the result mantissa left a bit and subtract
     * 1 from the quotient exponent value. */
    if (fftstbit(termfracPB, fracbyteN, fracbitN) == 0)
    {
	ushftlm(termfracPB, fracbyteN);
	expccB = isubm(dividexpPB, exponePB, expbyteN);
	/* check if the exponent is in a valid range */
	if ((expccB & OVERFLOW) == 0)
	{
	    expccB = ffexpchk(dividexpPB, expbyteN, expbiasPB, expbitN, expccB);
	}

	/* Check if the exponent overflowed */
	if ((expccB & OVERFLOW) != 0)
	{
	    condcodeB =  ffexpover(dividPB, totalenN, expccB, expbiasPB,
		expbyteN, exponePB, condcodeB, fracbitN, expbitN);
	    condcodeB = ffsetsign(termsignB, dividsignB, dividPB, totalenN,
		(fracbitN + expbitN), condcodeB);
	    FFREE(dividfracPB);
	    FFREE(termfracPB);
	    FFREE(dividexpPB);
	    FFREE(termexpPB);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return(condcodeB);
	}
    }
    ffbitext(termfracPB, fracbyteN, 0, fracbitN, dividPB, totalenN);

    /* Add exp bias back on */
    expccB = iaddm(dividexpPB, expbiasPB, expbyteN);

    /* Check if the exponent is zero.  This is a case of underflow that
     * is not caught earlier. */
    if ((expccB & ZERO) != 0)
    {
	/* Underflow, return zero. */
	for (i = 0; i < totalenN; i++)
	    *(dividPB + i) = 0;
	condcodeB |= FFZERO;
    }
    else
    {
	/* Stuff the quotient exponent into the result field. */
	ffbitins(dividexpPB, expbyteN, fracbitN, expbitN, dividPB, totalenN);
    }

    /* Now determine the sign of the result */
    condcodeB = ffsetsign(termsignB, dividsignB, dividPB, totalenN,
	(fracbitN + expbitN), condcodeB);

    /* Free allocated memory */
    FFREE(dividfracPB);
    FFREE(termfracPB);
    FFREE(dividexpPB);
    FFREE(termexpPB);
    FFREE(expbiasPB);
    FFREE(exponePB);
    /* Return condition codes */
    return(condcodeB);
}

/*
** Function:	unsigned char fcmpm(unsigned char *flt1PB,
**				    unsigned char *flt2PB,
**				    int expbitN, int fracbitN)
**
** This function compares the floating point number pointed to by
** flt1PB with the floating point number pointed to by flt2PB.  If flt1PB
** is larger than flt2PB, then 1 is returned.  If they are equal, then zero
** is returned.  If flt1PB is less than flt2PB, then -1 is returned.
**
** The values flt1PB and flt2PB are not changed.
*/
int
#ifdef PROTOTYPES
fcmpm(unsigned char *flt1PB, unsigned char *flt2PB, int expbitN, int fracbitN)
#else
fcmpm(flt1PB, flt2PB, expbitN, fracbitN)
unsigned char *flt1PB;
unsigned char *flt2PB;
int expbitN;
int fracbitN;
#endif
{
    int signbitN, totalenN, i;
    unsigned char *tempPB;
    unsigned char condcodeB;

    signbitN = expbitN + fracbitN;

    /* Determine the total byte length of the floating point number */
    totalenN = fftotlen(expbitN, fracbitN);

    /* Copy the value pointed to by flt1PB to a local buffer, so as not to
     * change the original value.  */
    tempPB = (unsigned char *) FMALLOC(totalenN, "FCMPM");
    for (i = 0; i < totalenN; i++)
	*(tempPB + i) = *(flt1PB + i);

    condcodeB = fsubm(tempPB, flt2PB, expbitN, fracbitN);
    FFREE( tempPB );

    if ((condcodeB & FFZERO) == FFZERO)
	return(0);
    else if ((condcodeB & FFNEG) == FFNEG)
	return(-1);
    return(1);
}

/*
** Function:	unsigned char fsubm(unsigned char *diffPB,
**				    unsigned char *termPB,
**				    int expbitN, int fracbitN)
**
** This function subtracts the floating point number pointed to by
** termPB from the floating point number pointed to by diffPB.  The difference
** is written out to diffPB.  The value of expbitN determines the number
** of bits in the signed exponent value.  The value of fracbitN determines
** the number of bits in the mantissa (ms bit implied.)
*/
unsigned char
#ifdef PROTOTYPES
fsubm(unsigned char *diffPB, unsigned char *termPB, int expbitN, int fracbitN)
#else
fsubm(diffPB, termPB, expbitN, fracbitN)
unsigned char *diffPB;
unsigned char *termPB;
int expbitN;
int fracbitN;
#endif
{
    int signbitN, totalenN, i;
    unsigned char *tempPB;
    unsigned char condcodeB;

    signbitN = expbitN + fracbitN;

    /* Determine the total byte length of the floating point number */
    totalenN = fftotlen(expbitN, fracbitN);

    /* Copy the value pointed to by termPB to a local buffer, so as not to
     * change the original value. */
    tempPB = (unsigned char *) FMALLOC(totalenN, "FSUBM");
    for (i = 0; i < totalenN; i++)
	*(tempPB + i) = *(termPB + i);

    /* Flip the sign of the float pointed to by tempPB. */
    if (fftstbit(tempPB, totalenN, signbitN) == 0)
	ffbitset(tempPB, totalenN, signbitN);
    else
	ffbitclr(tempPB, totalenN, signbitN);
    
    condcodeB = faddm(diffPB, tempPB, expbitN, fracbitN);
    FFREE(tempPB);
    return (condcodeB);
}

/*
** Function:	unsigned char faddm(unsigned char *sumPB,
**				    unsigned char *termPB,
**				    int expbitN, int fracbitN)
**
** This function adds the floating point number pointed to by
** termPB to the floating point number pointed to by sumPB.  The sum
** is written out to sumPB.  The value of expbitN determines the number
** of bits in the signed exponent value.  The value of fracbitN determines
** the number of bits in the mantissa (ms bit implied.)
*/
unsigned char
#ifdef PROTOTYPES
faddm(unsigned char *sumPB, unsigned char *termPB, int expbitN, int fracbitN)
#else
faddm(sumPB, termPB, expbitN, fracbitN)
unsigned char *sumPB;
unsigned char *termPB;
int expbitN;
int fracbitN;
#endif
{
    int expbyteN, fracbyteN;
    unsigned char *expbiasPB;
    unsigned char *exponePB;
    unsigned char *sumfracPB, *termfracPB, *sumexpPB, *termexpPB;
    unsigned char *mantlenPB, *tempexpPB;
    unsigned char sumsignB, termsignB;
    unsigned char statusB, condcodeB, expccB;
    int i, totalenN, manlenN, expdiffN;

    /* Initialize the condition code byte to zero */
    condcodeB = 0;

    /* Determine the total byte length of the floating point number */
    totalenN = fftotlen(expbitN, fracbitN);

    /* Check if the both operands are zero, if so then just return zero */
    if ((ffchkzero(sumPB, totalenN) == 0) &&
	(ffchkzero(termPB, totalenN) == 0))
    {
	return(FFZERO);
    }

    /* Extract the signs of the addend and augend. */
    ffbitext(sumPB, totalenN, (fracbitN + expbitN), 1, &sumsignB, 1);
    ffbitext(termPB, totalenN, (fracbitN + expbitN),1, &termsignB, 1);

    /* Check if either operand is zero, if so then just return the other
     * operand. */
    if (ffchkzero(termPB, totalenN) == 0)
    {
	if (sumsignB != 0)
	    return(FFNEG);
	return(0);
    }
    if (ffchkzero(sumPB, totalenN) == 0)
    {
	for (i = 0; i < totalenN; i++)
	    *sumPB++ = *termPB++;
	if (termsignB != 0)
	    return(FFNEG);
	return(0);
    }

    /* Done with easy cases, the numbers must be actually be added. 
     * Determine the number of bytes required to hold the mantissa and
     * exponent.  Allocate space for each... */
    expbyteN = ffexplen(expbitN);
    fracbyteN = ffraclen(fracbitN);

    /* Note that one extra byte is allocated for the mantissas - this is to
     * allow a "fracbyteN + 1" length sum mantissa to be generated. */
    fracbyteN += 1;
    /* Allocate space for everything */
    sumfracPB = (unsigned char *) FCALLOC( fracbyteN, 1, "FADDM1");
    termfracPB = (unsigned char *) FCALLOC( fracbyteN, 1, "FADDM2");
    sumexpPB = (unsigned char *) FMALLOC( expbyteN, "FADDM3");
    termexpPB = (unsigned char *) FMALLOC( expbyteN, "FADDM4");
    mantlenPB = (unsigned char *) FCALLOC( expbyteN, 1, "FADDM5");
    tempexpPB = (unsigned char *) FMALLOC( expbyteN, "FADDM6");
    expbiasPB = (unsigned char *) FCALLOC( expbyteN, 1, "FADDM7");
    exponePB = (unsigned char *) FCALLOC( expbyteN, 1, "FADDM8");

    /* Isolate the mantissas, exponents, and signs.  Calling ffextall()
     * also sets the implied ms bit of the mantissa to 1. */
    ffextall(sumPB, totalenN, fracbitN, fracbyteN, expbitN, expbyteN,
	     sumfracPB, sumexpPB, &sumsignB);
    ffextall(termPB, totalenN, fracbitN, fracbyteN, expbitN, expbyteN,
	     termfracPB, termexpPB, &termsignB);

    *(exponePB + (expbyteN - 1)) = 1;

    /* Determine the difference between exponents , this requires that bias
     * be calculated and subtracted off first. */
    ffgenbias(expbiasPB, expbyteN, expbitN);

    /* Subtract off exponent bias, note that if the result is negative, the
     * sign is properly extended.  This is important since it allows
     * exponent overflow and underflow to be detected much more easily. */
    isubm(sumexpPB, expbiasPB, expbyteN);
    isubm(termexpPB, expbiasPB, expbyteN);

    /* Determine the difference in the exponents.  This is used to
     * determine with operand must be shifted right and by how much it
     * must be shifted.  Since we don't yet want to change the exponents
     * yet, copy the sumPB exponent to a temporary location. */
    for (i = 0; i < expbyteN; i++)
	*(tempexpPB + i) = *(sumexpPB + i);
    expccB = isubm(tempexpPB, termexpPB, expbyteN);

    /* Check if the difference in exponents is small enough to determine
     * if both operands are significant.  If one operand is insignificant
     * relative to the other, then adding it to the other makes no difference
     * in the larger operand. */
    if ((expccB & OVERFLOW) == 0)
    {
	/* No obvious overflow or underflow problems, check explicitly if
	 * the exponent difference is too great.
	 *
	 * First determine the maximum exponent difference for both operands
	 * to be significant. */
	manlenN = fracbitN + 1;

	/* Convert this to "exponent format".  Note that the largest exponent
	 * allowed depends on the length of an int.  However, also note that
	 * even if an int is just a byte that this allows exponents to be up
	 * to 255 bits long. */
	i = expbyteN - 1;
	while (manlenN > 0)
	{
	    *(mantlenPB + i) = (unsigned char) manlenN;
	    manlenN >>= 8;
	    i--;
	}

	if ((expccB & SIGN) == 0)
	{
	    /* The addend exponent is larger than the augend exponent.
	     * Subtract the difference from the mantissa length.  If the
	     * result is negative, then the difference is larger than
	     * than the mantissa length and the augend is insignificant. */
	    statusB = isubm(mantlenPB, tempexpPB, expbyteN);
	    if ((statusB & SIGN) != 0)
		expccB |= OVERFLOW;
	}
	else
	{
	    /* The augend exponent is larger than the addend exponent.
	     * Add the difference to the mantissa length.  If the
	     * result is negative, then the difference is larger than
	     * than the mantissa length and the addend is insignificant. */
	    statusB = iaddm(mantlenPB, tempexpPB, expbyteN);
	    if ((statusB & SIGN) != 0)
		expccB |= OVERFLOW;
	}
    }
    if ((expccB & OVERFLOW) != 0)
    {
	/* One of the operands is insignificant relative to the other
	 * operand. */
	if ((expccB & SIGN) == 0)
	{
	    /* Addend exponent is much larger than the augend exponent.
	     * Return the value pointed to by sumPB unchanged. */
	    if (sumsignB != 0)
		condcodeB =  FFNEG;
	}
	else
	{
	    /* Augend exponent is much larger than the addend exponent.
	     * Copy the value pointed to by termPB to sumPB. */
	    for (i = 0; i < totalenN; i++)
		*sumPB++ = *termPB++;
	    if (termsignB != 0)
		condcodeB = FFNEG;
	}
	/* Free allocated memory */
	FFREE(sumfracPB);
	FFREE(termfracPB);
	FFREE(sumexpPB);
	FFREE(termexpPB);
	FFREE(mantlenPB);
	FFREE(tempexpPB);
	FFREE(expbiasPB);
	FFREE(exponePB);
	return(condcodeB);
    }

    /* The operands are significant relative to each other.  First convert
     * the exponent difference back to a positive int. */
    if ((expccB & SIGN) != 0)
    {
	/* The exponent difference is negative, change its sign */
	inegm(tempexpPB, expbyteN);
    }
    expdiffN = 0;
    for (i = 0; i < expbyteN; i++)
    {
	expdiffN <<= 8;
	expdiffN += (unsigned char) (*(tempexpPB + i));
    }

    if ((expccB & SIGN) == 0)
    {
	/* The addend exponent is larger than the augend exponent.  Shift
	 * the augend right n times where n is the difference between the
	 * addend and augend exponents.  The initial sum exponent in 
	 * sumexpPB is ok. */
	for (i = 0; i < expdiffN; i++)
	    ushftrm(termfracPB, fracbyteN);
    }
    else
    {
	/* The augend exponent is larger than the addend exponent.  Shift
	 * the addend right n times where n is the difference between the
	 * addend and augend exponents.  Copy the initial sum exponent
	 * from termexpPB. */
	for (i = 0; i < expdiffN; i++)
	    ushftrm(sumfracPB, fracbyteN);
	for (i = 0; i < expbyteN; i++)
	    *(sumexpPB + i) = *(termexpPB + i);
    }	

    /* Add the mantissas together */
    if (termsignB == sumsignB)
    {
	/* Mantissas have the same sign.  Note that the sign of the sum
	 * is the same as the original addend. */
	statusB = iaddm(sumfracPB, termfracPB, fracbyteN);
    }
    else
    {
	if (termsignB == 0)
	{
	    /* Augend mantissa is positive, addend mantissa is negative.
	     * Subtract addend from augend.  Copy the result back to
	     * sumfracPB. */
	    statusB = isubm(termfracPB, sumfracPB, fracbyteN);
	    for (i = 0; i < fracbyteN; i++)
		*(sumfracPB + i) = *(termfracPB + i);
	}
	else
	{
	    /* Addend mantissa is positive, augend mantissa is negative.
	     * Subtract augend from addend. */
	    statusB = isubm(sumfracPB, termfracPB, fracbyteN);
	}
	/* Check if the sum is negative, if so negate it and set the sum
	 * mantissa sign value to indicate a negative sum. */
	if ((statusB & CARRY) != 0)
	{
	    sumsignB = 1;
	    inegm(sumfracPB, fracbyteN);
	}
	else
	{
	    sumsignB = 0;
	}
    }

    /* Check if bit (fracbitN + 1) of the result is 1.  This is the
     * most significant possible bit of the sum.  If the bit is one, then
     * extract the sum mantissa as fracbitN bits starting at bit 1 and
     * add 1 to the product sum exponent value.  If the bit is not one,
     * then search for the most significant bit in the sum that is 1.
     * (When adding two numbers of opposite sign, but similar magnitude,
     * the result can have a magnitude significantly smaller than either.)
     * Then extract fracbitN bits starting with bit 0.  This effectively
     * writes out the sum mantissa. */
    if (fftstbit( sumfracPB, fracbyteN, (fracbitN + 1)) == 0)
    {
	i = fracbitN;
	/* Search only until all original mantissa bits have been checked. */
	while ((fftstbit(sumfracPB, fracbyteN, fracbitN) == 0) && (i-- > 0))
	{
	    /* Shift the mantissa left, and subtract one from the
	     * exponent.   Each time, check if the exponent is
	     * out of range. */
	    ushftlm(sumfracPB, fracbyteN);
	    expccB = isubm(sumexpPB, exponePB, expbyteN);
	    /* check if the exponent is in a valid range */
	    if ((expccB & OVERFLOW) == 0)
	    {
		expccB = ffexpchk(sumexpPB, expbyteN, expbiasPB,
				  expbitN, expccB );
	    }

	    /* Check if the exponent overflowed */
	    if ((expccB & OVERFLOW) != 0)
	    {
		condcodeB = ffexpover(sumPB, totalenN, expccB, expbiasPB,
		    expbyteN, exponePB, condcodeB, fracbitN, expbitN);
		FFREE(sumfracPB);
		FFREE(termfracPB);
		FFREE(sumexpPB);
		FFREE(termexpPB);
		FFREE(mantlenPB);
		FFREE(tempexpPB);
		FFREE(expbiasPB);
		FFREE(exponePB);
		return(condcodeB);
	    }
	}
	if (i >= 0)
	{
	    /* Valid, non-zero result. */
	    ffbitext(sumfracPB, fracbyteN, 0, fracbitN, sumPB, totalenN);
	}
	else
	{
	    /* The result is zero, set zero flag and zero the result */
	    for (i = 0; i < totalenN; i++)
		*(sumPB + i) = 0;
	    condcodeB = FFZERO;
	    /* Free allocated memory */
	    FFREE(sumfracPB);
	    FFREE(termfracPB);
	    FFREE(sumexpPB);
	    FFREE(termexpPB);
	    FFREE(mantlenPB);
	    FFREE(tempexpPB);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    /* Return condition codes */
	    return(condcodeB);
	}
    }
    else
    {
	ffbitext(sumfracPB, fracbyteN, 1, fracbitN, sumPB, totalenN);
	expccB = iaddm(sumexpPB, exponePB, expbyteN);
	/* check if the exponent is in a valid range */
	if ((expccB & OVERFLOW) == 0)
	{
	    expccB = ffexpchk(sumexpPB, expbyteN, expbiasPB, expbitN, expccB);
	}
	/* Check if the exponent overflowed */
	if ((expccB & OVERFLOW) != 0)
	{
	    condcodeB = ffexpover(sumPB, totalenN, expccB, expbiasPB,
	   	expbyteN, exponePB, condcodeB, fracbitN, expbitN);
	    FFREE(sumfracPB);
	    FFREE(termfracPB);
	    FFREE(sumexpPB);
	    FFREE(termexpPB);
	    FFREE(mantlenPB);
	    FFREE(tempexpPB);
	    FFREE(expbiasPB);
	    FFREE(exponePB);
	    return(condcodeB);
	}
    }

    iaddm(sumexpPB, expbiasPB, expbyteN);	/* Add exp bias back on */

    /* Stuff the sum exponent into the result field. */
    ffbitins(sumexpPB, expbyteN, fracbitN, expbitN, sumPB, totalenN);

    /* Set the sign bit of the sum */
    if (sumsignB == 0)
    {
	/* Clear the sign bit of the sum. */
	(*sumPB) &= (unsigned char) 0x7f;
    }
    else
    {
	/* Set the sign bit of the sum. */
	(*sumPB) |= (unsigned char) 0x80;
	condcodeB |= FFNEG;
    }

    /* Free allocated memory */
    FFREE(sumfracPB);
    FFREE(termfracPB);
    FFREE(sumexpPB);
    FFREE(termexpPB);
    FFREE(mantlenPB);
    FFREE(tempexpPB);
    FFREE(expbiasPB);
    FFREE(exponePB);
    /* Return condition codes */
    return(condcodeB);
}
