/* Extended IEEE Compatible Floating Point Arithmetic Library
**
** Version 1.1
** Copyright (C) 1990, by Fred Motteler
** All Rights Reserved
**
** This is part one of a simple extended floating point arithmetic package
** tester.
**
** This is a table driven tester.  It does a comprehensive test of:
**	1. Integer math operations
**	2. Integer / ASCII string conversions
**	3. Float math operations
**	4. Float / ASCII decimal conversions
**
** Where reasonable, all numeric input and output values are specified
** as ASCII strings.  This greatly simplifies setting up test cases, and
** provides a good test of the string conversion routines.
** Note that these routines recognize zero, but NOT infinite, not a number,
** or denormalized input parameters.  They blindly assume that the arguments
** are valid floating point numbers.
**
** These routines will return valid zero, infinite, and not a number values.
*/
#include <stdio.h>
#ifndef MWC
#include <stdlib.h>
#endif
#include "imlib.h"
#include "ffmlib.h"
#include "fmlib.h"
#include "fmtest.h"

/* Include the tables containing the tester data.  There are two data
** tables that contain test cases for:
**
**	dyadic integer operations (add, multiply, subtract, divide)
**	dyadic float operations (add, multiply, subtract, divide)
*/
#include "fmtest1.h"

/*
** Function:	int fmtst1(int *testPN)
**
** This is a table driven tester.  It does a comprehensive test of:
**	1. Integer math operations
**	2. Integer / ASCII string conversions
**	3. Float math operations
**	4. Float / ASCII decimal conversions
**
** testPN is a pointer to an integer used to total the error count.
**
** Where reasonable, all numeric input and output values are specified
** as ASCII strings.  This greatly simplifies setting up test cases, and
** provides a good test of the string conversion routines.
*/
int
#ifdef PROTOTYPES
fmtst1(int *testPN)
#else
fmtst1(testPN)
int *testPN;
#endif
{
    int i, fracbitsN, expbitsN, totalenN, radixN;
    int intlenN;
    int infracbitsN, inexpbitsN, intotalenN;
    int outfracbitsN, outexpbitsN, outtotalenN;
#ifdef MEMTEST
    int memparcelN;
#endif
    unsigned char op1AB[128], op2AB[128];
    unsigned char condcodeB;
    char resultAB[128], quotientAB[128];
    int errcountN;

    errcountN = 0;

#ifdef MEMTEST
    fmallinit();
#endif

    printf("*******************************************\n");
    printf("** Dyadic integer operation tests **\n");
    printf("*******************************************\n");
    i = 0;
    while (intableAH[i].operationS != NULL)
    {
	printf("Operation %d: %s\n", (*testPN)++, (intableAH[i].operationS));
	totalenN = intableAH[i].totalenN;
	radixN = intableAH[i].radixN;

	/* Convert the first operand from a string to an integer. */
	strtoint(intableAH[i].op1BP, strlen(intableAH[i].op1BP), op1AB,
		 totalenN, radixN);
#ifdef TEST
	printf("Operand 1 as string: %s\n", intableAH[i].op1BP);
	foutput("Operand 1 as integer: ", op1AB, totalenN);
#endif

	/* Convert the second operand from a string to a float. */
	strtoint(intableAH[i].op2BP, strlen(intableAH[i].op2BP), op2AB,
		 totalenN, radixN);
#ifdef TEST
	printf("Operand 2 as string: %s\n", intableAH[i].op2BP);
	foutput("Operand 2 as integer: ", op2AB, totalenN);
#endif

	/* Perform the arithmetic operation. */
	condcodeB = (*(intableAH[i].opfuncFP))(op1AB, op2AB,
		       intableAH[i].totalenN);

	if (intableAH[i].quotientBP == NULL)
	{
	    /* Standard check of results. */
#ifdef TEST
	    foutput("Result as integer: ", op1AB, totalenN);
#endif

	    /* Convert the result back to a string. */
	    intostr(op1AB, totalenN, resultAB, 128, radixN);
#ifdef TEST
	    printf("Result as string: %s\n", resultAB);
	    printf("Result should be: %s\n", intableAH[i].resultBP);
	    printf("Result condition code: 0x%x\n", condcodeB);
	    printf("Condition code should be: 0x%x\n", intableAH[i].ccresultN);
#endif
	    if ((strcmp(resultAB, intableAH[i].resultBP) == 0) &&
		(((int) condcodeB) == intableAH[i].ccresultN))
	    {
		printf(" [(-:} OK {:-)]\n");
	    }
	    else
	    {
		printf(" [)-:}***** ERROR *****{:-(]\n");
		errcountN++;
	    }
	}
	else
	{
	    /* Check both result operands.  For division, both quotient and
	     * remainder are returned. */
#ifdef TEST
	    foutput("Quotient as integer: ", op2AB, totalenN);
	    foutput("Remainder as integer: ", op1AB, totalenN);
#endif

	    /* Convert the quotient and remainder back to strings. */
	    intostr(op1AB, totalenN, resultAB, 128, radixN);
	    intostr(op2AB, totalenN, quotientAB, 128, radixN);
#ifdef TEST
	    printf("Quotient as string: %s\n", quotientAB);
	    printf("Quotient should be: %s\n", intableAH[i].quotientBP);
	    printf("Remainder as string: %s\n", resultAB);
	    printf("Remainder should be: %s\n", intableAH[i].resultBP);
	    printf("Result condition code: 0x%x\n", condcodeB);
	    printf("Condition code should be: 0x%x\n", intableAH[i].ccresultN);
#endif
	    if ((strcmp(resultAB, intableAH[i].resultBP) == 0) &&
		(strcmp(quotientAB, intableAH[i].quotientBP) == 0) &&
		(((int) condcodeB) == intableAH[i].ccresultN))
	    {
		printf(" [(-:} OK {:-)]\n");
	    }
	    else
	    {
		printf(" [)-:}***** ERROR *****{:-(]\n");
		errcountN++;
	    }
	}
#ifdef MEMTEST
	if ((memparcelN = fmallreport()) != 0)
	{
	    printf("%d memory blocks not free'd\n", memparcelN);
	    errcountN += memparcelN;
	}
#endif
	i++;
    }

    printf("*******************************************\n");
    printf("** Dyadic floating point operation tests **\n");
    printf("*******************************************\n");
    i = 0;
    while (fltableAH[i].operationS != NULL)
    {
	/* Determine the length and type of the floating point number */
	expbitsN = fltableAH[i].expbitsN;
	fracbitsN = fltableAH[i].fracbitsN;
	totalenN = fltableAH[i].totalenN;

	printf("Operation %d: %s\n", (*testPN)++, (fltableAH[i].operationS));
#ifdef TEST
	printf("Exponent bits: %d, Mantissa bits: %d, Total byte length: %d\n",
		expbitsN, fracbitsN, totalenN );
#endif

	/* Convert the first operand from a string to a float. */
	strtoflt(fltableAH[i].op1BP, strlen(fltableAH[i].op1BP), op1AB,
		 fracbitsN, expbitsN);
#ifdef TEST
	printf("Operand 1 as string: %s\n", fltableAH[i].op1BP);
	foutput("Operand 1 as float: ", op1AB, totalenN);
#endif

	/* Convert the second operand from a string to a float. */
	strtoflt(fltableAH[i].op2BP, strlen(fltableAH[i].op2BP), op2AB,
		 fracbitsN, expbitsN);
#ifdef TEST
	printf("Operand 2 as string: %s\n", fltableAH[i].op2BP);
	foutput("Operand 2 as float: ", op2AB, totalenN);
#endif

	/* Perform the arithmetic operation. */
	condcodeB = (*(fltableAH[i].opfuncFP))(op1AB, op2AB,
		    expbitsN, fracbitsN);
#ifdef TEST
	foutput("Result as float: ", op1AB, totalenN);
#endif
	/* Convert the result back to a string. */
	fltostr(op1AB, fracbitsN, expbitsN, resultAB, 128);
#ifdef TEST
	printf("Result as string: %s\n", resultAB);
	printf("Result should be: %s\n", fltableAH[i].resultBP);
	printf("Result condition code: 0x%x\n", condcodeB);
	printf("Condition code should be: 0x%x\n", fltableAH[i].ccresultN);
#endif
	if ((strcmp(resultAB, fltableAH[i].resultBP) == 0) &&
	    (((int) condcodeB) == fltableAH[i].ccresultN))
	{
	    printf(" [(-:} OK {:-)]\n");
	}
	else
	{
	    printf(" [)-:}***** ERROR *****{:-(]\n");
	    errcountN++;
	}
#ifdef MEMTEST
	if ((memparcelN = fmallreport()) != 0)
	{
	    printf("%d memory blocks not free'd\n", memparcelN);
	    errcountN += memparcelN;
	}
#endif
	i++;
    }

    return(errcountN);
}
