/* Extended IEEE Compatible Floating Point Arithmetic Library
**
** Version 1.0
** Copyright (C) 1990, by Fred Motteler
** All Rights Reserved
**
** This is a simple extended floating point arithmetic package.
** It must be used with the extended integer arithmetic package.
**
** Format:
**
** |S| n bit signed exponent | m bit fraction |
**
** The format is compatible with IEEE Standard for Binary Floating
** when:
**		n = 8,   m = 23		single precision
**		n = 11,  m = 52		double precision
**
** If n = 15 and m = 64, this format is almost compatible with IEEE
** extended precision format.  The main difference are
**	1) IEEE extended precision format is a 12 byte format with 10 bytes of
**	   information.  Bits 64 thru 80 are zero.
**	2) The most significant bit of the mantissa (implied in single and
**	   double precision IEEE formats) is explicitly included.
**
** Each routine returns a condition code value:
**
**	0 0 0 0 N Z I n
**	Where N = negative
**	      Z = zero
**	      I = infinite
**	      n = not a number
**
** This is an include file containing the actual tester data.  There are
** two data tables that contain test cases for:
**
**	dyadic integer operations (add, multiply, subtract, divide)
**	dyadic float operations (add, multiply, subtract, divide)
**
*/

/* Structure for testing dyadic integer operations. */
struct intvals {
    char *operationS;			/* Pointer to operation type string */
    char *op1BP;			/* Pointer to operand 1 string */
    char *op2BP;			/* Pointer to operand 2 string */
    int totalenN;			/* Byte length required by integer */
    int radixN;				/* Radix of I/O string values */
#ifdef PROTOTYPES
    unsigned char (*opfuncFP)(unsigned char *op1PB, unsigned char *op2PB,
			      int totalenN);
#else
    unsigned char (*opfuncFP)();	/* Pointer to function to perform */
#endif
    char *resultBP;			/* Pointer to expected result string */
    char *quotientBP;			/* Pointer to quotient result string */
    int ccresultN;			/* Resulting condition code */
    } intableAH[] = {
	{"Basic character addition",
	 "12", "34",
	 BYTEWIDE, 10,
	 iaddm, "46", NULL, 0},

	{"Basic character subtraction",
	 "12", "34",
	 BYTEWIDE, 10,
	 isubm, "-22", NULL, (SIGN | CARRY)},

	{"Basic unsigned character multiplication",
	 "12", "10",
	 BYTEWIDE, 10,
	 umultm, "120", NULL, 0},

	{"Basic unsigned character division",
	 "123", "12",
	 BYTEWIDE, 10,
	 udivm, "3", "10", 0},

	{"Basic signed character multiplication",
	 "6", "-20",
	 BYTEWIDE, 10,
	 smultm, "-120", NULL, SIGN},

	{"Basic signed character division",
	 "34", "-12",
	 BYTEWIDE, 10,
	 sdivm, "-10", "-2", SIGN},

	{"Basic integer addition",
	 "1234", "5678",
	 WORDWIDE, 10,
	 iaddm, "6912", NULL, 0},

	{"Basic integer subtraction",
	 "1234", "5678",
	 WORDWIDE, 10,
	 isubm, "-4444", NULL, (SIGN | CARRY)},

	{"Basic unsigned integer multiplication",
	 "12", "34",
	 WORDWIDE, 10,
	 umultm, "408", NULL, 0},

	{"Basic unsigned integer division",
	 "1234", "56",
	 WORDWIDE, 10,
	 udivm, "2", "22", 0},

	{"Basic signed integer multiplication",
	 "-12", "34",
	 WORDWIDE, 10,
	 smultm, "-408", NULL, SIGN},

	{"Basic signed integer division",
	 "1234", "-56",
	 WORDWIDE, 10,
	 sdivm, "-2", "-22", SIGN},

	{"Integer addition overflow",
	 "21234", "25678",
	 WORDWIDE, 10,
	 iaddm, "-18624", NULL, (SIGN | OVERFLOW)},

	{"Integer subtraction overflow",
	 "-11234", "25678",
	 WORDWIDE, 10,
	 isubm, "28624", NULL, OVERFLOW},

	{"Integer addition with zero result",
	 "-11234", "11234",
	 WORDWIDE, 10,
	 iaddm, "0", NULL, (ZERO | CARRY)},

	{"Unsigned integer multiplication overflow",
	 "1212", "343",
	 WORDWIDE, 10,
	 umultm, "-1", NULL, (SIGN | OVERFLOW)},

	{"Unsigned integer multiplication by zero",
	 "0", "343",
	 WORDWIDE, 10,
	 umultm, "0", NULL, ZERO},

	{"Unsigned integer division by zero",
	 "1234", "0",
	 WORDWIDE, 10,
	 udivm, "0", "-1", (SIGN | OVERFLOW)},

	{"Negative signed integer multiplication overflow",
	 "6346", "-201",
	 WORDWIDE, 10,
	 smultm, "-32768", NULL, (SIGN | OVERFLOW)},

	{"Signed integer multiplication overflow",
	 "6346", "201",
	 WORDWIDE, 10,
	 smultm, "32767", NULL, OVERFLOW},

	{"Signed integer multiplication by zero",
	 "6346", "0",
	 WORDWIDE, 10,
	 smultm, "0", NULL, ZERO},

	{"Signed integer division by zero",
	 "1234", "0",
	 WORDWIDE, 10,
	 sdivm, "0", "32767", OVERFLOW},

	{"Quad Quad unsigned (512 bit) multiplication",
	 "12345678901234567890123456789012345678901234567890",
	 "9876543210987654321098765432109876543210987654321",
	 QQWIDE, 10,
	 umultm, "121932631137021795226185032733866788594511507391561194939744871208653362292333223746380111126352690", NULL, 0},

	{ NULL,
	  NULL, NULL,
	  0, 0,
	  NULL, NULL, NULL, 0}};

/* Structure for testing dyadic floating point operations. */
struct fltvals {
    char *operationS;			/* Pointer to operation type string */
    char *op1BP;			/* Pointer to operand 1 string */
    char *op2BP;			/* Pointer to operand 2 string */
    int fracbitsN;			/* Precision of float mantissa */
    int expbitsN;			/* Precision of float exponent */
    int totalenN;			/* Byte length required by float */
#ifdef PROTOTYPES
    unsigned char (*opfuncFP)(unsigned char *op1PB, unsigned char *op2PB,
			      int expbitsN, int fracbitsN);
#else
    unsigned char (*opfuncFP)();	/* Pointer to function to perform */
#endif
    char *resultBP;			/* Pointer to expected result string */
    int ccresultN;			/* Resulting condition code */
    } fltableAH[] = {
	{"Basic float multiplication",
	 "1.2", "2.3",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.27599986e+1", 0},

	{"Float multiplication with exponents",
 	 "1.2e+12", "2.3e-5",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.27599990e+8", 0},

	{"Signed float multiplication (positive result)",
	 "-1.2e-12", "-2.3e-5",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.27599984e-16", 0},

	{"Signed float multiplication (negative result)",
	 "1.2e-12", "-2.3e-5",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "-.27599984e-16", FFNEG},

	{"Float multiplication by zero",
	 "0", "3.14159",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.0e+0", FFZERO},

	{"Float multiply underflow",
	 "1.2e-24", "3.14159e-17",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.0e+0", FFZERO},

	{"Almost float multiply underflow",
	 "1e-24", "2e-14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.19999982e-37", 0},

	{"Barely float multiply underflow",
	 "1e-24", "1e-14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.0e+0", FFZERO},

	{"Input conversion test, no decimal point",
	 "1", "2",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.19999998e+1", 0},

	{"Input conversion test, with decimal point",
	 "1.", "2.",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.19999998e+1", 0},

	{"Float multiply overflow",
	 "1.2e+24", "3.14159e+17",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+Infinity", FFINF},

	{"Almost float multiply overflow",
	 "1.2e+24", "2.0e+14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.23999992e+39", 0},

	{"Another almost float multiply overflow",
	 "1.2e+24", "1.0e+14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+.11999996e+39", 0},

	{"Barely float multiply overflow",
	 "1.2e+24", "4.0e+14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fmultm, "+Infinity", FFINF},

	{"Basic float addition",
 	 "2.4", "3.5",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 faddm, "+.58999984e+1", 0},

	{"Addition of negative numbers",
 	 "-2.4", "-3.5",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 faddm, "-.58999984e+1", FFNEG},

	{"Addition of zero",
 	 "2.4", "0",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 faddm, "+.23999994e+1", 0},

	{"Addition with zero result",
 	 "2.4", "-2.4",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 faddm, "+.0e+0", FFZERO},

	{"Addition of insignificant number",
 	 "2.4e+20", "3.5e-5",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 faddm, "+.23999996e+21", 0},

	{"Addition of almost insignificant number",
 	 "2.4e+20", "-3.5e+14",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 faddm, "+.23999964e+21", 0},

	{"Addition overflow",
 	 "2e+38", "3e+38",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 faddm, "+Infinity", FFINF},

	{"Almost addition overflow",
 	 "2e+38", "1e+38",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 faddm, "+.29999994e+39", 0},

	{"Basic float subtraction",
 	 "5.2", "3.3",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fsubm, "+.18999994e+1", 0},

	{"Float subtraction with negative result",
	 "3.3", "5.2",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fsubm, "-.18999994e+1", FFNEG},

	{"Float subtraction with positive result",
	 "3.3", "-5.2",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fsubm, "+.84999976e+1", 0},

	{"Basic float division",
 	 "1.6", "1.9",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm,  "+.84210520e+0", 0},

	{"Float division by zero",
 	 "1.6", "0",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm,  "+Infinity", (FFINF | FFNAN)},

	{"Division of zero",
 	 "0", "1.9",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm,  "+.0e+0", FFZERO},

	{"Zero divided by zero",
 	 "0", "0",
 	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm,  "+.1e+1", FFNAN},

	{"Float division underflow",
	 "1.2e-24", "3.14159e+17",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm, "+.0e+0", FFZERO},

	{"Almost division underflow",
	 "1e-24", ".5e+14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm, "+.19999992e-37", 0},

	{"Barely float multiply underflow",
	 "1e-24", "1e+14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm, "+.0e+0", FFZERO},

	{"Float Division overflow",
	 "1.2e+24", "3.14159e-17",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm, "+Infinity", FFINF},

	{"Almost float division overflow",
	 "1.2e+24", ".5e-14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm, "+.24000004e+39", 0},

	{"Another almost division multiply overflow",
	 "1.2e+24", "1.0e-14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm, "+.12000002e+39", 0},

	{"Barely float division overflow",
	 "1.2e+24", "0.25e-14",
	 SINGLEFRAC, SINGLEXP, SINGLEN,
	 fdivm, "+Infinity", FFINF},

	{"Basic double multiplication",
	 "1.2", "2.3",
	 DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 fmultm, "+.2759999999999995e+1", 0},

	{"Basic double addition",
	 "2.4", "3.5",
	 DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 faddm, "+.5899999999999995e+1", 0},

	{"Basic double subtraction",
	 "5.2", "3.3",
	 DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 fsubm, "+.1899999999999999e+1", 0},

	{"Basic double division",
	 "1.6", "1.9",
	 DOUBLEFRAC, DOUBLEXP, DOUBLEN,
	 fdivm, "+.8421052631578946e+0", 0},

	{"Basic extended multiplication",
	 "1.2", "2.3",
	 EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 fmultm, "+.27599999999999999988e+1", 0},

	{"Basic extended addition",
	 "2.4", "3.5",
	 EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 faddm, "+.58999999999999999984e+1", 0},

	{"Basic extended subtraction",
	 "5.2", "3.3",
	 EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 fsubm, "+.18999999999999999994e+1", 0},

	{"Basic extended division",
	 "1.6", "1.9",
	 EXTENDFRAC, EXTENDEXP, EXTENDLEN,
	 fdivm, "+.84210526315789473680e+0", 0},

	{"Basic sixteen byte multiplication",
	 "1.2", "2.3",
	 SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 fmultm, "+.27599999999999999999999999999991e+1", 0},

	{"Basic sixteen byte addition",
	 "2.4", "3.5",
	 SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 faddm, "+.58999999999999999999999999999990e+1", 0},

	{"Basic sixteen byte subtraction",
	 "5.2", "3.3",
	 SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 fsubm, "+.18999999999999999999999999999998e+1", 0},

	{"Basic sixteen byte division",
	 "1.6", "1.9",
	 SIXTEENFRAC, SIXTEENEXP, SIXTEENLEN,
	 fdivm, "+.84210526315789473684210526315788e+0", 0},

	{"Basic thirty-two byte multiplication",
	 "1.2", "2.3",
	 THIRTYTWOFRAC, THIRTYTWOEXP, THIRTYTWOLEN,
	 fmultm, "+.27599999999999999999999999999999999999999999999999999999999999999999974e+1", 0},

	{"Basic thirty-two byte addition",
	 "2.4", "3.5",
	 THIRTYTWOFRAC, THIRTYTWOEXP, THIRTYTWOLEN,
	 faddm, "+.58999999999999999999999999999999999999999999999999999999999999999999968e+1", 0},

	{"Basic thrity-two byte subtraction",
	 "5.2", "3.3",
	 THIRTYTWOFRAC, THIRTYTWOEXP, THIRTYTWOLEN,
	 fsubm, "+.18999999999999999999999999999999999999999999999999999999999999999999996e+1", 0},

	{"Basic thrity-two byte division",
	 "1.6", "1.9",
	 THIRTYTWOFRAC, THIRTYTWOEXP, THIRTYTWOLEN,
	 fdivm, "+.84210526315789473684210526315789473684210526315789473684210526315789464e+0", 0},

	{"Basic forty-eight byte multiplication",
	 "1.2", "2.3",
	 FORTYEIGHTFRAC, FORTYEIGHTEXP, FORTYEIGHTLEN,
	 fmultm, "+.2759999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999993e+1", 0},

	{"Basic forty-eight byte addition",
	 "2.4", "3.5",
	 FORTYEIGHTFRAC, FORTYEIGHTEXP, FORTYEIGHTLEN,
	 faddm, "+.5899999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999992e+1", 0},

	{"Basic forty-eight subtraction",
	 "5.2", "3.3",
	 FORTYEIGHTFRAC, FORTYEIGHTEXP, FORTYEIGHTLEN,
	 fsubm, "+.1899999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999998e+1", 0},

	{"Basic forty-eight byte division",
	 "1.6", "1.9",
	 FORTYEIGHTFRAC, FORTYEIGHTEXP, FORTYEIGHTLEN,
	 fdivm, "+.8421052631578947368421052631578947368421052631578947368421052631578947368421052631578947368421052631578947368e+0", 0},

	{ NULL,
	  NULL, NULL,
	  0, 0, 0,
	  NULL, NULL, 0}};

