/* Extended Integer Arithmetic Library
**
** Version 1.1
** Copyright (C) 1990, by Fred Motteler
** All Rights Reserved
**
** This is a simple extended integer arithmetic package.
**
** This version should be K&R, ANSI, and C++ compatible.
** Format:
**
** 	byte 0	byte 1	byte 2	...	byte N
**	ms byte ...	...	...	ls byte
**
** Each routine returns a condition code value:
**
**	0 0 0 0 Z V S C
**	Where Z = result zero if 1
**	      V = overflow if 1
**	      S = result negative if 1 (if ms bit of result is 1)
**	      C = carry
*/

#define NBYTE 8			/* Number of bytes precision */
#define NBIT (NBYTE*8)		/* Number of bits precision */

#include <stdio.h>
#ifndef MWC
#include <stdlib.h>
#endif
#include "imlib.h"


#ifdef TEST

/* Test routine for extended interger arithmetic library functions. */
unsigned char number1AB[NBYTE];
unsigned char number2AB[NBYTE];

void
#ifdef PROTOTYPES
main(void)
#else
main()
#endif
{
    int i;
    unsigned char condcodeB;		/* Returned condition codes */

    printf("Extended Integer Arithmetic Library\n");
    printf("Copyright (C) 1989 1992, by Fred Motteler\n");
    printf("All Rights Reserved\n");
    printf("Integer addition/subtraction test:\n");
    for (i = 0; i < NBYTE; i++)
    {
	number1AB[i] = (unsigned char)(i << 4);
	number2AB[i] = (unsigned char)((i + NBYTE) << 4);
    }
    printf("      addend: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");

    printf("      augend: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number2AB[i]);
    printf("\n");

    condcodeB = iaddm( number1AB, number2AB, NBYTE );
    printf("         sum: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");
    printf("Condition codes: %2x\n", condcodeB);

    condcodeB = isubm( number1AB, number2AB, NBYTE );
    printf("  difference: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");
    printf("Condition codes: %2x\n", condcodeB);

    printf("Unsigned multiplication test:\n");
    for (i = 0; i < NBYTE; i++)
    {
	number1AB[i] = 0;
	number2AB[i] = 0;
    }
    number1AB[(NBYTE-1)] = 56;
    number2AB[(NBYTE-1)] = 77;

    printf("multiplicand: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");

    printf("  multiplier: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number2AB[i]);
    printf("\n");

    condcodeB = umultm( number1AB, number2AB, NBYTE );
    printf("     product: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");
    printf("Condition codes: %2x\n", condcodeB);

    printf("Unsigned division test:\n");
    for (i = 0; i < NBYTE; i++)
	number2AB[i] = 0;
    number2AB[(NBYTE-1)] = 78;

    printf("    dividend: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");

    printf("     divisor: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number2AB[i]);
    printf("\n");

    condcodeB = udivm( number1AB, number2AB, NBYTE );
    printf("    quotient: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number2AB[i]);
    printf("\n");

    printf("   remainder: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");
    printf("Condition codes: %2x\n", condcodeB);

    printf("Signed multiplication test:\n");
    for (i = 0; i < NBYTE; i++)
    {
	number1AB[i] = 0;
	number2AB[i] = 0;
    }
    number1AB[(NBYTE-1)] = 56;
    number2AB[(NBYTE-1)] = 77;
    inegm( &(number2AB[0]), NBYTE );

    printf("multiplicand: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");

    printf("  multiplier: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number2AB[i]);
    printf("\n");

    condcodeB = smultm( number1AB, number2AB, NBYTE );
    printf("     product: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");
    printf("Condition codes: %2x\n", condcodeB);

    printf("Signed division test:\n");
    for (i = 0; i < NBYTE; i++)
	number2AB[i] = 0;
    number2AB[(NBYTE-1)] = 78;

    printf("    dividend: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");

    printf("     divisor: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number2AB[i]);
    printf("\n");

    condcodeB = sdivm( number1AB, number2AB, NBYTE );
    printf("    quotient: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number2AB[i]);
    printf("\n");

    printf("   remainder: ");
    for (i = 0; i < NBYTE; i++)
	printf("%2x",number1AB[i]);
    printf("\n");
    printf("Condition codes: %2x\n", condcodeB);
    exit( 0 );
}
#endif

/*
** function:	unsigned char itestm(unsigned char *valuePB,
**				     unsigned char condcodeB,
**				     int nbyteN, int signoneN,
**				     int signtwoN)
**
** Set the condition code bits: Z V S C appropriately.  The value of the
** carry bit must be passed in condcodeB.  The value of the condition codes
** are returned by the function.
**
** The value in valuePB is not affected.
*/
unsigned char
#ifdef PROTOTYPES
itestm(unsigned char *valuePB, unsigned char condcodeB, int nbyteN,
       int signoneN, int signtwoN)
#else
itestm(valuePB, condcodeB, nbyteN, signoneN, signtwoN)
unsigned char *valuePB;			/* Pointer to value to test */
unsigned char condcodeB;		/* Condition code carry bit value */
int nbyteN;				/* Length of value to test */
int signoneN;				/* Signs of values being compared */
int signtwoN;
#endif
{
    /* Make sure that only the CARRY bit has been set */
    condcodeB &= CARRY;

    /* Test ms bit of value and set SIGN bit accordingly */
    condcodeB |= (((*valuePB) >> 6) & SIGN);

    /* Test if the value is zero */
    if (ucheckm(valuePB,nbyteN) == 0) condcodeB |= ZERO;

    /* Determine if an overflow has occurred.  An overflow happens if
     * only when the sign of the two operands are the same, and the
     * sign of the result is different.  Note that if the sign of the
     * operands are different, overflow (or underflow) is not possible.
     * Also note that this routine is called only from iaddm() or isubm().
     * These routines set up signoneN and signtwoN appropriately. */
    if (signoneN == signtwoN)
    {
	/* Check if the sign of the result is different.  Note that the
	 * sign bit is set according to the result.  If an overflow or
	 * underflow occurs, then the sign of the result is just
	 * opposite what it should have been if no overflow occurred. */
	if (((condcodeB & SIGN) == 0) && (signoneN == (-1)))
	{
	    /* Overflow, sign of result should be positive */
	    condcodeB &= (~SIGN);
	    condcodeB |= OVERFLOW;
	}
	if (((condcodeB & SIGN) == SIGN) && (signoneN == 1))
	{
	    /* Underflow, sign of result should be negative */
	    condcodeB |= (SIGN | OVERFLOW);
	}
    }
    return( condcodeB );
}

/*
** function:	unsigned char iaddm(unsigned char *sumPB,
**				    unsigned char *termPB,
**				    int nbyteN)
**
** Add two nbyteN long numbers together.  The contents of the integer
** pointed to by termPB are added to the contents of the integer pointed
** to by sumPB.
**
** The value in termPB is not affected.
**
** All of the condition code bits are affected...
*/
unsigned char
#ifdef PROTOTYPES
iaddm(unsigned char *sumPB, unsigned char *termPB, int nbyteN)
#else
iaddm(sumPB, termPB, nbyteN)
unsigned char *sumPB;			/* Initial augend / final sum */
unsigned char *termPB;			/* addend */
int nbyteN;				/* Number of bytes precision */
#endif
{
    int i;
    int temptermN;
    int sumsignN, termsignN;		/* Sign of values to add */

    /* Save the signs of the values to add together */
    if (((*termPB) & NSIGN) == 0)
	termsignN = 1;
    else
	termsignN = (-1);
    if (((*sumPB) & NSIGN) == 0)
	sumsignN = 1;
    else
	sumsignN = (-1);

    temptermN = 0;
    termPB += nbyteN;
    sumPB += nbyteN;
    for (i = (nbyteN - 1); i >= 0; i--)
    {
	temptermN = temptermN + (int)(*(--sumPB)) + (int)(*(--termPB));
	*sumPB = (unsigned char)temptermN;
	temptermN >>= 8;
    }

    return(itestm(sumPB, temptermN, nbyteN, sumsignN, termsignN));
}

/*
** function:	unsigned char isubm(unsigned char *sumPB,
**				    unsigned char *termPB,
**				    int nbyteN)
**
** Subtract one unsigned nbyteN length number from another.  The contents
** of the integer pointed to by termPB are subtracted from the contents
** of the integer pointed to by sumPB.
**
** The value in termPB is not affected.
**
** All of the condition code bits are affected...
*/
unsigned char
#ifdef PROTOTYPES
isubm(unsigned char *sumPB, unsigned char *termPB, int nbyteN)
#else
isubm(sumPB, termPB, nbyteN)
unsigned char *sumPB;
unsigned char *termPB;
int nbyteN;				/* Number of bytes precision */
#endif
{
    int i;
    int temptermN;
    int sumsignN, termsignN;		/* Sign of values to subtract */

    /* Save the signs of the values.  Note that the integer pointed to by
     * termPB is subtracted from the integer pointed to by sumPB.  This is
     * just like adding the negative of termPB, thus flip its sign in
     * advance... */
    if (((*termPB) & NSIGN) == 0)
	termsignN = (-1);
    else
	termsignN = 1;
    if (((*sumPB) & NSIGN) == 0)
	sumsignN = 1;
    else
	sumsignN = (-1);

    temptermN = 0;

    for (i = (nbyteN - 1); i >= 0; i--)
    {
	temptermN = (int)(*(sumPB+i)) - (int)(*(termPB+i)) - temptermN;
	*(sumPB+i) = (unsigned char)temptermN;
	temptermN >>= 8;
	temptermN &= 0x1;
    }

    return(itestm(sumPB, temptermN, nbyteN, sumsignN, termsignN));
}

/*
** function:	int ucheckm(unsigned char *termPB, int nbyteN )
**
** Check if the integer pointed to by termPB is zero.  If so, then a zero
** value is returned.
**
** The value in termPB is not affected.
*/
int
#ifdef PROTOTYPES
ucheckm(unsigned char *termPB, int nbyteN)
#else
ucheckm(termPB,nbyteN)
unsigned char *termPB;
int nbyteN;
#endif
{
    int runsumN;
    unsigned char *finalPB;

    runsumN = 0;
    finalPB = termPB + nbyteN;
    while (termPB != finalPB)
	runsumN += *termPB++;
    return(runsumN);
}

/*
** function:	int ucmpm(unsigned char *firstPB,
**			  unsigned char *secondPB,
**			  int nbyteN)
**
** Compare the integer pointed to by firstPB with the integer pointed to by
** secondPB.  If the first integer is bigger, then return 1.  If both are
** the same size, return 0.  If the first integer is smaller, then return -1.
*/
int
#ifdef PROTOTYPES
ucmpm(unsigned char *firstPB, unsigned char *secondPB, int nbyteN)
#else
ucmpm(firstPB, secondPB, nbyteN)
unsigned char *firstPB;
unsigned char *secondPB;
int nbyteN;
#endif
{
    int i;

    for (i = 0; i < nbyteN; i++)
    {
	if (*firstPB > *secondPB) return( 1 );
	if (*firstPB++ < *secondPB++) return( -1 );
    }
    return(0);
}


/*
** function:	 void ushftlm(unsigned char *termPB, int nbyteN)
**
** Shift the unsigned integer pointed to by termPB left one bit.
*/
void
#ifdef PROTOTYPES
ushftlm(unsigned char *termPB, int nbyteN)
#else
ushftlm(termPB, nbyteN)
unsigned char *termPB;
int nbyteN;
#endif
{
    int temptermN;
    unsigned char *originalPB;

    temptermN = 0;
    originalPB = termPB;		/* Point to start of integer */
    termPB += nbyteN;			/* Point just past end of integer */
    while(termPB != originalPB)
    {
	temptermN += ((int)(*(--termPB)) << 1);
	*termPB = (unsigned char)temptermN;
	if ((temptermN &= 256) != 0)
	    temptermN = 1;
    }
    return;
}

/*
** function:	void ushftrm(unsigned char *termPB, int nbyteN )
**
** Shift the unsigned integer pointed to by termPB right one bit.
*/
void
#ifdef PROTOTYPES
ushftrm(unsigned char *termPB, int nbyteN)
#else
ushftrm(termPB, nbyteN)
unsigned char *termPB;
int nbyteN;
#endif
{
    int temptermN;
    unsigned char *finalPB;

    temptermN = 0;
    finalPB = termPB + nbyteN;
    while (termPB != finalPB)
    {
	temptermN += (int)(*termPB);
	*(termPB++) = (unsigned char)(temptermN >> 1);
	if ((temptermN &= 1) != 0)
	    temptermN = 256;
    }
    return;
}

/*
** function:	unsigned char umultm(unsigned char *prodPB,
**				     unsigned *termPB,
**				     int nbyteN)
**
** Mulitply two unsigned NBIT bit numbers together.  The contents of the
** integer pointed to by termPB are multiplied with the contents of the
** integer pointed to by prodPB.   The product is written to the integer
** pointed to by prodPB.
**
** The value in termPB is modified.
*/
unsigned char
#ifdef PROTOTYPES
umultm(unsigned char *prodPB, unsigned char *termPB, int nbyteN)
#else
umultm(prodPB, termPB, nbyteN)
unsigned char *prodPB;
unsigned char *termPB;
int nbyteN;
#endif
{
    int i;
    unsigned char *partprodPB;
    unsigned char condcodeB;
    unsigned char tempccB;

    condcodeB = 0;
    tempccB = 0;
    partprodPB = (unsigned char *) FMALLOC((unsigned)nbyteN, "UMULTM");
    for (i = 0; i < nbyteN; i++)
	*(partprodPB+i) = 0;

    while ((ucheckm(termPB, nbyteN)) != 0)
    {
	if (((*(termPB+(nbyteN-1))) & 1) == 1)
	    tempccB |= iaddm(partprodPB, prodPB, nbyteN);

	/* Check if an overflow occurred during the addition or during
	 * previous shift of the multiplicand left. */
	if ((tempccB & CARRY) == CARRY)
	{
	    /* Overflow has or will occur. */
	    for (i = 0; i < nbyteN; i++)
	    {
		*(prodPB + i) = 0xff;		/* Write out max int */
		*(termPB + i) = 0;		/* Clear multiplier */
	    }
	    FFREE(partprodPB);
	    return((unsigned char) (OVERFLOW | SIGN));
	}
	/* Check if the shift of the multiplicand left will result in
	 * an overflow. */
	if ((*prodPB & NSIGN) == NSIGN)
	    tempccB = CARRY;
	else
	    tempccB = 0;

	ushftlm(prodPB, nbyteN);
	ushftrm(termPB, nbyteN);
    }

    for (i = 0; i < nbyteN; i++)
	prodPB[i] = *(partprodPB+i);

    /* Set the appropriate condition code bits, only S and N are set */
    /* Test ms bit of value and set SIGN bit accordingly */
    condcodeB |= (((*prodPB) >> 6) & SIGN);

    /* Test if the value is zero */
    if (ucheckm(prodPB,nbyteN) == 0) condcodeB |= ZERO;
    FFREE(partprodPB);
    return(condcodeB);
}

/*
** function:	void inegm(unsigned char *valuePB, int nbyteN)
**
** Negate the integer value pointed to by valuePB.  This does a two's
** complement.
*/
void
#ifdef PROTOTYPES
inegm(unsigned char *valuePB, int nbyteN)
#else
inegm(valuePB, nbyteN)
unsigned char *valuePB;
int nbyteN;
#endif
{
    int i,temptermN;

    /* Complement the value */
    for (i = 0; i < nbyteN; i++)
	(*(valuePB+i)) = ~(*(valuePB+i));

    /* Add 1 to the value, allow for all possible carry's. */
    temptermN = 1;
    valuePB += nbyteN;
    for (i = (nbyteN - 1); i >= 0; i--)
    {
	temptermN += ((int)(*(--valuePB)));
	*valuePB = (unsigned char)temptermN;
	temptermN >>= 8;
    }
    return;
}

/*
** function:	unsigned char smultm(unsigned char *prodPB,
**				     unsigned char *termPB,
**				     int nbyteN)
**
** Mulitply two signed NBIT bit numbers together.  The contents of the
** integer pointed to by termPB are multiplied with the contents of the
** integer pointed to by prodPB.   The product is written to the integer
** pointed to by prodPB.
**
** The value in termPB is modified.
*/
unsigned char
#ifdef PROTOTYPES
smultm(unsigned char *prodPB, unsigned char *termPB, int nbyteN)
#else
smultm(prodPB, termPB, nbyteN)
unsigned char *prodPB;
unsigned char *termPB;
int nbyteN;
#endif
{
    unsigned char prodsignB, termsignB;
    unsigned char condcodeB;
    int i;
    
    /* Isolate the sign bits of multiplier and multiplicand */
    prodsignB = (((*prodPB) >> 6) & SIGN);
    termsignB = (((*termPB) >> 6) & SIGN);

    /* Change sign if necessary */
    if (prodsignB != 0)
	inegm(prodPB, nbyteN);
    if (termsignB != 0)
	inegm(termPB, nbyteN);

    condcodeB = umultm(prodPB, termPB, nbyteN);

    /* Product should be + if no overflow.  Note that the unsigned multiply
    ** routine will write out all ones as the result if there is an overflow.
    ** Also, if the value is ok as an unsigned, but is an overflow as a
    ** signed value, then this test will catch both. */
    if ((condcodeB & SIGN) != 0)
    {
	/* Overflow has occured. */
	condcodeB |= OVERFLOW;
	if (prodsignB != termsignB)
	{
	    /* Negative overflow, write out minimum integer value */
	    for (i = 0; i < nbyteN; i++)
	    {
		*(termPB + i) = 0;
		*(prodPB + i) = 0;
	    }
	    *prodPB = 0x80;		/* Set result sign bit. */
	    condcodeB |= SIGN;
	}
	else
	{
	    /* Positive overflow, write out maximum integer value */
	    for (i = 0; i < nbyteN; i++)
	    {
		*(termPB + i) = 0;
		*(prodPB + i) = 0xff;
	    }
	    *prodPB = 0x7f;		/* Clear result sign bit. */
	    condcodeB &= (~SIGN);	/* Clear sign condition code bit. */
	}
	return(condcodeB);
    }

    /* Change sign of the product if appropriate */
    if (prodsignB != termsignB)
	inegm(prodPB, nbyteN);

    /* Test ms bit of value and set SIGN bit accordingly */
    condcodeB |= (((*prodPB) >> 6) & SIGN);
    return(condcodeB);
}

/*
** function:	unsigned char udivm(unsigned char *dividPB,
**				    unsigned char *termPB,
**				    int nbyteN )
**
** Divide one unsigned NBIT bit by another.  The contents of the
** integer pointed to by termPB are divided into the contents of the
** integer pointed to by dividPB.   The quotient is written to the integer
** pointed to by termPB.  The remainder is written to the integer pointed
** to by dividPB.
*/
unsigned char
#ifdef PROTOTYPES
udivm(unsigned char *dividPB, unsigned char *termPB, int nbyteN)
#else
udivm(dividPB, termPB, nbyteN)
unsigned char *dividPB;			/* Dividend in, remainder out */
unsigned char *termPB;			/* Divisor in, quotient out */
int nbyteN;
#endif
{
    int i;
    unsigned char condcodeB;
    unsigned char *quotientPB;		/* Working copy of quotient */
    int shftcntN;			/* Divisor shift counter */

    /* Check for divide by zero.  In this particular case, this is the
     * only way for an unsigned integer divide routine to return an
     * OVERFLOW. */
    if ((ucheckm(termPB, nbyteN)) == 0 )
    {
	for (i = 0; i < nbyteN; i++)
	{
	    *(termPB+i) = 0xff;		/* Return maximum int as quotient */
	    *(dividPB+i) = 0;		/* Zero remainder out */
	}
	return((OVERFLOW | SIGN));
    }

    /* Check for zero dividend */
    if ((ucheckm(dividPB, nbyteN)) == 0 )
    {
	for (i = 0; i < nbyteN; i++)
	{
	    *(termPB+i) = 0xff;		/* Return 0 quotient and remainder */
	    *(dividPB+i) = 0;
	}
	return(ZERO);
    }

    /* Valid input operands, continue with division operation. */
    condcodeB = 0;

    quotientPB = (unsigned char *) FMALLOC((unsigned)nbyteN, "UDIVM");
    for (i = 0; i < nbyteN; i++)	/* Initialize the quotient */
	*(quotientPB+i) = 0;
    shftcntN = 0;			/* Initialize shift count */

    /* Check if the divisor is larger than the dividend, if so, then return
    ** the current dividend as the remainder, and the current quotient. */
    if ((ucmpm(dividPB, termPB, nbyteN)) < 0)
    {
	for (i = 0; i < nbyteN; i++)
	    *(termPB+i) = *(quotientPB+i);
	FFREE(quotientPB);
	return(condcodeB);
    }

    /* Check the relative size of the divisor and dividend.  Shift the divisor
    ** left until is is just larger than the dividend. */
    while ((ucmpm(dividPB, termPB, nbyteN)) >= 0 )
    {
	ushftlm(termPB, nbyteN);
	shftcntN++;
    }
    ushftrm(termPB, nbyteN);

    /* The divisor has been scaled to be just smaller (or equal to) the
    ** dividend.  The value in shftcntN is the number of significant bits
    ** in the quotient.   Everything is set up to finally do the division. */

    while (shftcntN-- > 0)
    {
        ushftlm(quotientPB, nbyteN);
	if ((ucmpm(dividPB, termPB, nbyteN)) >= 0 )
	{
	    isubm(dividPB, termPB, nbyteN);
	    *(quotientPB+(nbyteN-1)) += 1;
	}
	ushftrm(termPB, nbyteN);
    }
    for (i = 0; i < nbyteN; i++)
	*(termPB+i) = *(quotientPB+i);

    /* Test ms bit of value and set SIGN bit accordingly, depends on value
     * of quotient, and not the remainder. */
    condcodeB |= (((*termPB) >> 6) & SIGN);

    /* Test if the value is zero */
    if (ucheckm(termPB,nbyteN) == 0) condcodeB |= ZERO;
    FFREE(quotientPB);
    return (condcodeB);
}

/*
** function:	unsigned char sdivm(unsigned char *dividPB,
**				    unsigned char *termPB,
**				    int nbyteN)
**
** Divide one signed NBIT bit numbers into another.  The contents of the
** integer pointed to by termPB is divided into the contents of the
** integer pointed to by dividPB.  The quotient is written to the integer
** pointed to by termPB.  The remainder is written to the integer pointed
** to by dividPB.
**
** The values in both dividPB and termPB are modified.
*/
unsigned char
#ifdef PROTOTYPES
sdivm(unsigned char *dividPB, unsigned char *termPB, int nbyteN)
#else
sdivm(dividPB, termPB, nbyteN)
unsigned char *dividPB;
unsigned char *termPB;
int nbyteN;
#endif
{
    unsigned char prodsignB, termsignB;
    unsigned char condcodeB;
    int i;
    
    /* Isolate the sign bits of dividend and divisor */
    prodsignB = (((*dividPB) >> 6) & SIGN);
    termsignB = (((*termPB) >> 6) & SIGN);

    /* Change sign if necessary */
    if (prodsignB != 0)
	inegm(dividPB, nbyteN);
    if (termsignB != 0)
	inegm(termPB, nbyteN);

    condcodeB = udivm(dividPB, termPB, nbyteN);

    /* Quotient should be + if no overflow */
    if ((condcodeB & SIGN) != 0)
    {
	condcodeB |= OVERFLOW;
	if (prodsignB != termsignB)
	{
	    /* Negative overflow. */
	    for (i = 1; i < nbyteN; i++)
		*(termPB + i) = 0;
	    *termPB = 0x80;
	}
	else
	{
	    /* Positive overflow. */
	    *termPB = 0x7f;		/* Reset sign bit. */
	    condcodeB &= (~SIGN);	/* Reset condition code sign bit. */
	}
	return(condcodeB);
    }

    /* Change sign of the product if appropriate */
    if (prodsignB != termsignB)
    {
	inegm(dividPB, nbyteN);
	inegm(termPB, nbyteN);
    }

    /* Test ms bit of value and set SIGN bit accordingly */
    condcodeB |= (((*termPB) >> 6) & SIGN);
    return(condcodeB);
}
