/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	error;
include	process;
include	vmemory;
include	memory;
include	node;
include	kprintf;
include	hardware, list;
include	sound;

SECTOR_SIZE:		public	const	int = 512;
MAX_PARTITIONS:		public	const	int = 1 << INDEX_BITS;

systemIndicators:	public	type	unsigned[16] = {
	SI_DOS_12 =		0x001,
	SI_XENIX2 = 		0x002,
	SI_XENIX3 =		0x003,
	SI_DOS_16 =		0x004,
	SI_EXTENDED =		0x005,
	SI_BIGDOS =		0x006,
	SI_HPFS =		0x007,
	SI_SPLIT =		0x008,
	SI_BOOTMGR =		0x00a,			// OS/2 2.0 boot manager
	SI_DOS_12_HIDDEN =	0x011,			// hidden by boot manager
	SI_DOS_16_HIDDEN =	0x014,			// hidden by boot manager
	SI_BIGDOS_HIDDEN = 	0x016,			// hidden by boot manager
	SI_DM_RO =		0x050,
	SI_DM_RW =		0x051,
	SI_GB =			0x056,
	SI_SPEED61 =		0x061,
	SI_386IX =		0x063,
	SI_NOVELL =		0x064,
	SI_PCIX =		0x075,
	SI_CPM =		0x0db,
	SI_SPEEDE1 =		0x0e1,
	SI_SPEEDE3 =		0x0e3,
	SI_SPEEDF1 =		0x0f1,
	SI_SPEEDF4 =		0x0f4,
	SI_BBT =		0x1ff,

		// DOS floppy media

	SI_DOS_360K =		0x100,
	SI_DOS_1_2M =		0x102,
	SI_DOS_720K =		0x103,
	SI_DOS_1_4M =		0x104,
	};

Partition:		public	[MAX_PARTITIONS] partition_t;

partition_t:	public	type	{
	public:

	sectorCount:		unsigned;
	sectorOffset:		unsigned;
	system:			unsigned[16];
	index:			byte;
	drive:			ref diskDrive_t;
	removable:		boolean;

create:	factory	(sys: unsigned[16], rem: boolean) ref partition_t =
	{
	n:	threadLock;
	i:	int;

	n lock();
	for	(i = 0; i < MAX_PARTITIONS; i++)
		if	(Partition[i].system == 0){
			Partition[i].system = sys;
			Partition[i].index = i;
			n unlock();
			Partition[i].removable = rem;
			return &Partition[i];
			}
	n unlock();
	return 0;
	}

display:	() =
	{
	kprintf("Partition %2d sys %03x [%7d:%7d]\n", index,
				system, sectorOffset, sectorCount);
	}

readDisk:	(sector: long, buf: pointer, count: int) =
	{
	drive read(sector + sectorOffset, buf, count);
	}

writeDisk:	(sector: long, buf: pointer, count: int) =
	{
	drive write(sector + sectorOffset, buf, count);
	}

read:	(psect: long, buf: * char, sectCount: long) =
	{
	c:		* cache;
	i:		int;
	j:		int;

	while	(sectCount > 0){
		j = sectCount;
		if	(j > 256)
			j = 256;
		for	(i = 0; i < j; i++){
			c = findThisBlock(psect + i);
			if	(c)
				break;
			}

			// We've got some number of blocks to read, read them

		if	(i){
			readDisk(psect, buf, i);

				// Now put the data into the cache

			while	(i){
				fillSector(buf, psect);
				i--;
				buf += SECTOR_SIZE;
				psect++;
				sectCount--;
				}
			if	(sectCount == 0)
				break;
			}
		memCopy(buf, CacheData + (c - Cache) * SECTOR_SIZE, SECTOR_SIZE);
		c release();
		sectCount--;
		psect++;
		buf += SECTOR_SIZE;
		}
	}

/*
	This routine is used for full-block disk writes.  Multiple block
	writes are also executed here.
 */
write:	(psect: long, buf: * char, sectCount: long) =
	{
	writeDisk(psect, buf, sectCount);
	if	(sectCount > CacheSize)
		sectCount = CacheSize;
	while	(sectCount > 0){
		fillSector(buf, psect);
		buf += SECTOR_SIZE;
		psect++;
		sectCount--;
		}
	}
/*
	This routine is used for full-block disk fills.  Multiple block
	fills are also executed here.
 */
fill:	(psect: long, sectCount: long) =
	{
	while	(sectCount > 0){
		emptySector(psect);
		psect++;
		sectCount--;
		}
	}

emptySector:	(psect: long) =
	{
	c:		ref cache;
	cp:		* char;

	c = findSector(psect, FALSE);
	cp = CacheData + (c - Cache) * SECTOR_SIZE;
	memSet(cp, 0, SECTOR_SIZE);
	writeDisk(psect, cp, 1);
	c release();
	}

fillSector:	(buf: pointer, psect: long) =
	{
	c:		ref cache;
	cp:		* char;

	c = findSector(psect, FALSE);
	cp = CacheData + (c - Cache) * SECTOR_SIZE;
	memCopy(cp, buf, SECTOR_SIZE);
	c release();
	}

/*
	This function writes a sector from the disk cache back to disk.
	This routine is used for directory operations and for partial block
	disk writes.

	The correct protocol to read a sector into the cache, modify some
	of it, and write it back out to disk is this:

	Use readSector to get the data pointer into the cache entry.  Do
	the data modification and call writeSector with the buffer pointer.
	The sector and partition specification are found in the cache.
 */
writeSector:	(buf: pointer) =
	{
	c:		* cache;

	c = Cache + unsigned(ref char(buf) - CacheData) >> 9;
	writeDisk(c sector(), buf, 1);
	c release();
	}

readSector:	(psect: long) pointer =
	{
	c:		ref cache;
	cp:		* char;

	c = findSector(psect, TRUE);
	return CacheData + (c - Cache) * SECTOR_SIZE;
	}

releaseSector:	(buf: pointer) =
	{
	c:		* cache;

	c = Cache + unsigned(ref char(buf) - CacheData) >> 9;
	c release();
	}

private:

findSector:	(sector: long, readit: boolean) ref cache =
	{
	c:		ref cache;
	cmap:		ref cache;
	cp:		ref char;
	i:		int;
	j:		int;
	lowClock:	unsigned[16];
	lowCache:	ref cache;
	lowData:	ref char;
	ck:		cookie;

	ck = makeCookie(sector);
	j = sector % SPREAD;
	CacheAcquire down(FALSE);
	cmap = CacheMap[j];
	for	(c = cmap; c; c = c->nextHash)
		if	(c->cook == ck){

				// Move the block to the new end of the age q

			c extract();
			CacheAge enqueue(c);
			c acquire();
			CacheAcquire up();
			AlysNode.cacheHits++;
			return c;
			}

		// Find the oldest available block in the cache

	for	(c = ref cache(CacheAge.next);; c = ref cache(c->next)){
		if	(c == &CacheAge){	// entire cache is busy!
			c = ref cache(CacheAge.next);
			break;			// wait for the oldest to free up
			}
		if	(!c->busy)
			break;
		}
	c extract();
	c acquire();
	lowCache = c;
	i = lowCache->cook % SPREAD;

		// Hash queues are changing!

	if	(i != j){

			// Pull lowCache off its hash queue

		c = lowCache->prevHash;
		if	(c)
			c->nextHash = lowCache->nextHash;
		c = lowCache->nextHash;
		if	(c)
			c->prevHash = lowCache->prevHash;
		if	(lowCache == CacheMap[i])
			CacheMap[i] = c;

			// Put lowCache onto its new hash queue

		if	(cmap)
			cmap->prevHash = lowCache;
		lowCache->nextHash = cmap;
		lowCache->prevHash = 0;
		CacheMap[j] = lowCache;
		}
	lowCache->cook = ck;
	CacheAge enqueue(lowCache);
	CacheAcquire up();
	cp = CacheData + (lowCache - Cache) * SECTOR_SIZE;
	if	(readit)
		readDisk(sector, cp, 1);
	AlysNode.cacheMisses++;
	return lowCache;
	}

makeCookie:	(sector: long) cookie =
	{
	return index << INDEX_SHIFT + sector;
	}

findThisBlock:	(sector: long) * cache =
	{
	c:		* cache;
	ck:		cookie;

	ck = makeCookie(sector);
	for	(c = CacheMap[sector % SPREAD]; c; c = c->nextHash)
		if	(c->cook == ck){

			// Move the block to the new end of the cache

			c extract();
			CacheAge enqueue(c);
			AlysNode.cacheHits++;
			return c;
			}
	return 0;
	}

	};
/*
	The resizeCache routine accepts a size in bytes.  This size is only
	the size of actual data and does not include cache block headers.
	This makes it easier for someone to calculate the number of blocks
	in the cache from the requested size (if the cache header changes
	the number of blocks in the cache won't).
 */
resizeCache:	public	(newSize: unsigned[32]) int =
	{
	newCacheSize:	unsigned[32];
	i, j:		int;

	newCacheSize = newSize / SECTOR_SIZE;
	if	(newCacheSize < 1)
		newCacheSize = 1;		// rock bottom minimum,
						// very slow cache
	newSize = newCacheSize * (SECTOR_SIZE + sizeof cache);
	if	(!CacheAcquire down(TRUE))
		return ERRINTERRUPTED;

		// Now we control the cache, don't give it up until we're
		// done.  No more cache entries will be allocated from now
		// until we're done.

		// First acquire all the blocks.  This will quiesce the cache
		// and let us modify it without worry.  Any blocks already
		// acquired will eventually free up.

	for	(i = 0; i < CacheSize; i++)
		Cache[i] acquire();

	CacheSegment unlock();
	j = CacheSegment grow(newSize);
	if	(j < 0){			// couldn't grow the cache,
						// it's unchanged.
		CacheSegment lock(0);

			// release the blocks

		for	(i = 0; i < CacheSize; i++)
			Cache[i] release();
		}
	else
		prepareCache();
	CacheAcquire up();
	return SUCCESS;
	}

startup:	entry	() =
	{
	i:		int;
	c:		ref cache;
	seg:		ref segment;

	if	(HighMemory == 0)		// No extended RAM
		i = 32;				//	16K of cache
	else if	(HighMemory < 0x80000)		// < 512K of extended RAM
		i = 128;			//	64K of cache
	else if	(HighMemory < 0x100000)		// < 1024K of extended RAM
		i = 256;			//	128K of cache
	else if	(HighMemory < 0x200000)		// < 2048K of extended RAM
		i = 512;			// 	256K of cache
	else if	(HighMemory < 0x400000)		// < 4096K of extended RAM
		i = 1024;			// 	512K of cache
	else if	(HighMemory < 0x800000)		// < 8192K of extended RAM
		i = 2048;			//	1 meg of cache
	else					// >= 8192K of extended RAM
		i = 4096;			// 	2 megs of cache
	for	(;;){
		j:	vaddr_t;

		j = i * (sizeof cache + SECTOR_SIZE);
		seg = segment create(0, j, j);
		if	(seg)
			break;
		i >>= 1;
		}
	if	(i < 16)
		kprintf("No room for significant data caching\n");
	CacheSegment = seg;
	CacheAcquire = [ 1 ];
	BlockAcquire = [ 0 ];
	CacheSize = i;
	prepareCache();
	}

prepareCache:	() =
	{
	i:	int;
	c:	ref cache;

	CacheData = CacheSegment lock(0);
	Cache = ref cache(CacheData + CacheSize * SECTOR_SIZE);
	memSet(Cache, 0, CacheSize * sizeof cache);
	CacheAge constructor();
	for	(i = 0, c = Cache; i < CacheSize; i++, c++)
		CacheAge enqueue(c);
	memSet(&CacheMap, 0, sizeof CacheMap);
	}

SPREAD:		const	unsigned = 128;
/*
	The hash cookie is a combination of the partition and the partition
	sector of the cache block.  The partition bits are high, so we can
	hash the cookie by simply extracting the low order bits.  This will
	tend to separate blocks reasonably well.
 */
cookie:	type	unsigned[COOKIE_BITS];

INDEX_SHIFT:		public	const	int = COOKIE_BITS - INDEX_BITS;
COOKIE_BITS:		public	const	int = 32;
INDEX_BITS:		public	const	int = 5;

cache:	type	inherit	queue	{
	public:

	nextHash:	ref cache;
	prevHash:	ref cache;
	cook:		cookie;
	busy:		boolean;

acquire:	() =
	{
	n:	threadLock;

	n lock();
	if	(busy){
		DesiredBlock = self;
		BlockAcquire down(FALSE);
		}
	DesiredBlock = 0;
	busy = TRUE;
	n unlock();
	}

release:	() =
	{
	n:	threadLock;

	n lock();
	busy = FALSE;
	if	(DesiredBlock == self)
		BlockAcquire wakeup();
	n unlock();
	}

sector:	() long =
	{
	return cookieSector(cook);
	}

partition:	() int =
	{
	return cookiePartition(cook);
	}

	};

CacheMap:	[SPREAD] ref cache;

Cache:			ref cache;
CacheAge:		queue;		// oldest cache blocks are at head
CacheData:		ref char;
CacheSize:		int;
CacheSegment:	public	ref segment;
CacheAcquire:		semaphore;
BlockAcquire:		semaphore;
DesiredBlock:		ref cache;

cookiePartition:	(ck: cookie) int =
	{
	return ck >> INDEX_SHIFT;
	}

cookieSector:	(ck: cookie) long =
	{
	return ck & (1 << INDEX_SHIFT - 1);
	}

diskDrive_t:	public	type	{
	public:

read:	dynamic	(sector: long, buffer: pointer, count: int) =
	{
	}

write:	dynamic	(sector: long, buffer: pointer, count: int) =
	{
	}

	};

