/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	sound;
include	alys;
include	hardware, list;
include	error;
include	process;
include	arena;
include	message;
include	timer;
include	_startup;
include	kprintf;
include	karena;
include	trace;
include	filesys;

NOBJECTS:		const	int = 256;
OID_BASE:		const	int = NLOCALS;
NLOCALS:	public	const	int = 32;

ObjectTable:	public	[NOBJECTS] ref object = [ &BootObject, &BootJobObject ];
BootObject:	public	kernelObject;
BootContext:	public	objectContext = [ 1 ];
BootJob:	public	job;
BootJobObject:	public	kernelObject;
MyNetNodeId:	public	objectId;
OutgoingObject:	public	ref object;

sendWait:	public	(target: ref far external, func: int, 
					buf: vaddr_t, len: int,
					rbuf: vaddr_t, rlen: int) int =
	{
	o:	ref object;
	r:	unsigned[16];
	src:		ref object;

	src = CurProc->animates;
//	kprintf("sendWait(%d, %d, %x:%d, %x:%d) <- %d\n", target, func, buf, len,
//			rbuf, rlen, src->me);
//	CurProc->where dumpHex(buf, len);
	o = objectEntry(target);
	if	(o){
//		kprintf("target: %x (%S)\n", o->me, o->name);
		if	(func == int(&external.childExit)){
			if	(src->parent == o->me)
				r = ~0;
			else
				r = 0;
//			kprintf("sendWait exit: %d (%S)\n", o->me, o->name);
			}
		else if	(CurProc->myJob == o->myJob)
			r = o->jobRights;
		else if	(src->user == o->user)
			r = o->userRights;
		else if	(src->group == o->group)
			r = o->groupRights;
		else
			r = o->worldRights;
		if	(r){
			i:	int;

			if	(func == int(&external.dup)){
				i = o sendWait(target, r, func, 0, 0, 0, 0);
				if	(i >= 0){
					o dup(o->me);
					}
				}
			else
				i = o sendWait(target, r, func, buf, len, 
								rbuf, rlen);
			return i;
			}
		else	{
			kprintf("perm sendWait(%d(%d), %d, %x:%d, %x:%d) <- %d\n", target, o->me, func, buf, len,
					rbuf, rlen, src->me);
			CurProc->where dumpHex(buf, len);
			displayObject(o);
			src rejectedSend(ERRPERMISSION);
			return 0;
			}
		}
	else	{
		kprintf("sendWait unknown object %d\n", target);
		src rejectedSend(ERRINVALIDARGUMENT);
		return 0;
		}
	}

sendWaitInt:	public	(target: ref far external, func: int, 
					buf: vaddr_t, len: int) int =
	{
	o:	ref object;
	r:	unsigned[16];
	src:		ref object;

	src = CurProc->animates;
//	kprintf("sendWaitInt(%d, %d, %x:%d) <- %d\n", target, func, buf, len,
//			src->me);
//	CurProc->where dumpHex(buf, len);
	o = objectEntry(target);
	if	(o){
//		kprintf("target: %x (%S)\n", o->me, o->name);
		if	(CurProc->myJob == o->myJob)
			r = o->jobRights;
		else if	(src->user == o->user)
			r = o->userRights;
		else if	(src->group == o->group)
			r = o->groupRights;
		else
			r = o->worldRights;
		if	(r){
			i:	int;

			if	(func == int(&external.close)){
				if	(o->useCount == 1)
					func = int(&external.delete);
				i = o sendWaitInt(target, r, func, 0, 0);
//				kprintf("int close = %d\n", i);
				if	(i){
//					kprintf("closing %x %S\n", target, o->name);
					src closeLocal(target);
					o close();
					}
				}
			else
				i = o sendWaitInt(target, r, func, buf, len);
//			kprintf("%d (%S) responding to %d: %x\n", o->me,
//					o->name, src->me, i);
			return i;
			}
		else	{
			kprintf("perm sendWaitInt(%d(%d), %d, %x:%d) <- %d\n", target, o->me, func, buf, len,
					src->me);
			CurProc->where dumpHex(buf, len);
			displayObject(o);
			src rejectedSend(ERRPERMISSION);
			return 0;
			}
		}
	else	{
		kprintf("sendWaitInt unknown object %d\n", target);
		src rejectedSend(ERRINVALIDARGUMENT);
		return 0;
		}
	}

probeObject_:	public	(o: ref far external) ref far external =
	{
	op:	ref object;

	op = objectEntry(o);
	if	(op == 0)
		return 0;
	else
		return op->me;
	}

objectEntry:	public	(obj: ref far external) ref object =
	{
	o:	int;

	o = int(obj);
	if	(o < 0)
		return 0;
	if	(o < NLOCALS){
		if	(CurProc->animates->context == 0)
			return 0;
		o = int(CurProc->animates->context->localObject[o]);
		if	(o == 0)
			return 0;
		}
	if	(o & NODE_MASK){
		if	(o & NODE_MASK != MyNetNodeId)
			return OutgoingObject;
		o &= ~NODE_MASK;
		}
	o -= NLOCALS;
	if	(o >= NOBJECTS)
		return 0;
	return ObjectTable[o];
	}

locateObject:	public	(obj: ref far external) pointer =
	{
	o:	ref object;

	o = objectEntry(obj);
	if	(o == 0 ||
		 o == OutgoingObject)
		return 0;
	if	(!o->isKernel)
		return 0;
	else
		return ref kernelObject(o)->dispatcher;
	}

forkObject:	public	(o: ref far external) ref far external =
	{
	return o;
	}

dupObject:	public	(obj: ref far external) =
	{
	o:	ref object;

	o = objectEntry(obj);
//	kprintf("dupObject(%x) = %x\n", obj, o);
	if	(o)
		o dup(obj);
	}

closeObject:	public	(obj: ref far external) =
	{
	o:	ref object;

	o = objectEntry(obj);
	if	(o == 0)
		return;
	o close();
	}

setNewJobObject:	public	(obj: ref far external,
				 j: ref far job) ref far external =
	{
	o:	ref object;

	o = objectEntry(obj);
	if	(o)
		return o newJob(j);
	else
		return 0;
	}

getJobObject:	public	(obj: ref far external) ref far job =
	{
	o:	ref object;

	o = objectEntry(obj);
	if	(o)
		return o->myJob;
	else
		return 0;
	}

processJob:	public	(obj: ref far external) =
	{
	o:	ref object;

	o = objectEntry(obj);
	if	(o)
		o setJob(CurProc->myJob);
	}

objectContext:	public	type	{
	public:

	refCount:		int;
	parent:			ref objectContext;
	owner:			ref object;
	localObject:		[NLOCALS] ref far external;
	forkAction:		[NLOCALS] forkAction_t;

makeLocal:	(obj: ref far external, slot: ref far external) int =
	{
	i:	int;

	i = int(slot);
	if	(i < 0 || i >= NLOCALS)
		return ERRINVALIDDATA;
	if	(unsigned(obj) < NLOCALS){
		obj = localObject[unsigned(obj)];
		if	(obj == 0)
			return ERRINVALIDDATA;
		obj dup();
		}
	if	(localObject[i])
		localObject[i] close();
	localObject[i] = obj;
	forkAction[i] = FA_REF;
	return SUCCESS;
	}

makeAnyLocal:	(obj: ref far external) ref far external =
	{
	i:	int;

//	kprintf("obj = %x\n", obj);
	if	(unsigned(obj) < NLOCALS){
		obj = localObject[unsigned(obj)];
		if	(obj == 0)
			return ref far external(ERRINVALIDDATA);
		obj dup();
		}
	for	(i = 0; i < NLOCALS; i++){
		if	(localObject[i] == 0){
			localObject[i] = obj;
			forkAction[i] = FA_REF;
//			kprintf("arena %x fd = %x\n", self, i);
			return ref far external(i);
			}
		}
	return ref far external(ERRMFILE);
	}

closeLocal:	(obj: ref far external) =
	{
	i:	unsigned;

	i = unsigned(obj);
	if	(i < NLOCALS)
		localObject[i] = 0;
	}

setForkAction:	(slot: ref far external, action: forkAction_t) int =
	{
	i:	int;
	x:	forkAction_t;

	i = int(slot);
	if	(i < 0 || i >= NLOCALS)
		return ERRINVALIDDATA;
	if	(action != FA_REF &&
		 action != FA_COPY &&
		 action != FA_STATIC &&
		 action != FA_CLOSE)
		return ERRINVALIDDATA;
	x = forkAction[i];
	forkAction[i] = action;
	return x;
	}

copy:	(o: ref object) ref objectContext =
	{
	i:	int;
	oc:	ref objectContext;

	oc = new objectContext;
	oc->refCount = 1;
	memCopy(&oc->localObject, &localObject, sizeof localObject);
	memCopy(&oc->forkAction, &forkAction, sizeof forkAction);
	for	(i = 0; i < NLOCALS; i++){
		x:	ref far external;

		x = oc->localObject[i];
		if	(x == 0 ||
			 x == CurProc->animates->me)
			continue;
		switch	(oc->forkAction[i]){
		case	FA_CLOSE:
			oc->localObject[i] = 0;
			break;

		case	FA_COPY:
			oc->localObject[i] = forkObject(x);
			break;

		case	FA_REF:
			x dup();
			break;

		case	FA_STATIC:
			break;
			}
		}
	oc->owner = o;
	if	(o)
		oc->parent = self;
	else	{
//		kprintf("copy(0) %d %p -> %p\n", CurProc->animates->me,
//					self, oc);
		oc->parent = 0;
		}
	return oc;
	}

dup:	() =
	{
	n:	threadLock;

	n lock();
	refCount++;
	n unlock();
//	kprintf("%d %p dup()\n", CurProc->animates->me, self);
	}

newJob:	(j: ref far job) =
	{
	i:	int;
	
	for	(i = 0; i < NLOCALS; i++){
		if	(localObject[i] == 0)
			continue;
		switch	(forkAction[i]){
		case	FA_CLOSE:
			localObject[i] close();
			localObject[i] = 0;
			break;

		case	FA_COPY:
		case	FA_REF:
			try
				localObject[i] = 
					setNewJobObject(localObject[i], j);
			except	{
				localObject[i] close();
				localObject[i] = 0;
				}
			break;

		case	FA_STATIC:
			break;
			}
		}
	}

close:	() ref objectContext =
	{
	p:	ref objectContext;
	i:	int;
	e:	ref far external;
	n:	threadLock;

//	displayObjects();
//	kprintf("%d %p close() count %d\n", CurProc->animates->me, self, refCount);
	p = parent;
	n lock();
	i = --refCount;
	n unlock();
	if	(i <= 0){
		for	(i = 0; i < NLOCALS; i++){
			e = localObject[i];
			localObject[i] = 0;
			if	(e)
				e close();
			}
		if	(parent)
			parent close();
		free(self);
		}
	return p;
	}

	};

userObject:	public	type	inherit	object	{
	animator:	ref process;
	control:	semaphore;

	public:

		// Signal handling information

	signalHandler:	vaddr_t;

constructor:	(name: [:] char, d: vaddr_t, jr: accessRights_t, ur: accessRights_t,
				gr: accessRights_t, wr: accessRights_t) =
	{
	animator = CurProc;
	control = [ 0 ];
	super constructor(name, FALSE, jr, ur, gr, wr);
	where = CurArena;
	}

serves:	(par: ref far external, p: ref process, ar: ref arena) =
	{
	animator = p;
	parent = par;
	p->animates = self;
	p->where = ar;
	p->myJob = myJob;
	where = ar;
	}

sendWait:	dynamic	(ref far external, r: unsigned[16],
					func: int, buf: vaddr_t, len: int, 
					rbuf: vaddr_t, rlen: int) int =
	{
	m:	ref message;
	src:	ref object;

	src = CurProc->animates;
	m = message create(self, r, src->me, func, buf, len, rbuf, rlen);
	if	(m == 0){
		src rejectedSend(ERRNOMEMORY);
		return 0;
		}

	n:	threadLock;

	n lock();
	msgs enqueue(m);
	n unlock();
	control up();
	return m waitForDone(src);
	}

sendWaitInt:	dynamic	(ref far external, r: unsigned[16],
				func: int, buf: vaddr_t, len: int) int =
	{
	m:	ref message;
	src:	ref object;
	retn:	int;

	src = CurProc->animates;
	m = message createIntReply(self, r, src->me, func, buf, len);
	if	(m == 0){
		src rejectedSend(ERRNOMEMORY);
		return 0;
		}

	n:	threadLock;

	n lock();
	msgs enqueue(m);
	n unlock();
	control up();
	return m waitForDone(src);
	}

receive:	dynamic	(mh: vaddr_t) int =
	{
		// wait for a message

	if	(!control down(TRUE))
		return ERRINTERRUPTED;
	else	{
		m:	ref message;

		for	(;;){
			m = ref message(msgs.next);
//			kprintf("(%S) recv %p ", name, mh);
//			m display();
//			dumpData(&m->header, sizeof m->header);
			if	(m->state != MS_RECEIVED)
				break;
			m reply_(0, 0);
			}
		m->state = MS_RECEIVED;
		return CurArena write(mh, &m->header, sizeof messageHeader);
		}
	}

newJob:	dynamic	(j: ref far job) ref far external =
	{
	return 0;
	}
/*
reply_:	dynamic	(m: ref message, buf: vaddr_t, len: int) =
	{
	m reply_(buf, len);
	}
 */
catchSignal:	(func: vaddr_t) int =
	{
	p:	int;

	p = signalHandler;
	if	(where canCall(func))
		signalHandler = func;
	return p;
	}

signalInfo:	(seq: _sequence_t, hdr: vaddr_t) int =
	{
		// wait for a message

//	kprintf("control down\n");
	if	(!control down(TRUE))
		return ERRINTERRUPTED;
	else	{
//		kprintf("control gained\n");
		m:	ref message;

		for	(;;){
			m = ref message(msgs.next);
			if	(m->state != MS_RECEIVED)
				break;
			m reply_(0, 0);
			}
//		kprintf("m->header.sequence = %d seq = %d\n", m->header.sequence, seq);
		if	(m->header.sequence != seq)
			return ERRINVALIDARGUMENT;
		m->state = MS_RECEIVED;
		return CurArena write(hdr, &m->header, sizeof messageHeader);
		}
	}

processTermination:	dynamic	() =
	{
	n:	threadLock;

	n lock();
	CurProc->childtimes.user += animator->times.user;
	CurProc->childtimes.kernel += animator->times.kernel;
	n unlock();
	animator termination();
	useCount = 1;
	close();
	}

attention:	dynamic	(ref process) =
	{
	m:	ref message;

	kprintf("User object break %d\n", me);
	m = message createSelfGenerated(self, int(&external.attention), 0, 0);
	if	(m){
		n:	threadLock;

		n lock();
		msgs push(m);
		where scheduleSignal(self, animator, m->header.sequence);
		control up();
		n unlock();
		}
	}

rejectedSend:	dynamic	(code: int) =
	{
	m:	ref message;
	a:	ref task_t;

	a = new task_t;
	where recoverSendParameters(a);
	a->state = TS_REJECT;
	a->errorCode = code;
	if	(!a calledFromUserMode()){
		kprintf("Called from %d where = %p KernelArena = %p\n", me, where, &KernelArena);
//		RejectTrap raise(code);
		panic("Kernel mode reject - System halted");
		}
	m = message createSelfGenerated(self, int(&external.rejectedMessage),
						vaddr_t(a), sizeof task_t);
	if	(m){
		n:	threadLock;

		n lock();
		msgs push(m);
		where scheduleSignal(self, animator, m->header.sequence);
		control up();
		n unlock();
		}
	}

interruptedSend:dynamic	() =
	{
	m:	ref message;
	a:	ref task_t;

	a = new task_t;
	where recoverSendParameters(a);
	a->state = TS_INTERRUPT;
	if	(!a calledFromUserMode()){
		a display();
		panic("Kernel mode interrupted send - System halted");
		}
	m = message createSelfGenerated(self, 
				int(&external.interruptedMessage), 
				vaddr_t(a), sizeof task_t);
	if	(m){
		n:	threadLock;

		n lock();
		msgs push(m);
		where scheduleSignal(self, animator, m->header.sequence);
		control up();
		n unlock();
//		CurProc abort(int(&external.interruptedMessage));
		}
	}

machineTrap:	dynamic	(t: ref task_t, func: int) int =
	{
	m:	ref message;
	a:	ref task_t;

	if	(!t calledFromUserMode()){
		t display();
		kprintf("Halted in object %d\n", CurProc->animates->me);
		trace(t->esp, t->ebp, unsigned(&CurProc->kernelStack),
			unsigned(&CurProc->kernelStack[KERNEL_STACK]));
		panic("Kernel mode trap - System halted");
		}
	a = new task_t;
	*a = *t;
	a->state = TS_TRAP;
	a->errorCode = func;
	m = message createSelfGenerated(self, func, vaddr_t(a), sizeof task_t);
	if	(m){
		n:	threadLock;

		m->dumpOnAbort = TRUE;
		n lock();
		msgs push(m);
//		where scheduleSignal(self, animator);
//		control up();
		n unlock();
		CurProc abort(func);
		m close();
		}
	}

coreDump:	dynamic	() boolean =
	{
	m:	ref message;

	m = ref message(msgs.next);
	if	(m == &msgs)
		return FALSE;
	if	(m->dumpOnAbort)
		return where coreDump(m);
	else
		return FALSE;
	}

	};

kernelObject:	public	type	inherit	object	{
	public:

	dispatcher:	ref external;
	control:	semaphore;

constructor:	(name: [:] char, d: ref external, jr: accessRights_t, ur: accessRights_t,
				gr: accessRights_t, wr: accessRights_t) =
	{
	dispatcher = d;
	control = [ 1 ];		// kernel object is not busy
	super constructor(name, TRUE, jr, ur, gr, wr);
	where = &KernelArena;
	}

object0constructor:	() =
	{
	CurProc->animates = &BootObject;
	context = &BootContext;
	dispatcher = 0;
	useCount = 1;
	isKernel = TRUE;
	me = ref far external(NLOCALS);
	name = "BootObject";
	msgs constructor();
	where = &KernelArena;
	}

object1constructor:	() =
	{
	CurProc->animates = &BootObject;
	dispatcher = &BootJob;
	useCount = 1;
	isKernel = TRUE;
	me = ref far external(NLOCALS + 1);
	name = "Boot Job";
	msgs constructor();
	where = &KernelArena;
	}

sendWait:	dynamic	(xx: ref far external, r: unsigned[16], 
					func: int, buf: vaddr_t, len: int, 
					rbuf: vaddr_t, rlen: int) int =
	{
	m:		message;
	src:		ref object;
	n:		threadLock;

	src = CurProc->animates;
	m kernelSend(self, func, r, src, buf, len, rbuf, rlen);
	n lock();
	msgs enqueue(&m);
	n unlock();
	if	(control down(TRUE)){
		n lock();
		CurProc->animates = self;
		CurArena = CurProc->where = where;
		n unlock();

		bufp:	pointer;

		if	(src == 0)
			panic("No source for message\n");
		if	(src->where == 0)
			panic("No arena for source\n");
		bufp = src->where obtainWrite(buf, len);
		if	(len == 0 || bufp){
			m.state = MS_RECEIVED;
			dispatcher kernelLocal(&m.header, bufp);
			if	(bufp)
				src->where unlock();
			}
		else	{
			m.errorCode = ERRINVALIDDATA;
			m.state = MS_REJECTED;
			}

//		kprintf("clearing queue for %x\n", self);
		n lock();
		m wrapup();
		CurProc->animates = src;
		CurArena = CurProc->where = src->where;
		n unlock();

		control up();
		}
	else	{
		src interruptedSend();
		return 0;
		}
	if	(m.state == MS_REJECTED){
//		kprintf("rejected\n");
		src rejectedSend(m.errorCode);
		return 0;
		}
	return m.cumReply;
	}
/*
	This function handles sends with an integral reply.  This is a 
	special purpose, slighter faster pathway to process messages.
 */
sendWaitInt:	dynamic	(ref far external, r: unsigned[16], func: int, 
						buf: vaddr_t, len: int) int =
	{
	m:		message;
	src:		ref object;

	src = CurProc->animates;
	m kernelSend(self, func, r, src, buf, len, 0, sizeof int);
	n:	threadLock;

	n lock();
	msgs enqueue(&m);
	n unlock();
	if	(control down(TRUE)){
		n lock();
//		kprintf("self = %x message %x enqueued\n", self, &m);
		CurProc->animates = self;
		CurArena = CurProc->where = where;
		n unlock();

		bufp:	pointer;

		if	(src == 0)
			panic("No source for message\n");
		if	(src->where == 0)
			panic("No arena for source\n");
		bufp = src->where obtainWrite(buf, len);
		if	(len == 0 || bufp){
			m.state = MS_RECEIVED;
			dispatcher kernelLocal(&m.header, bufp);
			if	(bufp)
				src->where unlock();
			}
		else	{
			m.errorCode = ERRINVALIDDATA;
			m.state = MS_REJECTED;
			}

		n lock();
		m wrapup();
		CurProc->animates = src;
		CurArena = CurProc->where = src->where;
		n unlock();
		control up();
		if	(m.state == MS_REJECTED)
			src rejectedSend(m.errorCode);
		}
	else
		src interruptedSend();
	return m.errorCode;
	}

attention:	dynamic	(p: ref process) =
	{
	kprintf("Kernel object break %d\n", me);
	if	(CurProc == p){			// call yourself
		m:		message;
	
		kprintf("to myself\n");
		m kernelSend(self, int(&external.attention), AR_ANY, 
							0, 0, 0, 0, 0);
		msgs enqueue(&m);
		m.state = MS_RECEIVED;
		dispatcher kernelLocal(&m.header, 0);
		m extract();
		}
	else
		p arrangeCall(&hackKernelAttention);
	if	(p->status == PWAIT)		// wake up a sleeping
						// process
		p->resource abort(p);
	}

rejectedSend:	dynamic	(code: int) =
	{
	RejectTrap raise(code);
	}

interruptedSend:dynamic	() =
	{
	InterruptTrap raise();
	}

newJob:	dynamic	(j: ref far job) ref far external =
	{
	e:	ref far external;

	e = me copy();
	if	(e){
		o:	ref object;

		o = objectEntry(e);
		if	(o != self)
			o->myJob = j;
		}
	return e;
	}

	};

hackKernelAttention:	() =
	{
	n:	threadLock;

	kprintf("haskKernelAttention %d\n", CurProc->animates->me);
	n lock();
	CurProc->animates attention(CurProc);
	n unlock();
	}

object:	public	type	{
	public:

	name:		[:] char;
	me:		ref far external;
	parent:		ref far external;
	myJob:		ref far job;
	user:		userId;
	group:		userId;
	jobRights:	accessRights_t;
	userRights:	accessRights_t;
	groupRights:	accessRights_t;
	worldRights:	accessRights_t;
	useCount:	int;
	isKernel:	boolean;
	where:		ref arena;
	msgs:		queue;
	alarm:		* alarmTimer;
	sequence:	_sequence_t;
	context:	ref objectContext;

constructor:	(n: [:] char, k: boolean, jr: accessRights_t, ur: accessRights_t,
				gr: accessRights_t, wr: accessRights_t) =
	{
	o:	ref object;

	o = CurProc->animates;
	parent = o->me;
	myJob = o->myJob;
	user = o->user;
	group = o->group;
	jobRights = jr;
	userRights = ur;
	groupRights = gr;
	worldRights = wr;
	useCount = 1;
	isKernel = k;
	name = new [|n] char;
	name [:]= n;
	msgs constructor();
	sequence = 0;
	context = CurProc->animates->context;
	context dup();
	}

describe:	(o: ref object_t) =
	{
	o->me = me;
	o->parent = parent;
	o->myJob = myJob;
	o->user = user;
	o->group = group;
	o->jobRights = jobRights;
	o->userRights = userRights;
	o->groupRights = groupRights;
	o->worldRights = worldRights;
	o->useCount = useCount;
	o->isKernel = isKernel;
	o->alertable = 0;
	}

receive:	dynamic	(mh: vaddr_t) int =
	{
	}

sendWait:	dynamic	(ref far external, unsigned[16], int, vaddr_t, int, vaddr_t, int) int =
	{
	return ERRPERMISSION;
	}

sendWaitInt:	dynamic	(ref far external, unsigned[16], int, vaddr_t, int) int =
	{
	return ERRPERMISSION;
	}

attention:	dynamic	(ref process) =
	{
	}

rejectedSend:	dynamic	(code: int) =
	{
	}

interruptedSend:dynamic	() =
	{
	}

_forward:	dynamic	(* message, ref far external, int, vaddr_t, int) int =
	{
	return ERRPERMISSION;
	}

ownerFree:	dynamic	(* arena) boolean =
	{
	return FALSE;
	}

processTermination:	dynamic	() =
	{
	}

kernelFree:	dynamic	(pointer, paddr_t) boolean =
	{
	return FALSE;
	}

dispose:	dynamic	() int =
	{
	return ERRPERMISSION;
	}

dup:	dynamic	(ref far external) int =
	{
	useCount++;
	return SUCCESS;
	}

newJob:	dynamic	(j: ref far job) ref far external =
	{
	return 0;
	}

setJob:	dynamic	(j: ref far job) =
	{
	myJob = j;
	}

close:	dynamic	() =
	{
	useCount--;
	if	(useCount == 0){
		target:	int;
		m:	ref message;

		while	(!msgs isEmpty()){
			m = ref message(msgs.next);
			m reject(ERRINVALIDARGUMENT);
			}
		target = int(me);
		target &= ~NODE_MASK;
		target -= NLOCALS;
		ObjectTable[target] = 0;
		if	(context)
			context close();
		dispose();
		}
	}

copyContext:	() =
	{
	context = context copy(self);
	}

closeContext:	() int =
	{
	if	(context->parent &&
		 context->owner == self){
		context->parent dup();
		context = context close();
		return SUCCESS;
		}
	else
		return ERRNOPARENTCONTEXT;
	}

closeLocal:	(target: ref far external) =
	{
	if	(context)
		context closeLocal(target);
	}

machineTrap:	dynamic	(t: ref task_t, func: int) int =
	{
	t display();
	panic("Kernel mode trap - System halted");
	}

coreDump:	dynamic	() boolean =
	{
	return FALSE;
	}

	};

alarm:	public	(sec: int) int =
	{
	new alarmTimer[ secondsToTicks(sec), CurProc->animates ];
	return SUCCESS;
	}

alarmTimer:	type	inherit	timer	{
	public:

constructor:	(cnt: tick_t, o: ref object) =
	{
	obj = o;
	o->alarm = self;
	super constructor(cnt);
	}

fire:	dynamic	() =
	{
	o:	ref object;

	o = obj;
	o->alarm = 0;
	free(self);
	o->me alarmExpired!();
	}

dispose:	dynamic	() =
	{
	obj->alarm = 0;
	free(self);
	}

	private:

	obj:		ref object;
	};

oStatus_t:	public	type	byte = {
	O_NEW,
	O_RUN,
	O_WAIT,
	O_RECEIVE,
	O_DEAD,
	O_STOP
	};

broadcastAttention:	public	(j: ref far job) =
	{
	i:	int;
	p:	ref process;
	n:	threadLock;

	for	(i = 0; i < NPROCS; i++){
		n lock();
		p = ProcessTable[i];
		if	(p &&
			 p != CurProc &&
			 p->myJob == j)
			p->animates attention(p);
		n unlock();
		}
	n lock();
	if	(CurProc->myJob == j)
		CurProc->animates attention(CurProc);
	n unlock();
	}

newJob_:	public	() ref far job =
	{
	src:		ref object;
	j:		ref job;
	jf:		ref far job;

	src = CurProc->animates;
	j = new job;
	jf = ref far job(jobPublishKernel("job object", j, AR_ANY));
	src->context newJob(jf);
	src->myJob = jf;
	CurProc->myJob = src->myJob;
	return src->myJob;
	}

myJob_:	public	() ref far job =
	{
	return CurProc->myJob;
	}

publishKernel:	public	(name: [:] char,
			 obj: ref external, 
			 jobRights: accessRights_t,
			 userRights: accessRights_t,
			 groupRights: accessRights_t,
			 worldRights: accessRights_t) ref far external =
	{
	ko:	ref kernelObject;
	i:	int;
	n:	threadLock;

	ko = new kernelObject[ name, obj, jobRights, userRights, groupRights, 
								worldRights ];
	n lock();
	for	(i = 0; i < NOBJECTS; i++)
		if	(ObjectTable[i] == 0){
			ObjectTable[i] = ko;
			ko->me = ref far external(OID_BASE + i);
			n unlock();
//			kprintf("publish %S: %d\n", name, ko->me);
			obj->objectId = ko->me;
			return ko->me;
			}
	n unlock();
	free(ko);
	return 0;
	}

jobPublishKernel:	public	(name: [:] char,
			 obj: ref external, 
			 jobRights: accessRights_t) ref far external =
	{
	ko:	ref kernelObject;
	i:	int;
	n:	threadLock;

	ko = new kernelObject[ name, obj, jobRights, AR_NONE, AR_NONE, 
								AR_NONE ];
	n lock();
	for	(i = 0; i < NOBJECTS; i++)
		if	(ObjectTable[i] == 0){
			ObjectTable[i] = ko;
			ko->me = ref far external(OID_BASE + i);
			n unlock();
//			kprintf("publish %S: %d\n", name, ko->me);
			obj->objectId = ko->me;
			ko->myJob = CurProc->myJob;
			return ko->me;
			}
	n unlock();
	free(ko);
	return 0;
	}

publish:	public	(name: [:] char,
			 obj: vaddr_t, 
			 jobRights: accessRights_t,
			 userRights: accessRights_t,
			 groupRights: accessRights_t,
			 worldRights: accessRights_t) ref far external =
	{
	uo:	ref userObject;
	i:	int;
	n:	threadLock;

	uo = new userObject[ name, obj, jobRights, userRights, groupRights, 
								worldRights ];
	n lock();
	for	(i = 0; i < NOBJECTS; i++)
		if	(ObjectTable[i] == 0){
			ObjectTable[i] = uo;
			uo->me = ref far external(OID_BASE + i);
			n unlock();
			return uo->me;
			}
	n unlock();
	free(uo);
	return 0;
	}

ThreadTrap:	trap;

threadLaunch:	public	(ex: vaddr_t, th: vaddr_t, stack: vaddr_t,
				func: vaddr_t) ref far external =
	{
	uo:	ref userObject;
	i:	int;
	n:	threadLock;
	p:	ref process;

	p = process create();
	if	(p == 0)
		ThreadTrap raise();
	CurArena initializeThread(p, th, stack, func);
	uo = new userObject[ "thread", ex, AR_ANY, AR_NONE, AR_NONE, AR_NONE ];
	uo serves(CurProc->animates->me, p, CurArena);
	n lock();
	for	(i = 0; i < NOBJECTS; i++)
		if	(ObjectTable[i] == 0){
			ObjectTable[i] = uo;
			uo->me = ref far external(OID_BASE + i);
			n unlock();
			p setRunnable();
			CurArena dup();
			return uo->me;
			}
	n unlock();
	free(uo);
	return ref far external(ERRNOMEMORY);
	}

userArenaStartup:	public	(par: ref far external, p: ref process, 
					ar: ref arena) ref far external =
	{
	uo:	ref userObject;
	i:	int;
	n:	threadLock;

	uo = new userObject[ "$start", 0, AR_ANY, AR_NONE, AR_NONE, AR_NONE ];
	uo serves(par, p, ar);
	n lock();
	for	(i = 0; i < NOBJECTS; i++)
		if	(ObjectTable[i] == 0){
			ObjectTable[i] = uo;
			uo->me = ref far external(OID_BASE + i);
			n unlock();
			return uo->me;
			}
	n unlock();
	free(uo);
	return ref far external(ERRNOMEMORY);
	}

identifyObject0:	entry	() =
	{
	BootObject object0constructor();
	P_publish = &publishKernel;
	BootJobObject object1constructor();
	BootObject.myJob = ref far job(BootJobObject.me);
	CurProc->myJob = BootObject.myJob;
	}

displayObjects:	public	() =
	{
	i:	int;
	o:	ref object;

	for	(i = 0; i < NOBJECTS; i++){
		o = ObjectTable[i];
		if	(o)
			displayObject(o);
		}
	}

displayObject:	public	(o: ref object) =
	{
	kprintf("%3d %3d %3d u%04x:g%04x %04x %04x %04x %04x %5d %c",
			o->me, o->parent, o->myJob, o->group, o->user,
			o->worldRights, o->groupRights, o->userRights,
			o->jobRights, o->useCount, 
			o->isKernel ? 'K' : 'U');
//			o->alertable ? 'A' : 'N');
	kprintf(" %S\n", o->name);
	}
