/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	alys;
include	hardware;
include	pc_hdw;
include	error;
include	arena, karena;
include	process;
include	object;
include	message;
include	kprintf;
include	filesys;

startup:	entry	() =
	{
		// Object management system calls

	kernelCall(_GDT_PUBLISH, &publishGate, 0);
	kernelCall(_GDT_PROBEOBJECT, &probeObjectGate, 0);

		// Local aliases system calls

	kernelCall(_GDT_FORKARENA, &forkArenaGate, 0);
	kernelCall(_GDT_DISCARDARENA, &discardArenaGate, 0);
	kernelCall(_GDT_MAKELOCAL, &makeLocalGate, 0);
	kernelCall(_GDT_MAKEANYLOCAL, &makeAnyLocalGate, 0);
	kernelCall(_GDT_SETFORKACTION, &setForkActionGate, 0);

		// Message handling system calls

	kernelCall(_GDT_SEND_INT, &sendIntGate, 0);
	kernelCall(_GDT_SEND_VOID, &sendVoidGate, 0);
	kernelCall(_GDT_SEND_ANY, &sendAnyGate, 0);
	kernelCall(_GDT_RECEIVE, &receiveGate, 0);
	kernelCall(_GDT_SIGNALINFO, &signalInfoGate, 0);
	kernelCall(_GDT_READTEXT, &readTextGate, 0);
	kernelCall(_GDT_DISCARDTEXT, &discardTextGate, 0);
	kernelCall(_GDT_REJECT, &rejectGate, 0);
	kernelCall(_GDT_REPLY, &replyGate, 0);
	kernelCall(_GDT_REPLYPARTIAL, &replyPartialGate, 0);

		// Memory management system calls

	kernelCall(_GDT_GROW, &growGate, 0);

		// Program startup & termination system calls

	kernelCall(_GDT_CATCHSIGNAL, &catchSignalGate, 0);
	kernelCall(_GDT_ALARM, &alarmGate, 0);
	kernelCall(_GDT_EXIT, &exitGate, 0);
	kernelCall(_GDT_ABORT, &abortGate, 0);
	kernelCall(_GDT_RAISE, &raiseGate, 0);
	kernelCall(_GDT_THREADLAUNCH, &threadLaunchGate, 0);
	kernelCall(_GDT_KERNELBLOCK, &kernelBlockGate, 0);
	kernelCall(_GDT_KERNELUNBLOCK, &kernelUnblockGate, 0);
	kernelCall(_GDT_KERNELDOWN, &kernelDownGate, 0);
	kernelCall(_GDT_KERNELUP, &kernelUpGate, 0);

		// Job control system calls

	kernelCall(_GDT_NEWJOB, &newJobGate, 0);
	kernelCall(_GDT_MYJOB, &myJobGate, 0);
	}

/*
	This gate function will only be called from within user mode.  So,
	do not check for kernel mode caller.  Such a caller is by definition
	incorrect.
 */
receiveGate:	gate	() int =
	{
	hdr:	vaddr_t;
	buf:	vaddr_t;
	len:	int;

	hdr = _EBX;
	if	(ref gateFrame_t(_EBP) calledFromUserMode())
		return CurProc->animates receive(hdr);
	else			// kernel mode processes must abort here
		MessageTrap raise();
	}

rejectGate:	gate	() =
	{
	code:	int;
	seq:	_sequence_t;

	seq = _BX;
	code = _EDX;
	m_reject(seq, code);
	}

readTextGate:	gate	() int =
	{
	offs:	vaddr_t;
	buf:	vaddr_t;
	len:	vaddr_t;
	seq:	_sequence_t;

	seq = _BX;
	offs = _EAX;
	len = _ECX;
	buf = _EDX;
	return m_readText(seq, offs, buf, len);
	}

discardTextGate:	gate	() =
	{
	seq:	_sequence_t;

	seq = _BX;
	m_discardText(seq);
	}

replyGate:	gate () =
	{
	len:	vaddr_t;
	buf:	vaddr_t;
	seq:	_sequence_t;

	seq = _BX;
	len = _ECX;
	buf = _EDX;
	m_reply(seq, buf, len);
	}

replyPartialGate:	gate () =
	{
	len:	vaddr_t;
	buf:	vaddr_t;
	seq:	_sequence_t;

	seq = _BX;
	len = _ECX;
	buf = _EDX;
	m_replyPartial(seq, buf, len);
	}
/*
	This code filters the exit code so that only a normal exit can be
	returned.  Thus, a parent can always tell that the exit is a normal
	one.
 */
exitGate:	gate	() =
	{
	code:	unsigned;

	code = _EBX;
	CurProc exit(code & EX_LEVEL);
	}

abortGate:	gate	() =
	{
	code:	unsigned;

	code = _EBX;
//	printf("Abort(%x)\n", code);
	CurProc abort(code & EX_LEVEL);
	}

raiseGate:	gate	() =
	{
	CurProc raise();
	}

threadLaunchGate:	gate	() ref far external =
	{
	exObject:	vaddr_t;
	stack:		vaddr_t;
	func:		vaddr_t;
	threadObj:	vaddr_t;

	exObject = _EBX;
	func = _EAX;
	stack = _ECX;
	threadObj = _EDX;
	return threadLaunch(exObject, threadObj, stack, func);
	}

alarmGate:	gate	() =
	{
	alarm(_EBX);
	}

growGate:	gate () boolean =
	{
	len:	vaddr_t;

	len = _EBX;
	return CurArena grow(len);
	}

sendIntGate:	gate () int =
	{
	o:		ref far external;
	f:		int;
	len:		unsigned;
	buf:		unsigned;

	o = ref far external(_ECX);
	f = _EAX;
	len = _EBX;
	buf = _EDX;
	return sendWaitInt(o, f, buf, len);
	}

sendAnyGate:	gate () int =
	{
	o:		ref far external;
	f:		int;
	len:		vaddr_t;
	buf:		vaddr_t;
	rlen:		vaddr_t;
	rbuf:		vaddr_t;

	o = ref far external(_ECX);
	f = _EAX;
	len = _EBX;
	buf = _EDX;
	rlen = _ESI;
	rbuf = _EDI;
	return sendWait(o, f, buf, len, rbuf, rlen);
	}

sendVoidGate:	gate () int =
	{
	o:		ref far external;
	f:		int;
	len:		int;
	buf:		int;

	o = ref far external(_ECX);
	f = _EAX;
	len = _EBX;
	buf = _EDX;
	return sendWait(o, f, buf, len, 0, 0);
	}

publishGate:	gate	() ref far external =
	{
	obj:	vaddr_t;
	job:	accessRights_t;
	user:	accessRights_t;
	group:	accessRights_t;
	world:	accessRights_t;
	x, y:	unsigned[32];

	obj = _EBX;
	x = _EDX;
	y = _ECX;
	job = x;
	user = x >> 16;
	group = y;
	world = y >> 16;
	return publish("user", obj, job, user, group, world);
	}

discardArenaGate:	gate	() int =
	{
	return CurProc->animates closeContext();
	}

forkArenaGate:	gate	() int =
	{
	try	{
		CurProc->animates copyContext();
		return SUCCESS;
		}
	except	{
		return ERRNOMEMORY;
		}
	}

catchSignalGate:	gate	() int =
	{
	func:	vaddr_t;

	func = _EBX;
	return ref userObject(CurProc->animates) catchSignal(func);
	}

signalInfoGate:		gate	() int =
	{
	hdr:	vaddr_t;
	seq:	_sequence_t;

	hdr = _EDX;
	seq = _BX;
	return ref userObject(CurProc->animates) signalInfo(seq, hdr);
	}

probeObjectGate:		gate	() ref far external =
	{
	o:	ref far external;

	o = ref far external(_EBX);
	return probeObject_(o);
	}

newJobGate:	gate	() ref far job =
	{
	return newJob_();
	}

myJobGate:	gate	() ref far job =
	{
	return myJob_();
	}

makeLocalGate:	gate	() int =
	{
	obj, slot:	ref far external;

	obj = ref far external(_EBX);
	slot = ref far external(_EDX);
	return CurProc->animates->context makeLocal(obj, slot);
	}

makeAnyLocalGate:	gate	() ref far external =
	{
	obj:	ref far external;

	obj = ref far external(_EBX);
	return CurProc->animates->context makeAnyLocal(obj);
	}

setForkActionGate:	gate	() int =
	{
	slot:		ref far external;
	action:		forkAction_t;

	slot = ref far external(_EBX);
	action = _DL;
	return CurProc->animates->context setForkAction(slot, action);
	}

kernelBlockGate:	gate	() =
	{
	CurArena->latch = ARENA_LATCH_COUNT;
	}

kernelUnblockGate:	gate	() =
	{
	CurArena->latch = 0;
	}

kernelDownGate:	gate	() =
	{
	event:	vaddr_t;

	event = _EBX;
	CurArena eventDown(event);
	}

kernelUpGate:	gate	() int =
	{
	event:	vaddr_t;

	event = _EBX;
	return CurArena eventUp(event);
	}
