/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	alys;
include	kprintf;
include	object;
include	twindow;
include	console;
include	sound;
include	video;
include	message;
include	pc_keybd;
include	pc_comm;
include	error;
include	filesys;
include	arena;
include	process;
include	hardware, list;
/*
	The ALYS console design requires the role of several different
	objects, each performing distinct functions.  First, the console
	object itself defines the basic functioning of a display, keyboard,
	mouse, pen or other input devices.  Window objects in turn define
	an architecture of multiple overlapping windows to appear on the
	console display.  All of the windows active on a display form a 
	stack that can be rearranged or cycled, as needed.

	For the time being, a console file object is used to provide tty-
	like behavior.  A console channel object is connected to a window
	when the console file object is opened.  This is clearly an interim
	solution, since a more powerful tty window will eventually be
	supplied that provides much more power (like scrollability and line
	editing).
 */
ConsoleObject:	public	inherit	console	{
	public:

	breakKey:	keystroke;
	events:		queue;
	mouseButtons:	byte;
	mouseLoc:	point;

mode:	gate	(m: consoleMode_t) =
	{
	}

newWindow:	gate	(title: [:] char, sz: point) ref far textWindow =
	{
	w:	ref PC_textWindow;
	id:	ref far PC_textWindow;

	w = new PC_textWindow[ sz, self ];
	id = ref far PC_textWindow(jobPublishKernel(title, w, AR_ANY));
	return id;
	}

screenWindow:	gate	() ref far textWindow =
	{
	reject(ERRINVALIDFUNC);
	}

describe:	gate	() console_t =
	{
	}

getWindowInfo:	gate	(idx: int) window_t =
	{
	reject(ERRINVALIDFUNC);
	}

beep:	gate	() =
	{
	reject(ERRINVALIDFUNC);
	}

setRepeatRate:	gate	(delay: int, repeat: int) =
	{
	repeat &= 0x1f;
	pc_keybd::setRepeatRate(delay << 5 + repeat);
	}

postEvent:	dynamic	(ev: event_t) =
	{
	e:	ref event;
	n:	threadLock;

	if	(ev.info == breakKey &&
		 ev.class == E_CHARACTER){
		n lock();
		if	(WindowStack){
			kprintf("Break! %d\n", WindowStack->objectId);
			j:	ref far job;

			j = getJobObject(WindowStack->objectId);
			broadcastAttention(j);
			}
		n unlock();
		return;
		}
	if	(ev.class == E_MOUSE){
		n lock();
		if	(ev.location.x > 5 ||
			 ev.location.x < -5)
			ev.location.x <<= 2;
		if	(ev.location.y > 5 ||
			 ev.location.y < -5)
			ev.location.y <<= 2;
		mouseLoc.x += ev.location.x;
		mouseLoc.y += ev.location.y;
		if	(mouseLoc.x < 0)
			mouseLoc.x = 0;
		else if	(mouseLoc.x >= 800)
			mouseLoc.x = 799;
		if	(mouseLoc.y < 0)
			mouseLoc.y = 0;
		else if	(mouseLoc.y >= 600)
			mouseLoc.y = 599;
		ev.location = mouseLoc;
		p:	point;

		p.x = mouseLoc.x / 10;
		p.y = mouseLoc.y / 24;
//		kprintf("[%d,%d]", p.x, p.y);
		Screen mouseCursor(p);
		if	(mouseButtons == ev.info){
			n unlock();
			return;
			}
		mouseButtons = ev.info;
		n unlock();
		}
	e = event create(ev);
//	printf("post e = %x\n", e);
	n lock();
	if	(!events isEmpty() ||
		 WindowStack == 0 ||
		 !WindowStack->events isEmpty())
		events enqueue(e);
	else if	(WindowStack->eventWait.count < 0){
		WindowStack->events enqueue(e);
		WindowStack->eventWait up();
		}
	else
		events enqueue(e);
	n unlock();
//	printf("posted\n");
	}

/*
getKey:		gate	() keystroke =
	{
	reject(ERRINVALIDFUNC);
	}

getRawKey:	gate	() keystroke =
	{
	reject(ERRINVALIDFUNC);
	}

testKey:	gate	() keystroke =
	{
	reject(ERRINVALIDFUNC);
	}
 */
	};

setup:	entry	() =
	{
	ConsoleObject.events = [];
	ConsoleObject.breakKey = CTRL_BREAK;
	registerConsole(&ConsoleObject);
	publishKernel("Console", &ConsoleObject, AR_ANY, AR_ANY, AR_ANY, AR_ANY);
	}

windowFlags:	public	type	char = {
	W_OPEN =	0x08,
	};

windowStatus:	type	char = {
	WS_HIDDEN = 0x01,
	};

windowInfo:	type	packed	{
	public:

	corner:		point;
	size:		point;
	color:		char;
	flags:		windowFlags;
	status:		windowStatus;
	title:		[] char;
	};

WindowStack:	ref PC_textWindow;
TopWindow:	public	ref far textWindow;

cycleWindows:	public	(keystroke) =
	{
	if	(WindowStack)
		WindowStack _toBottom();
	}

PC_textWindow:	public	type	inherit	textWindow	{
	on:			ref console;
	currentZoom:		windowZoom_t;
	color:			color_t;
	image:			ref colorChar;
	imageLength:		int;
	flags:			windowFlags;
	status:			windowStatus;

	public:
	corner:			point;
	size:			point;
	next:			ref PC_textWindow;
	prev:			ref PC_textWindow;
	events:			queue;
	eventWait:		semaphore;
	cursor:			point;

constructor:	(sz: point, cons: ref console) =
	{
	on = cons;
	size = sz;
	cursor = [ 0, 0 ];
	status = 0;
	color = WHITE;
	flags = 0;
	image = 0;
	currentZoom = WZ_HIDDEN;
	imageLength = 0;
	events = [];
	eventWait = [ 0 ];
	}

delete:	gate	() boolean =
	{
	w:	ref PC_textWindow;

	if	(image){
		free(image);
		image = 0;
		}
	removeFromStack();
	flags = 0;

		// now draw all the windows but this one.

	if	(WindowStack){
		for	(w = WindowStack; w; w = w->next)
			w draw();
		Screen setColor(WindowStack->color);
		WindowStack _showCursor();
		}
	return TRUE;
	}

appearsOn:	gate	() ref far console =
	{
	return ref far console(on->objectId);
	}

read:	gate	(p: point) [] colorChar =
	{
	img:	ref colorChar;
	len:	int;

	img = image + (p.y * size.x + p.x);
	len = MessageHeader->expected;
	if	(imageLength < len)
		len = imageLength;
	len >>= 1;
	return img[:len];
	}

write:	gate	(p: point, buf: [:] colorChar) =
	{
	rows:	int;
	clen:	unsigned;
	rem:	unsigned;
	img:	ref colorChar;
	endimg:	ref colorChar;
	cp:	ref colorChar;
	bf:	ref colorChar;
	s:	ref colorChar;

	rem = size.x - p.x;
	if	(rem == 0)
		return;
	img = image + (p.y * size.x + p.x);
	endimg = image + imageLength;
	if	(img + |buf < endimg)
		endimg = img + |buf;
	for	(cp = img, bf = buf; cp < endimg; cp++, bf++)
		*cp = *bf;
	p.x += corner.x;
	p.y += corner.y;
	s = buf;
	for	(clen = |buf; clen; ){
		if	(clen < rem)
			rem = clen;
		if	(WindowStack == self){
			Screen writeCA(p, s, rem);
			s += rem;
			}
		else	{
			drawSegment(p, img, rem);
			img += rem;
			}
		p.x = corner.x;
		p.y++;
		clen -= rem;
		rem = size.x;
		}
	}

writeChar:	gate	(p: point, buf: [:] char) =
	{
	rows:	int;
	clen:	unsigned;
	rem:	unsigned;
	img:	ref colorChar;
	endimg:	ref colorChar;
	cp:	ref colorChar;
	bf:	ref char;
	s:	ref char;

	img = image + (p.y * size.x + p.x);
	endimg = image + imageLength;
	if	(img + |buf < endimg)
		endimg = img + |buf;
	for	(cp = img, bf = buf; cp < endimg; cp++, bf++)
		cp[0] = [ *bf, color ];
	rem = size.x - p.x;
	if	(rem == 0)
		return;
	p.x += corner.x;
	p.y += corner.y;
	s = buf;
	for	(clen = |buf; clen; ){
		if	(clen < rem)
			rem = clen;
		if	(WindowStack == self){
			Screen write(p, s, rem);
			s += rem;
			}
		else	{
			drawSegment(p, img, rem);
			img += rem;
			}
		p.x = corner.x;
		p.y++;
		clen -= rem;
		rem = size.x;
		}
	}

zoom:	gate	(p: point, zoomState: windowZoom_t) =
	{
	i:	int;

	switch	(zoomState){
	case	WZ_NORMAL:
		corner = p;
		i = size.x * size.y * sizeof colorChar;
		imageLength = i;
		image = alloc(i);
		if	(image){
			clen:	unsigned;
			buf:	* colorChar;

			buf = image;
			for	(clen = 0; clen < size.y; clen++){
				Screen read(p, buf, size.x);
				buf += size.x;
				p.y++;
				}
			}
		next = 0;
		prev = 0;
		_toTop();
		break;		
		}
	currentZoom = zoomState;
	}

clear:	gate	() =
	{
	reject(ERRINVALIDFUNC);
	}

textCursor:	gate	(p: point) =
	{
	cursor = p;
	_showCursor();
	}

mouseCursor:	gate	(p: point) =
	{
	reject(ERRINVALIDFUNC);
	}

setColor:	gate	(c: color_t) =
	{
	color = c;
	Screen setColor(c);
	}

putc:	gate	(p: point, c: char) =
	{
	if	(p.y >= size.y)
		return;
	if	(p.x >= size.x)
		return;
	img:	ref colorChar;

	img = image + (p.y * size.x + p.x);
	*img = [ c, color ];
	p.x += corner.x;
	p.y += corner.y;
	if	(WindowStack == self)
		Screen putc(p, c);
	else
		drawSegment(p, img, 1);
	}

verticalScroll:	gate	(ul: point, sz: point, amount: int) =
	{
	offset:		unsigned;
	diff:		unsigned;
	len:		unsigned;
	rows:		int;
	copyrows:	int;
	i:		unsigned;
	ip:		ref colorChar;
	jp:		ref colorChar;

	if	(amount == 0)
		return;

		// If the scrolling frame in view?  If not, give up'

	if	(ul.x > size.x ||
		 ul.y > size.y ||
		 sz.x < 1 ||
		 sz.y < 1 ||
		 ul.x + sz.x < 1 ||
		 ul.y + sz.y < 1)
		return;

		// now clip the scrolling frame to the window

	if	(ul.x < 0)
		ul.x = 0;
	if	(ul.y < 0)
		ul.y = 0;
	if	(ul.x + sz.x > size.x)
		sz.x = size.x - ul.x;
	if	(ul.y + sz.y > size.y)
		sz.y = size.y - ul.y;
	len = 2 * sz.x;
	rows = sz.y;
	i = 0;
	offset = ul.y * size.x + ul.x;
	if	(amount < 0){		/* scroll down */
		amount = -amount;
		if	(amount < sz.y){
			copyrows = sz.y - amount;
			offset += sz.y * size.x - size.x;
			diff = amount * size.x;
			for	(; i < copyrows; i++){
				memCopy(image + offset,
					image + offset - diff, len);
				offset -= size.x;
				}
			}
		for	(; i < sz.y; i++){
			ip = image + offset;
			jp = ip + sz.x;
			while	(ip < jp){
				*ip = [ ' ', color ];
				ip++;
				}
			offset -= size.x;
			}
		amount = -amount;
		}
	else	{			/* scroll up */
		if	(amount < sz.y){
			copyrows = sz.y - amount;
			diff = amount * size.x;
			for	(; i < copyrows; i++){
				memCopy(image + offset,
					image + offset + diff, len);
				offset += size.x;
				}
			}
		for	(; i < sz.y; i++){
			ip = image + offset;
			jp = ip + sz.x;
			while	(ip < jp){
				*ip = [ ' ', color ];
				ip++;
				}
			offset += size.x;
			}
		}
	if	(WindowStack == self){		// if on top use quick call
		lr:	point;

		ul.x += corner.x;
		ul.y += corner.y;
		lr.x = ul.x + sz.x - 1;
		lr.y = ul.y + sz.y - 1;
		Screen verticalScroll(ul, lr, amount);
		}
	else
		draw();
	}

horizontalScroll:	gate	(ul: point, sz: point, amount: int) =
	{
	reject(ERRINVALIDFUNC);
	}

showCursor:	gate	() =
	{
	_showCursor();
	}

putcc:		gate	(p: point, c: char, co: color_t) =
	{
	if	(p.y >= size.y)
		return;
	if	(p.x >= size.x)
		return;
	img:	ref colorChar;

	img = image + (p.y * size.x + p.x);
	*img = [ c, co ];
	p.x += corner.x;
	p.y += corner.y;
	if	(WindowStack == self)
		Screen putcc(p, c, co);
	else
		drawSegment(p, img, 1);
	}

hideCursor:	gate	() =
	{
	Screen hideCursor();
	status |= WS_HIDDEN;
	}

toTop:	gate	() =
	{
	if	(currentZoom == WZ_NORMAL)
		_toTop();
	}

_toTop:	() =
	{
	if	(WindowStack == self)
		return;			// already on the top
	removeFromStack();
	next = WindowStack;
	if	(WindowStack)
		WindowStack->prev = self;
	WindowStack = self;
	Screen setColor(color);
	_showCursor();
	draw();
//	takeKeyboard();
	}

toBottom:	gate	() =
	{
	if	(currentZoom == WZ_NORMAL)
		_toBottom();
	}

_toBottom:	() =
	{
	switch	(currentZoom){
	case	WZ_NORMAL:
		if	(next == 0)
			return;			// already on the bottom
		removeFromStack();
		w:	ref PC_textWindow;

		for	(w = WindowStack; w && w->next; w = w->next)
			;
		if	(w){
			w->next = self;
			prev = w;
			}
		else
			WindowStack = self;

			// now draw all the windows but this one.

		for	(w = WindowStack; w->next; w = w->next)
			w draw();
		WindowStack _showCursor();
		}
	}

removeFromStack:	() =
	{
	if	(currentZoom != WZ_NORMAL)
		return;
	if	(next)
		next->prev = prev;
	if	(prev)
		prev->next = next;
	if	(WindowStack == self)
		WindowStack = next;
	prev = 0;
	next = 0;
	}

beep:	gate	() =
	{
	_beep();
	}

getKey:		gate	() keystroke =
	{
	RawKeyMap:	static	[] signed[16] = [
		'\b',	'\t',	'+',	'-',
		'*',	'0',	'1',	'2',
		'3',	'4',	'5',	'6',
		'7',	'8',	'9',	'.',
		'\r',	ESC
		];
	k:	keystroke;

	k = _getRawKey();
	if	(k >= GREY_BS && k <= GREY_ESC)
		k = RawKeyMap[k - GREY_BS];
	return k;
	}

getRawKey:	gate	() keystroke =
	{
	return _getRawKey();
	}

_getRawKey:	() keystroke =
	{
	n:	threadLock;
	e:	ref event;
	k:	keystroke;

	for	(;;){
		n lock();
		if	(events isEmpty() &&
			 !ConsoleObject.events isEmpty())
			e = ref event(ConsoleObject.events dequeue());
		else if	(eventWait down(TRUE))
			e = ref event(events dequeue());
		else	{
			n unlock();
			return keystroke(-1);
			}
		n unlock();
		if	(e->ev.class != E_CHARACTER &&
			 e->ev.class != E_BUTTON)
			continue;
		k = e->ev.info;
		e free();
		return k;
		}
	}

testKey:	gate	() keystroke =
	{
	return -1;
	}

_showCursor:	() =
	{
	status &= ~WS_HIDDEN;	
	positionCursor(cursor);
	Screen showCursor();
	}

positionCursor:	(p: point) =
	{
	if	(p.x < size.x && p.y < size.y){
		cursor = p;
		if	(WindowStack == self){
			p.x += corner.x;
			p.y += corner.y;
			Screen positionCursor(p);
			}
		}
	}

draw:	() =
	{
	img:	* colorChar;
	p:	point;
	i:	int;

	img = image;
	p = corner;
	for	(i = 0; i < size.y; i++, img += size.x, p.y++)
		drawSegment(p, img, size.x);
	}
/*
	This code performs all the clipping logic for a background write.
	It is called for each line of a window and backtracks through the
	windows above it to determine whether there are any obscuring
	windows.

	Note: coordinates are absolute coordinates for this routine.
 */
drawSegment:	(p: point, img: * colorChar, len: int) =
	{
	w:	* PC_textWindow;
	xx:	int;
	endx:	int;

	endx = p.x + len;
	for	(w = prev; w; w = w->prev){
		if	(p.y < w->corner.y)
			continue;
		if	(p.y >= w->corner.y + w->size.y)
			continue;
		if	(endx <= w->corner.x)
			continue;
		xx = w->corner.x + w->size.x;
		if	(p.x >= xx)
			continue;
		if	(p.x >= w->corner.x){
			if	(endx <= xx)
				return;
			img += (xx - p.x) * 2;
			p.x = xx;
			}
		else	{
			if	(endx > xx)
				drawSegment([ xx, p.y ],
					img + (xx - p.x) * 2, endx - xx);
			endx = w->corner.x;
			}
		len = endx - p.x;
		}
	Screen writeCA(p, img, len);
	}

	};

_beep:	() =
	{
	beep();
	}

/*
window:	public	type	{
	public:
	corner:			point;
	size:			point;
	next:			* window;
	prev:			* window;
	keyQ:			* inputQueue;

getWindowInfo:	(buf: * windowInfo, len: int) int =
	{
	wi:	windowInfo;
	tlen:	int;

	wi.corner = corner;
	wi.size = size;
	wi.color = color;
	wi.flags = flags;
	wi.status = status;
	tlen = 0;
	if	(len > sizeof windowInfo){
		if	(title.address){
			tlen = len - sizeof windowInfo;
			if	(tlen > title length())
				tlen = title length() + 1;
			memCopy(buf + 1, title.address, tlen);
			}
		len = sizeof windowInfo;
		}
	memCopy(buf, &wi, len);
	return len + tlen;
	}

constructor:	(titl: string, 		// Window title
		sz: point) =		// Size of window
	{
	size = sz;
	cursor = [ 0, 0 ];
	status = 0;
	color = WHITE;
	flags = 0;
	title = titl;
	image = 0;
	imageLength = 0;
	}

create:	factory	(titl: string, sz: point) * window =
	{
	if	(titl.address){
		self = alloc(sizeof window + titl length() + 1);
		sp:	string;

		if	(self){
			sp = [ pointer(self + 1) ];
			sp copy(titl);
			self = [ sp, sz ];
			}
		}
	else	{
		self = alloc(sizeof window);
		if	(self)
			self = [ [ 0 ], sz ];
		}
	return self;
	}

toBottom:	() =
	{
	if	(flags & W_OPEN == 0)
		return;
	if	(next == 0)
		return;			// already on the bottom
	removeFromStack();
	w:	* window;

	for	(w = WindowStack; w && w->next; w = w->next)
		;
	if	(w){
		w->next = self;
		prev = w;
		}
	else
		WindowStack = self;

		// now draw all the windows but this one.

	registerInputQueue(WindowStack->keyQ);
	for	(w = WindowStack; w->next; w = w->next)
		w draw();
	WindowStack showCursor();
	}

	private:

	color:			char;
	image:			* char;
	imageLength:		int;
	flags:			windowFlags;
	status:			windowStatus;

	public:

	title:			string;
	cursor:			point;

getCursor:	() point =
	{
	return cursor;
	}

hideCursor:	() =
	{
	Screen hideCursor();
	status |= WS_HIDDEN;
	}

showCursor:	() =
	{
	status &= ~WS_HIDDEN;	
	positionCursor(cursor);
	Screen showCursor();
	}

getSize:	() point = 
	{
	return size;
	}

getUlcorner:	() point = 
	{
	return corner;
	}

activate:	() =
	{
	if	(status & WS_HIDDEN)
		Screen hideCursor();
	else
		showCursor();
	Screen setColor(color);
	}

read:	(p: point, buf: * char, len: unsigned) =
	{
	img:	* char;

	img = image + (p.y * size.x + p.x) * 2;
	len <<= 1;
	if	(imageLength < len)
		len = imageLength;
	memCopy(buf, img, len);
	}

write:	(p: point, buf: * char, len: unsigned) =
	{
	rows:	int;
	clen:	unsigned;
	rem:	unsigned;
	img:	* char;
	endimg:	* char;
	cp:	* char;
	bf:	* char;

	img = image + (p.y * size.x + p.x) * 2;
	endimg = image + imageLength;
	if	(img + (len << 1) < endimg)
		endimg = img + (len << 1);
	for	(cp = img, bf = buf; cp < endimg; cp += 2, bf++){
		cp[0] = *bf;
		cp[1] = color;
		}
	rem = size.x - p.x;
	if	(rem == 0)
		return;
	p.x += corner.x;
	p.y += corner.y;
	for	(clen = len; clen; ){
		if	(clen < rem)
			rem = clen;
		if	(WindowStack == self){
			Screen write(p, buf, rem);
			buf += rem;
			}
		else	{
			drawSegment(p, img, rem);
			img += rem * 2;
			}
		p.x = corner.x;
		p.y++;
		clen -= rem;
		rem = size.x;
		}
	}

writeCA:	(p: point, buf: * char, len: unsigned) =
	{
	rows:	int;
	clen:	unsigned;
	rem:	unsigned;
	img:	* char;
	endimg:	* char;
	cp:	* char;
	bf:	* char;

	img = image + (p.y * size.x + p.x) * 2;
	endimg = image + imageLength;
	if	(img + (len << 1) < endimg)
		endimg = img + (len << 1);
	memCopy(img, buf, endimg - img);
	rem = size.x - p.x;
	if	(rem == 0)
		return;
	p.x += corner.x;
	p.y += corner.y;
	for	(clen = len; clen; ){
		if	(clen < rem)
			rem = clen;
		if	(WindowStack == self){
			Screen writeCA(p, buf, rem);
			buf += rem * 2;
			}
		else	{
			drawSegment(p, img, rem);
			img += rem * 2;
			}
		p.x = corner.x;
		p.y++;
		clen -= rem;
		rem = size.x;
		}
	}

putcAt:	(p: point, c: char, a: char) =
	{
	if	(p.y >= size.y)
		return;
	if	(p.x >= size.x)
		return;
	img:	* char;

	img = image + (p.y * size.x + p.x) * 2;
	img[0] = c;
	img[1] = a;
	p.x += corner.x;
	p.y += corner.y;
	if	(WindowStack == self)
		Screen putcAt(p, c, a);
	else
		drawSegment(p, img, 1);
	}

putc:	(p: point, c: char) =
	{
	}

putString:	(p: point, s: * char) =
	{
	buf:	[2] char;

	buf[1] = color;
	while	(*s){
		buf[0] = *s;
		write(p, buf, 1);
		p.x++;
		s++;
		}
	}

clear:	() =
	{
	Screen setColor(color);
	verticalScroll([ 0, 0 ], [ size.x - 1, size.y - 1 ], size.y);
	positionCursor([ 0, 0]);
	}

setColor:	(a: char) =
	{
	color = a;
	if	(flags & W_OPEN)
		Screen setColor(a);
	}

hide:	() int =
	{
	removeFromStack();
	flags &= ~W_OPEN;
	return 1;
	}

close:	() int =
	{
	w:	* window;

	removeFromStack();
	flags = 0;

		// now draw all the windows but this one.

	if	(WindowStack){
		registerInputQueue(WindowStack->keyQ);
		for	(w = WindowStack; w; w = w->next)
			w draw();
		Screen setColor(WindowStack->color);
		WindowStack showCursor();
		}
	else
		registerInputQueue(0);
	return 1;
	}

dispose:	() int =
	{
	w:	* window;

	if	(image){
		free(image);
		image = 0;
		}
	removeFromStack();
	flags = 0;

		// now draw all the windows but this one.

	if	(WindowStack){
		registerInputQueue(WindowStack->keyQ);
		for	(w = WindowStack; w; w = w->next)
			w draw();
		Screen setColor(WindowStack->color);
		WindowStack showCursor();
		}
	else
		registerInputQueue(0);
	return 1;
	}

};
 */
/*
beepSend:	(pointer, * kernelMessage, pointer, int) int =
	{
	beep();
	}

getWindowSend:	(pointer, * kernelMessage, pointer, int) int =
	{
	dupObject(ConsoleFile.myWindowId);
	return ConsoleFile.myWindowId;
	}

getKeySend:	(pointer, * kernelMessage, pointer, int) int =
	{
	if	(WindowStack)
		return WindowStack->keyQ getKey();
	else
		return -1;
	}

getRawKeySend:	(pointer, * kernelMessage, pointer, int) int =
	{
	if	(WindowStack)
		return WindowStack->keyQ getRawKey();
	else
		return -1;
	}

testKeySend:	(pointer, * kernelMessage, pointer, int) int =
	{
	if	(WindowStack)
		return WindowStack->keyQ testKey();
	else
		return -1;
	}

getShiftSend:	(pointer, * kernelMessage, pointer, int) int =
	{
	return getShiftState();
	}

setShiftSend:	(pointer, * kernelMessage, buf: pointer, int) int =
	{
	setShiftState(* ref int(buf));
	}

setBreakSend:	(pointer, * kernelMessage, buf: pointer, int) int =
	{
	return setBreakKey(* ref int(buf));
	}

getWInfoSend:	(pointer, m: * kernelMessage, buf: pointer, int) int =
	{
	w:	* window;
	i:	int;

	i = * ref int(buf);
	for	(w = WindowStack; w && i; i--, w = w->next)
		;
	m->scratch = w;
	if	(w == 0)
		return 4;
	else
		return sizeof windowInfo + w->title length() + 1;
	}

getWInfoReply:	(pointer, m: * kernelMessage, buf: pointer, len: int) int =
	{
	w:	* window;

	w = m->scratch;
	if	(w == 0)
		return ERRNOTFOUND;
	else
		return w getWindowInfo(buf, len);
	}

setRateSend:	(pointer, * kernelMessage, buf: pointer, int) int =
	{
	setRepeatRate(* ref int(buf));
	return SUCCESS;
	}

newWindowSend:	(pointer, * kernelMessage, buf: pointer, int) int =
	{
	op:	* packed {
		public:
		offs:	int;
		len:	int;
		size:	point;
		};
	name:	* char;
	w:	* window;
	i:	int;

	op = buf;
	if	(op->len)
		name = ref char(&op->len) + op->offs;
	else
		name = 0;
	w = window create([ name ], op->size);
	i = publishKernel(&WindowObject, w, AR_ALL);
	if	(i < 0)
		return i;
	else
		return makeLocal(i);
	}

wdisposeSend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	dc:	* window;

	dc = o;
	dc dispose();
	}

wreadSend:	(pointer, m: * kernelMessage, buf: pointer, int) int =
	{
	args:	* packed { public:
		p:	point;
			unsigned[16];
		len:	int;
		};

	args = buf;
	m->scratch = &args->p;
	return args->len;
	}

wreadReply:	(o: pointer, m: * kernelMessage, buf: pointer, len: int) int =
	{
	dc:	* window;
	pp:	* point;

	dc = o;
	pp = m->scratch;
	dc read(*pp, buf, len);
	return len;
	}

wwriteSend:	(o: pointer, * kernelMessage, buf: pointer, int) int =
	{
	dc:	* window;
	args:	* packed { public:
		p:	point;
			signed[16];
		offs:	int;
		len:	int;
		};
	i:	int;
	data:	* char;

	args = buf;
	data = ref char(&args->len) + args->offs;
	dc = o;
	dc write(args->p, data, args->len);
	}

wwriteCASend:	(o: pointer, * kernelMessage, buf: pointer, int) int =
	{
	dc:	* window;
	args:	* packed { public:
		p:	point;
			signed[16];
		offs:	int;
		len:	int;
		};
	i:	int;
	data:	* char;

	args = buf;
	data = ref char(&args->len) + args->offs;
	dc = o;
	dc writeCA(args->p, data, args->len >> 1);
	}

wseekSend:	(o: pointer, * kernelMessage, buf: pointer, int) int =
	{
	dc:	* window;
	args:	* { public:
		p:	point;
		};
	i:	int;

	dc = o;
	args = buf;
	dc positionCursor(args->p);
	}

wcolorSend:	(o: pointer, * kernelMessage, buf: pointer, int) int =
	{
	dc:	* window;
	args:	* { public:
		c:	char;
		};

	dc = o;
	args = buf;
	dc setColor(args->c);
	}

wputcAtSend:	(o: pointer, * kernelMessage, buf: pointer, int) int =
	{
	dc:	* window;
	args:	* packed { public:
		p:	point;
			signed[16];
		c:	int;
		a:	int;
		};

	dc = o;
	args = buf;
	dc putcAt(args->p, args->c, args->a);
	}

wvScrollSend:	(o: pointer, * kernelMessage, buf: pointer, int) int =
	{
	dc:	* window;
	args:	* packed { public:
		p1:	point;
			signed[16];
		p2:	point;
			signed[16];
		c:	int;
		};

	dc = o;
	args = buf;
	dc verticalScroll(args->p1, args->p2, args->c);
	}

wgetCSend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	dc:	* window;
	p:	point;

	dc = o;
	p = dc getCursor();
	return * ref int(&p);
	}

wtoBottomSend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	dc:	* window;

	dc = o;
	dc toBottom();
	}

wclearSend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	dc:	* window;

	dc = o;
	dc clear();
	}

wshowCSend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	dc:	* window;

	dc = o;
	dc showCursor();
	}

whideCSend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	dc:	* window;

	dc = o;
	dc hideCursor();
	}

wgetKeySend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	w:	* window;

	w = o;
	return w->keyQ getKey();
	}

wgetRawKeySend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	w:	* window;

	w = o;
	return w->keyQ getRawKey();
	}

wtestKeySend:	(o: pointer, * kernelMessage, pointer, int) int =
	{
	w:	* window;

	w = o;
	return w->keyQ testKey();
	}
 */
