/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	console;
include	window;
include	editor;
include	string;

formStatus:	public	type	char = {
	FS_SUCCESS,
	FS_ESC,
	FS_ERROR,

	FS_ACTIONS,

	FS_UP,
	FS_DOWN,
	FS_LEFT,
	FS_RIGHT,
	FS_TAB,
	FS_BTAB,
	FS_HOME,
	FS_END,
	FS_BLINE,
	FS_ELINE
	};

form:	public	type	inherit	viewer	{
	public:

	fields:		* field;
	data:		pointer;

constructor:	(titl: [:] char, sz: point) =
	{
	super constructor(titl, sz, 0);
	fields = 0;
	}

dispose:	dynamic	() =
	{
	f:	* field;

	for	(f = fields; f; f = fields){
		fields = f->next;
		f dispose();
		}
	}

connect:	(dat: pointer) =
	{
	data = dat;
	}

redraw:	dynamic	() =
	{
	clear();
	draw(data);
	}

enter:	dynamic	() * field =
	{
	f:	* field;
	cf:	* field;

	for	(f = fields; f; f = f->next){
		cf = f enter();
		if	(cf)
			return cf;
		}
	return 0;
	}

startup:	() int =
	{
	prior:	* viewer;
	fe:	fieldEditor;
	f:	* field;
	fs:	formStatus;
	p:	point;

	f = enter();
	if	(f == 0)
		return FS_ERROR;
	prior = setPrimaryViewer(0);
	fs = FS_SUCCESS;
	p = [ 0, 0 ];
	for	(;;){
		p2:	point;

		fe = [ "", f->size, 0 ];
		fe connect(self, f, data);
		fe open(f->ground, f->corner);
		fe resetEditor();
		fe setColor(f->color);
		setPrimaryViewer(&fe);
		p2 = f absoluteCoordinates([ 0, 0 ]);
		switch	(fs){
		case	FS_RIGHT:
		case	FS_TAB:
		case	FS_BTAB:
		case	FS_UP:
		case	FS_DOWN:
		case	FS_BLINE:
			fe atLine(p.y - p2.y);		// go to the relative
							// line, start of line
			break;

		case	FS_END:
			fe atEndFile();
			break;

		case	FS_LEFT:
		case	FS_ELINE:
			fe atLine(p.y - p2.y);		// go to the relative
							// line, end of line
			fe atEndLine();
			}
		consoleInput();
		fe close();
		if	(fe.status < FS_ACTIONS)
			break;
		fs = fe.status;
		f = fe.successor;
		p = fe.exitPoint;
		}
	setPrimaryViewer(prior);
	self close();
	return fe.status;
	}

draw:	dynamic	(data: pointer) =
	{
	f:	* field;

	for	(f = fields; f; f = f->next)
		f draw(data);
	}

locateField:	dynamic	(best: * field, p: point, action: formStatus) * field =
	{
	f:	* field;
	fx:	* field;

	for	(f = fields; f; f = f->next){
		fx = f locateField(best, p, action);
		if	(fx)
			best = fx;
		}
	return best;
	}

addItem:	(f: * field) =
	{
	fx:	* field;

	if	(fields){
		for	(fx = fields; fx->next; fx = fx->next)
			;
		fx->next = f;
		}
	else
		fields = f;
	f->next = 0;
	f->ground = self;
	}
/*
subframe:	(c: point, sz: point) * frame =
	{
	sf:	* form;

	sf = alloc(sizeof form);
	if	(sf == 0)
		return 0;
	sf = [ [ 0 ], sz ];
	sf->corner = c;
	addItem(sf);
	return sf;
	}
 */
text:	(p: point, color: char, text: [:] char) =
	{
	t:	* labelField;

	t = alloc(sizeof labelField);
	if	(t == 0)
		return;
	t = [ p ];
	t->size = [ |text, 1 ];
	t->text = text;
	t->color = color;
	addItem(t);
	}

rotary:	(c: point, sz: point, off: unsigned, fsize: unsigned, fmt: [:] char,
				flags: byte,
				fcolor: unsigned, choices: * * char) =
	{
	r:	* rotaryField;

	r = alloc(sizeof rotaryField);
	if	(r == 0)
		return;
	r = [ c ];
	r->size = sz;
	r->offset = off;
	r->dsize = fsize;
	r->fformat = fmt;
	r->flags = flags;
	r->color = fcolor;
	r->choices = choices;
	addItem(r);
	}

integer:	(c: point, sz: point, off: unsigned, fsize: unsigned, 
				fmt: [:] char,
				flags: byte,
				fcolor: unsigned) =
	{
	r:	* integerField;

	r = alloc(sizeof integerField);
	if	(r == 0)
		return;
	r = [ c ];
	r->size = sz;
	r->offset = off;
	r->dsize = fsize;
	r->fformat = fmt;
	r->flags = flags;
	r->color = fcolor;
	addItem(r);
	}

stringf:	(c: point, sz: point, off: unsigned, fsize: unsigned, 
				fmt: [:] char,
				flags: byte,
				fcolor: unsigned) =
	{
	r:	* stringField;

	r = alloc(sizeof stringField);
	if	(r == 0)
		return;
	r = [ c ];
	r->size = sz;
	r->offset = off;
	r->dsize = fsize;
	r->fformat = fmt;
	r->flags = flags;
	r->color = fcolor;
	addItem(r);
	}

	};

fieldEditor:	public	type	inherit editor {
	public:

connect:	(f: * form, fld: * field, data: pointer) =
	{
	textState = TS_INITIAL;
	owner = f;
	buff = [ fld, data ];
	super connect(&buff);
	}

escape:	dynamic	() =
	{
	buff close();
	clearPrimaryViewer();
	status = FS_ESC;
	}

enterKey:	dynamic	() =
	{
	leaveField(FS_SUCCESS);
	}

atLine:	(line: int) =
	{
	gotoLine(line + 1);
	}

atEndFile:	() =
	{
	super endFile();
	}

atEndLine:	() =
	{
	super endLine();
	}

leftArrow:	dynamic	() =
	{
	textState = TS_EDITING;
	if	(cursor.offset == 0)
		leaveField(FS_LEFT);
	else
		super leftArrow();
	}

rightArrow:	dynamic	() =
	{
	textState = TS_EDITING;
	if	(cursor.column >= size.x - 1)
		leaveField(FS_RIGHT);
	else
		super rightArrow();
	}

upArrow:	dynamic	() =
	{
	textState = TS_EDITING;
	if	(cursor.line == 0)
		leaveField(FS_UP);
	else
		super upArrow();
	}

downArrow:	dynamic	() =
	{
	textState = TS_EDITING;
	if	(cursor.line >= size.y - 1)
		leaveField(FS_DOWN);
	else
		super downArrow();
	}

deleteCharacter:	dynamic	() =
	{
	textState = TS_EDITING;
	super deleteCharacter();
	}

backspace:	dynamic	() =
	{
	textState = TS_EDITING;
	super backspace();
	}

enterCharacter:	dynamic	(k: keystroke) =
	{
	if	(textState == TS_INITIAL){
//		buff makeEmpty();
		buff deleteAll();
		redraw();
		}
	textState = TS_EDITING;
	super enterCharacter(k);
	}

tab:	dynamic	() =
	{
	leaveField(FS_TAB);
	}

backtab:	dynamic	() =
	{
	leaveField(FS_BTAB);
	}

beginFile:	dynamic	() =
	{
	leaveField(FS_HOME);
	}

endFile:	dynamic () =
	{
	leaveField(FS_END);
	}

beginLine:	dynamic	() =
	{
	leaveField(FS_BLINE);
	}

endLine:	dynamic	() =
	{
	leaveField(FS_ELINE);
	}

leaveField:	(action: formStatus) =
	{
	p:	point;

	if	(buff save() != FS_SUCCESS)
		return;
	p = absoluteCoordinates([ cursor.column, cursor.line ]);
	if	(action > FS_ACTIONS){
		successor = owner locateField(0, p, action);
		if	(successor == 0)
			return;
		if	(successor == buff.mappedField){
			textState = TS_EDITING;
			switch	(action){
			case	FS_UP:
				super upArrow();
				break;

			case	FS_DOWN:
				super downArrow();
				break;

			case	FS_RIGHT:
				super rightArrow();
				break;

			case	FS_HOME:
			case	FS_TAB:
			case	FS_BTAB:
				super beginFile();
				break;

			case	FS_BLINE:
				super beginLine();
				break;

			case	FS_END:
				super endFile();
				break;

			case	FS_LEFT:
				super leftArrow();
				break;

			case	FS_ELINE:
				super endLine();
				}
			return;
			}
		}
	buff close();
	clearPrimaryViewer();
	status = action;
	exitPoint = p;
	}

	status:		formStatus;
	successor:	* field;
	exitPoint:	point;

private:

	owner:		* form;
	buff:		fieldBuffer;
	textState:	tStates;
	};

tStates:	type	char = {
	TS_INITIAL,
	TS_EDITING
	};

fieldBuffer:	type	inherit	transferBuffer	{

	mappedData:	pointer;
	usedLen:	int;
	changed:	int;

public:

	mappedField:	* field;

constructor:	(fld: * field, dat: pointer) =
	{
	len:	int;

	len = fld->size.x * int(fld->size.y) + 1;
	data = 0;
	beginExtract(len);
	memSet(data, 0, len);
	mappedData = dat;
	mappedField = fld;
	changed = 0;
	usedLen = fld load(dat, data, len);
	}

deleteAll:	() =
	{
	usedLen = 1;
	data[0] = EOF_MARK;
	}

close:	dynamic	() =
	{
	makeEmpty();
	usedLen = 0;
	changed = 0;
	}

save:	() int =
	{
	i:	int;

	i = mappedField store(mappedData, data, usedLen);
	if	(i == FS_SUCCESS)
		changed = 0;
	return i;
	}

makeEmpty:	dynamic	() =
	{
	usedLen = 0;
	changed = 1;
	super makeEmpty();
	}

insert:	dynamic	(buf: * char, len: int) =	// buffer address and length
	{
	i:	int;
	cp:	* char;

	if	(usedLen + len > dataLen)
		len = dataLen - usedLen;
	if	(offset > usedLen)
		offset = usedLen;
	i = usedLen - offset;
	cp = &data[offset];
	if	(i)
		memMove(&data[offset + len], cp, i);
	memCopy(cp, buf, len);
	usedLen += len;
	changed = 1;
	}

delete:	dynamic	(len: int) =
	{
	rem:	int;

	rem = usedLen - offset;
	if	(len > rem)
		len = rem;
	memMove(&data[offset], &data[offset + len], rem - len);
	usedLen -= len;
	changed = 1;
	}

seek:	dynamic	(newPos: textPos, whence: int) textPos =
	{
	switch	(whence){
	case	0:			// SEEK_ABS
		offset = newPos;
		break;

	case	1:			// SEEK_CUR
		offset += newPos;
		break;

	case	2:
		offset = dataLen + newPos;
		break;

	case	3:
		cp:	* char;
		i:	textPos;
		lc:	lineNum;

		lc = newPos;
		i = 0;
		if	(lc){
			for	(i = 1, cp = data; i < usedLen; i++, cp++){
				if	(*cp == '\n'){
					lc--;
					if	(lc == 0)
						break;
					}
				}
			}
		offset = i;
		break;
		}
	return offset;
	}

fetchLine:	dynamic	(ln: lineNum, col: int) * char = 
					// line number and offset in line
	{
	cp:	* char;
	off:	int;

	off = seek(ln, 3);		// get to ln, column 0
	cp = &data[off];
	while	(col){
		if	(off >= usedLen ||
			 *cp == '\n')
			return 0;
		cp++;
		off++;
		col--;
		}
	offset = off;
	return cp;
	}

getLineno:	dynamic	(lnp: * lineNum, colp: * int) =
					// return current location in lines
	{
	cp:	* char;
	i:	textPos;
	col:	int;
	ln:	lineNum;

	ln = 0;
	i = 0;
	col = 0;
	for	(cp = data; i < offset; i++, cp++){
		if	(*cp == '\n'){
			col = i;
			ln++;
			}
		}
	*colp = i - (col + 1);
	*lnp = ln;
	}

extract:	dynamic	(buff: * editBuffer, len: int) =
					// buffer to put the data in
	{
	buff beginExtract(len);
	buff write(&data[offset], len);
	}

hasChanged:	dynamic	() int =
	{
	return 0;
	}

	};


FLD_READONLY:	public	const byte = 0x01;

field:	public	type	inherit frame	{
	public:

	flags:		byte;
	color:		byte;
	next:		* field;

dispose:	dynamic	() =
	{
	free(self);
	}

printf:	dynamic	(p: point, fmt: [:] char, ...) =
	{
	fe:	frameEmitter;

	fe = [ self ];
	fe.p = p;
	setColor(color);
	fe format(fmt, ...);
	while	(fe.p.x < size.x){
		putc(fe.p, ' ');
		fe.p.x++;
		}
	}

enter:	dynamic	() * field =
	{
	if	(flags & FLD_READONLY)
		return 0;
	else
		return self;
	}

locateField:	dynamic	(best: * field, p: point, action: formStatus) * field =
	{
	c:	point;
	b:	point;

	if	(flags & FLD_READONLY)
		return best;
	c = absoluteCoordinates([ 0, 0 ]);
	switch	(action){
	case	FS_UP:
		if	(c.x > p.x)
			break;
		if	(c.x + size.x <= p.x)
			break;
		if	(c.y >= p.y)
			break;
		if	(best == 0)
			return self;
		p = best absoluteCoordinates([ 0, 0 ]);
		if	(c.y > p.y)
			return self;
		break;

	case	FS_DOWN:
		if	(c.x > p.x)
			break;
		if	(c.x + size.x <= p.x)
			break;
		if	(c.y <= p.y)
			break;
		if	(best == 0)
			return self;
		p = best absoluteCoordinates([ 0, 0 ]);
		if	(c.y < p.y)
			return self;
		break;

	case	FS_LEFT:
		if	(c.y > p.y)
			break;
		if	(c.y + size.y <= p.y)
			break;
		if	(c.x >= p.x)
			break;
		if	(best == 0)
			return self;
		p = best absoluteCoordinates([ 0, 0 ]);
		if	(c.x > p.x)
			return self;
		break;

	case	FS_RIGHT:
		if	(c.y > p.y)
			break;
		if	(c.y + size.y <= p.y)
			break;
		if	(c.x <= p.x)
			break;
		if	(best == 0)
			return self;
		p = best absoluteCoordinates([ 0, 0 ]);
		if	(c.x < p.x)
			return self;
		break;

	case	FS_TAB:
		if	(c.y + size.y <= p.y)
			break;

			// If this field ends on the same line and is to the
			// left, forget it.

		if	(c.y + size.y == p.y + 1){
			if	(c.x <= p.x)
				break;
			}
		if	(best == 0)
			return self;

			// Trim the current field dimensions to below or
			// to the right of the starting point

		if	(c.y <= p.y){
			if	(c.x < p.x)
				c.y = p.y + 1;
			else
				c.y = p.y;
			}
		b = best absoluteCoordinates([ 0, 0 ]);

			// Trim the best field dimensions to below or
			// to the right of the starting point

		if	(b.y < p.y){
			if	(b.x < p.x)
				b.y = p.y + 1;
			else
				b.y = p.y;
			}
		if	(c.y < b.y)
			return self;
		if	(c.y > b.y)
			break;
		if	(c.x < b.x)
			return self;
		break;

	case	FS_BTAB:
		if	(c.y > p.y)
			break;

			// If this field starts on the same line and is to the
			// right, forget it.

		if	(c.y == p.y){
			if	(c.x >= p.x)
				break;
			}
		if	(best == 0)
			return self;

			// Trim the current field dimensions to above or
			// to the left of the starting point.  Also move
			// the c point to the lower right corner

		c.y += size.y - 1;
		if	(c.y >= p.y){
			c.y = p.y - 1;
			if	(c.x < p.x)
				c.y++;
			}
		c.x += size.x - 1;

		b = best absoluteCoordinates([ 0, 0 ]);

			// Trim the best field dimensions to below or
			// to the right of the starting point

		b.y += best->size.y - 1;
		if	(b.y >= p.y){
			b.y = p.y - 1;
			if	(b.x < p.x)
				b.y++;
			}
		b.x += best->size.x - 1;
		if	(c.y > b.y)
			return self;
		if	(c.y < b.y)
			break;
		if	(c.x > b.x)
			return self;
		break;

	case	FS_HOME:
		if	(best == 0)
			return self;
		p = best absoluteCoordinates([ 0, 0 ]);
		if	(c.y < p.y)
			return self;
		if	(c.y > p.y)
			break;
		if	(c.x < p.x)
			return self;
		break;

	case	FS_END:
		if	(best == 0)
			return self;
		p = best absoluteCoordinates([ 0, 0 ]);
		if	(c.y + size.y > p.y + best->size.y)
			return self;
		if	(c.y + size.y < p.y + best->size.y)
			break;

			// Fields end on the same line, so pick the higher x

		if	(c.x > p.x)
			return self;
		break;

	case	FS_BLINE:
		if	(c.y > p.y)
			break;
		if	(c.y + size.y <= p.y)
			break;
		if	(best == 0)
			return self;
		p = best absoluteCoordinates([ 0, 0 ]);
		if	(c.x < p.x)
			return self;
		break;

	case	FS_ELINE:
		if	(c.y > p.y)
			break;
		if	(c.y + size.y <= p.y)
			break;
		if	(best == 0)
			return self;
		p = best absoluteCoordinates([ 0, 0 ]);
		if	(c.x > p.x)
			return self;
		break;
		}
	return best;
	}

load:	dynamic	(pointer, buf: * char, int) int =
	{
	buf[0] = EOF_MARK;
	return 1;
	}

store:		dynamic	(pointer, * char, int) formStatus =
	{
	return FS_ERROR;
	}

	};

labelField:	public	type	inherit	field	{
	public:

	text:		[:] char;

constructor:	(p: point) =
	{
	corner = p;
	flags = FLD_READONLY;
	}

draw:	dynamic	(pointer) =
	{
	setColor(color);
	printf([ 0, 0 ], "%s", text);
	}

};

dataField:	public	type	inherit	field	{
	public:

	offset:		unsigned;
	dsize:		unsigned;
	fformat:	[:] char;
	};

rotaryField:	public	type	inherit dataField	{
	public:

	choices:	* * char;

constructor:	(p: point) =
	{
	corner = p;
	}

draw:	dynamic	(data: pointer) =
	{
	i:	long;
	fmt:	[:] char;
	s:	* * char;
	cs:	* char;

	i = 0;
	memCopy(&i, ref char(data) + offset, dsize);
	if	(|fformat)
		fmt = fformat;
	else
		fmt = "%s";
	s = choices;
	for	(;;){
		if	(*s == 0){
			cs = "<illegal>";
			break;
			}
		if	(i == 0){
			cs = *s;
			break;
			}
		i--;
		s++;
		}
	printf([ 0, 0 ], fmt, cs);
	}

	};

integerField:	public	type	inherit dataField	{
	public:

constructor:	(p: point) =
	{
	corner = p;
	}

draw:	dynamic	(data: pointer) =
	{
	i:	long;
	fmt:	[:] char;

	i = 0;
	memCopy(&i, ref char(data) + offset, dsize);
	if	(|fformat)
		fmt = fformat;
	else
		fmt = "%d";
	printf([ 0, 0 ], fmt, i);
	}

load:	dynamic	(data: pointer, buf: * char, len: int) int =
	{
	s:	* char;
	i:	int;
	dst:	[:] char;
	j:	int;
	numBuf:	[12] char;
	sc:	stringConverter;

	s = ref char(data) + offset;
	i = 0;
	memCopy(&i, s, dsize);
	sc = [ numBuf ];
	dst = sc decodeSigned(i);
	j = |dst;
	if	(j >= len)
		j = len - 1;
	memCopy(buf, &numBuf, j);
	buf[j] = EOF_MARK;
	return j + 1;
	}

store:		dynamic	(data: pointer, buf: * char, len: int) formStatus =
	{
	s:	* char;
	i:	int;
	sc:	stringConverter;

	s = ref char(data) + offset;
	i = sc encodeSigned(buf[:len]);
	memCopy(s, &i, dsize);
	return FS_SUCCESS;
	}

	};

stringField:	public	type	inherit dataField	{
	public:

constructor:	(p: point) =
	{
	corner = p;
	}

draw:	dynamic	(data: pointer) =
	{
	fmt:	[:] char;
	s:	* char;

	s = ref char(data) + offset;
	if	(|fformat)
		fmt = fformat;
	else
		fmt = "%s";
	printf([ 0, 0 ], fmt, s);
	}

load:	dynamic	(data: pointer, buf: * char, len: int) int =
	{
	s:	[:] char;

	s = (ref char(data) + offset)[:dsize];
	if	(|s >= len)
		|s = len - 1;
	memCopy(buf, s, |s);
	buf[|s] = EOF_MARK;
	return |s + 1;
	}

store:		dynamic	(data: pointer, buf: * char, len: int) formStatus =
	{
	s:	* char;
	i:	int;

	s = ref char(data) + offset;
	i = len - 1;
	if	(i >= dsize)
		i = dsize - 1;
	memCopy(s, buf, i);
	s[i] = 0;
	return FS_SUCCESS;
	}

	};

