/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	file, string;

searchPattern:	public	type	{
	public:

	variable:	[26] * char;

constructor:	(meta: boolean) =
	{
	metaflag = meta;
	memSet(&variable, 0, sizeof variable);
	compiledPattern = 0;
	}

dispose:	() =
	{
	if	(compiledPattern)
		sr_free_re(compiledPattern);
	}

/*
 *	sr_search() takes the pattern compiled by sr_compile() and searches
 *	the string str.  If an instance of the pattern is found,
 *	sr_search() returns the length of the match and sets str_st to point
 *	to the first character of the match in str.  The application can use
 *	this information to show the user exactly what was matched in the line.
 *	If the user doesn't wish to do a substitution at that point, sr_search()
 *	can be called again with str set to (str_st + length_of_match) which
 *	will then continue the search from where the last match was found.
 */
search: 	(str: * char,	/* The string to search for the pattern. */
		endstr: * char,	/* Pointer to the end of the search string */
		str_st: ** char	/* Returns where the match was found. */
		) int =
	{
	start:	* char;
	start = str;
	c:	* char;
	i:	int;
	t:	* pnode;

	/* Set up the search for the first character in the pattern.	*/
	t = compiledPattern;
	if	(t == 0)
		return -1;
	while (t->ntype == CAT)
		t = t->left;

	if (t->ntype == CHARS)
		i = int(t->left);
	else
		i = 0;

	while (TRUE) {
		if	(i)  // fast loop to find first character
			for	(;;){
				if	(str >= endstr)
					return -1;
				if	(*str == i)
					break;
				str++;
				}
		c = scanner(compiledPattern, str, endstr, start);
		if (c != NULL) {
			*str_st = str;
			return(c - str);
			}

		if (str >= endstr)
			break;

		str++;
		}

	return(-1);
	}

/*
 *	This is a very straightforward routine.  You put one pointer (c)
 *	at the beginning of the old string, and the other pointer (d)
 *	at the beginning of the new string.  You copy from the old string
 *	until you get to where the match began.  You substitute the
 *	new string (with variable referencing, if any).  And then you
 *	copy the rest of the old string over.  Easy.
 */
replace:	(
		old_str: * char,	/* The original string. */
		old_str_st: * char,	/* Where to start the substitution. */
		len: int,	    	/* The length of the match in the old_str. */
		new_str: * char,   	/* Pointer to a buffer for the new string. */
		sub_str: [:] char   	/* The substitution string. */
		) boolean =
	{
	c:	* char;
	d:	* char;
	index:	int;

	if (!decipher(sub_str))
		return(FALSE);

	/* The only special character allowed is '<' so			*/
	/* turn off all other special characters and			*/
	/* check syntax first before starting to substitute.		*/
	for (index = 0; nextch[index]; index++)
		if (Flags[index]) {
			if (nextch[index] != '<') {
				Flags[index] = FALSE;
				continue;
				}

			index++;
			if ('A' <= nextch[index] && nextch[index] <= 'Z')
				nextch[index] += 'a' - 'A';
			else if (nextch[index] < 'a' || 'z' < nextch[index])
				goto err;
			index++;
			if (nextch[index] != '>')
				goto err;
			}

	/* do substitution */
	d = new_str;
	for (c = old_str; c < old_str_st; )
		*d++ = *c++;

	for (index = 0; nextch[index]; index++) {
		if (Flags[index]) {
			index++;
			c = variable[nextch[index++] - 'a'];
			if (c == NULL)
				continue;

			while (*c)
				*d++ = *c++;
			}

		else *d++ = nextch[index];
		}

	c = old_str_st + len;
	while (*d++ = *c++)
		;

	free(nextch);
	free(Flags);
	return(TRUE);

	label	err:

	free(nextch);
	free(Flags);
	return(FALSE);
	}

/*
 *	This routine returns the parsed form of the search patern
 *	that it was passed.
 */
compile:	(s: [:] char) boolean =
	{
	xstack:	[STACK] pstack;
	new_:	* pnode;
	trace1:	* pnode;
	trace2:	* pnode;
	nstack:	int;
	i:	int;
	j:	int;
	k:	int;
	next:	int;
	lastt:	int;
	nextt:	int;

	if	(compiledPattern)
		sr_free_re(compiledPattern);
	compiledPattern = 0;
	if (!decipher(s))
		return FALSE;

	if	(DEBUG){
		printf("pattern string deciphered\n");
		for	(i = 0; ; i++){
			printf("%d: s[i] = %c nextch[i] = %x Flags[i] = %d\n",
				i, s[i], nextch[i], Flags[i]);
			if	(s[i] == 0)
				break;
			}
		}
	Index = 0;
	xstack[0].ntype = END;
	xstack[0].term = TRUE;
	xstack[0].val  = NULL;
	nstack = 1;
	next = END;
	while (TRUE) {
	        xstack[nstack].ntype = next = gettoken(&xstack[nstack], next);
		if	(DEBUG){
			printf("token = %d Index = %d\n", next, Index);
			for	(i = 0; i <= nstack; i++)
				printf("%d: ntype = %d term = %d val = %lx\n",
                                                i, xstack[i].ntype,
						xstack[i].term,
						xstack[i].val);
			}
        	if	(next < 0)
			goto err;

		while (TRUE) {
            /*  find top terminal on the stack */
			for	(i = nstack - 1; !xstack[i].term; i--)
				;

			if	(DEBUG)
				printf("Top terminal is %d\n", i);

            /* see if should shift or reduce */
			if	(ff[xstack[i].ntype] <= gg[next])
				break;

            /* perform reduction */
			for	(i = 0; i < NPROD; i++) {
				j = nstack - 1;
				k = 0;
				while	(TRUE) {
					if	(xstack[j].term != prod[i].terms[k])
						goto nope;

					if	(xstack[j].term) {
			/* Here lint complains since nextt is assigned 	*/
			/* to lastt without having been set.  This is 	*/
			/* fine here, since you will come through this 	*/
			/* place at least twice before having the 	*/
			/* prod[i].types[k] == 0 test succeed (which is	*/
			/* the only place where lastt is used).		*/
						lastt = nextt;
						nextt = xstack[j].ntype;
						if	(prod[i].types[k] == 0)
							if	(ff[nextt] < gg[lastt])
								goto reduce;
							else
								goto nope;

						if	(nextt != prod[i].types[k])
							goto nope;
						}

					j--;
					k++;
					}

				label	nope:

				continue;
				}

			if	(DEBUG)
				printf("Shift/reduce error\n");
			goto err;

			label	reduce:

			if	(DEBUG)
				printf("Reducing case %d\n", i);
			switch	(i){
			case 1:             /* N & N => N */
                    /* The code is a little involved here in the interest */
		    /* of speeding up the scanning time later.  	  */
                    /* If you have a sequence of characters, 		  */
                    /* character classes, arbitraries, positions, and     */
                    /* variables, they are put into a linked list.        */
                    /* Scanning is quick since each of these can be       */
                    /* tested without worrying about backtracking.        */
				trace1 = xstack[nstack-3].val;
				while	(trace1->ntype == CAT)
					trace1 = trace1->right;
				if	(trace1->ntype != CHARS     &&
					 trace1->ntype != CCLASS    &&
					 trace1->ntype != ARBITRARY &&
					 trace1->ntype != POSITION  &&
					 trace1->ntype != VARIABLE)
					goto cat;

				while	(trace1->right != NULL)
					trace1 = trace1->right;

        			trace2 = xstack[nstack-1].val;
				if	(trace2->ntype == CHARS     ||
					 trace2->ntype == CCLASS    ||
					 trace2->ntype == ARBITRARY ||
					 trace2->ntype == POSITION  ||
					 trace2->ntype == VARIABLE)  {
					trace1->right   = trace2;
					xstack[nstack-2] = xstack[nstack];
					nstack         -= 2;
					break;
					}

			label	cat:
			/* fall through and handle ordinary case */
			case 0:             /* N | N => N */
				new_        = newPnode();
				new_->ntype  = xstack[nstack-2].ntype,
				new_->left  = xstack[nstack-3].val;
				new_->right = xstack[nstack-1].val;
				(xstack[nstack-3].val)->parent =
					(xstack[nstack-1].val)->parent = new_;
				xstack[nstack-3].val = new_;
				xstack[nstack-2]     = xstack[nstack];
				nstack             -= 2;
				break;

                /* REPLICATION CODE (Cases 2 and 3)			*/
                /* The code is a little opaque here too since I		*/
                /* contract replications of single characters, single	*/
                /* character classes and a single arbitrary into a new	*/
                /* node with a different type to speed up scanning.	*/
	                case 2:             /* N R1 => N */
				trace1 = xstack[nstack-2].val;
				if	(trace1->right != NULL)
					goto rep;

				if	(trace1->ntype == CHARS) {
					trace1->ntype = CHARS1;
					goto repdone;
					}

				if	(trace1->ntype == CCLASS) {
					trace1->ntype = CCLASS1;
					goto repdone;
					}

				if	(trace1->ntype == ARBITRARY) {
					trace1->ntype = ARB1;
					goto repdone;
					}
				goto rep;

			case 3:             /* N R2 => N */
				trace1 = xstack[nstack-2].val;
				if	(trace1->right != NULL)
					goto rep;

				if	(trace1->ntype == CHARS) {
					trace1->ntype = CHARS2;
					goto repdone;
					}

				if	(trace1->ntype == CCLASS) {
					trace1->ntype = CCLASS2;
					goto repdone;
					}

				if	(trace1->ntype == ARBITRARY) {
					trace1->ntype = ARB2;
					goto repdone;
					}

                   /* CODE FOR REPLICATION CASES */
			label	rep:
                    new_        = newPnode();
                    new_->ntype  = xstack[nstack-1].ntype,
                    new_->left  = xstack[nstack-2].val;
                    new_->right = xstack[nstack-1].val;
                    xstack[nstack-2].val->parent = new_;
                    xstack[nstack-2].val = new_;
                    xstack[nstack-1]     = xstack[nstack];
                    nstack--;
                    break;

		label	repdone:
                    trace1->right   = xstack[nstack-1].val;
                    xstack[nstack-1] = xstack[nstack];
                    nstack--;
                    break;

                case 4:             /* N $ C => N */
                    if ('A' <= int(xstack[nstack-1].val) &&
                        int(xstack[nstack-1].val) <= 'Z')
                            xstack[nstack-1].val += 'a' - 'A';

                    else if (int(xstack[nstack-1].val) < 'a' ||
                        'z' < int(xstack[nstack-1].val))
			    goto err;

                    new_        = newPnode();
                    new_->ntype  = xstack[nstack-2].ntype,
                    new_->left  = xstack[nstack-3].val;
                    new_->right = xstack[nstack-1].val;
                    xstack[nstack-3].val->parent = new_;
                    xstack[nstack-3].val = new_;
                    xstack[nstack-2]     = xstack[nstack];
                    nstack             -= 2;
                    break;

                case 5:             /* ( N ) => N */
                    xstack[nstack-3] = xstack[nstack-2];
                    xstack[nstack-2] = xstack[nstack];
                    nstack         -= 2;
                    break;

                case 6:             /* C => N */
                case 7:             /* CC => N */
                case 8:             /* A => N */
                case 9:             /* P => N */
                case 10:            /* V => N */
                    new_        = newPnode();
                    new_->ntype  = xstack[nstack-1].ntype;
                    new_->left  = xstack[nstack-1].val;
                    new_->right = NULL;
                    xstack[nstack-1].val  = new_;
                    xstack[nstack-1].term = FALSE;
                    break;

                default:
			if	(DEBUG)
				error("in reduction");
			goto err;
			}
		}

		if	(xstack[nstack++].ntype == END)
			break;

		if	(nstack == STACK) {
			printf("SEARCH and REPLACE error - stack overflow\n");
			nstack--;
			goto err;
			}
		}

	if	(nstack-- != 3)
		goto err;

	free(nextch);
	free(Flags);
	compiledPattern = xstack[1].val;
	return TRUE;

	label	err:

	for	(i = 1; i <= nstack; i++)
		if	(!xstack[i].term)
			sr_free_re(xstack[i].val);

	free(nextch);
	free(Flags);
	if	(DEBUG)
		printf("Error in compile\n");
	return FALSE;
	}

	private:

/*
 * This routine does the actual scanning of the subject string for the
 * pattern that was compiled by sr_compile().
 *
 * The subject string is the string being scanned.  The pattern is the list of
 * operations to be done in scanning that have been constructed into a parse
 * tree by sr_compile().
 *
 *
 *      			LIST OF OPERATORS:
 *
 *      Generators:
 *
 *      The essence of a generator is that if at some point sufficient
 *      information is pushed onto a stack so that if later in scanning
 *      the pattern matching should fail for some reason, popping the
 *      stack will restore the scanning process to the same state as when
 *      the push was done.  At that time the generator can then take some
 *      alternate course of action.
 *
 *      Current generators include alternation and replication.  Note that
 *      character classes and range of positions are not generators, even
 *      though they imply use of alternatives, because the alternatives
 *      can all be tested without scanning ahead in the subject string.
 *
 *      RESTORE: is a special generator.  When it is pushed, the character
 *          pointer normally associated with the scanning position is instead
 *          a pointer to a node in the tree which needs to have its alter
 *          flag restored to some previous value.  After doing the restoration,
 *          RESTORE fails.
 *
 *      ALT:
 *          First: push; execute left branch
 *          Success: (from either branch) succeed
 *          Failure: execute right branch
 *
 *      REP1: (as many as possible)
 *          First: reset alter counter; goto Success
 *          Success: bump counter;
 *                   if insufficient count then
 *                       push RESTORE; execute left branch
 *                   if upper limit hit then
 *                       push RESTORE; succeed
 *                   push normally
 *                   execute left branch
 *          Failure: push RESTORE
 *                   succeed
 *
 *      REP2: (as few as possible)
 *          First: reset alter counter; goto Success
 *          Success: bump counter;
 *                   if insufficient count then
 *                       push RESTORE; execute left branch
 *                   push normally
 *                   succeed
 *          Failure: if upper limit hit then fail
 *                   push RESTORE
 *                   execute left branch

 *      REP1 and REP2 use as their executable operand (the left branch) any
 *      arbitrary pattern.  In addition, to make scanning faster in typical
 *      cases, there are CHARS1 and CHARS2, ARB1 and ARB2, and CCLASS1 and
 *      CCLASS2.  All of these are special cases of REP1 and REP2.  The code
 *      is simpler and faster since the execution of the left branch can be
 *      skipped and only one push on the stack is required.
 *
 *      Non-Generators:
 *
 *      Note that non-generators will not normally receive either the failure
 * 	signal or the success signal.  CAT and ASSIGN are exceptions to this.
 *	Their action is described below.
 *
 *      CAT:
 *          First: set alter flag to indicate left side; execute left branch
 *          Success: if coming from left branch then push RESTORE;
 *                   set the alter flag for right branch; execute right branch
 *                   if coming from right branch, then succeed.
 *
 *      ASSIGN:
 *          First: save starting pos in alter flag; execute left branch
 *          Success: push RESTORE; succeed
 *
 *      CHARS:
 *          Succeed if the characters in the operator string match those
 *          in the subject at current position.  Otherwise fail.
 *
 *      CCLASS:
 *          Succeed if the current character from the subject is any of
 *          those in the CCLASS (or not in the CCLASS if the group is
 *          negated).  Otherwise fail.
 *
 *      ARBITRARY:
 *          Succeed if not at the end of the subject.  Otherwise fail.
 *
 *      POSITION:
 *          Succeed if in the range of positions allowed.  Otherwise fail.
 *
 *      VAR:
 *          Like CHARS, but instead of using an operand to match against,
 *          use the value of a variable.
 */

scanner:	(t: * pnode, subj: * char, endsubj: * char,
				st_start: * char) * char =
	{
	pos:	* char;
	d:	* char;
	c:	* char;
	twork:	* pnode;
	pflag:	int;
	i:	int;

        Xstack = NULL;
	pos   = subj;

    pflag = FIRST;
    while (TRUE) {
        if (pflag == SUCCESS)
            t = t->parent;

        if (pflag == FAILURE) {
            pos = pop(&twork);
            t   = twork;
            }

        if (t == NULL) {
            if (pflag == SUCCESS) {
                /* Make sure that the stack is not NULL upon success.	*/
                if (push(&Done, pos))
                    goto err;

		return(pos);
                }

            /* If FAILURE, then stack is necessarily NULL. 		*/
            return(0);
            }

        switch(t->ntype) {
            case DONE:                                  /* FAILURE 	*/
                continue;

            case RESTORE:                               /* FAILURE 	*/
                (ref pnode(pos))->alter = t->alter;
                continue;

            case ALT:
                switch(pflag) {
                    case FIRST:
                        if (push(t, pos))
                            goto err;
                        t = t->left;		   	/* left branch	*/
                        continue;

                    case SUCCESS:
                        continue;                       /* we worked 	*/

                    case FAILURE:                       /* TRY AGAIN 	*/
                        pflag = FIRST;                  /* right branch */
                        t     = t->right;
                        continue;
                    }

            case CAT:
                switch (pflag) {
                    case FIRST:
                        t->alter = FALSE;               /* left branch flag */
                        t        = t->left;
                        continue;

                    case SUCCESS:
                        if (pusher(t))
                            goto err;        		/* push RESTORE	*/

                        if (t->alter)
                            continue;         		/* test flag  	*/

                        t->alter = TRUE;                /* right branch flag */
                        pflag    = FIRST;
                        t        = t->right;
                        continue;
                    }

            case ASSIGN:
                switch (pflag) {
                    case FIRST:
                        t->asgpos = pos;
                        t         = t->left;
                        continue;

                    case SUCCESS:
                        i = int(t->right) - 'a';
                        if (variable[i] != NULL)
                            free(variable[i]);

                        d = variable[i] =
				alloc(pos - t->asgpos + 1);
			if (d == 0)
				fatal("out of memory\n");
                        c = t->asgpos;
                        while (c < pos)
                            *d++ = *c++;
                        *d = EOS;
                        if (pusher(t))
                            goto err;    		/* push RESTORE */
                        continue;                   	/* succeed 	*/
                    }

            case REP1:
                switch (pflag) {
                    case FIRST:
                        t->alter = -1;
                        /* Fall through here */
                    case SUCCESS:
                        t->alter++;                     /* bump counter */
                        if (t->alter < LOWER(t)) {      /* insufficient count */
                            if (pusher(t))
                                goto err;               /* push RESTORE */

                            pflag = FIRST;              /* execute LEFT */
                            t = t->left;
                            continue;
                            }

                        if (t->alter > LOWER(t) &&
                            pos == ref char(BEGIN(t))) { /* null string check */
                                pflag = FAILURE;
                                continue;
                                }

                        if (pusher(t->right))
			    goto err;
                        if (t->alter == UPPER(t)) {     /* upper limit hit */
                            if (pusher(t))
                                goto err;               /* push RESTORE	*/

                            pflag = SUCCESS;            /* SUCCEED 	*/
                            continue;
                            }

			t->right->alter = int(pos);	/* null check 	*/
					// t->right->alter == BEGIN(t);
                        if (push(t, pos))
                            goto err;                   /* push 	*/

                        pflag = FIRST;                  /* execute LEFT */
                        t = t->left;
                        continue;

                    case FAILURE:
                        if (pusher(t))
                            goto err;                   /* push RESTORE */

                        pflag = SUCCESS;                /* SUCCEED 	*/
                        continue;
                    }

            case CHARS1:
            case ARB1:
            case CCLASS1:
                switch (pflag) {
                    case FIRST:
                        t->alter = 0;
                        if (t->ntype == CHARS1)
                            while (pos[t->alter] == int(t->left))
                                t->alter++;

                        else if (t->ntype == ARB1)
                            while (pos + t->alter < endsubj)
                                        t->alter++;

                        else /* t->ntype == CCLASS1 */
                            while (inrange(pos[t->alter], t->left))
                                        t->alter++;

                        if (UPPER(t) >= 0 && t->alter > UPPER(t))
                            t->alter = UPPER(t);

                        /* t->alter ends at the maximum number of       */
                        /* successful matches so it needs to be bumped. */
                        t->alter++;
                        pflag = FAILURE;
                        /* fall through here */
                    case FAILURE:
                        /* decrement t->alter first */
                        if (--t->alter < LOWER(t))
                            continue;    		/* FAIL 	*/

                        if (push(t, pos))
                            goto err;

                        pos  += t->alter;
                        pflag = SUCCESS;
                        continue;
                    }

            case REP2:
                switch (pflag) {
                    case FIRST:
                        t->alter = -1;
                        /* Fall through here */
                    case SUCCESS:
                        t->alter++;                     /* bump counter */
                        if (t->alter < LOWER(t)) {      /* insufficient count */
                            if (pusher(t))
                                goto err;               /* push RESTORE */

                            pflag = FIRST;              /* execute LEFT */
                            t     = t->left;
                            continue;
                            }

                        if (t->alter > LOWER(t)) {
                            if (pos == ref char(BEGIN(t))) {
                                /* null string check */
                                pflag = FAILURE;
                                continue;
                                }

                            if (pusher(t->right))
                                goto err;
                            }

                        if (push(t, pos)) 
                            goto err;                   /* push 	*/

                        pflag = SUCCESS;                /* SUCCEED 	*/
                        continue;

                    case FAILURE:
                        if (t->alter == UPPER(t))       /* FAIL 	*/
                            continue;

			t->right->alter = int(pos);	/* save null check */
					// t->right->alter == BEGIN(t);
                        if (pusher(t)) 
                            goto err;                   /* push RESTORE */

                        pflag = FIRST;                  /* execute LEFT */
                        t = t->left;
                        continue;
                    }

            case CHARS2: 
            case ARB2:
            case CCLASS2:
                switch (pflag) {
                    case FIRST:
                        t->alter = 0;
                        if (t->ntype == CHARS2)
                            while (pos[t->alter] == int(t->left))
                                t->alter++;

                        else if (t->ntype == ARB2)
                            while (pos + t->alter < endsubj)
                                        t->alter++;

                        else /* t->ntype == CCLASS2 */
                            while (inrange(pos[t->alter], t->left))
                                    t->alter++;

                        if (UPPER(t) >= 0 && t->alter > UPPER(t))
                            t->alter = UPPER(t);

                        /* t->alter ends at exactly the maximum
                            number of successful matches */
                        if (t->alter < LOWER(t)) {
                            pflag = FAILURE;
                            continue;
                            }

                        pos      += LOWER(t);
                        t->alter -= LOWER(t);
                        /* t-> alter now contains how many different  */
                        /* options we can try                         */
                        if (push(t, pos))
                            goto err;

                        pflag = SUCCESS;
                        continue;

                    case FAILURE:
                        /* t->alter is how many left to try */
                        if (t->alter-- <= 0) 
				continue;      		/* FAIL 	*/

                        pos++;
                        if (push(t, pos))
                            goto err;

                        pflag = SUCCESS;
                        continue;
                    }

            case CHARS:
            case ARBITRARY:
            case CCLASS:
            case POSITION:
            case VARIABLE:
                twork = t;
                while (twork) {
                    if (twork->ntype == CHARS) {
                        if (*pos++ != int(twork->left))
                            goto cfail;
                        }

                    else if (twork->ntype == ARBITRARY) {
                        if (pos >= endsubj)
                            goto cfail;

                        pos++;
                        }

                    else if (twork->ntype == CCLASS) {
                        if (!inrange(*pos++, twork->left))
                            goto cfail;
                        }

                    else if (twork->ntype == VARIABLE) {
                        c = variable[int(twork->left) - 'a'];
                        if (c != NULL)
                            for (c = variable[int(twork->left) - 'a']; *c; )
                                if (*pos++ != *c++)
					goto cfail;
                        }

                    else { /* twork->ntype == POSITION */
                        if (!posscan(twork->left, &pos, endsubj, st_start))
				goto cfail;
			}

                    twork = twork->right;
                    }

                pflag = SUCCESS;
                continue;

	label	cfail:

                pflag = FAILURE;
                continue;

            default:
		if	(DEBUG)
	                error("in scanning");
                goto err;
            }
        }
label	err:
    sr_free_re(Xstack);
    return(NULL);
    }

decipher:	(pattern: [:] char) boolean =
	{
	d:	* boolean;
	n:	* int;
	i:	int;
	j:	int;
	s:	ref char;
	ends:	ref char;

	s = pattern;
	ends = s + |pattern;
	nextch = n = alloc(sizeof int * (|pattern + 1));
	if (n == NULL)
		return(FALSE);

	Flags = d = alloc(sizeof boolean * (|pattern + 1));
	if (d == NULL) {
		free(n);
		return(FALSE);
		}

	if	(DEBUG)
		printf("Memory allocated\n");
	while	(s < ends) {
		if (*s != ESCCHAR) {
			*n++ = *s;
			*d++ = metaflag & special(*s++);
			continue;
			}
		s++;

			// treat a trailing escape char as itself

		if	(s >= ends){
			*n++ = ESCCHAR;
			*d++ = FALSE;
			break;
			}
		switch (*s) {
			case 't':       *n++ = '\t';	goto gotit;
			case 'b':       *n++ = '\b';	goto gotit;
			case 'r':       *n++ = '\r';	goto gotit;
			case 'n':       *n++ = '\n';	goto gotit;
			case ESCCHAR:   *n++ = ESCCHAR;
label	gotit:
				s++;
				*d++ = FALSE;
				break;

			case '0':
			case '1':
			case '2':
			case '3':
				i = 3;
				goto octal;

			case '4':
			case '5':
			case '6':
			case '7':
				i = 2;
label	octal:
				j = *s++ - '0';
				while	(s < ends &&
					 *s >= '0' && *s <= '7' && i--)
					j = j * 8 + *s++ - '0';

				*n++ = j;
				*d++ = FALSE;
				break;

			case NULL:
				break;

			default:
				*n++ = *s;
				*d++ = !metaflag & special(*s++);
				break;
			}
		}

	*n = NULL;
	*d = FALSE;
	return(TRUE);
	}

gettoken:	(xstack: * pstack, last: int) int =
	{
	cc:	* char;
	OnOff:	boolean;
	ind:	int;
	c:	int;
	n:	int;
	m:	int;
	k:	int;
	ccval:	* pnode;
	save:	* pnode;

	ind         = Index;
	xstack->term = TRUE;
	c           = nextch[ind];
	if	(DEBUG)
		printf("gettoken index = %d c = %x\n", ind, c);

	/* If any of these were the last token returned, 		*/
	/* don't put it in CAT operator.				*/
	if (last == END  ||
	    last == OPEN ||
	    last == ALT  ||
	    last == CAT  ||
	    last == ASSIGN)
		goto ok;

	/* If any of these are next, don't put in CAT operator 		*/
	if	(c == EOS)
		return(END);

	if (Flags[ind] &&
	   (c == '*'   ||
	    c == '+'   ||
	    c == '{'   ||
	    c == '|'   ||
	    c == '$'   ||
	    c == ')'))
		goto ok;

	/* Put in CAT operator */
	return(CAT);

label	ok:

	if (!Flags[ind]) {
		xstack->val = ref pnode(c);
		Index++;
		return(CHARS);
		}

	Index = ++ind;
	switch (c) {
		case '*':
			n = -1;
			m = 0;
			goto REPS;

		case '+':
			n = -1;
			m = 1;
			goto REPS;

		case '{':
			n = getint(&ind);
			if (nextch[ind] == ',') {
				ind++;
				m = getint(&ind);
				}
			else
				m = n;

			if (nextch[ind++] != '}')
				return(-1);

			Index = ind;

label	REPS:
			if (n < 0 &&
			    m < 0)
				return(-1);

			xstack->val = newPnode();
			if (xstack->val == NULL)
				return(-1);

			/* lower limit must be on left, upper limit on right */
			if (n < 0  ||
			   (m >= 0 &&
			    m < n)) {
				xstack->val->left  = ref pnode(m);
				xstack->val->right = ref pnode(n);
				return(REP1);
				}

			xstack->val->left  = ref pnode(n);
			xstack->val->right = ref pnode(m);
			return(REP2);

		case '|':
			return(ALT);

		case '$':
			return(ASSIGN);

		case '<':
			/* The code here is a little complicated since the */
			/* syntax is a bit involved.  If this is a variable*/
			/* reference, the allowable characters after the < */
			/* are a-z or A-Z, so life is easy for that case.  */
			/* Otherwise, you expect a comma-separated list of */
			/* possible positions or range of positions.  If   */
			/* the first character of a position is ~ then the */
			/* position is to be counted from the end of the   */
			/* string					   */
			n = nextch[ind++];
			if ('A' <= n &&
			     n  <= 'Z')
				n += 'a' - 'A';

			/* handle variable reference */
			if ('a' <= n &&
			     n  <= 'z') {
				if (nextch[ind++] != '>')
					return(-1);

				Index = ind;
				xstack->val = ref pnode(n);
				return(VARIABLE);
				}

			/* now, for ranges of positions */
			/* ccval points to the next node to be filled */
			ccval = newPnode();
			if (ccval == NULL)
				return(-1);

			save = ccval;
			/* restore ind before going into loop */
			ind = Index;
			while (TRUE) {
				ccval->right = NULL;
				if (!getpos(&ind, &n))
					goto poserr;

				k = nextch[ind++];
				if (k == '-') {
					if (!getpos(&ind, &m))
						goto poserr;

					k = nextch[ind++];
					}

				else
					m = n;

				ccval->left  = ref pnode(n);
				ccval->alter = m;
				if (k == '>')
					break;

				if (k != ',')
					goto poserr;

				ccval->right = newPnode();
				ccval = ccval->right;
				}

			Index = ind;
			xstack->val = save;
			return(POSITION);

label	poserr:
			freerange(save);
			return(-1);

		case '[':
			/* The code here is complicated.  Part of the problem */
			/* is that the rules for forming a character class are*/
			/* complicated.  The things to remember are:          */
			/*	1)  a circumflex ^ first indicates that that  */
			/*	    character class is negated.		      */
			/*	2)  a close bracket ] first (after a possible */
			/*          leading ^) is treated as a regular	      */
			/*          character.  Otherwise it ends the         */
			/*          character class.			      */
			/*	3)  a minus - either first (after a possible  */
			/*          leading ^) or last (before the closing ]) */
			/*	    is treated as a regular character.        */
			/*	    Also, a minus after a minus is treated as */
			/*          a regular character.		      */
			/*          Otherwise it indicates a range of         */
			/*          characters.  Normally the first           */
			/*          character will be less than the second.   */
			/*				                      */
			/* I form a Boolean array 128 long (one for each      */
			/* character).  I then go through the character class */
			/* specification and at the end of the pass through,  */
			/* the array will have ones corresponding to          */
			/* included characters and otherwise zeros.  Then I   */
			/* generate nodes for the individual ranges           */
			/* represented in the array.			      */
			cc = alloc(128);
			if (cc == NULL)
				return(-1);

			if (nextch[ind] == '^') {
				for (n = 1; n < 128; n++)
					cc[n] = TRUE;

				ind++;
				OnOff = FALSE;
				}
			else {
				for (n = 1; n < 128; n++)
					cc[n] = FALSE;

				OnOff = TRUE;
				}

			while (TRUE) {
				n = nextch[ind++];

label	gotn:
				if (n == EOS) {
					free(cc);
					return(-1);
					}

				cc[n] = OnOff;

				/* look ahead one character */
				m = nextch[ind++];
				if (m == ']')
					break;

				if (m == '-') {
					/* look ahead another character */
					m = nextch[ind++];
					if (m != ']') {
						/* minus sign indicates range */
						/* of characters              */
						while (n < m)
							cc[++n] = OnOff;

						while (n > m)
							cc[--n] = OnOff;

						/* take care of ] right after */
						/* range of characters        */
						if (nextch[ind++] == ']')
							break;

						ind--;
						}
					else {
						/* minus sign was last before ] */
						cc['-'] = OnOff;
						break;
						}
					}
				else {
					/* already looked ahead so have     */
					/* next n value                     */
					n = m;
					goto gotn;
					}
				}

			Index = ind;
			ccval = newPnode();
			/* ccval constantly points to the next node to be  */
			/* filled.  At the end, ccval must be freed since  */
			/* it is a useless node. 			   */
			ccval->right = NULL;
			save = ccval;
			for (n = 1; n < 128; ) {
				while (n < 128 && !cc[n])
					n++;	  /* skip over FALSE      */

				if (n == 128)
					break;

				for (m = n + 1; m < 128 && cc[m]; m++)
					;

				ccval->right = newPnode();
				ccval->right->parent = ccval;
				ccval->left  = ref pnode(n);
				ccval->alter = m-1;
				ccval = ccval->right;
				ccval->right = NULL;
				n = m;
				}

			if (save->right == NULL)
				save = NULL;

			else
				ccval->parent->right = NULL;

			freepnode(ccval);
			xstack->val = save;
			free(cc);
			return(CCLASS);

		case '(':
			return(OPEN);

		case ')':
			return(CLOSE);

		case '.':
			return(ARBITRARY);

		default:
			return(-1);
		}
	}

/*
 * getint is similar to atoi but is special purpose and updates indices
 */
getint:	(indpt: * int) int =
	{
	i:	int;

	if (nextch[*indpt] < '0' ||
	    nextch[*indpt] > '9')
		return(-1);

	i = 0;
	while (nextch[*indpt] >= '0' &&
	       nextch[*indpt] <= '9')
		i = 10 * i + nextch[(*indpt)++] - '0';

	return(i);
	}

/*      getpos gets a position entry.  If ~ is first, it is from the end.
 *	0 value is beginning of string and working forward.
 *	-1 value is end of string working backward.
 */
getpos:	(indpt: * int, valpt: * int) boolean =
	{
	flag:	int = FALSE;

	if (nextch[*indpt] == '~') {
		(*indpt)++;
		flag = TRUE;
		}

	*valpt = getint(indpt);
	if (*valpt < 0)
		return(FALSE);

	if (flag)
		*valpt = -1 - *valpt;		 /* from the end */

	return(TRUE);
	}

/* normal push */
push:	(t: * pnode, pos: * char) boolean =
	{
	new_:	* pnode;

	new_ = newPnode();
	if (new_ == NULL)
		return(1);

	new_->parent = Xstack;
	new_->left   = t;
	new_->right  = ref pnode(pos);
	new_->alter  = t->alter;
	Xstack       = new_;
	return(0);
	}

/* push RESTORE */
pusher:	(t: * pnode) boolean =
	{
	new_:	* pnode;

	new_ = newPnode();
	if (new_ == NULL)
		return(1);

	new_->parent = Xstack;
	new_->left   = &Restore;
	new_->right  = t;
	new_->alter  = t->alter;
	Xstack       = new_;
	return(0);
	}

pop:	(t: ** pnode) * char =
	{
	old:	* pnode;
	c:	* char;

	old = Xstack;
	if (old == NULL) {
		*t = NULL;
		return(NULL);
		}

	*t          = old->left;
	(*t)->alter = old->alter;
	c           = ref char(old->right);
	Xstack       = old->parent;
	freepnode(old);
	return(c);
	}

	compiledPattern:	* pnode;
	metaflag:		boolean;

/* This is the pointer to the top of the scanning stack saved by generators */

	Xstack:		* pnode;
	nextch:		* int;
	Flags:		* boolean;
	Index:		int;

	};

/*		THESE ARE THE SEARCH AND REPLACE ROUTINES.		*/
/*Copyright 1989 by English Knowledge Systems, Inc. All Rights Reserved.*/


/* LIST OF TOKENS							*/

tokens:	type	char = {
	END,
	ALT,			/* alternation 				*/
	CAT,			/* concatenation 			*/
	ASSIGN,			/* assignment 				*/
	REP1,			/* repetition count type 1 		*/
	REP2,			/* repetition count type 2 		*/
	CHARS,			/* characters 				*/
	CCLASS,			/* character class 			*/
	ARBITRARY,		/* arbitrary 				*/
	POSITION,		/* position from beginning of string 	*/
	VARIABLE,		/* variable 				*/
	OPEN,			/* ( 					*/
	CLOSE,			/* ) 					*/

/*
 *  The above token values are used as indices into the precedence functions
 *  to determine when reductions should occur.  They are also used during
 *  scanning as the node type.
 *
 *  The following are not returned by gettoken(), but are node types used
 *  during scanning.
 */
	CHARS1,			/* CHARS repetition count type 1 	*/
	CHARS2,			/* CHARS repetition count type 2 	*/
	CCLASS1,		/* CCLASS repetition count type 1 	*/
	CCLASS2,		/* CCLASS repetition count type 2 	*/
	ARB1,			/* ARB repetition count type 1 		*/
	ARB2,			/* ARB repetition count type 2 		*/
	RESTORE,		/* special value for FAILURE in scanning*/
	DONE,			/* special value for finished scanning 	*/
	};

/* This is the structure of a general parse tree entry			*/
pnode:	type	{
	public:

	ntype:	tokens;		/* type field 				*/
	left:	* pnode;	/* left operand pointer			*/
	right:	* pnode;	/* right operand pointer 		*/
	parent:	* pnode;	/* parent pointer 			*/
	alter:	int;		/* alternative count 			*/
	asgpos:	* char;
	} ;

/*
 * A DETAILED DESCRIPTION OF PARSE TREE STRUCTURE
 *
 *   The normal use of the fields in a parse tree node is as described above.
 *   Exceptions to this use are described below.
 *
 *   ALT and CAT:
 *       For these two types of nodes, the left and right branches are actually
 *       of type (pnode *), and point to sub-parse trees.  alter is used
 *       in scanning as a flag to indicate which branch is being executed.
 *
 *   ASSIGN:
 *       The left branch is a (pnode *) pointing to a sub-tree.  The
 *       right branch is actually a (char) with value a-z.  The alter flag
 *       is used as (char *) pointing to beginning of the assignment location.
 *
 *   REP1 and REP2:
 *       The left branch is a (pnode *) pointing to a sub-tree.  The
 *       right branch is a (pnode *) pointing to a single node which
 *       has the replication information.  In that node, the left branch
 *       is an (int) specifying the lower bound on replication; the right
 *	 branch is an (int) specifying the upper bound.  Also in the sub-node,
 *	 alter is a (char *) pointing to the place in the string being scanned
 *	 where the current try at replication began.  It is included in order
 *	 to prevent the replication from matching the null string an infinite
 *	 number of times.  The parent and type fields are unused in the
 *	 sub-node.  In the REP1 or REP2 node, the alter field is used as a
 *	 count on number of replications.
 *
 *   CHARS1 and CHARS2:
 *       The left branch is a (char) specifying which character to match.
 *       The right branch and the alter flag are as in REP1 and REP2, except
 *       that the alter flag is unused in the sub-node.
 *
 *   CCLASS1 and CCLASS2:
 *       The left branch is a (pnode *) pointing to a description
 *       of the character class (see CCLASS).  The right branch and alter
 *       flags are as in CHARS1 and CHARS2.
 *
 *   ARB1 and ARB2:
 *       The left branch is unused, and the right branch and alter flags
 *       are as in CHARS1 and CHARS2.
 *
 *   CHARS:
 *       The left branch is a (char) specifying which character to match.
 *       The right branch is a (pnode *) pointer to additional
 *       sequences of CHARS, CCLASS, ARBITRARY, POSITION, and VARIABLE
 *       nodes (since all of these can be processed linearly with no need
 *       to backtrack).  The alter flag is unused.

 *   CCLASS:
 *       The left branch is a (pnode *) pointer to a chain of
 *       character class ranges.  In each node in the chain, the left and
 *       alter fields are the (char) lower and upper bounds on characters
 *       accepted, and the right branch is a (pnode *) to more
 *       links in the chain, with the parent field unused.  The chain of
 *       ranges is in ascending order, to shorten time needed to look.
 *       The right branch and alter flag are as in CHARS:
 *
 *   ARBITRARY:
 *       The left branch and alter flag are unused.  The right branch is
 *       as in CHARS.
 *
 *   POSITION:
 *       The left branch is a (pnode *) pointer to a chain of
 *       position ranges.  In each node in the chain, the left and
 *       alter fields are the (int) bounds on positions.  It is impossible
 *       to tell a priori which is the lower and which is the upper bound
 *       since it may depend on the length of the string being scanned.
 *       In the chained nodes, the right branch is a (pnode *)
 *       pointer to the rest of the chain.  In the POSITION node, the
 *       right branch and alter flag are as in CHARS.
 *
 *   VARIABLE:
 *       The left branch is a (char) from a-z indicating which variable to
 *       expand.  The alter flag is unused, and the right branch is as in
 *       CHARS.
 *
 *
 *
 *  USE OF NODES IN SCANNING:
 *
 *   In scanning, use is made of a linked list stack of free parse tree nodes
 *   in order to do backtracking.  Normally, the parent pointer points back
 *   to the previous entry on the stack.  The left branch is a (pnode *)
 *   pointer to the tree position to which to return.  The right branch is a
 *   (char *) pointing to the place in the string being scanned at which to
 *   continue the scanning process.  The alter flag is the saved alter flag
 *   of the node in the tree to which the return is to be made.
 *
 *   The exception to this is when the type of the tree position is the
 *   special value RESTORE (which does not occur in the tree).  In that case,
 *   the right branch is a (pnode *) pointer to a node in the tree
 *   and the alter flag is to be restored to that node.  This gives a
 *   mechanism to distinguish between just resetting the alter flag value
 *   (which is necessary sometimes, even when the node is not a generator)
 *   and a FAILURE return to a node which will try another choice.
 */

/* MACROS FOR EASY REFERENCE FOR REPLICATIONS */
LOWER:	(t: * pnode) int = 
	{
	return(int(t->right->left));
	}

UPPER:	(t: * pnode) int = 
	{
	return(int(t->right->right));
	}

BEGIN:	(t: * pnode) int = 
	{
	return(int(t->right->alter));
	}

FAILURE:	const int = 0;
SUCCESS:	const int = 1;
FIRST:		const int = 2;
EOS:		const char = '\0';
ESCCHAR:	const char = '\\';
TYPES:		const int = 19;
GETNODE:	const int = 20;

DEBUG:		const int = 0;
kntnode:	int = 0;
NULL:		const int = 0;

Restore:	pnode = [    RESTORE,    NULL,   NULL,   NULL,   0 ] ;

Done:		pnode = [    DONE,       NULL,   NULL,   NULL,   0 ] ;



/* This is the structure of a parse stack entry.			*/
pstack:	type	{
	public:

	ntype:	char; 		/* type of stack entry (token type) 	*/
	term:	boolean;  	/* terminal or not 			*/
	val:	* pnode;   	/* pointer to value 			*/
    	} ;


/*
 *	This frees the memory allocated by sr_compile()
 */
sr_free_re:	(tree: * pnode) =
	{
	t:	* pnode;

	/* Check for NULL value.					*/
	if (tree == NULL)
		return;

	if (tree->ntype == DONE) {	   		/* free a stack */
		while (tree != NULL) {
			t = tree->parent;
			freepnode(tree);
			tree = t;
			}
		return;
		}

	/* free a tree	 						*/
	if	(DEBUG){
		if	(tree->ntype < 0 ||
			 tree->ntype > TYPES)
			printf("unrecognized type: %d", tree->ntype);
		}

	freepnode(tree);
	switch (tree->ntype) {
		case ALT:
		case CAT:
			sr_free_re(tree->left);
			sr_free_re(tree->right);
			break;

		case REP1:
		case REP2:
			sr_free_re(tree->left);
			freepnode(tree->right);
			break;

		case CHARS1:
		case CHARS2:
		case ARB1:
		case ARB2:
			freepnode(tree->right);
			break;

		case CCLASS1:
		case CCLASS2:
			freepnode(tree->right);
			freerange(tree->left);
			break;

		case CHARS:
		case ARBITRARY:
		case CCLASS:
		case POSITION:
		case VARIABLE:
			while (TRUE) {
				if (tree->ntype == CCLASS ||
				    tree->ntype == POSITION)
					freerange(tree->left);

				tree = tree->right;
				if (tree == NULL)
					break;

				freepnode(tree);
				}
			break;
		}
	}

/*
 *   The parsing scheme uses an operator precedence grammar.
 *
 *   In the following comments the productions, precedence matrix, and
 *   precedence functions are listed.
 *
 *   The scanning scheme is a backtracking one which remembers where
 *   earlier alternatives were chosen. If the match should fail,
 *   the algorithm can retreat back to the last place where a different
 *   alternative could have been selected and try again.
 *
 *   For a description of how backtracking works, the SNOBOL and ICON
 *   programming languages are excellent instructors in the use of
 *   backtracking, particularly in string analysis.
 *
 *   LIST OF PRODUCTIONS
 *
 *	Production               Number             Schematic
 *
 *	OR      OR | AND            0                 N | N
 *	        AND
 *	AND     AND & UNARY         1                 N & N
 *	        UNARY
 *	UNARY   UNARY *      --
 *	        UNARY +        |--  2                 N REP-COUNT-1
 *	        UNARY {n,}     |--  3                 N REP-COUNT-2
 *	        UNARY {n}      |                      (whether 2 or 3 depends
 *	        UNARY {,m}     |                      on n compared to m)
 *	        UNARY {n,m}  --
 *	        UNARY $ CHARS       4                 N $ CHARS
 *	        TERM
 *	TERM    ( OR )              5                 ( N )
 *	        CHARS               6                 CHARS
 *	        CHAR-CLASS          7                 CHAR-CLASS
 *	        ARBITRARY           8                 ARBITRARY
 *	        POSITION            9                 POSITION
 *	        VARIABLE           10                 VARIABLE
 *
 *	Precedence Matrix
 *	    R1 = REP-COUNT-1
 *	    R2 = REP-COUNT-2
 *	    C  = CHARS
 *	    CC = CHAR-CLASS
 *	    A  = ARBITRARY
 *	    P  = POSITION
 *	    V  = VARIABLE
 *
 *	         2  4  6  6  6  6  6  6  6  6  6  1
 *	         |  &  $  R1 R2 C  CC A  P  V  (  )
 *	3   |    >  <  <  <  <  <  <  <  <  <  <  >
 *	5   &    >  >  <  <  <  <  <  <  <  <  <  >
 *	6   $                   =
 *	7   R1   >  >  >  >  >                    >
 *	7   R2   >  >  >  >  >                    >
 *	7   C    >  >  >  >  >                    >
 *	7   CC   >  >  >  >  >                    >
 *	7   A    >  >  >  >  >                    >
 *	7   P    >  >  >  >  >                    >
 *	7   V    >  >  >  >  >                    >
 *	1   (    <  <  <  <  <  <  <  <  <  <  <  =
 *	7   )    >  >  >  >  >                    >
 *
 */

/* PRECEDENCE FUNCTIONS 					   */
/*                  END |  &  $  R1 R2 C  CC A  P  V  (  )         */
ff:	[] char = [ 0, 3, 5, 6, 7, 7, 7, 7, 7, 7, 7, 1, 7 ] ;
gg:	[] char = [ 0, 2, 4, 6, 6, 6, 6, 6, 6, 6, 6, 6, 1 ] ;

STACK:		const int = 20;              /* size of stack */
NPROD:		const int = 11;

prod:	[NPROD] {
	public:

	terms:	* char;
	types:	* char;
    	} = [
        [ N00,    M00 ],            /* N | N => N */
        [ N01,    M01 ],            /* N & N => N */
        [ N02,    M02 ],            /* N R1  => N */
        [ N03,    M03 ],            /* N R2  => N */
        [ N04,    M04 ],            /* N $ C => N */
        [ N05,    M05 ],            /* ( N ) => N */
        [ N06,    M06 ],            /* C     => N */
        [ N07,    M07 ],            /* CC    => N */
        [ N08,    M08 ],            /* A     => N */
        [ N09,    M09 ],            /* P     => N */
        [ N10,    M10 ]             /* V     => N */
        ] ;

/* ENCODING OF PRODUCTIONS */

/* Productions are reversed because the stack works from the most	*/
/* recently pushed at the top.  Comparisons are done from the top of 	*/
/* the stack down in order to get a reduction.			       	*/
/*									*/
/* In the N arrays, 0 signifies a non-terminal position.		*/
/*                  1 signifies a terminal position.			*/
/* In the M arrays, 0 fills in non-terminal positions.			*/
/*                  the terminal needed for the particular production	*/
/*		    is in the terminal position.			*/
/* The last entry in both the N and M arrays is a dummy value that	*/
/* facilitates searching for which reduction to apply.			*/
N00:	[] char = [0, 1, 0, 1 ];
M00:	[] char = [0, ALT, 0, 0 ] ;
N01:	[] char = [0, 1, 0, 1 ];
M01:	[] char = [0, CAT, 0, 0 ] ;
N02:	[] char = [1, 0, 1 ];
M02:	[] char = [REP1, 0, 0 ] ;
N03:	[] char = [1, 0, 1 ];
M03:	[] char = [REP2, 0, 0 ] ;
N04:	[] char = [1, 1, 0, 1 ];
M04:	[] char = [CHARS, ASSIGN, 0, 0 ] ;
N05:	[] char = [1, 0, 1, 1 ];
M05:	[] char = [CLOSE, 0, OPEN, 0 ];
N06:	[] char = [1, 1 ];
M06:	[] char = [CHARS, 0 ];
N07:	[] char = [1, 1 ];
M07:	[] char = [CCLASS, 0 ] ;
N08:	[] char = [1, 1 ];
M08:	[] char = [ARBITRARY, 0 ] ;
N09:	[] char = [1, 1 ];
M09:	[] char = [POSITION, 0 ] ;
N10:	[] char = [1, 1 ];
M10:	[] char = [VARIABLE, 0 ] ;

pfree:	* pnode = NULL;      /* beginning of list of free nodes 	*/

/*
 * This routine searches the productions tables for a production to reduce.
 */
/*
findProduction:	(xstack: * pstack, nstack: int) int =
	{
	i:	int;
	j:	int;
	k:	int;
	lastt:	int;
	nextt:	int;

	for (i = 0; i < NPROD; i++) {
                j = nstack - 1;
                k = 0;
                while (TRUE) {
			if (xstack[j].term != prod[i].terms[k])
				break;

			if	(xstack[j].term) {

			/* Here lint complains since nextt is assigned 	*/
			/* to lastt without having been set.  This is 	*/
			/* fine here, since you will come through this 	*/
			/* place at least twice before having the 	*/
			/* prod[i].types[k] == 0 test succeed (which is	*/
			/* the only place where lastt is used).		*/

	                        lastt = nextt;
				nextt = xstack[j].ntype;
	                        if (prod[i].types[k] == 0)
					if (ff[nextt] < gg[lastt])
						return(i);
					else
						break;

		                        if (nextt != prod[i].types[k])
					break;
	                        }

			j--;
			k++;
			}
		}
	return(-1);
	}
 */

newPnode:	() * pnode = {

	t:	* pnode;
	i:	int;

	if (pfree == NULL) {
		if	(DEBUG){
			printf("getting nodes\n");
			kntnode += GETNODE;
			}
		pfree = alloc(GETNODE * sizeof(pnode));
		memSet(pfree, 0, GETNODE * sizeof(pnode));
		if (pfree == NULL)
			fatal("no memory in parsing");

		for (i = 0; i < GETNODE-1; i++)
			pfree[i].parent = &pfree[i+1];

		pfree[GETNODE-1].parent = NULL;
		}

	t         = pfree;
	pfree     = pfree->parent;
	t->parent = NULL;
	if	(DEBUG)
		kntnode--;
	return(t);
	}

/*
 * 	Note that freed nodes are not actually returned to the memory cache.
 *	Note also that the values are not affected except for the parent
 *	pointer so that in particular, the left and right values may be used.
 */
freepnode:	(p: * pnode) =
	{
/*      WARNING!!!!  Havoc will result if you free the same 		 */
/*	node twice.  A simple check helps prevent this.			 */

	t:	* pnode;

	for (t = pfree; t != NULL; t = t->parent)
		if (t == p)
			return;

	p->parent = pfree;
	pfree     = p;
	if	(DEBUG){
		i:	int;
		t:	* pnode;

		kntnode++;
		t = pfree;
		for (i = 0; i < kntnode; i++)
			t = t->parent;

		if (t != NULL)
			fatal("the free list is corrupted");
		}
	}

/* freerange(tree) frees ranges for CCLASS and POSITION nodes */
freerange:	(tree: * pnode) =
	{
	while (tree) {
		freepnode(tree);
		tree = tree->right;
		}

	}

special:	(c: int) boolean =
	{
	sp:	* char;

	for (sp = ".[*()|$+<{"; *sp; sp++)
		if (c == *sp)
			return(TRUE);

	return(FALSE);
	}


/* tests ranges for character classes */
inrange:	(c: int, t: * pnode) boolean =
	{
	/* this requires ranges being in numerical order	 	*/

	while (t) {
		if (c < int(t->left))
			return(FALSE);	  		/* lower limit	*/

		if (c <= t->alter)
			return(TRUE);		   	/* upper limit	*/

		t = t->right;
		}

	return(FALSE);
	}

posscan:	(t: * pnode, posadr: ** char, endsubj: * char,
				subject: * char) boolean =
	{
	pos:	* char;
	pos = *posadr;
	c:	* char;
	slen:	int;
	bplace:	int;
	eplace:	int;
	first:	int;
	second:	int;
	i:	int;

	i = 0;
	eplace = -1;
	for (c = subject; c < endsubj; c++) {
		if (c == pos)
			bplace = i;

		if (*c == '\t')
			i = (i + 8) & ~7;		 /* tab stop */
		else
			i++;

		if (c == pos)
			eplace = i - 1;
		}

	slen = i;
	if (eplace < 0) {
		/* in case pos was out of range */
		bplace = i;
		eplace = i;
		}

	/* At this point, slen has how long the string is, bplace has 	*/
	/* the character position from which to begin, and eplace has 	*/
	/* the character position from which to end.			*/
	while (t) {
		first  = int(t->left);
		second = t->alter;
		if (first < 0)
			first = slen + 1 + first;

		if (second < 0)
			second = slen + 1 + second;

		if (first < second) {
			if (first  <= eplace &&
			    bplace <= second) {
				*posadr += (second - first);
				return(TRUE);
				}
			}
		else {
			if (second <= eplace &&
			    bplace <= first)  {
				*posadr += (first - second);
				return(TRUE);
				}
			}

		t = t->right;
		}

	return(FALSE);
	}

/*
 *	This routine is called in case of fatal error,
 *	including out of memory.
 */
fatal:	(s: * char) =
	{
	printf("%s\n", s);
	exit(1);
	}

/*
/* the following is the debugging code.					*/
indexx:	[TYPES] * char = [
	"END",
	"ALT",
	"CAT",
	"ASSIGN",
	"REP1",
	"REP2",
	"CHARS",
	"CCLASS",
	"ARBITRARY",
	"POSITION",
	"VAR",
	"OPEN",
	"CLOSE",
	"CHARS1",
	"CHARS2",
	"CCLASS1",
	"CCLASS2",
	"ARB1",
	"ARB2"
	] ;
 */
/* the following is parse stack tracing for debugging purposes      */
/*
stackp:	(xstack: * pstack, nstack: int) =
	{
	i:	int;

	for (i = 0; i <= nstack; i++) {
		if (xstack[i].term) {
			if (xstack[i].ntype >= 0 &&
			    xstack[i].ntype <= TYPES)
				printf("%s\n", indexx[xstack[i].ntype]);
			else
				printf("undefined terminal %d\n", xstack[i].ntype);
			}

		else
			printf("non-terminal\n");
		}

	printf("\n");
	}
 */

/*
 *	This routine will print out the parse tree.
 *	For debugging call this routine with p = 0, indent = 0,
 *	and the regular expression returned from sr_compile()
 *	in tree.
 */
/*
treep:	public	(tree: * pnode, p: * pnode, indent: int) =
	{
	i:	int;

	for (i = 0; i < indent; i++)
		printf(" ");

	if (tree->ntype >= 0 &&
	    tree->ntype <= TYPES)
		printf("%s", indexx[tree->ntype]);
	else
		printf("unrecognized type: %d", tree->ntype);

	if (tree->parent != p)
		printf("  parent mismatch");

	switch (tree->ntype) {
		case ALT: case CAT:
			printf("\n");
			treep(tree->left, tree, indent+4);
			treep(tree->right, tree, indent+4);
			break;

		case ASSIGN:
			printf("   %c\n", int(tree->right));
			treep(tree->left, tree, indent+4);
			break;

		case CHARS:
		case ARBITRARY:
		case CCLASS:
		case POSITION:
		case VARIABLE:
			while (TRUE) {
				if (tree->ntype == CHARS) {
					printf("   ");
					while (tree->ntype == CHARS) {
						printf("%c", int(tree->left));
						tree = tree->right;
						}

					if (tree == NULL)
						break;

					printf("\n%*s%s", indent, "",
						          indexx[tree->ntype]);
					continue;
					}

				printf("   ");
				if (tree->ntype == CCLASS)
					rangep(tree->left);

				else if (tree->ntype == ARBITRARY)
					printf(" ARB ");

				else if (tree->ntype == POSITION)
					for (p = tree->left; TRUE; ) {
						printf(" %d %d", int(p->left), 
								p->alter);
						p = p->right;
						if (p == NULL)
							break;

						printf(" OR ");
						}

				else /* tree->ntype == VARIABLE */
					printf("   %c", int(tree->left));

				tree = tree->right;
				if (tree == NULL)
					break;

				printf("\n%*s%s", indent, "",
						  indexx[tree->ntype]);
				}

			printf("\n");
			break;

		case CHARS1:
		case CHARS2:
			printf("	%c  %d  %d\n",
				int(tree->left), LOWER(tree), UPPER(tree));
			break;

		case CCLASS1:
		case CCLASS2:
			rangep(tree->left);
			printf("	%d  %d\n", LOWER(tree), UPPER(tree));
			break;

		case ARB1:
		case ARB2:
			printf("	%d  %d\n", LOWER(tree), UPPER(tree));
			break;

		case REP1:
		case REP2:
			printf("	%d  %d\n", LOWER(tree), UPPER(tree));
			treep(tree->left, tree, indent+4);
			break;

		default:
			printf("\n");
			break;
		}
	}

rangep:	(t: * pnode) =
	{
	while (t) {
		printf(" ");
		pp(int(t->left));
		printf("-");
		pp(t->alter);
		t = t->right;
		}

	return;
	}

pp:	(i: int) =
	{
	if	(i < 0)
		printf("NEGATIVE");

	else if (i < 040)
		printf("\\%03o", i);

	else if (i < 0177)
		printf("%c", i);

	else
		printf("DEL");

	return;
	}
 */
error:	(s: * char) =
	{
	printf("%s\n", s);
	}
