/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	console;
include	format;
include	alys;
include	error;
include	string;
include	filesys;
include	file;

Screen:	public	inherit	frame	{
	cons:	public	ref far console;
	};

startup:	entry	() =
	{
	if	(stdin getClass() == CC_CONSOLE){
		c:	ref far channel;

		c = stdin getChannel();
		Screen.cons = ref far consoleChannel(c) appearsOn();
		}
	}

clearMessage:	public () =
	{
	if	(ActiveViewer)
		ActiveViewer clearMessage();
	}

postAlarm:	public	(errno: int) =
	{
	postMessage(cmdError(errno));
	}

postMessage:	public	(fmt: [:] char, ...) =
	{
	if	(ActiveViewer)
		ActiveViewer postMessageV(fmt, ...);
	}

frameEmitter:	public	type	inherit	emitter	{
	public:

	fr:	* frame;
	p:	point;

putc:	dynamic	(c: char) =
	{
	fr putc(p, c);
	p.x++;
	}

	};

frame:	public	type	{
	public:

	corner:		point;
	size:		point;
	ground:		ref frame;
/*
enter:	dynamic	() * frame =
	{
	return 0;
	}
 */
textCursor:	dynamic	(p: point) =
	{
	p.x += corner.x;
	p.y += corner.y;
	ground textCursor(p);
	}

setColor:	dynamic	(c: color_t) color_t =
	{
	return ground setColor(c);
	}

showCursor:	dynamic	() =
	{
	ground showCursor();
	}

hideCursor:	dynamic	() =
	{
	ground hideCursor();
	}

getKey:	dynamic	() keystroke =
	{
	return ground getKey();
	}

getRawKey:	dynamic	() keystroke =
	{
	return ground getRawKey();
	}

testKey:	dynamic	() keystroke =
	{
	return ground testKey();
	}

printf:	dynamic	(p: point, fmt: [:] char, ...) =
	{
	fe:	frameEmitter;

	fe = [ self ];
	fe.p = p;
	fe format(fmt, ...);
/*
	while	(fe.p.x < size.x){
		putc(fe.p, ' ');
		fe.p.x++;
		}
 */
	}

vprintf:	dynamic	(p: point, fmt: [:] char, ap: varArgs) =
	{
	fe:	frameEmitter;

	fe = [ self ];
	fe.p = p;
	fe format(fmt, ap);
/*
	while	(fe.p.x < size.x){
		putc(fe.p, ' ');
		fe.p.x++;
		}
 */
	}

putc:	dynamic	(p: point, c: char) =
	{
	if	(p.y >= size.y)
		return;
	if	(p.x >= size.x)
		return;
	ground putc([p.x + corner.x, p.y + corner.y], c);
	}

putcc:	dynamic	(p: point, c: char, a: color_t) =
	{
	if	(p.y >= size.y)
		return;
	if	(p.x >= size.x)
		return;
	ground putcc([p.x + corner.x, p.y + corner.y], c, a);
	}

write:	dynamic	(p: point, text: [:] colorChar) =
	{
	rem:	int;

	if	(p.y >= size.y)
		return;
	if	(p.x >= size.x)
		return;
	rem = size.x - p.x;
	if	(|text > rem)
		|text = rem;
	ground write([p.x + corner.x, p.y + corner.y], text);
	}

writeChar:	dynamic	(p: point, text: [:] char) =
	{
	rem:	int;

	if	(p.y >= size.y)
		return;
	if	(p.x >= size.x)
		return;
	if	(p.x + |text >= size.x)
		|text = size.x - p.x;
	ground writeChar([p.x + corner.x, p.y + corner.y], text);
	}

read:	dynamic	(p: point, buf: [:] colorChar) =
	{
	rem:	int;

	if	(p.y >= size.y ||
		 p.x >= size.x){
		|buf = 0;
		return;
		}
	rem = size.x - p.x;
	if	(||buf > rem)
		||buf = rem;
	ground read([p.x + corner.x, p.y + corner.y], buf);
	}

draw:	dynamic	(pointer) =
	{
	}

close:	dynamic	() int =
	{
	}

verticalScroll:	dynamic	(ul: point, sz: point, amount: int) =
	{
	if	(amount == 0)
		return;

		// If the scrolling frame in view?  If not, give up'

	if	(ul.x > size.x ||
		 ul.y > size.y ||
		 sz.x < 1 ||
		 sz.y < 1 ||
		 ul.x + sz.x < 1 ||
		 ul.y + sz.y < 1)
		return;

		// now clip the scrolling frame to the window

	if	(ul.x < 0)
		ul.x = 0;
	if	(ul.y < 0)
		ul.y = 0;
	if	(ul.x + sz.x > size.x)
		sz.x = size.x - ul.x;
	if	(ul.y + sz.y > size.y)
		sz.y = size.y - ul.y;
	ul.x += corner.x;
	ul.y += corner.y;
	ground verticalScroll(ul, sz, amount);
	}

horizontalScroll:	dynamic	(ul: point, lr: point, amount: int) =
	{
	if	(ul.x >= size.x)
		return;
	if	(ul.y >= size.y)
		return;
	if	(lr.x >= size.x)
		lr.x = size.x - 1;
	if	(lr.y >= size.y)
		lr.y = size.y - 1;
	ul.x += corner.x;
	ul.y += corner.y;
	lr.x += corner.x;
	lr.y += corner.y;
	ground horizontalScroll(ul, lr, amount);
	}

absoluteCoordinates:	dynamic	(p: point) point =
	{
	p.x += corner.x;
	p.y += corner.y;
	return ground absoluteCoordinates(p);
	}

putString:	(p: point, s: [:] char) =
	{
	writeChar(p, s);
	}

beep:	dynamic	() =
	{
	ground beep();
	}

	};




windowFlags:	public	type	char = {
	W_OS =		0x02,
	W_POPUP =	0x04,
	W_OPEN =	0x08,
	};

windowStatus:	type	byte = {
	WS_HIDDEN = 0x01,
	};

borderStyles:	public	type	byte = {
	WB_NONE,
	WB_SINGLE,
	WB_DOUBLE,
	WB_DBL_TOP,
	WB_DBL_SIDE,
	WB_PULL_DOWN
	};

borderDescriptor:	type	{
	public:

	uleft:		byte;
	uright:		byte;
	lright:		byte;
	lleft:		byte;
	side:		byte;
	top:		byte;
	};

Borders:	[] borderDescriptor = [
	[   0,   0,   0,   0,   0,   0 ],		// WB_NONE
	[ '', '', '', '', '', '' ],		// WB_SINGLE
	[ '', '', '', '', '', '' ],		// WB_DOUBLE
	[ '', '', '', '', '', '' ],		// WB_DBL_TOP
	[ '', '', '', '', '', '' ],		// WB_DBL_SIDE
	[ '', '', '', '', '', '' ]		// WB_PULL_DOWN
	];

newWindow:	public	(titl: [:] char, sz: point, corn: point) * windowFrame =
	{
	xp:	ref far textWindow;

	xp = Screen.cons newWindow(titl, sz);
	xp = ref far textWindow(makeAnyLocal(xp));
	return new windowFrame[ xp, sz, corn ];
	}

windowInfo:	public	type	packed	{
	public:

	corner:		point;
	size:		point;
	color:		color_t;
	flags:		windowFlags;
	status:		windowStatus;
	title:		[80] char;
	};

screenWindow:	public	() * windowFrame =
	{
	return new windowFrame[ Screen.cons screenWindow(), 
						[ 80, 25 ], [ 0, 0 ] ];
	}

windowFrame:	type	inherit	frame	{
	public:

	viewport:	ref far textWindow;

constructor:	(vp: ref far textWindow, sz: point, corn: point) =	
							// Kernel window object
	{
	viewport = vp;
	cursor = [ 0, 0 ];
	size = sz;
	corner = corn;
	viewport zoom(corn, WZ_NORMAL);
	}

	private:

	cursor:			point;
	color:			color_t;

	public:

textCursor:	dynamic	(p: point) =
	{
	viewport textCursor(p);
	cursor = p;
	}

verticalScroll:	dynamic	(ul: point, sz: point, amount: int) =
	{
	viewport verticalScroll(ul, sz, amount);
	}

absoluteCoordinates:	dynamic	(p: point) point =
	{
	p.x += corner.x;
	p.y += corner.y;
	return p;
	}

getCursor:	() point =
	{
	return cursor;
	}

hideCursor:	dynamic	() =
	{
	viewport hideCursor();
	}

showCursor:	dynamic	() =
	{
	viewport showCursor();
	}

read:	dynamic	(p: point, buf: [:] colorChar) =
	{
	buf [:]= viewport read(p);
	}

putcc:	dynamic	(p: point, c: char, a: color_t) =
	{
	viewport putcc(p, c, a);
	}

putc:	dynamic	(p: point, c: char) =
	{
	viewport putc(p, c);
	}

write:	dynamic	(p: point, buf: [:] colorChar) =
	{
	viewport write(p, buf);
	}

writeChar:	dynamic	(p: point, buf: [:] char) =
	{
	viewport writeChar(p, buf);
	}

clear:	() =
	{
	viewport clear();
	}

setColor:	dynamic	(a: color_t) color_t =
	{
	x:	color_t;

	x = color;
	color = a;
	viewport setColor(a);
	return x;
	}

hide:	dynamic	() int =
	{
	return 1;
	}

close:	dynamic	() int =
	{
	viewport close();
	viewport = ref far textWindow(-1);
	return 1;
	}

getKey:	dynamic	() keystroke =
	{
	return viewport getKey();
	}

getRawKey:	dynamic	() keystroke =
	{
	return viewport getRawKey();
	}

testKey:	dynamic	() keystroke =
	{
	return viewport testKey();
	}

beep:	dynamic	() =
	{
	viewport beep();
	}

};

window:	public	type	inherit	frame	{
	public:

constructor:	(	titl: [:] char,		// Window title
			sz: point) =		// Size of window
	{
	size = sz;
	outFrame.size = size;
	cursor = [ 0, 0 ];
	status = 0;
	color = WHITE;
	flags = 0;
	title = titl;
	border = WB_NONE;
	borderColor = WHITE;
	titleColor = WHITE;
	saved = 0;
	}

assumeScreen:	dynamic	() =
	{
	if	(flags & W_OPEN)
		return;
	flags |= W_OPEN|W_OS;
	ground = screenWindow();
	outFrame.corner = [ 0, 0 ];
	outFrame.ground = ground;
	if	(border != WB_NONE){
		corner = [ 1, 1 ];
		drawBorder();
		}
	else	{
		corner = [ 0, 0 ];
		}
	}

open:	dynamic	(grnd: * frame, p: point) int =
	{
	if	(flags & W_OPEN)
		return 0;
	flags |= W_OPEN;
	if	(grnd == &Screen){
		grnd = newWindow(title, outFrame.size, p);
		p = [ 0, 0 ];
		flags |= W_OS;
		}
	ground = grnd;
	outFrame.corner = p;
	outFrame.ground = grnd;
	if	(border != WB_NONE){
		corner = [ p.x + 1, p.y + 1 ];
		drawBorder();
		}
	else	{
		corner = p;
		}
	return 1;
	}

popup:	dynamic	(grnd: * frame, p: point) int =
	{
	i:	int;

	if	(flags & W_OPEN)
		return 0;
	flags |= W_OPEN|W_POPUP;
	if	(grnd == &Screen){
		grnd = newWindow(title, outFrame.size, p);
		p = [ 0, 0 ];
		flags |= W_OS;
		}
	ground = grnd;
	outFrame.corner = p;
	outFrame.ground = grnd;
	corner = p;
	i = outFrame.size.x * outFrame.size.y;
	saved = new [i] colorChar;
	if	(saved){
		p:	point;

		p = size;
		size = outFrame.size;
		readAll(saved);
		size = p;
		}
	if	(border != WB_NONE){
		corner = [ p.x + 1, p.y + 1 ];
		drawBorder();
		}
	return 1;
	}

	private:

readAll:	(cp: ref colorChar) =
	{
	p:	point;
	i:	int;

	p = [ 0, 0 ];
	for	(i = 0; i < size.y; i++, p.y++){
		read(p, cp[:size.x]);
		cp += size.x;
		}
	}

writeAll:	(cp: ref colorChar) =
	{
	p:	point;
	i:	int;

	p = [ 0, 0 ];
	for	(i = 0; i < size.y; i++, p.y++){
		write(p, cp[:size.x]);
		cp += size.x;
		}
	}

	cursor:			point;
	saved:			ref colorChar;
	flags:			windowFlags;
	status:			windowStatus;
	title:		public	[:] char;

	public:

	outFrame:		frame;
	color:			color_t;
	borderColor:		color_t;
	titleColor:		color_t;
	border:			borderStyles;

textCursor:	dynamic	(p: point) =
	{
	if	(p.x < size.x && p.y < size.y){
		cursor = p;
		if	(flags & W_OPEN)
			super textCursor(p);
		}
	}

getCursor:	() point =
	{
	return cursor;
	}

hideCursor:	dynamic	() =
	{
	ground hideCursor();
	status |= WS_HIDDEN;
	}

showCursor:	dynamic	() =
	{
	status &= ~WS_HIDDEN;	
	textCursor(cursor);
	ground showCursor();
	}

getSize:	() point = 
	{
	return size;
	}

getUlcorner:	() point = 
	{
	return corner;
	}

activate:	() =
	{
	if	(status & WS_HIDDEN)
		hideCursor();
	else
		showCursor();
	ground setColor(color);
	}

scrollContentsUp:	(amount: int) =
	{
	verticalScroll([ 0, 0 ], size, amount);
	}

scrollContentsDown:	(amount: int) =
	{
	verticalScroll([ 0, 0 ], size, -amount);
	}

read:	dynamic	(p: point, buf: [:] colorChar) =
	{
	rows:	int;
	clen:	unsigned;
	rem:	unsigned;

	rem = size.x - p.x;
	p.x += corner.x;
	p.y += corner.y;
	for	(clen = |buf; clen; ){
		if	(clen < rem)
			rem = clen;
		ground read(p, buf[:rem]);
		buf = buf[rem:];
		p.x = corner.x;
		p.y++;
		clen -= rem;
		rem = size.x;
		}
	}

write:	dynamic	(p: point, buf: [:] colorChar) =
	{
	rows:	int;
	clen:	unsigned;
	rem:	unsigned;

	rem = size.x - p.x;
	if	(rem == 0)
		return;
	p.x += corner.x;
	p.y += corner.y;
	for	(clen = |buf; clen; ){
		if	(clen < rem)
			rem = clen;
		ground write(p, buf[:rem]);
		buf = buf[rem:];
		p.x = corner.x;
		p.y++;
		clen -= rem;
		rem = size.x;
		}
	}

putString:	(p: point, s: [:] char) =
	{
	writeChar(p, s);
	}

drawTitle:	() =
	{
	cp:	ref char;
	len:	int;
	i:	int;
	j:	int;
	p:	point;
	b:	* borderDescriptor;

	b = &Borders[border];
	cp = title;
	if	(cp)
		len = |title;
	else
		len = 0;
	i = size.x - 2;
	if	(len >= i)
		i = 0;
	else
		i -= len;
	p = [ 1, 0 ];
	ground setColor(borderColor);
	for	(j = 0; j < i; j++){
		outFrame putc(p, b->top);
		p.x++;
		}
	ground setColor(titleColor);
	for	(; j < size.x && len; j++, cp++, len--){
		outFrame putc(p, *cp);
		p.x++;
		}
	ground setColor(borderColor);
	for	(; j < size.x; j++){
		outFrame putc(p, b->top);
		p.x++;
		}
	}

drawBorder:	dynamic	() =
	{
	b:	* borderDescriptor;
	p:	point;
	i:	int;

	if	(border == WB_NONE)
		return;
	b = &Borders[border];
	drawTitle();

//		The various border corners

	outFrame putc([ 0, 0 ], b->uleft);
	outFrame putc([ size.x + 1, 0 ], b->uright);
	outFrame putc([ 0, size.y + 1 ], b->lleft);
	outFrame putc([ size.x + 1, size.y + 1 ], b->lright);
	drawBottom();

//		Sides of the border

	for	(i = size.y; i > 0; i--){
		outFrame putc([ 0, i ], b->side);
		outFrame putc([ size.x + 1, i ], b->side);
		}
	}

drawBottom:	() =
	{
	i:	int;
	b:	* borderDescriptor;

	if	(border == WB_NONE)
		return;
	b = &Borders[border];
	ground setColor(borderColor);

//		Bottom of the border

	for	(i = size.x; i > 0; i--)
		outFrame putc([ i, size.y + 1 ], b->top);
	}

clear:	() =
	{
	ground setColor(color);
	verticalScroll([ 0, 0 ], size, size.y);
	textCursor([ 0, 0]);
	}

setColor:	dynamic	(a: color_t) color_t =
	{
	x:	color_t;

	x = color;
	color = a;
	if	(flags & W_OPEN)
		ground setColor(a);
	return x;
	}

setBorder:	(b: borderStyles) =
	{
	border = b;
	if	(b != WB_NONE){
		size.x = outFrame.size.x - 2;
		size.y = outFrame.size.y - 2;
		corner.x = outFrame.corner.x + 1;
		corner.y = outFrame.corner.y + 1;
		if	(flags & W_OPEN)
			drawBorder();
		}
	else	{
		size = outFrame.size;
		corner = outFrame.corner;
		}
	}

hide:	dynamic	() int =
	{
	flags &= ~W_OPEN;
	return 1;
	}

close:	dynamic	() int =
	{
	w:	* windowFrame;

	if	(saved){
		setBorder(WB_NONE);
		writeAll(saved);
		free(saved);
		saved = 0;
		}
	if	(flags & W_OS)
		ground close();
	flags = 0;
	return 1;
	}

};

createScreen:	public	(size: point) int =
	{
	a:	int;

	_BX = * ref unsigned[16](&size);
	_AL = 0x3A;
	_softInterrupt(0x50);
	a = _EAX;
	if	(a == 0) // Success
		Screen.size = size;
	return(a);
	}

vScreen:	public	type	packed	{
	public:

 	size:			point;
	cursor:			point;
	useCount:		unsigned[16];
	color:			color_t;
				char;
	currentScreen:		int;
	};

getVideoEntry:	public	(vid: int, buf: * vScreen) int =
	{
	_EBX = vid;
	_EDX = int(buf);
	_AL = 0x49;
	_softInterrupt(0x50);
	return _EAX;
	}

MenuError:	trap;

miFlags:	public	type	char = {
	MI_SEL_CLOSES = 0x01,
	};

menuItem:	public	type	{
	public:

	class:		char;
	flags:		miFlags;
	mtag:		[:] char;
	next:		* menuItem;

select:	dynamic	(* frame, point, v: * viewer) int =
	{
	v beep();		// Default is a dead key.
	return(0);
	}

	};

miSubmenu:	public	type	inherit	menuItem {
	public:

	submenu:	* menu;

constructor:	(l: [:] char, s: * menu) =
	{
	submenu = s;
	mtag = l;
	flags = 0;
	}

select:	dynamic	(grnd: * frame, p: point, prior: * viewer) int =
	{
	if	(p.y < 24)
		p.y++;
	if	(submenu->items == 0)
		return(0);

	mv:	subMenuViewer;
	mi:	* menuItem;
	maxLen:	int;
	i:	int;

	maxLen = 0;
	for	(mi = submenu->items; mi; mi = mi->next){
		i = |mi->mtag;
		if	(i > maxLen)
			maxLen = i;
		}
	if	(p.x + maxLen > 76)
		p.x = 76 - maxLen;
	mv = [ "", [ maxLen + 4, submenu->itemCount + 2 ], 0 ];
	mv setBorder(WB_SINGLE);
	mv.borderColor = mv.color = COLOR(HIGH|YELLOW, BLUE);
	mv popup(grnd, p);
	mv pickMenu(submenu, prior);
	prior = setPrimaryViewer(&mv);
	consoleInput();
	if	(mv.status & SM_CLOSED == 0)
		mv close();
	setPrimaryViewer(prior);
	if	(mv.status & SM_ABORTED)
		return 0;
	else if	(mv.status & SM_RIGHT)
		return 2;
	else if	(mv.status & SM_LEFT)
		return 3;
	else
		return 1;
	}

	};

miButton:	public	type	inherit	menuItem {
	public:

	func:		* (* viewer);

constructor:	(l: [:] char, f: * (* viewer)) =
	{
	mtag = l;
	func = f;
	flags = MI_SEL_CLOSES;
	}

select:	dynamic	(* frame, point, v: * viewer) int =
	{
	func(v);
	return(1);
	}

	};

menuStatus:	type	char = {
//	MS_DORMANT =	0x00,
	MS_ACTIVE =	0x01,
	MS_SELECT =	0x02,
//	MS_SUB =	0x02,
	};
	
menu:	public	type	{
	public:

	items:		* menuItem;
	currentItem:	* menuItem;
	status:		menuStatus;
	itemCount:	char;

addItem:	(mi: * menuItem) =
	{
	m:	* menuItem;

	if	(items){
		for	(m = items; m->next; m = m->next)
			;
		m->next = mi;
		}
	else	{
		items = mi;
		currentItem = items;
		}
	itemCount++;
	mi->next = 0;
	}

button:	(lab: [:] char, f: * (* viewer)) =
	{
	b:	* miButton;

	b = alloc(sizeof miButton);
	if	(b == 0)
		return;
	b = [ lab, f ];
	addItem(b);
	}

submenu:	(lab: [:] char, m: * menu) =
	{
	s:	* miSubmenu;

	s = alloc(sizeof miSubmenu);
	if	(s == 0)
		return;
	s = [ lab, m ];
	addItem(s);
	}

startHorizontal:	(grnd: * frame, selection: keystroke) =
	{
	mm:	mainMenuViewer;
	m:	* menuItem;

	if	(selection){
		selection += ('A' - ALT_A);
		for	(m = items; m; m = m->next)
			if	(m->mtag[0] == selection)
				break;
		if	(m == 0){
			grnd beep();
			return;
			}
		}
	else
		m = 0;
	mm = [ "", [ grnd->size.x, 1 ], 0 ];
	mm open(grnd, [ 0, 0 ]);
	mm startup(self, m);
	consoleInput();
	mm close();
	}

horizontalOffset:	() int =
	{
	mi:		* menuItem;
	i:		int;

	i = 3;
	for	(mi = items; mi; mi = mi->next){
		if	(mi == currentItem)
			break;
		i += |mi->mtag + 2;
		}
	return i;
	}

drawHorizontal:	(grnd: * frame, xmi: * menuItem) =
	{
	mi:		* menuItem;
	i:		int;

	if	(itemCount == 0)
		return;
	grnd setColor(COLOR(HIGH|YELLOW, BLUE));
	i = 3;
	for	(mi = items; mi; mi = mi->next){
		if	(xmi == 0 ||
			 xmi == mi)
			showItem(grnd, i, mi);
		i += |mi->mtag + 2;
		}
	}

showItem:	(grnd: * frame, i: int, mi: * menuItem) =
	{
	if	(status & (MS_ACTIVE|MS_SELECT) == MS_ACTIVE &&
		 mi == currentItem){
		grnd setColor(COLOR(HIGH|YELLOW, PURPLE));
		grnd putString([ i, 0 ], mi->mtag);
		grnd setColor(COLOR(HIGH|YELLOW, BLUE));
		}
	else	{
		grnd putcc([ i, 0 ], mi->mtag[0], COLOR(HIGH|CYAN, BLUE));
		s:	[:] char;

		s = mi->mtag;
		s = s[1:];
		grnd putString([ i + 1, 0 ], s);
		}
	}

	};	

mainMenuViewer:	type	inherit viewer {
	menup:		* menu;
//	totalLength:	int;

	public:

	prior:		* viewer;

close:	dynamic	() int =
	{
	menup->status = 0;
	setPrimaryViewer(prior);
	redraw();
	super close();
	}

redraw:	dynamic	() =
	{
	menup drawHorizontal(ground, 0);
	}

beforeKeyPressed:	dynamic	() =
	{
	hideCursor();
	menup drawHorizontal(ground, menup->currentItem);
	}

afterKeyPressed:	dynamic	() =
	{
	mi:		* menuItem;

	mi = menup->currentItem;
	menup->currentItem = 0;
	menup drawHorizontal(ground, mi);
	menup->currentItem = mi;
	}

rightArrow:	dynamic	() =
	{
	mi:	* menuItem;

	mi = menup->currentItem;
	if	(mi->next)
		menup->currentItem = mi->next;
	else
		menup->currentItem = menup->items;
	}

leftArrow:	dynamic	() =
	{
	mis:	* menuItem;
	mi:	* menuItem;

	mis = menup->currentItem;
	if	(mis == menup->items)
		mis = 0;
	for	(mi = menup->items; mi; mi = mi->next){
		if	(mi->next == mis){
			menup->currentItem = mi;
			break;
			}
		}
	}

upArrow:	dynamic () =
	{
	}

beginLine:	dynamic	() =
	{
	menup->currentItem = menup->items;
	}

endLine:	dynamic	() =
	{
	beginLine();
	leftArrow();
	}

enterKey:	dynamic	() =
	{
	mi:		* menuItem;
	i:		int;
	p:		point;
	v:		* viewer;

	for	(;;){
		p = corner;
		p.x += menup horizontalOffset();
		menup->status |= MS_SELECT;
		i = menup->currentItem select(ground, p, prior);
		menup->status &= ~MS_SELECT;
		if	(i == 1){
			v = setPrimaryViewer(0);

				// If the select command changed the primary 
				// viewer, update the viewer to return to when
				// the menus quit.

			if	(pointer(v) != self)
				prior = v;
			menup->status &= ~MS_ACTIVE;
			clearPrimaryViewer();
			break;
			}
		else if	(i == 2)
			rightArrow();
		else if	(i == 3)
			leftArrow();
		else
			break;
		}
	refuseInput();
	}

enterCharacter:	dynamic	(k: keystroke) =
	{
	mi:		* menuItem;

	k = toupper(k);
	for	(mi = menup->items; mi; mi = mi->next){
		if	(mi->mtag[0] == k){
			menup->currentItem = mi;
			enterKey();
			break;
			}
		}
	}

downArrow:	dynamic	() =
	{
	self enterKey();
	}

escape:	dynamic	() =
	{
	menup->status &= ~MS_ACTIVE;
	setPrimaryViewer(prior);
	refuseInput();
	}

startup:	(m: * menu, mi: * menuItem) int =
	{
	v:	* viewer;

	menup = m;
	if	(menup->status & MS_ACTIVE == 0){
		menup->status |= MS_ACTIVE;
		menup->currentItem = 0;
		prior = setPrimaryViewer(self);
		acceptInput();
		if	(mi){
			menup->currentItem = mi;
			enterKey();
			}
		else
			menup->currentItem = menup->items;
		return 1;
		}
	else
		return 0;
	}

	};

smStatus:	type	char = {
	SM_ABORTED =	0x01,
	SM_CLOSED = 	0x02,
	SM_RIGHT = 	0x04,
	SM_LEFT =	0x08,
	};

subMenuViewer:	type	inherit viewer {
	menup:		* menu;
//	totalLength:	int;
//	itemCount:	int;
	incr:		int;

showItem:	(i: int, mi: * menuItem) =
	{
	p:	point;

	p = getSize();
	setColor(COLOR(HIGH|YELLOW, BLUE));
	verticalScroll([0, i], [ p.x, 1 ], 1);
	putcc([ 1, i ], mi->mtag[0], COLOR(HIGH|CYAN, BLUE));
	s:	[:] char;
	s = mi->mtag;
	s = s[1:];
	putString([ 2, i], s);
	}

	public:

	prior:		* viewer;
	status:		smStatus;

pickMenu:	(m: * menu, p: * viewer) =
	{
	menup = m;
	prior = p;
	redraw();
	}

redraw:	dynamic	() =
	{
	mi:		* menuItem;
	i:		int;

	setColor(COLOR(HIGH|YELLOW, BLUE));
	clear();
//	if	(itemCount == 0)
//		return;
	i = 0;
	for	(mi = menup->items; mi; mi = mi->next, i++)
		showItem(i, mi);
	}

beforeKeyPressed:	dynamic	() =
	{
	mi:		* menuItem;
	i:		int;
	p:		point;

	i = 0;
	for	(mi = menup->items; mi; mi = mi->next, i++)
		if	(mi == menup->currentItem)
			break;
	hideCursor();
	setColor(COLOR(HIGH|YELLOW, PURPLE));
	p = getSize();
	verticalScroll([0, i], [ p.x, 1 ], 1);
	putString([ 1, i ], mi->mtag);
	}

afterKeyPressed:	dynamic	() =
	{
	mi:		* menuItem;
	i:		int;

	i = 0;
	for	(mi = menup->items; mi; mi = mi->next, i++)
		if	(mi == menup->currentItem)
			break;
	self showItem(i, mi);
	}

enterKey:	dynamic	() =
	{
	p:		point;
	mi:		* menuItem;
	i:		int;
	v:		* viewer;

	i = 1;
	for	(mi = menup->items; mi; mi = mi->next, i++)
		if	(mi == menup->currentItem)
			break;
	p = self getUlcorner();
	p.x++;
	p.y += i;
	if	(menup->currentItem->flags & MI_SEL_CLOSES){
		self close();
		status = SM_CLOSED;
		clearPrimaryViewer();
		}
	else
		status = 0;
	do	{
		menup->status |= MS_SELECT;
		i = menup->currentItem select(ground, p, prior);
		menup->status &= ~MS_SELECT;
		if	(i == 1){
			v = setPrimaryViewer(0);

				// If the select command changed the primary 
				// viewer, update the viewer to return to when
				// the menus quit.

			if	(v && pointer(v) != self)
				prior = v;
			clearPrimaryViewer();
			}
		}
		while	(i > 1);
	}

downArrow:	dynamic	() =
	{
	mi:	* menuItem;

	mi = menup->currentItem;
	if	(mi->next)
		menup->currentItem = mi->next;
	else
		menup->currentItem = menup->items;
	}

upArrow:	dynamic	() =
	{
	mis:	* menuItem;
	mi:	* menuItem;

	mis = menup->currentItem;
	if	(mis == menup->items)
		mis = 0;
	for	(mi = menup->items; mi; mi = mi->next){
		if	(mi->next == mis){
			menup->currentItem = mi;
			break;
			}
		}
	}

leftArrow:	dynamic () =
	{
	clearPrimaryViewer();
	status = SM_LEFT;
	}

rightArrow:	dynamic () =
	{
	clearPrimaryViewer();
	status = SM_RIGHT;
	}

enterCharacter:	dynamic	(k: keystroke) =
	{
	mi:		* menuItem;

	k = toupper(k);
	for	(mi = menup->items; mi; mi = mi->next){
		if	(mi->mtag[0] == k){
			menup->currentItem = mi;
			self enterKey();
			break;
			}
		}
	}

beginLine:	dynamic	() =
	{
	menup->currentItem = menup->items;
	}

endLine:	dynamic	() =
	{
	self beginLine();
	self upArrow();
	}

escape:	dynamic	() =
	{
	clearPrimaryViewer();
	status = SM_ABORTED;
	}

	};

ActiveViewer:		* viewer;
ViewerList:		* viewer;

viewerFlags:	public	type	char = {
	VS_SHOW		= 0x01,		// Show
	VS_INTERACT	= 0x02,		// Allow input
	};

viewer:	public	type	inherit	window	{
	public:

constructor:	(titl: [:] char, sz: point, mm: * menu) =
	{
	super constructor(titl, sz);
	next = self;		// This will make a close before an open work
	prev = self;		// just a little precaution
	flags = 0;		// The window starts out closed
	mainMenu = mm;
	}

open:	dynamic	(grnd: * frame, p: point) int =
	{
	if	(flags & VS_SHOW)
		return 1;			// Already open
	super open(grnd, p);
	addToViewerList();
	flags |= VS_SHOW;
	return 1;
	}

popup:	dynamic	(grnd: * frame, p: point) int =
	{
	if	(flags & VS_SHOW)
		return 1;			// Already open
	super popup(grnd, p);
	addToViewerList();
	flags |= VS_SHOW;
	return 1;
	}

drawBorder:	dynamic	() =
	{
	if	(border == WB_NONE)
		return;
	super drawBorder();
	if	(mainMenu)
		mainMenu drawHorizontal(&outFrame, 0);
	}

	private:

	next:			* viewer;
	prev:			* viewer;
	flags:			viewerFlags;

	public:

	mainMenu:		* menu;

afterKeyPressed:	dynamic () = {}
beforeKeyPressed:	dynamic () = {}

enterCharacter:	dynamic	(keystroke) = { beep(); }

redraw:		dynamic	() = { }
refresh:	dynamic	() = { }
saveFile:	dynamic	() = { beep(); }
escape:		dynamic	() = { beep(); }
cancel:		dynamic	() = { beep(); }
upArrow:	dynamic	() = { beep(); }
downArrow:	dynamic	() = { beep(); }
rightArrow:	dynamic	() = { beep(); }
leftArrow:	dynamic	() = { beep(); }
dragUp:		dynamic	() = { beep(); }
dragDown:	dynamic	() = { beep(); }
dragRight:	dynamic	() = { beep(); }
dragLeft:	dynamic	() = { beep(); }
rightCharacter:	dynamic	() = { beep(); }
leftCharacter:	dynamic	() = { beep(); }
rightWord:	dynamic	() = { beep(); }
leftWord:	dynamic	() = { beep(); }
upScreen:	dynamic	() = { beep(); }
downScreen:	dynamic	() = { beep(); }
topScreen:	dynamic	() = { beep(); }
bottomScreen:	dynamic	() = { beep(); }
scrollUp:	dynamic	() = { beep(); }
scrollDown:	dynamic	() = { beep(); }
beginLine:	dynamic	() = { beep(); }
endLine:	dynamic	() = { beep(); }
beginFile:	dynamic	() = { beep(); }
endFile:	dynamic	() = { beep(); }
enterKey:	dynamic	() = { beep(); }
deleteCharacter:dynamic	() = { beep(); }
deleteLine:	dynamic	() = { beep(); }
deleteWord:	dynamic	() = { beep(); }
deleteBlock:	dynamic () = { beep(); }
deleteEndOfLine:dynamic	() = { beep(); }
backspace:	dynamic	() = { beep(); }
startOfBlock:	dynamic	() = { beep(); }
endOfBlock:	dynamic	() = { beep(); }
moveBlock:	dynamic	() = { beep(); }
copyBlock:	dynamic	() = { beep(); }
writeBlock:	dynamic () = { beep(); }
cutToScrap:	dynamic	() = { beep(); }
copyToScrap:	dynamic	() = { beep(); }
pasteFromScrap:	dynamic	() = { beep(); }
pasteFromFile:	dynamic	() = { beep(); }
help:		dynamic	() = { beep(); }
tab:		dynamic	() = { beep(); }
backtab:	dynamic	() = { beep(); }
search:		dynamic	() = { beep(); }
replace:	dynamic	() = { beep(); }
again:		dynamic	() = { beep(); }

localKeys:	dynamic	() * keyMap = { return 0; }

addToViewerList:	private	() =
	{
	if	(ViewerList == 0){
		next = self;
		prev = self;
		}
	else	{
		next = ViewerList;
		prev = ViewerList->prev;
		ViewerList->prev->next = self;
		ViewerList->prev = self;
		}
	ViewerList = self;
	}

acceptInput:	() =
	{
	flags |= VS_INTERACT;
	}

refuseInput:	() =
	{
	flags &= ~VS_INTERACT;
	}

viewerState:	() viewerFlags =
	{
	return flags;
	}

close:	dynamic	() int =
	{
	w:	* viewer;

	if	(flags & VS_SHOW == 0)
		return 1;				// Already closed
	super close();
	flags = 0;
	if	(ActiveViewer == self){
		ActiveViewer = ActiveViewer nextInLoop();
		setPrimaryViewer(ActiveViewer);
		}
	if	(ViewerList == self){
		ViewerList = ViewerList->next;
		if	(ViewerList == self)
			ViewerList = 0;
		}
	prev->next = next;
	next->prev = prev;
	return 1;
	}

nextInLoop:	() * viewer =
	{
	v:	* viewer;

	v = next;
	while	(v->flags & VS_INTERACT == 0){
		if	(v == self)
			return 0;
		v = v->next;
		}
	return v;
	}

clearMessage:	public () =
	{
	if	(border != WB_NONE)
		drawBottom();
	}

postMessage:	public	(fmt: [:] char, ...) =
	{
	postMessageV(fmt, ...);
	}

postMessageV:	public	(fmt: [:] char, ap: varArgs) =
	{
	x:	char;

	if	(border != WB_NONE){
		outFrame setColor(HIGH|WHITE);
		outFrame vprintf([ 2, outFrame.size.y - 1 ], fmt, ap);
		outFrame setColor(borderColor);
		}
	}
};

nextViewer:	public	(* viewer) =
	{
	if	(ActiveViewer == 0)
		return;
	ActiveViewer = ActiveViewer nextInLoop();
	setPrimaryViewer(ActiveViewer);
	}

setPrimaryViewer:	public	(nv: * viewer) * viewer =
	{
	v:	* viewer;

	v = ActiveViewer;
	if	(nv){
		ActiveViewer = nv;
		ActiveViewer acceptInput();
		ActiveViewer redraw();
		ActiveViewer drawBorder();
		}
	return(v);
	}

clearPrimaryViewer:	public	() =
	{
	ActiveViewer = 0;
	}

keyMap:	public	type	{
	private:

	keys:		* keydef;
	funcs:		* funcdef;
	base:		* keyMap;

	public:

constructor:	(baseMap: * keyMap) =
	{
	keys = 0;
	funcs = 0;
	base = baseMap;
	}

defineFunction:	(name: [:] char, lab: [:] char, f: * (* viewer), k: keystroke) int =
	{
	fd:	* funcdef;

	for	(fd = funcs; fd; fd = fd->next)
		if	(stringCompare(fd->name, name) == 0)
			return ERREXIST;
	fd = alloc(sizeof funcdef + |name + |lab);
	if	(fd == 0)
		return ERRNOMEMORY;
	fd->name = ref char(fd + 1)[:|name];
	fd->lab = (&fd->name[|name])[:|lab];
	fd->name [:]= name;
	fd->lab [:]= lab;
	fd->f = f;
	fd->next = funcs;
	funcs = fd;
	if	(k)
		return bindKey(k, name, 0);
	return SUCCESS;
	}
/*
	The scope is zero for local key bindings, non-zero for global
	key bindings.  This primarily applies to how key bindings are
	saved on disk.
 */
bindKey:	(k: keystroke, name: [:] char, scope: int) int =
	{
	kd:	* keydef;
	kp:	* keydef;

	if	(|name == 0){

			// This is an undefine of a keystroke

		kp = 0;
		for	(kd = keys; kd; kp = kd, kd = kd->next)
			if	(kd->key == k){
				if	(kp)
					kp->next = kd->next;
				else
					keys = kd->next;
				free(kd);
				return SUCCESS;
				}
		return ERRNOTFOUND;
		}

	for	(kd = keys; kd; kd = kd->next)
		if	(kd->key == k)
			return ERREXIST;
	fd:	* funcdef;

	for	(fd = funcs; fd; fd = fd->next)
		if	(stringCompare(fd->name, name) == 0)
			break;
	if	(fd == 0)
		return ERRNOTFOUND;
	kd = alloc(sizeof (keydef));
	if	(kd){
		kd->next = keys;
		keys = kd;
		kd->key = k;
		kd->func = fd;
		kd->scope = scope;
		return SUCCESS;
		}
	else
		return ERRNOMEMORY;
	}

lookup:	(k: keystroke) * (* viewer) =
	{
	kd:	* keydef;

	for	(kd = keys; kd; kd = kd->next)
		if	(kd->key == k)
			return kd->func->f;
	return 0;
	}

	};

keydef:	type	{
	public:

	next:	* keydef;
	key:	keystroke;
	scope:	char;
	func:	* funcdef;
	};

funcdef:	type	{
	public:

	next:	* funcdef;
	name:	[:] char;
	lab:	[:] char;
	f:	* (* viewer);
	};

GlobalMap:	public	keyMap;

shutDownViewers:	public	(* viewer) =
	{
	while	(ActiveViewer)
		if	(!ActiveViewer close())
			return;
	exit(0);
	}

consoleInput:	public	() =
	{
	k:	keystroke;
	l:	* keyMap;
	f:	* (* viewer);

	for	(;;){
		if	(ActiveViewer == 0)
			return;
		ActiveViewer beforeKeyPressed();
		ActiveViewer activate();
		do
			k = ActiveViewer getRawKey();
			while	(k == -1);		// ignore signal catches
		ActiveViewer afterKeyPressed();
		l = ActiveViewer localKeys();
		if	(l){
			f = l lookup(k);
			if	(f){
				f(ActiveViewer);
				continue;
				}
			}
		f = GlobalMap lookup(k);
		if	(f){
			f(ActiveViewer);
			continue;
			}
		switch	(k){
		case	ALT_X:
			shutDownViewers(0);
			break;

			/* Navigation commands */

		case	DOWN_ARW:
			ActiveViewer downArrow();
			break;

		case	UP_ARW:
			ActiveViewer upArrow();
			break;

		case	LEFT_ARW:
			ActiveViewer leftArrow();
			break;

		case	RIGHT_ARW:
			ActiveViewer rightArrow();
			break;

		case	NUM_2:
			ActiveViewer dragDown();
			break;

		case	NUM_8:
			ActiveViewer dragUp();
			break;

		case	NUM_4:
			ActiveViewer dragLeft();
			break;

		case	NUM_6:
			ActiveViewer dragRight();
			break;

		case	HOME:
			ActiveViewer beginLine();
			break;

		case	END:
			ActiveViewer endLine();
			break;

		case	CTRL_HOME:
			ActiveViewer beginFile();
			break;

		case	CTRL_END:
			ActiveViewer endFile();
			break;

		case	PGUP:
			ActiveViewer upScreen();
			break;

		case	PGDN:
			ActiveViewer downScreen();
			break;

		case	CTRL_LEFT:
			ActiveViewer leftWord();
			break;

		case	CTRL_RIGHT:
			ActiveViewer rightWord();
			break;

		case	CTRL_PGUP:
			ActiveViewer topScreen();
			break;

		case	CTRL_PGDN:
			ActiveViewer bottomScreen();
			break;

		case    CTRL_Z:
			ActiveViewer scrollUp();
			break;

		case	CTRL_W:
			ActiveViewer scrollDown();
			break;

			// Window management commands

		case	ALT_C:
			ActiveViewer close();
			if	(ActiveViewer)
				ActiveViewer redraw();
			break;

			/* File management commands */

		case	F2:
			ActiveViewer saveFile();
			break;

			/* Block commands */

		case	ALT_F7:
			ActiveViewer startOfBlock();
			break;

		case	ALT_F8:
			ActiveViewer endOfBlock();
			break;

		case	ALT_F2:
			ActiveViewer moveBlock();
			break;

		case	ALT_F4:
			ActiveViewer copyBlock();
			break;

		case	ALT_Y:
			ActiveViewer deleteBlock();
			break;

		case	GREY_PLUS:
			ActiveViewer copyToScrap();
			break;

		case	GREY_DASH:
			ActiveViewer cutToScrap();
			break;

		case	INS:
			ActiveViewer pasteFromScrap();
			break;

		case	ALT_W:
			ActiveViewer writeBlock();
			break;

		case	ALT_R:
			ActiveViewer pasteFromFile();
			break;

			/* Miscellaneous commands */

		case	F1:
			ActiveViewer help();
			break;

		case	GREY_ESC:
			ActiveViewer escape();
			break;

		case	ALT_S:
			ActiveViewer search();
			break;

		case	ALT_Q:
			ActiveViewer replace();
			break;

		case	ALT_L:
			ActiveViewer again();
			break;

			/* Text entry commands */

		case	GREY_TAB:
			ActiveViewer tab();
			break;

		case	BACK_TAB:
			ActiveViewer backtab();
			break;

		case	GREY_STAR:
			ActiveViewer enterCharacter('*');
			break;

		case	DEL:
			ActiveViewer deleteCharacter();
			break;

		case	GREY_BS:
			ActiveViewer backspace();
			break;

		case	CTRL_Y:
			ActiveViewer deleteLine();
			break;

		case	CTRL_T:
			ActiveViewer deleteWord();
			break;

		case	GREY_CR:
			ActiveViewer enterKey();
			break;

			/* Menu Commands */

		case	F10:
			if	(ActiveViewer->mainMenu)
				ActiveViewer->mainMenu 
				   startHorizontal(&ActiveViewer->outFrame, 0);
			break;

		default:
			if	(k < 256)
				ActiveViewer enterCharacter(k);
			else if	(ActiveViewer->mainMenu &&
				 k >= ALT_A &&
				 k <= ALT_Z)
				ActiveViewer->mainMenu 
				   startHorizontal(&ActiveViewer->outFrame, k);
			else
				ActiveViewer beep();
			}
		}
	}

MAXWIDTH:	const	int = 76;

confirm:	public	(msg: [:] char, attr: color_t) int =
	{
	i:	unsigned;
	w:	window;
	k:	keystroke;

	i = |msg;
	if	(i > MAXWIDTH)
		i = MAXWIDTH;
	w = [ "Yes or No? (Y/N)", [ i + 4, 3 ] ];
	w setColor(attr);
	w setBorder(WB_DOUBLE);
	w popup(&Screen, [ 40 - i / 2, 11 ]);
	w clear();
	w printf([ 1, 0 ], "%s", msg);
	w hideCursor();
	for	(;;){
		k = w getKey();
		if	(k == 'y' || k == 'Y'){
			i = 1;
			break;
			}
		else if	(k == 'n' || k == 'N'){
			i = 0;
			break;
			}
		else if	(k == ESC || k == GREY_ESC){
			i = -1;
			break;
			}
		w beep();
		}
	w close();
	return i;
	}

keystroke:	public	type	signed[16] = {
	CTRL_AT,			/* ^@	nul character */
	CTRL_A,		CTRL_B,		CTRL_C,		CTRL_D,
	CTRL_E,		CTRL_F,		CTRL_G,		CTRL_H,
	CTRL_I,		CTRL_J,		CTRL_K,		CTRL_L,
	CTRL_M,		CTRL_N,		CTRL_O,		CTRL_P,
	CTRL_Q,		CTRL_R,		CTRL_S,		CTRL_T,
	CTRL_U,		CTRL_V,		CTRL_W,		CTRL_X,
	CTRL_Y,		CTRL_Z,

	ESC,

	BS = CTRL_H,	TAB = CTRL_I,	LF = CTRL_J,	FF = CTRL_L,
	CR = CTRL_M,

	START_KEY = CTRL_Q,
	STOP_KEY = CTRL_S,

	F1 = 256,	F2,		F3,		F4,
	F5,		F6,		F7,		F8,
	F9,		F10,		F11,		F12,

	SHFT_F1,	SHFT_F2,	SHFT_F3,	SHFT_F4,
	SHFT_F5,	SHFT_F6,	SHFT_F7,	SHFT_F8,
	SHFT_F9,	SHFT_F10,	SHFT_F11,	SHFT_F12,

	CTRL_F1,	CTRL_F2,	CTRL_F3,	CTRL_F4,
	CTRL_F5,	CTRL_F6,	CTRL_F7,	CTRL_F8,
	CTRL_F9,	CTRL_F10,	CTRL_F11,	CTRL_F12,

	ALT_F1,		ALT_F2,		ALT_F3,		ALT_F4,
	ALT_F5,		ALT_F6,		ALT_F7,		ALT_F8,
	ALT_F9,		ALT_F10,	ALT_F11,	ALT_F12,

	HOME,		END,		PGUP,		PGDN,
	CENTER,

	UP_ARW,		DOWN_ARW,	LEFT_ARW,	RIGHT_ARW,

	INS,
	DEL,

	GREY_BS,	GREY_TAB,	GREY_PLUS,	GREY_DASH,
	GREY_STAR,

	NUM_0,		NUM_1,		NUM_2,		NUM_3,
	NUM_4,		NUM_5,		NUM_6,		NUM_7,
	NUM_8,		NUM_9,

	NUM_DOT,

	GREY_CR,
	GREY_ESC,

	BACK_TAB,

	CTRL_HOME,	CTRL_END,	CTRL_PGUP,	CTRL_PGDN,
	CTRL_CENTER,

	CTRL_UP,	CTRL_DOWN,	CTRL_LEFT,	CTRL_RIGHT,

	ALT_DASH,	ALT_EQ,		CTRL_BREAK,	CTRL_PRTSC,
	PRTSC,		SYSREQ,		SHFT_SYSREQ,

	ALT_0,		ALT_1,		ALT_2,		ALT_3,
	ALT_4,		ALT_5,		ALT_6,		ALT_7,
	ALT_8,		ALT_9,

	CTRL_CR,

	ALT_A,		ALT_B,		ALT_C,		ALT_D,
	ALT_E,		ALT_F,		ALT_G,		ALT_H,
	ALT_I,		ALT_J,		ALT_K,		ALT_L,
	ALT_M,		ALT_N,		ALT_O,		ALT_P,
	ALT_Q,		ALT_R,		ALT_S,		ALT_T,
	ALT_U,		ALT_V,		ALT_W,		ALT_X,
	ALT_Y,		ALT_Z,

	ALT_SYSREQ,

	KEYCOUNT,
	};

shiftStates:	public	type	byte = {
	CAPS_STATE	= 0x40,		/* The CAPS lock toggle */
	NUM_STATE	= 0x20,		/* Num lock toggle */
	SCROLL_STATE	= 0x10,		/* Scroll lock toggle */
	ALT_SHIFT	= 0x08,		/* Alt key */
	CTL_SHIFT	= 0x04,		/* Ctrl key */
	LEFT_SHIFT	= 0x02,		/* Left hand shift key */
	RIGHT_SHIFT	= 0x01		/* Right hand shift key */
	};

//lastKey:	keystroke;
//ungotten:	char;

keyboardObject:	type	{
	public:

//getKey:		remote	() keystroke	= 0x301;
//getRawKey:	remote	() keystroke	= 0x302;
//testKey:	remote	() keystroke	= 0x303;
getShiftState:	remote	() shiftStates	= 0x304;
setShiftState:	remote	(shiftStates)	= 0x305;
setBreakKey:	remote	(int) int	= 0x306;
	};
/*
ungetKey:	public	() =
	{
	ungotten = 1;
	}

getKey:	public	() keystroke =
	{
	if	(ungotten){
		ungotten = 0;
		return(lastKey);
		}
	return ref far keyboardObject(0) getKey();
	}

getRawKey:	public	() keystroke =
	{
	if	(ungotten){
		ungotten = 0;
		return(lastKey);
		}
	return ref far keyboardObject(0) getRawKey();
	}

testKey:	public	() keystroke =
	{
	if	(ungotten)
		return(lastKey);
	return ref far keyboardObject(0) testKey();
	}
 */
getShiftState:	public	() shiftStates =
	{
	return ref far keyboardObject(0) getShiftState();
	}

setShiftState:	public	(newShiftState: shiftStates) =
	{
	ref far keyboardObject(0) setShiftState(newShiftState);
	}

setBreakKey:	public	(newBreakValue: int) int =
	{
	ref far keyboardObject(0) setBreakKey(newBreakValue);
	}
