///////////////////////////////////////////////////////////
//                                                       //
// MsgDisp - Message Text Display                        //
// Copyright (c) Ivan Gerencir, Aug 1993                 //
// Language: Borland C++ 3.1 & Application Frameworks    //
// OS: Windows 3.1                                       //
//                                                       //
///////////////////////////////////////////////////////////

// File: mainwin.cpp; Listing 2

#include <owl.h>
#include <commdlg.h>
#include <string.h>
#include <dir.h>
#include <stdio.h>
#include <print.h>

#include "msgdisp.h"
#include "mainwin.h"
#include "menu.h"

////////////////////////////////////////////////////////////////////
/////////////////////////// TMainWindow members ////////////////////
//------------------------------------------------------------------
// Constructor

#define RESOURCEID 200

TMainWindow::TMainWindow()
: TDialog(NULL, APPNAME)
{
    // Create interface to ListBox
    TextControl = new TListBox(this, RESOURCEID);
}

#undef RESOURCEID

//----------------------------------------------------------------
// Destructor

void TMainWindow::~TMainWindow()
{
    // Destroy interface to ListBox
    delete TextControl;
}

//----------------------------------------------------------------
// SetupWindow

void TMainWindow::SetupWindow()
{
    TDialog::SetupWindow();
    Menu = GetMenu(HWindow);

    // PopUp menu item is initially checked
    PopUp = TRUE;
    CheckMenuItem(Menu, MENU_POPUP, MF_BYCOMMAND | MF_CHECKED);

    // Register message used for communicating with us
    if((MsgDispMessage = RegisterWindowMessage(APPNAME)) == 0) {
        MessageBox(HWindow,
                "Unable to register Windows Message",
                APPNAME,
                MB_OK);
        PostQuitMessage(0);
    }
}

//----------------------------------------------------------------
// DefWndProc
// Called directly from StdWndProc (owl.cpp) for messages >= 0x8000

void TMainWindow::DefWndProc(RTMessage Msg)
{
    // If we received our message
    if(Msg.Message == MsgDispMessage) {
        // And magic data proves to be OK
        if(Msg.WParam == MAGIC_ID) {
            RecvString(Msg);
            return;
        }
    }
    TDialog::DefWndProc(Msg);
}

//----------------------------------------------------------------
// RecvString
// Client sends this message with atom number in LOWORD(LParam).

void TMainWindow::RecvString(RTMessage Msg)
{
    char buffer[MAXMSGSTRLEN];
    GlobalGetAtomName(LOWORD(Msg.LParam), buffer, sizeof(buffer));
    TextControl->AddString(buffer);
    TextControl->SetSelIndex(TextControl->GetCount() - 1);
    if(PopUp) {
        // If PopUp menu item checked
        BringWindowToTop(HWindow);
        ShowWindow(HWindow, SW_SHOWNORMAL);
    }
}

//----------------------------------------------------------------
// Implements WM_SIZE message

#define REDISPLAY TRUE

void TMainWindow::WMSize(RTMessage Msg)
{
    // Fill the whole parent window
    RECT Rect;
    Rect.top = Rect.left = 0;
    Rect.right = LOWORD(Msg.LParam);
    Rect.bottom = HIWORD(Msg.LParam);
    MoveWindow(TextControl->HWindow, Rect.left, Rect.top, Rect.right,
            Rect.bottom, REDISPLAY);
}

#undef REDISPLAY

//-----------------------------------------------------------------
// CMSave

void TMainWindow::CMSave(RTMessage)
{
    int NumStrings = TextControl->GetCount();
    if(NumStrings == 0)
        return;

    OPENFILENAME ofn;
    static char FileBuf[256] = "";
    char FileTitleBuf[256];

    // Prepare OPENFILENAME structure for GetSaveFileName function
    memset(&ofn, 0, sizeof(ofn));
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = HWindow;
    ofn.lpstrFilter = APPNAME " files (*.msg)\0"
                        "*.msg\0"
                        "All files (*.*)\0"
                        "*.*\0";
    ofn.lpstrFile = FileBuf;
    ofn.nMaxFile = sizeof(FileBuf);
    ofn.lpstrFileTitle = FileTitleBuf;
    ofn.nMaxFileTitle = sizeof(FileTitleBuf);
    ofn.lpstrTitle = APPNAME " Save";
    ofn.lpstrDefExt = "msg";
    ofn.Flags = OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST |
            OFN_HIDEREADONLY;

    if(GetSaveFileName(&ofn) == 0) {
        DWORD Err = CommDlgExtendedError();
        if(Err) {
            char b[64];
            sprintf(b, APPNAME " Error %08lX", Err);
            MessageBox(HWindow, "Data not saved", b,
                    MB_OK | MB_ICONSTOP);
        }
        return;
    }

    // Remember entered filename for the next time
    strcpy(FileBuf, FileTitleBuf);

    // Open file for write in text mode
    FILE *f = fopen(FileBuf, "wt");
    if(f == NULL) {
        char b[256];
        sprintf(b, "Unable to open file \"%s\" for write", FileBuf);
        MessageBox(HWindow, b, APPNAME " Save data",
                MB_OK | MB_ICONSTOP);
        return;
    }

    // Now dump contents of ListBox into file
    for(int i = 0; i < NumStrings; ++i) {
        char Str[MAXMSGSTRLEN];
        TextControl->GetString(Str, i);
        fputs(Str, f);
        fputs("\n", f);
    }

    fclose(f);
    MessageBox(HWindow, "Data saved", APPNAME, MB_OK);
}

//-----------------------------------------------------------------
// CMPrint

void TMainWindow::CMPrint(RTMessage)
{
    int NumStrings = TextControl->GetCount();
    if(NumStrings == 0)
        return;

    PRINTDLG pd;

    // Prepare PRINTDLG structure for PrintDlg function
    memset(&pd, 0, sizeof(pd));
    pd.lStructSize = sizeof(pd);
    pd.hwndOwner = HWindow;

    // Constant PD_HIDEPRINTTOFILE defined in COMMDLG.H
    // is long but doesn't end in L, so the IDE compiler
    // issues a warning.  Unfortunately, #pragma warn -cln
    // does not suppress it. Command line compiler BCC however,
    // handles it OK.
    
    #pragma warn -cln
    pd.Flags = PD_RETURNDC | PD_HIDEPRINTTOFILE;

    if(PrintDlg(&pd) == 0) {
        DWORD Err = CommDlgExtendedError();
        if(Err) {
            char b[64];
            sprintf(b, APPNAME " Error %08lX", Err);
            MessageBox(HWindow, "Printing canceled", b,
                    MB_OK | MB_ICONSTOP);
        }
        return;
    }

    // If PrintDlg allocated additional structures, release them
    if(pd.hDevMode) {
        GlobalFree(pd.hDevMode);
        pd.hDevMode = NULL;
    }
    if(pd.hDevNames) {
        GlobalFree(pd.hDevNames);
        pd.hDevNames = NULL;
    }

    HDC DC = pd.hDC;
    POINT PageDims;
    // Get physical page size in pixels
    Escape(DC, GETPHYSPAGESIZE, NULL, NULL, &PageDims);

    // Calculate LineHeight according to the current font
    TEXTMETRIC tm;
    GetTextMetrics(DC, &tm);
    int LineHeight = tm.tmHeight + tm.tmExternalLeading; 

    POINT PrintOffs;
    // Get printing offset on the page
    Escape(DC, GETPRINTINGOFFSET, NULL, NULL, &PrintOffs);
    // Reduce vertical page size to reflect printable area,
    // for the printer cannot print on whole physical page 
    PageDims.y -= 2 * PrintOffs.y + LineHeight;

    DOCINFO di;
    // Prepare DOCINFO structure for StartDoc function
    di.cbSize = sizeof(di);
    di.lpszDocName = APPNAME;
    di.lpszOutput = NULL; // no redirection to file

    StartDoc(DC, &di);  // Start print job
    StartPage(DC);

    int YPos = 0;
    for(int i = 0; i < NumStrings; ++i) {
        if(YPos > PageDims.y) {
            EndPage(DC);
            StartPage(DC);
            YPos = 0;
        }
        char Str[MAXMSGSTRLEN];
        TextControl->GetString(Str, i);
        TextOut(DC, PrintOffs.x, YPos, Str, strlen(Str));
        YPos += LineHeight;
    }

    EndPage(DC);
    EndDoc(DC);

    DeleteDC(DC);
}

//-----------------------------------------------------------------
// CMRepaint

void TMainWindow::CMRepaint(RTMessage)
{
    // Redraw complete window with all the children
    RedrawWindow(HWindow, NULL, NULL,
            RDW_ERASE | RDW_FRAME | RDW_INTERNALPAINT |
            RDW_INVALIDATE | RDW_UPDATENOW | RDW_ALLCHILDREN);
}

//----------------------------------------------------------------
// Implements Menu Exit choice

void TMainWindow::CMExitApp(RTMessage)
{
    PostQuitMessage(0);
}

//----------------------------------------------------------------
// Implements Menu Options|Insert break choice

void TMainWindow::CMInsertBreak(RTMessage)
{
    TextControl->AddString("======================================");
    // Force selected item to the last so that
    // it is always displayed
    TextControl->SetSelIndex(TextControl->GetCount() - 1);
}

//----------------------------------------------------------------
// Implements Menu Options|Clear choice

void TMainWindow::CMClear(RTMessage)
{
    TextControl->ClearList();
}

//----------------------------------------------------------------
// Implements Menu Options|PopUp choice

void TMainWindow::CMOptionsPopUp(RTMessage)
{
    if(PopUp) {
        PopUp = FALSE;
        CheckMenuItem(Menu, MENU_POPUP, MF_BYCOMMAND | MF_UNCHECKED);
    }
    else {
        PopUp = TRUE;
        CheckMenuItem(Menu, MENU_POPUP, MF_BYCOMMAND | MF_CHECKED);
    }
}

// End of File
