/*|*********************************************************
|* 'ncpint.c'
|*
|* DOS NCP interface.
|*
|* -93     PS    Created
|*
|**********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <time.h>
#include "ipxint.h"
#include "ncptest.h"


/*********************** Constants  ***********************/

/* NCP base socket */
#define NCP_BASE_SOCKET       (0x5000)

/* Sockets/NCP connection */
#define SOCK_PER_NCP          (3)

/* Max. size of a NCP packet */
#define MAX_NCPPACKET         (1024)

/* Packet types */
#define PACKET_NCP            (0x11)
#define PACKET_SAP            (0x00)

/* SAP queries/responses */
#define SAP_NEAREST_QUERY     (0x0003)
#define SAP_NEAREST_RESPONSE  (0x0004)

/* SAP socket # */
#define SAP_SOCKET      (0x0452)

/* NCP packet types */
#define NCPPACK_OPEN    (0x1111)
#define NCPPACK_CLOSE   (0x5555)
#define NCPPACK_REQUEST (0x2222)
#define NCPPACK_REPLY   (0x3333)


/*********************** Data types ***********************/

/* NCP connection info */
typedef struct NCPCONNECTION_record
{
  u8         sequence;
  u8         task;
  u8         cLow;
  u8         cHigh;
  int        *cmdStatus;
  unsigned   requestSent:1;
  unsigned   isReply:1;
  char       *serverName;
  ECB        recvECB[SOCK_PER_NCP];
  ECB        sendECB[SOCK_PER_NCP];
  IPXHEADER  recvIPX[SOCK_PER_NCP];
  IPXHEADER  sendIPX[SOCK_PER_NCP];
} NCPCONNECTION;

/* SAP request packet */
typedef struct SAPREQUEST_record
{
  u16 requestType;
  u16 serverType;
} SAPREQUEST;

/* SAP reply packet */
typedef struct SAPREPLY_record
{
  u16        replyType;
  u16        serverType;
  char       serverName[LIMIT_OBJNAME+1];
  IPXADDRESS serverAddress;
  u16        hops;
} SAPREPLY;

/* Complete SAP packets incl. IPX headers */
typedef struct SAPREQ_PACKET_record
{
  IPXHEADER  requestHeader;
  SAPREQUEST requestData;
} SAPREQ_PACKET;

typedef struct SAPREP_PACKET_record
{
  IPXHEADER  replyHeader;
  SAPREPLY   replyData;
} SAPREP_PACKET;

/* NCP request w/o subfunction */
typedef struct NCPF_REQ_record
{
  u16   packetType;
  u8    sequenceNumber;
  u8    cLow;
  u8    taskNumber;
  u8    cHigh;
  u8    functionNumber;
  u8    packet[VARSIZE];
} NCPF_REQ;

/* NCP request with subfunction */
typedef struct NCPFS_REQ_record
{
  u16   packetType;
  u8    sequenceNumber;
  u8    cLow;
  u8    taskNumber;
  u8    cHigh;
  u8    functionNumber;
  u16   subfunctionLength;
  u8    subfunctionNumber;
  u8    packet[VARSIZE];
} NCPFS_REQ;

/* MCP reply */
typedef struct NCP_REP_record
{
  u16   packetType;
  u8    sequenceNumber;
  u8    cLow;
  u8    taskNumber;
  u8    cHigh;
  u8    functionStatus;
  u8    connectionStatus;
  u8    packet[VARSIZE];
} NCP_REP;

/* Buffer size request */
typedef struct NCPF_BUFFER_record
{
  u16  bufferSize;
} NCPF_BUFFER;


/* Plain login request */
typedef struct NCPF_LOGIN_record
{
  u16  objectType;
  u8   data[VARSIZE];

/* It actually is:
  u16  objectType;
  u8   usernameLength;
  char userName[usernameLength];
  u8   userpasswordLength;
  char userPassword[userpasswordLength];
*/

} NCPF_LOGIN;


/* Encrypted login request */
typedef struct NCPF_LOGINE_record
{
  u8         objectPassword[8];
  u16        objectType;
  NCP_STRING objectName;
} NCPF_LOGINE;


/*********************** Local prototypes ******************/
static void NCPSend(unsigned int conn);


/*********************** Local data ***********************/
static NCPCONNECTION  NCP[MAX_NCP_CONNECTIONS];
static NCPHaveMessage *MessageCallback=NULL;
static char           nearestServerName[LIMIT_OBJNAME+1];


/*$*********************************************************
$*
$* Initialize NCP.
$*
$**********************************************************/
int NCPInitialize(NCPHaveMessage *message)
{
bool         Fretval,connOK;
unsigned int x,y;
void         *recvBuf;
void         *sendBuf;
u16          socket;

  Fretval=NCP_ERROR;
  if(IPXInitialize()==IPX_OK)
  {
    MessageCallback=message;
    memset(NCP,0,sizeof(NCP));
    for(x=0;x<MAX_NCP_CONNECTIONS;x++)
    {
      /* For every NCP connection. */
      NCP[x].sequence=0;
      NCP[x].task=0;
      NCP[x].cLow=0xFF;
      NCP[x].cHigh=0xFF;
      NCP[x].requestSent=0;
      NCP[x].serverName=nearestServerName;
      NCP[x].isReply=0;
      connOK=FALSE;
      for(y=0;y<SOCK_PER_NCP;y++)
      {
        /* For every socket on the connection. */
        socket=NCP_BASE_SOCKET+((x*SOCK_PER_NCP)+y);
        if((recvBuf=malloc(MAX_NCPPACKET))!=NULL)
        {
          if((sendBuf=malloc(MAX_NCPPACKET))!=NULL)
          {
            /* We do not really need that much buffer memory on
               watchdog and mail sockets. */
            if(IPXOpenSocket(&socket,IPXSOCK_SHORT)==IPX_OK)
            {
              /* Receive portion */
              NCP[x].recvECB[y].ESRAddress=NULL;
              NCP[x].recvECB[y].socketNumber=Swap16(socket);
              NCP[x].recvECB[y].buffers=2;
              NCP[x].recvECB[y].headerLength=sizeof(IPXHEADER);
              NCP[x].recvECB[y].header=&NCP[x].recvIPX[y];
              NCP[x].recvECB[y].dataLength=MAX_NCPPACKET;
              NCP[x].recvECB[y].data=recvBuf;
              /* Transmit portion */
              NCP[x].sendECB[y].ESRAddress=NULL;
              NCP[x].sendECB[y].socketNumber=Swap16(socket);
              NCP[x].sendECB[y].buffers=2;
              NCP[x].sendECB[y].headerLength=sizeof(IPXHEADER);
              NCP[x].sendECB[y].header=&NCP[x].sendIPX[y];
              NCP[x].sendECB[y].dataLength=0;
              NCP[x].sendECB[y].data=sendBuf;
              NCP[x].sendIPX[y].packetType=PACKET_NCP;
              /* All set */
              connOK=TRUE;
            }
          }
        }
      }
      if(connOK!=TRUE)
      {
        Fretval=NCP_ERROR;
        break;
      }
      else
      {
        /* Post a listen on watchdog/mail sockets. */
        IPXListenForPacket(&NCP[x].recvECB[1]);
        IPXListenForPacket(&NCP[x].recvECB[2]);
        Fretval=NCP_OK;
      }
    }
  }
  else
    fprintf(stderr,"IPX not installed\n");
  return(Fretval);
}


/*$*********************************************************
$*
$* Find the nearest file server using SAP.
$*
$**********************************************************/
bool SAPGetNearestServer(IPXADDRESS *nearestServer)
{
static bool haveNearest=FALSE;
static SAPREP_PACKET sapReply;
static SAPREQ_PACKET sapRequest;
static ECB     reqECB;
static ECB     repECB;
bool    Fretval;
u16     socket;
clock_t now;

  Fretval=FALSE;
  if(haveNearest!=TRUE)
  {
    socket=0;
    if(IPXOpenSocket(&socket,IPXSOCK_SHORT)==IPX_OK)
    {
      memset(&repECB,0,sizeof(repECB));
      memset(&reqECB,0,sizeof(reqECB));

      /* Request portion */
      reqECB.ESRAddress=NULL;
      reqECB.socketNumber=Swap16(socket);
      memset(reqECB.routerAddress,0xFF,sizeof(reqECB.routerAddress));
      memset(sapRequest.requestHeader.destination.network,0x00,
             sizeof(sapRequest.requestHeader.destination.network));
      memset(sapRequest.requestHeader.destination.node,0xFF,
             sizeof(sapRequest.requestHeader.destination.node));
      sapRequest.requestHeader.destination.socket=Swap16(SAP_SOCKET);
      sapRequest.requestHeader.packetType=PACKET_SAP;
      reqECB.buffers=2;
      reqECB.headerLength=sizeof(IPXHEADER);
      reqECB.header=&sapRequest.requestHeader;
      reqECB.dataLength=sizeof(sapRequest.requestData);
      reqECB.data=&sapRequest.requestData;
      sapRequest.requestData.requestType=Swap16(SAP_NEAREST_QUERY);
      sapRequest.requestData.serverType=Swap16(OTYPE_FSERVER);

      /* Reply portion */
      repECB.ESRAddress=NULL;
      repECB.socketNumber=Swap16(socket);
      repECB.buffers=2;
      repECB.headerLength=sizeof(IPXHEADER);
      repECB.header=&sapReply.replyHeader;
      repECB.dataLength=sizeof(sapReply.replyData);
      repECB.data=&sapReply.replyData;

      /* Start listening */
      IPXListenForPacket(&repECB);
      /* Send a request */
      IPXSendPacket(&reqECB);
      while(reqECB.inUse!=0)
        IPXRelinquishControl();
      now=clock();
      /* Wait for a reply */
      while(clock()<(now+90))
      {
        if((repECB.inUse==0)&&(repECB.completionCode==0))
        {
          if((Swap16(sapReply.replyData.replyType)==SAP_NEAREST_RESPONSE)&&
             (Swap16(sapReply.replyData.serverType)==OTYPE_FSERVER)
            )
          {
            memcpy(nearestServer,&sapReply.replyData.serverAddress,
                   sizeof(*nearestServer));
            strncpy(nearestServerName,sapReply.replyData.serverName,
                    LIMIT_OBJNAME);
            haveNearest=TRUE;
            Fretval=TRUE;
            break;
          }
        }
      }
      IPXCloseSocket(socket);
    }
  }
  else
  {
    memcpy(nearestServer,&sapReply.replyData.serverAddress,
           sizeof(*nearestServer));
    Fretval=TRUE;
  }
  return(Fretval);
} /* SAPGetNearestServer */


/*$*********************************************************
$*
$* Create a connection to a file server having the
$* specified address.
$*
$**********************************************************/
void NCPAttachToFileServerWithAddress(int *commandStatus,unsigned int conn,
                                      IPXADDRESS *serverAddress)
{
NCPF_REQ *nr;
u16      ticks; /* Unused */

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    if(IPXGetLocalTarget(serverAddress,
       NCP[conn].sendECB[0].routerAddress,&ticks)==IPX_OK)
    {
      NCP[conn].sequence=0;
      NCP[conn].task=0;
      NCP[conn].cLow=0xFF;
      NCP[conn].cHigh=0xFF;
      NCP[conn].isReply=0;
      *NCP[conn].cmdStatus=NCP_BUSY;
      nr=(NCPF_REQ *)NCP[conn].sendECB[0].data;
      nr->packetType=NCPPACK_OPEN;
      nr->functionNumber=0;
      NCP[conn].sendECB[0].dataLength=sizeof(*nr);
      memcpy(&NCP[conn].sendIPX[0].destination,serverAddress,
             sizeof(NCP[conn].sendIPX[0].destination));
      NCPSend(conn);
    }
  }
} /* NCPAttachToFileServerWithAddress */


/*$********************************************************
$*
$* Negotiate NCP buffer size.
$*
$**********************************************************/
void NCPNegotiateBufferSize(int *commandStatus,unsigned int conn)
{
NCPF_REQ    *nr;
NCPF_BUFFER *nd;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].task=0x99;
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    nr=(NCPF_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_REQUEST;
    nr->functionNumber=0x21;
    nd=(NCPF_BUFFER *)nr->packet;
    nd->bufferSize=Swap16(MAX_NCPPACKET);
    NCP[conn].sendECB[0].dataLength=sizeof(*nr)+sizeof(*nd);
    NCPSend(conn);
  }
  return;
} /* NCPNegotiateBufferSize */


/*$*********************************************************
$*
$* Get an object's bindery ID.
$*
$**********************************************************/
void NCPGetBinderyObjectID(int *commandStatus,unsigned int conn,
                           const char *objectName,u16 objectType)
{
NCPFS_REQ    *nr;
OBJECTID_REQ *oi;
size_t       l1;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    l1=strlen(objectName);
    nr=(NCPFS_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_REQUEST;
    nr->functionNumber=0x17;
    nr->subfunctionNumber=0x35;
    nr->subfunctionLength=Swap16(1+sizeof(*oi)+l1);
    oi=(OBJECTID_REQ *)nr->packet;
    oi->objectType=Swap16(objectType);
    oi->objectNameLength=(u8)l1;
    strcpy(oi->objectName,objectName);
    NCP[conn].sendECB[0].dataLength=sizeof(*nr)+sizeof(*oi)+l1;
    NCPSend(conn);
  }
  return;
} /* NCPGetBinderyObject */


/*$*********************************************************
$*
$* Get encryption key.
$*
$**********************************************************/
void NCPGetEncryptionKey(int *commandStatus,unsigned int conn)
{
NCPFS_REQ  *nr;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    nr=(NCPFS_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_REQUEST;
    nr->functionNumber=0x17;
    nr->subfunctionNumber=0x17;
    nr->subfunctionLength=Swap16(1);
    NCP[conn].sendECB[0].dataLength=sizeof(*nr);
    NCPSend(conn);
  }
} /* NCPGetEncryptionKey */


/*$*********************************************************
$*
$* Log in an object to a file server. The encrypted
$* version.
$*
$**********************************************************/
void NCPLoginObjectEncrypted(int *commandStatus,unsigned int conn,
                             const char *userName,u16 objectType,
                             u8 *key)
{
NCPFS_REQ   *nr;
NCPF_LOGINE *nd;
size_t      l1;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    l1=strlen(userName);
    nr=(NCPFS_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_REQUEST;
    nr->functionNumber=0x17;
    nr->subfunctionNumber=0x18;
    nr->subfunctionLength=Swap16(1+sizeof(*nd)+l1);
    nd=(NCPF_LOGINE *)nr->packet;
    nd->objectType=Swap16(objectType);
    memcpy(nd->objectPassword,key,sizeof(nd->objectPassword));
    nd->objectName.length=(u8)l1;
    strcpy(nd->objectName.string,userName);
    NCP[conn].sendECB[0].dataLength=sizeof(*nr)+sizeof(*nd)+l1;
    NCPSend(conn);
  }
  return;
} /* NCPLoginObjectEncrypted */


/*$********************************************************
$*
$* Log in an object to a file server. The unencrypted
$* version.
$*
$**********************************************************/
void NCPLoginObject(int *commandStatus,unsigned int conn,
                    u16 objectType,const char *userName,
                    const char *userPassword)
{
NCPFS_REQ  *nr;
NCPF_LOGIN *nd;
NCP_STRING *ns;
size_t     l1,l2;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    l1=strlen(userName);
    l2=strlen(userPassword);
    nr=(NCPFS_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_REQUEST;
    nr->functionNumber=0x17;
    nr->subfunctionNumber=0x14;
    nr->subfunctionLength=Swap16(1+sizeof(*nd)+2+l1+l2);
    nd=(NCPF_LOGIN *)nr->packet;
    nd->objectType=Swap16(objectType);
    ns=(NCP_STRING *)&nd->data[0];
    ns->length=(u8)l1;
    strcpy(ns->string,userName);
    ns=(NCP_STRING *)&nd->data[l1+1];
    ns->length=(u8)l2;
    strcpy(ns->string,userPassword);
    (void)strupr(ns->string);
    NCP[conn].sendECB[0].dataLength=sizeof(*nr)+sizeof(*nd)+2+l1+l2;
    NCPSend(conn);
  }
  return;
} /* NCPLoginObject */


/*$*********************************************************
$*
$* Logout from a file server.
$*
$**********************************************************/
void NCPLogoutFromFileServer(int *commandStatus,unsigned int conn)
{
NCPF_REQ    *nr;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    nr=(NCPF_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_REQUEST;
    nr->functionNumber=0x19;
    NCP[conn].sendECB[0].dataLength=sizeof(*nr);
    NCPSend(conn);
  }
  return;
} /* NCPLogoutFromFileServer */


/*$*********************************************************
$*
$* Enable user messages.
$*
$**********************************************************/
void NCPEnableUserMessages(int *commandStatus,unsigned int conn)
{
NCPFS_REQ    *nr;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    nr=(NCPFS_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_REQUEST;
    nr->subfunctionLength=Swap16(1);
    nr->functionNumber=0x15;
    nr->subfunctionNumber=0x3;
    NCP[conn].sendECB[0].dataLength=sizeof(*nr);
    NCPSend(conn);
  }
  return;
} /* NCPEnableUserMessages */


/*$*********************************************************
$*
$* Break the connection.
$*
$**********************************************************/
void NCPDetachFromFileServer(int *commandStatus,unsigned int conn)
{
NCPF_REQ    *nr;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    nr=(NCPF_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_CLOSE;
    nr->functionNumber=0x00;
    NCP[conn].sendECB[0].dataLength=sizeof(*nr);
    NCPSend(conn);
  }
  return;
} /* NCPDetachFromFileServer */


/*$*********************************************************
$*
$* Server has some message for us, read it.
$*
$**********************************************************/
void NCPGetBroadcastMessage(int *commandStatus,unsigned int conn)
{
NCPFS_REQ  *nr;

  *commandStatus=NCP_ERROR;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    NCP[conn].cmdStatus=commandStatus;
    *NCP[conn].cmdStatus=NCP_BUSY;
    nr=(NCPFS_REQ *)NCP[conn].sendECB[0].data;
    nr->packetType=NCPPACK_REQUEST;
    nr->functionNumber=0x15;
    nr->subfunctionNumber=0x01;
    nr->subfunctionLength=Swap16(1);
    NCP[conn].sendECB[0].dataLength=sizeof(*nr);
    NCPSend(conn);
  }
  return;
} /* NCPGetBroadcastMessage */


/*$*********************************************************
$*
$* Given local connection #, return file server connection #.
$*
$**********************************************************/
u16 NCPGetServerConnection(unsigned int conn)
{
u16 Fretval;

  Fretval=0xFFFF;
  if(conn<MAX_NCP_CONNECTIONS)
    Fretval=NCP[conn].cLow+(NCP[conn].cHigh<<8);
  return(Fretval);
}


/*$*********************************************************
$*
$* Given local connection #, return file server name.
$*
$**********************************************************/
char *NCPGetServerName(unsigned int conn)
{
char *Fretval;

  Fretval=NULL;
  if(conn<MAX_NCP_CONNECTIONS)
    Fretval=NCP[conn].serverName;
  return(Fretval);
}


/*$*********************************************************
$*
$* Send a NCP packet.
$*
$**********************************************************/
static void NCPSend(unsigned int conn)
{
NCPF_REQ *nr;

  nr=(NCPF_REQ *)NCP[conn].sendECB[0].data;
  nr->sequenceNumber=NCP[conn].sequence;
  nr->taskNumber=NCP[conn].task;
  nr->cLow=NCP[conn].cLow;
  nr->cHigh=NCP[conn].cHigh;
  NCP[conn].requestSent=1;
  IPXListenForPacket(&NCP[conn].recvECB[0]);
  IPXSendPacket(&NCP[conn].sendECB[0]);
  while(NCP[conn].sendECB[0].inUse!=0)
    IPXRelinquishControl();
  return;
} /* NCPSend */


/*$*********************************************************
$*
$* Receive a NCP packet.
$* A minimal implementation: no retry/timeout logic, no
$* reply length checking, error handling etc.
$*
$**********************************************************/
void NCPPoll(unsigned int conn)
{
NCP_REP *nr;

  if(conn<MAX_NCP_CONNECTIONS)
  {
    if(NCP[conn].requestSent)
    {
      /* We are waiting for a response. */
      if((NCP[conn].recvECB[0].inUse==0)&&
         (NCP[conn].recvECB[0].completionCode==0))
      {
        NCP[conn].requestSent=0;
        nr=(NCP_REP *)NCP[conn].recvECB[0].data;
        if((nr->packetType==NCPPACK_REPLY)&&
           (NCP[conn].sequence==nr->sequenceNumber)
          )
        {
          /* Let the app know that a response arrived. */
          *NCP[conn].cmdStatus=nr->functionStatus;
          NCP[conn].isReply=1;
          NCP[conn].sequence++;
          NCP[conn].cLow=nr->cLow;
          NCP[conn].cHigh=nr->cHigh;
          if(nr->connectionStatus&0x0F)
          {
            /* Map connection error to a generic error code. */
            *NCP[conn].cmdStatus=NCP_ERROR;
            MessageCallback(conn,1);
          }
          else if(nr->connectionStatus&0x40)
            MessageCallback(conn,0);  /* Server has a message. */
          else if(nr->connectionStatus&0x10)
            MessageCallback(conn,1); /* Server down. */
        }
      }
    }

    /* Take care of watchdog packets. */
    if((NCP[conn].recvECB[1].inUse==0)&&
       (NCP[conn].recvECB[1].completionCode==0))
    {
      if(((char *)NCP[conn].recvECB[1].data)[1]=='?')
      {
        char *p=(char *)NCP[conn].sendECB[1].data;
        p[0]=((char *)NCP[conn].recvECB[1].data)[0];
        p[1]='Y';
        NCP[conn].sendECB[1].dataLength=2;
        /* Set the router node. */
        memcpy(NCP[conn].sendECB[1].routerAddress,
               NCP[conn].recvECB[1].routerAddress,
               sizeof(NCP[conn].sendECB[1].routerAddress)
              );
        /* Set IPX address of the receiver. */
        memcpy(&NCP[conn].sendIPX[1].destination,
               &NCP[conn].recvIPX[1].source,
               sizeof(NCP[conn].sendIPX[1].destination)
              );
        /* Respond with the same packet type. */
        NCP[conn].sendIPX[1].packetType=NCP[conn].recvIPX[1].packetType;
        IPXSendPacket(&NCP[conn].sendECB[1]);
        while(NCP[conn].sendECB[1].inUse!=0)
          IPXRelinquishControl();
      }
      IPXListenForPacket(&NCP[conn].recvECB[1]);
    }

    /* Check the mail socket. */
    if((NCP[conn].recvECB[2].inUse==0)&&
       (NCP[conn].recvECB[2].completionCode==0))
    {
      if((((char *)NCP[conn].recvECB[2].data)[1]=='!')&&
         (MessageCallback!=NULL)
        )
        MessageCallback(conn,0);
      IPXListenForPacket(&NCP[conn].recvECB[2]);
    }
  }
  return;
} /* NCPReceive */


/*$*********************************************************
$*
$* Return a pointer to connection's NCP reply buffer.
$*
$**********************************************************/
void *NCPGetReplyBuffer(unsigned int conn,unsigned int *replyLength)
{
void         *Fretval;
unsigned int r;

  Fretval=NULL;
  *replyLength=0;
  if(conn<MAX_NCP_CONNECTIONS)
  {
    if((!NCP[conn].requestSent)&&(NCP[conn].isReply))
    {
      NCP[conn].isReply=0;
      Fretval=(void *)(((NCP_REP *)NCP[conn].recvECB[0].data)->packet);
      r=Swap16(NCP[conn].recvIPX[0].length);
      if(r>(sizeof(IPXHEADER)+sizeof(NCP_REP)))
        *replyLength=r-(sizeof(IPXHEADER)+sizeof(NCP_REP));
    }
  }
  return(Fretval);
} /* NCPGetReplyBuffer */


/*$*********************************************************
$*
$* End using NCP.
$*
$**********************************************************/
void NCPTerminate(void)
{
int x;

  for(x=0;x<(MAX_NCP_CONNECTIONS*SOCK_PER_NCP);x++)
  {
    /* Need to close shortlived sockets in case
       no shell is loaded. */
    IPXCloseSocket((u16)(NCP_BASE_SOCKET+x));
  }
  IPXTerminate();
} /* NCPTerminate */


