/*|*********************************************************
|* 'ncptest.c'
|*
|* DOS NCP test.
|*
|* -93     PS    Created
|*
|* Shows how to access services of a NetWare file server.
|* IPX has to be loaded, but the presence of the
|* NetWare shell 'net?.exe', 'vlm.exe' or similar is
|* not required (although it won't do any harm if it
|* is loaded).
|*
|* The following files constitute this program:
|* 'ncptest.c' - main loop
|* 'ipxint.c' - DOS IPX interface, in C.
|* 'ncpint.c' - NCP interface.
|* 'ncptest.h' and 'ipxint.h' - header files
|*
|* Compile using Borland C (BC 3.1 and TC 2.0 tested). Use large
|* memory model, packed structures and unsigned characters. Run
|* under plain DOS.
|*
|* 'ncptest' creates 'MAX_NCP_CONNECTIONS' connections to the
|* nearest NetWare file server. You are logged in on every
|* connection, using the user name and password you supply
|* on the command line. After a successful login, the program
|* sits in a loop waiting for you to press a key to terminate
|* a connection. At this point, the program will also accept
|* messages from other workstations and show them on the screen.
|* If a message contains the string 'logout', the corresponding
|* connection will log out from the file server and close the
|* connection. Use the command 'send "message" to server/user'
|* to send a message to all connections and 'send "message"
|* to server/connection' to send a message to the specified
|* connection only.
|* During operation,the program shows the local connection
|* number, server connection number, local connection states,
|* your object ID and file server name for every connection.
|* Messages received from other workstations will be shown
|* on the screen.
|* Watchdog packets are responded to.
|* Encrypted passwords are supported (source from March -93
|* issue of Byte).
|*
|* Use FCONSOLE or USERLIST to verify that the connections
|* have been created.
|*
|* Make sure your file server responds to SAP nearest server
|* queries (it normally does); type 'set Reply To Get Nearest
|* Server=on' at the server console.
|*
|* You may need to increase the number of sockets IPX can
|* handle by setting 'IPX SOCKETS' in 'shell.cfg' or 'net.cfg'
|* to a larger value.
|*
|**********************************************************
|*
|* Limitations:
|*
|* You cannot control which file server will be used. The nearest
|* server is the server whose SAP response arrives first to us.
|*
|* Defining more than 10 connections will mess up the screen,
|* since the lower part of it is used for showing messages
|* received by NCP. The program assumes 80*25 screen.
|*
|* IPX sockets are allocated starting at 'NCP_BASE_SOCKET'.
|* If other applications are using sockets in the same
|* range, 'ncptest' will fail.
|*
|* There is no retranmission/timeout logic implemented (in
|* 'NCPPoll()'). If a packet, either a request or reply, never
|* reaches the destination, the corresponding connection will
|* continue waiting forever. If it happens, press Ctrl/Break.
|*
|* Above all: this is not a production quality software upon
|* which you could build a real NCP engine. You may, however,
|* find it useful for testing various NCP commands.
|*
|**********************************************************
|*
|* 1993, Pawel Szczerbina
|*
|**********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <conio.h>
#include "ipxint.h"
#include "ncptest.h"


/******************* Data types ***************************/
typedef unsigned char buf32[32];
typedef unsigned char buf16[16];
typedef unsigned char buf8[8];
typedef unsigned char buf4[4];
typedef int      word;

/******************* Local data ***************************/
static const      char *userName;
static const      char *userPassword;
static IPXADDRESS nearestServer;
static NCPINFO    NCPInfo[MAX_NCP_CONNECTIONS];

/******************* Local prototypes *********************/
static bool NCPTest(void);
static void NCPTestShowStatus(void);
static char *NCPTestStateToName(TESTSTATE state);
static void NCPTestMessageCallback(unsigned int conn,int msgType);
static void NCPTestShowMessage(unsigned int conn,char *message);
static void NCPTestShowError(unsigned int conn,int errorCode);
static int  breakHandler(void);
static bool breakFlag=FALSE;
static void NCPTestHandleMessage(unsigned int conn,void *reply,
                                  unsigned int replyLength);
static void NCPTestHandleObjectID(unsigned int conn,void *reply,
                                  unsigned int replyLength);
static void NCPTestHandleGetKey(unsigned int conn,void *reply,
                                unsigned int replyLength);
static void encrypt(char *fra,char *buf,char *til);
static void shuffle(char *lon, const char *buf, word buflen, char *target);
static void shuffle1(buf32 temp, char *target);


/*$*********************************************************
$*
$* Entry point.
$*
$***********************************************************/
void main(int argc,char *argv[])
{
int brkState;

  if(argc==3)
  {
    userName=argv[1];
    (void)strupr(argv[2]);
    userPassword=argv[2];
    if(NCPInitialize(NCPTestMessageCallback)==NCP_OK)
    {
      ctrlbrk(breakHandler);
      brkState=getcbrk();
      (void)setcbrk(1);
      while((NCPTest()==TRUE)&&(breakFlag==FALSE))
      {
        /* Do other things. */
        ;
      }
      gotoxy(1,24);
      NCPTerminate();
      (void)setcbrk(brkState);
    }
    else
      fprintf(stderr,"Unable to initialize NCP.\n");
  }
  else
    fprintf(stderr,"usage: %s <user> <password>.\n",argv[0]);
  return;
} /* main */


/*$*********************************************************
$*
$* Simple NCP test.
$*
$**********************************************************/
static bool NCPTest()
{
static unsigned int   done;
bool         Fretval;
unsigned int x;
buf32        buf;
u32          tmpID;

  Fretval=TRUE;
  for(x=0;x<MAX_NCP_CONNECTIONS;x++)
  {
    switch(NCPInfo[x].state)
    {
      case TS_INIT:
        done=0;
        NCPInfo[x].attached=FALSE;
        NCPInfo[x].messagePresent=FALSE;
        NCPInfo[x].connectionDown=FALSE;
        NCPInfo[x].state=TS_GET_NEAREST_SERVER;
        NCPInfo[x].replyHandler=NULL;
        NCPInfo[x].objectID=0xFFFFFFFFL;
        break;

      case TS_GET_NEAREST_SERVER:
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].nextState=TS_DONE;
        if(SAPGetNearestServer(&nearestServer)==TRUE)
          NCPInfo[x].state=TS_ATTACH;
        break;

      case TS_ATTACH:
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_NEGOTIATE_BUFFER_SIZE;
        NCPAttachToFileServerWithAddress(&NCPInfo[x].commandStatus,
                                         x,&nearestServer);
        break;

      case TS_NEGOTIATE_BUFFER_SIZE:
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].attached=TRUE;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_GETKEY;
        /* We ignore the buffer size. */
        NCPNegotiateBufferSize(&NCPInfo[x].commandStatus,x);
        break;

      case TS_GETKEY:
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_OBJECTID;
        NCPInfo[x].replyHandler=NCPTestHandleGetKey;
        NCPGetEncryptionKey(&NCPInfo[x].commandStatus,x);
        break;

      case TS_OBJECTID:
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_LOGIN;
        NCPInfo[x].replyHandler=NCPTestHandleObjectID;
        NCPGetBinderyObjectID(&NCPInfo[x].commandStatus,x,userName,
                              OTYPE_USER);
        break;

      case TS_LOGIN:
        tmpID=Swap32(NCPInfo[x].objectID);
        shuffle((u8 *)&tmpID,userPassword,(int)strlen(userPassword),buf);
        encrypt(NCPInfo[x].encKey,buf,NCPInfo[x].encKey);
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_ENABLEMSG;
        NCPLoginObjectEncrypted(&NCPInfo[x].commandStatus,x,userName,
                                OTYPE_USER,NCPInfo[x].encKey);

#if 0   /* The plain version */
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_IDLE;
        NCPLoginObject(&NCPInfo[x].commandStatus,x,OTYPE_USER,userName,
                       userPassword);
#endif

        break;

      case TS_ENABLEMSG:
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_IDLE;
        NCPEnableUserMessages(&NCPInfo[x].commandStatus,x);
        break;

      case TS_MESSAGE:
        NCPInfo[x].messagePresent=FALSE;
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_IDLE;
        NCPInfo[x].nextState=TS_IDLE;
        break;

      case TS_IDLE:
        if(NCPInfo[x].messagePresent==TRUE)
        {
          NCPInfo[x].messagePresent=FALSE;
          NCPInfo[x].prevState=NCPInfo[x].state;
          NCPInfo[x].state=TS_WAIT;
          NCPInfo[x].nextState=TS_MESSAGE;
          NCPInfo[x].replyHandler=NCPTestHandleMessage;
          NCPGetBroadcastMessage(&NCPInfo[x].commandStatus,x);
        }
        break;

      case TS_LOGOUT:
        NCPLogoutFromFileServer(&NCPInfo[x].commandStatus,x);
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_CLOSE;
        break;

      case TS_CLOSE:
        NCPInfo[x].attached=FALSE;
        NCPDetachFromFileServer(&NCPInfo[x].commandStatus,x);
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_WAIT;
        NCPInfo[x].nextState=TS_DONE;
        break;

      case TS_WAIT:
        switch(NCPInfo[x].commandStatus)
        {
          void         *reply;
          unsigned int replyLength;

          case NCP_OK:
            /* Command executed OK. */
            NCPInfo[x].state=NCPInfo[x].nextState;
            if((NCPInfo[x].replyHandler!=NULL)&&
               ((reply=NCPGetReplyBuffer(x,&replyLength))!=NULL)
              )
            {
              NCPInfo[x].replyHandler(x,reply,replyLength);
              NCPInfo[x].replyHandler=NULL;
            }
            break;

          case NCP_BUSY:
            /* Command still in progress */
            break;

          default:
            /* Error occured. */
            NCPInfo[x].replyHandler=NULL;
            NCPInfo[x].messagePresent=FALSE;
            if(NCPInfo[x].connectionDown==TRUE)
              NCPInfo[x].state=TS_DONE;
            else
            {
              NCPTestShowError(x,NCPInfo[x].commandStatus);
              NCPInfo[x].state=TS_DONE;
              if(NCPInfo[x].attached==TRUE)
                NCPInfo[x].state=TS_CLOSE;
            }
            break;
        }
        break;

      case TS_DONE:
        NCPInfo[x].attached=FALSE;
        NCPInfo[x].prevState=NCPInfo[x].state;
        NCPInfo[x].state=TS_DEAD;
        NCPInfo[x].nextState=TS_DEAD;
        done++;
        break;

      case TS_DEAD:
        break;

      default:
        break;
    }
    NCPTestShowStatus();
    NCPPoll(x);
    if((kbhit()!=0)&&(NCPInfo[x].state==TS_IDLE))
    {
      if(!getch())
        (void)getch();
      NCPInfo[x].state=TS_DONE;
      if(NCPInfo[x].attached==TRUE)
        NCPInfo[x].state=TS_LOGOUT;
    }
  }
  if(done>=MAX_NCP_CONNECTIONS)
    Fretval=FALSE;
  return(Fretval);
} /* NCPTest */


/*$*********************************************************
$*
$* Called by NCP when a message arrives or a connection
$* goes down.
$*
$**********************************************************/
static void NCPTestMessageCallback(unsigned int conn,int msgType)
{
  if(conn<MAX_NCP_CONNECTIONS)
    switch(msgType)
    {
      case 0:  NCPInfo[conn].messagePresent=TRUE; break;
      case 1:  NCPInfo[conn].connectionDown=TRUE;  break;
      default: break;
    }
} /* NCPMessageCallback */


/*$*********************************************************
$*
$* Takes care of GetBroadcastMessage reply.
$*
$**********************************************************/
static void NCPTestHandleMessage(unsigned int conn,void *reply,
                                 unsigned int replyLength)
{
NCP_STRING *msg;

  msg=(NCP_STRING *)reply;
  if(replyLength>1 && msg->length)
    {
      msg->string[msg->length]=0;
      (void)strupr(msg->string);
      if(strstr(msg->string,TXT_LOGOUT)!=NULL)
        NCPInfo[conn].state=TS_LOGOUT;
      NCPTestShowMessage(conn,msg->string);
    }
} /* NCPTestHandleMessage */


/*$*********************************************************
$*
$* Takes care of GetBinderyObjectID reply
$*
$**********************************************************/
static void NCPTestHandleObjectID(unsigned int conn,void *reply,
                                  unsigned int replyLength)
{
OBJECTID_REP *or;

  or=(OBJECTID_REP *)reply;
  if(replyLength>=sizeof(or->objectID))
    NCPInfo[conn].objectID=Swap32(or->objectID);
} /* NCPTestHandleObjectID */


/*$*********************************************************
$*
$* Takes care of GetEncryptionKey reply
$*
$**********************************************************/
static void NCPTestHandleGetKey(unsigned int conn,void *reply,
                                unsigned int replyLength)
{
  if(replyLength>=sizeof(NCPInfo[conn].encKey))
    memcpy(NCPInfo[conn].encKey,reply,sizeof(NCPInfo[conn].encKey));
} /* NCPTestHandleEncryptionKey */


/*$*********************************************************
$*
$* Show connection status.
$*
$**********************************************************/
static void NCPTestShowStatus(void)
{
static   bool first=TRUE;
unsigned int  x;

  if(first==TRUE)
  {
    window(1,1,80,25);
    clrscr();
    gotoxy(1,1);
    cprintf("Local   Server    PrvState   State  NextState  ObjectID   "
		    "ServerName");
    first=FALSE;
  }
/*   _setcursortype(_NOCURSOR); */
  for(x=0;x<MAX_NCP_CONNECTIONS;x++)
  {
    gotoxy(1,3+(int)x);
    cprintf("%05d   %05d %10s%10s%10s   %08lX   %s",
            x,
            (unsigned int)NCPGetServerConnection(x),
            NCPTestStateToName(NCPInfo[x].prevState),
            NCPTestStateToName(NCPInfo[x].state),
            NCPTestStateToName(NCPInfo[x].nextState),
            (unsigned long)NCPInfo[x].objectID,
            NCPGetServerName(x)
          );
  }
/*  _setcursortype(_NORMALCURSOR); */
} /* NCPTestShowStatus */


/*$*********************************************************
$*
$* Show message received by NCP.
$*
$**********************************************************/
static void NCPTestShowMessage(unsigned int conn,char *msg)
{
  gotoxy(1,14+(int)conn);
  clreol();
  cprintf("Connection %d: %s",conn,msg);
} /* NCPTestShowMessage */


/*$*********************************************************
$*
$* Show NCP error.
$*
$**********************************************************/
static void NCPTestShowError(unsigned int conn,int errorCode)
{
  gotoxy(1,25);
  clreol();
  highvideo();
  cprintf("\007Error %d on local connection %d - Press a key...",
          errorCode,conn);
  normvideo();
  while(!kbhit())
    ;
  gotoxy(1,25);
  clreol();
  if(!getch())
    (void)getch();
} /* NCPTestShowError */


/*$*********************************************************
$*
$* Convert connection state to name.
$*
$**********************************************************/
static char *NCPTestStateToName(TESTSTATE state)
{
  switch(state)
  {
    case TS_INIT:						return("Init");
    case TS_GET_NEAREST_SERVER:			return("Nearest");
    case TS_ATTACH:						return("Attach");
    case TS_NEGOTIATE_BUFFER_SIZE:      return("Negotiate");
    case TS_LOGIN:						return("Login");
    case TS_OBJECTID:					return("ObjectID");
    case TS_GETKEY:						return("GetKey");
    case TS_ENABLEMSG:					return("EnableMsg");
    case TS_LOGOUT:						return("Logout");
    case TS_CLOSE:						return("Close");
    case TS_WAIT:						return("Wait");
    case TS_IDLE:						return("Idle");
    case TS_MESSAGE:					return("Message");
    case TS_DONE:						return("Done");
    case TS_DEAD:						return("Dead");
    default:							return("UNKNOWN !");
  }
} /* NCPStateToName */


/*$*********************************************************
$*
$* Set the break flag if Ctrl/Break was pressed.
$*
$**********************************************************/
static int  breakHandler(void)
{
  breakFlag=TRUE;
  return(1);
} /* breakHandler */


/*$*********************************************************
$*
$* Password encryption routines follow.
$* Converted to C from Barry Nance's Pascal
$* prog published in the March -93 issue of Byte.
$*
$**********************************************************/

static u8 encrypttable[256] =
{0x7,0x8,0x0,0x8,0x6,0x4,0xE,0x4,0x5,0xC,0x1,0x7,0xB,0xF,0xA,0x8,
 0xF,0x8,0xC,0xC,0x9,0x4,0x1,0xE,0x4,0x6,0x2,0x4,0x0,0xA,0xB,0x9,
 0x2,0xF,0xB,0x1,0xD,0x2,0x1,0x9,0x5,0xE,0x7,0x0,0x0,0x2,0x6,0x6,
 0x0,0x7,0x3,0x8,0x2,0x9,0x3,0xF,0x7,0xF,0xC,0xF,0x6,0x4,0xA,0x0,
 0x2,0x3,0xA,0xB,0xD,0x8,0x3,0xA,0x1,0x7,0xC,0xF,0x1,0x8,0x9,0xD,
 0x9,0x1,0x9,0x4,0xE,0x4,0xC,0x5,0x5,0xC,0x8,0xB,0x2,0x3,0x9,0xE,
 0x7,0x7,0x6,0x9,0xE,0xF,0xC,0x8,0xD,0x1,0xA,0x6,0xE,0xD,0x0,0x7,
 0x7,0xA,0x0,0x1,0xF,0x5,0x4,0xB,0x7,0xB,0xE,0xC,0x9,0x5,0xD,0x1,
 0xB,0xD,0x1,0x3,0x5,0xD,0xE,0x6,0x3,0x0,0xB,0xB,0xF,0x3,0x6,0x4,
 0x9,0xD,0xA,0x3,0x1,0x4,0x9,0x4,0x8,0x3,0xB,0xE,0x5,0x0,0x5,0x2,
 0xC,0xB,0xD,0x5,0xD,0x5,0xD,0x2,0xD,0x9,0xA,0xC,0xA,0x0,0xB,0x3,
 0x5,0x3,0x6,0x9,0x5,0x1,0xE,0xE,0x0,0xE,0x8,0x2,0xD,0x2,0x2,0x0,
 0x4,0xF,0x8,0x5,0x9,0x6,0x8,0x6,0xB,0xA,0xB,0xF,0x0,0x7,0x2,0x8,
 0xC,0x7,0x3,0xA,0x1,0x4,0x2,0x5,0xF,0x7,0xA,0xC,0xE,0x5,0x9,0x3,
 0xE,0x7,0x1,0x2,0xE,0x1,0xF,0x4,0xA,0x6,0xC,0x6,0xF,0x4,0x3,0x0,
 0xC,0x0,0x3,0x6,0xF,0x8,0x7,0xB,0x2,0xD,0xC,0x6,0xA,0xA,0x8,0xD};

static buf32 encryptkeys  =
{0x48,0x93,0x46,0x67,0x98,0x3D,0xE6,0x8D,
 0xB7,0x10,0x7A,0x26,0x5A,0xB9,0xB1,0x35,
 0x6B,0x0F,0xD5,0x70,0xAE,0xFB,0xAD,0x11,
 0xF4,0x47,0xDC,0xA7,0xEC,0xCF,0x50,0xC0};


static void shuffle1(buf32 temp, char *target)
{
buf16 *t = (buf16*)target;
word  b4;
unsigned char b3;
word  s;
word  b2;
word  i;

  b4 = 0;
  {
  int UPPER_LIMIT3 = 1;
  for (b2 = 0; b2 <= UPPER_LIMIT3; ++b2)
     {
     {
     int UPPER_LIMIT4 = 31;
     for (s = 0; s <= UPPER_LIMIT4; ++s)
        {
        b3 = (((temp[s] + b4) ^ (temp[((s + b4) & 31)]
             - encryptkeys[s])));
        b4 = b4 + b3;
        temp[s] = b3;
        }
     s--;
     }
     }
  b2--;
  }
  {
  int UPPER_LIMIT5 = 15;
  for (i = 0; i <= UPPER_LIMIT5; ++i)
     (*t)[i] = (encrypttable[temp[((unsigned)i << 1)]] |
((encrypttable[temp[((unsigned)i << 1)
     + 1]] << 4)));
  i--;
  }
}


static void shuffle(char *lon, const char *buf, word buflen, char *target)
{
buf4 *l = (buf4*)lon;
typedef unsigned char INTER_ARR7[128];
INTER_ARR7 *b = (INTER_ARR7*)buf;
word b2;
buf32 temp;
word s;
word d;

  if (buflen > 0)
     while (((buflen > 0) && ((*b)[buflen - 1] == 0)))
        buflen = buflen - 1;
  memset(temp,0,sizeof(temp));
  d = 0;
  while (buflen >= 32)
     {
     {
     int UPPER_LIMIT6 = 31;
     for (s = 0; s <= UPPER_LIMIT6; ++s)
        {
        temp[s] = (temp[s] ^ (*b)[d]);
        d = d + 1;
        }
     s--;
     }
     buflen = buflen - 32;
     }
  b2 = d;
  if (buflen > 0)
     {
     {
     int UPPER_LIMIT7 = 31;
     for (s = 0; s <= UPPER_LIMIT7; ++s)
        {
        if (d + buflen == b2)
           {
           b2 = d;
           temp[s] = (temp[s] ^ encryptkeys[s]);
           }
        else
           {
           temp[s] = (temp[s] ^ (*b)[b2]);
           b2 = b2 + 1;
           }
        }
     s--;
     }
     }
  {
  int UPPER_LIMIT8 = 31;
  for (s = 0; s <= UPPER_LIMIT8; ++s)
     temp[s] = (temp[s] ^ (*l)[(s & 3)]);
  s--;
  }
  shuffle1(temp,&((*target)));
}


static void encrypt(char *fra,char *buf,char *til)
{
buf8 *f = (buf8*)fra;
buf8 *t = (buf8*)til;
buf32 k;
word s;

  shuffle(&((*f)[0]),&((*buf)),16,&(k[0]));
  shuffle(&((*f)[4]),&((*buf)),16,&(k[16]));
  {
  int UPPER_LIMIT9 = 15;
  for (s = 0; s <= UPPER_LIMIT9; ++s)
     k[s] = (k[s] ^ k[31 - s]);
  s--;
  }
  {
  int UPPER_LIMIT10 = 7;
  for (s = 0; s <= UPPER_LIMIT10; ++s)
     (*t)[s] = (k[s] ^ k[15 - s]);
  s--;
  }
}
