/***********************************************************************/
/* COMM4.C - Commands P-S                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * 36 David Road                 Phone:                    +617 3849 7731
 * Holland Park                  http://www.gu.edu.au/gwis/the/markh.html
 * Brisbane                      **** Maintainer PDCurses & REXX/SQL ****
 * QLD 4121                      ************* Author of THE ************
 * Australia                     ************* Member RexxLA ************
 */

/*
$Id: comm4.c 2.1 1995/06/24 16:28:56 MH Rel MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

#ifdef UNIX
#include <signal.h>
#endif

/*#define DEBUG 1*/

/*man-start*********************************************************************
COMMAND
     print - send text to default printer or print spooler

SYNTAX
     PRint [target] [n]
     PRint LINE [text]
     PRint STRING [text]
     PRint FORMfeed
     PRint CLOSE

DESCRIPTION
     The PRINT command writes a portion of the current file to the default
     printer or print spooler, or text entered on the command line.
  
     PRINT [target] [n]
        Sends text from the file contents up to the target to the printer
        followed by a CR/LF (DOS) or LF(UNIX) after each line.
        When [n] is specified, this sends a formfeed after [n] successive
        lines of text.
     PRINT LINE [text]
        Sends the remainder of the text on the command line to the printer
        followed by a CR/LF (DOS) or LF(UNIX).
     PRINT STRING [text]
        Sends the remainder of the text on the command line to the printer
        without any trailing line terminator.
     PRINT FORMFEED
         Sends a formfeed (^L) character to the printer.
     PRINT CLOSE
         Closes the printer spooler.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     PRINTER

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Print(CHARTYPE *params)
#else
short Print(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define PRT_PARAMS  2
 CHARTYPE *word[PRT_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"10";
 unsigned short num_params=0;
 short page_break=0;
 short rc=RC_OK;
 short target_type=TARGET_NORMAL|TARGET_ALL|TARGET_BLOCK_CURRENT|TARGET_SPARE;
 TARGET target;
#if defined(UNIX)
 CHARTYPE *line_term = (CHARTYPE *)"\n";
#else
 CHARTYPE *line_term = (CHARTYPE *)"\n\r";
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Print");
#endif
/*---------------------------------------------------------------------*/
/* Split parameters up...                                              */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,PRT_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (CHARTYPE *)"1";
   }
/*---------------------------------------------------------------------*/
/* If first argument is LINE...                                        */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"line",word[0],4))
   {
    print_line(FALSE,0L,0L,0,(CHARTYPE *)word[1],line_term);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If first argument is STRING...                                      */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"string",word[0],5))
   {
    print_line(FALSE,0L,0L,0,(CHARTYPE *)word[1],(CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If first argument is FORMFEED...                                    */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"formfeed",word[0],4))
   {
    if (num_params > 1)
      {
       display_error(1,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    print_line(FALSE,0L,0L,0,(CHARTYPE *)"",(CHARTYPE *)"\f");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If first argument is CLOSE...                                       */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"close",word[0],5))
   {
    if (num_params > 1)
      {
       display_error(1,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    print_line(TRUE,0L,0L,0,(CHARTYPE *)"",(CHARTYPE *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* ...treat all other options as targets...                            */
/*---------------------------------------------------------------------*/
 initialise_target(&target);
 if ((rc = validate_target(params,&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 if (target.spare == (-1))
    page_break = 0;
 else
   {
    if (!valid_positive_integer(strtrunc(target.rt[target.spare].string)))
      {
       display_error(4,word[0],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    page_break = atoi(strtrunc(target.rt[target.spare].string));
   }
 print_line(FALSE,target.true_line,target.num_lines,page_break,(CHARTYPE *)"",line_term);
 free_target(&target);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     put - write part of a file to another

SYNTAX
     PUT [target] [fileid]

DESCRIPTION
     The PUT command writes a portion of the current file to another
     file, either explicit or temporary.
     When no fileid is supplied the temporary file is overwritten.
     When a fileid is supplied the portion of the file written out
     is appended to the specified file.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     PUTD, GET

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Put(CHARTYPE *params)
#else
short Put(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Put");
#endif
 rc = execute_put(params,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     putd - write part of a file to another and delete

SYNTAX
     PUTD [target] [fileid]

DESCRIPTION
     The PUTD command writes a portion of the current file to another
     file, either explicit or temporary and deletes those lines written.
     When no fileid is supplied the temporary file is overwritten.
     When a fileid is supplied the portion of the file written out
     is appended to the specified file.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     PUT, GET

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Putd(CHARTYPE *params)
#else
short Putd(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Putd");
#endif
 rc = execute_put(params,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     qquit - exit from the current file without saving changes

SYNTAX
     QQuit

DESCRIPTION
     The QQUIT command exits the user from the current file, whether
     changes have been made to the file or not.
     The previous file in the ring then becomes the current file.
     If the current file is the only file in the ring, the user is
     returned to the Operating System.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     QUIT

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Qquit(CHARTYPE *params)
#else
short Qquit(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Qquit");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 free_view_memory(TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     query - display various option settings

SYNTAX
     Query item

DESCRIPTION
     The QUERY command displays the various settings for options set
     by THE.

COMPATIBILITY
     XEDIT: Compatible functionality, but not all options.
     KEDIT: Compatible functionality, but not all options.

SEE ALSO
     STATUS, MODIFY

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Query(CHARTYPE *params)
#else
short Query(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VALUE item_values[18];
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short itemno=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Query");
#endif
 if ((itemno = find_item(params,QUERY_QUERY)) == (-1))
    {
     display_error(1,params,FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }

 itemno = get_item_values(itemno,(CHARTYPE *)"",QUERY_QUERY,0L,NULL,0L);
 strcpy(temp_cmd,"");
 for (i=0;i<itemno+1;i++)
   {
    strcat(temp_cmd,item_values[i].value);
    strcat(temp_cmd," ");
   }
 display_error(0,temp_cmd,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     quit - exit from the current file if no changes made

SYNTAX
     QUIT

DESCRIPTION
     The QUIT command exits the user from the current file, provided
     no changes have been made to the file. An error message will be
     displayed if changes have been made.
     The previous file in the ring then becomes the current file.
     If the current file is the only file in the ring, the user is
     returned to the Operating System.

COMPATIBILITY
     XEDIT: Does not support return code option.
     KEDIT: Compatible.

SEE ALSO
     QQUIT

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Quit(CHARTYPE *params)
#else
short Quit(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Quit");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (CURRENT_FILE->save_alt > 0)
   {
    display_error(22,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_FILE_CHANGED);
   }
 free_view_memory(TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     readv - read keystrokes and pass to macro

SYNTAX
     READV Cmdline|KEY

DESCRIPTION
     The READV command allows a REXX macro to interact with the user 
     by accepting keystrokes.

     The macro obtains the entered information by setting REXX
     variables. These are set as follows:

     For READV KEY

         readv.0 = 3
         readv.1 = name of key (empty if unknown)
         readv.2 = ASCII value of key (null if not an ASCII code)
         readv.3 = curses key value (or ASCII code if an ASCII code)

     For READV CMDLINE

         readv.0 = 1
         readv.1 = contents of command line

COMPATIBILITY
     XEDIT: Similar to READ CMDLINE option.
     KEDIT: Compatible.

STATUS
     Incomplete. CMDLINE option not implemented.
**man-end**********************************************************************/
#ifdef PROTO
short Readv(CHARTYPE *params)
#else
short Readv(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_macro;
 extern bool rexx_support;
 extern bool error_on_screen;
 extern WINDOW *error_window;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK,itemno=0,num_values=0;
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Readv");
 if (!in_macro)
   {
    display_error(53,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
#endif

 if (rexx_support)
   {
    getyx(CURRENT_WINDOW,y,x);
    (void)THERefresh("");                              
    if (error_on_screen
    &&  error_window != NULL)
       {
        touchwin(error_window);
        wnoutrefresh(error_window);
        wnoutrefresh(CURRENT_WINDOW);
        wmove(CURRENT_WINDOW,y,x);
        doupdate();
       }
#if defined(USE_EXTCURSES)
    getyx(CURRENT_WINDOW,y,x);
    wmove(CURRENT_WINDOW,y,x);
    wrefresh(CURRENT_WINDOW);
#endif
    if (equal((CHARTYPE *)"key",params,3))
      {
/*---------------------------------------------------------------------*/
/* Find the item in the list of valid extract options...               */
/*---------------------------------------------------------------------*/
       if ((itemno = find_item("READV",QUERY_READV)) == (-1))
         {
          display_error(1,params,FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
/*---------------------------------------------------------------------*/
/* Get the current settings for the valid item...                      */
/*---------------------------------------------------------------------*/
       num_values = get_item_values(itemno,NULL,QUERY_READV,0L,NULL,0L);
/*---------------------------------------------------------------------*/
/* If the arguments to the item are invalid, return with an error.     */
/*---------------------------------------------------------------------*/
       if (num_values == EXTRACT_ARG_ERROR)
         {
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
/*---------------------------------------------------------------------*/
/* If the REXX variables have already been set, don't try to set them. */
/*---------------------------------------------------------------------*/
       if (num_values != EXTRACT_VARIABLES_SET)
          rc = set_extract_variables(itemno);
       if (error_on_screen)
          clear_msgline();
      }
    else
      {
       if (equal((CHARTYPE *)"cmdline",params,1))
         {
         }
       else
         {
          display_error(1,params,FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
      }
   }
 else
   {
    display_error(53,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     recover - recover changed or deleted lines

SYNTAX
     RECover [n|*]

DESCRIPTION
     The RECOVER command restores the last n changed or deleted lines
     back into the body of the file.

COMPATIBILITY
     XEDIT: Also recovers changes to lines, not just lines deleted.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Recover(CHARTYPE *params)
#else
short Recover(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define REC_PARAMS  1
 CHARTYPE *strip=(CHARTYPE *)"11";
 CHARTYPE *word[REC_PARAMS+1];
 unsigned short num_params=0;
 short num=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Recover");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. The one and only   */
/* parameter should be a positive integer greater than zero or '*'.    */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,REC_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (CHARTYPE *)"1";
    }
 if (num_params != 1)
    {
     display_error(1,word[1],FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 if (strcmp(word[0],"*") == 0)
   {
    num = 99;
   }
 else
   {
    if (!valid_positive_integer(word[0]))
       {
        display_error(4,word[0],FALSE);
#ifdef TRACE
        trace_return();
#endif
        return(RC_INVALID_OPERAND);
       }
    num = atoi(word[0]);
   }

 get_from_recovery_list(num);

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     redraw - redraw the current screen

SYNTAX
     REDRAW

DESCRIPTION
     The REDRAW command redraws the current contents of the screen.
     This is usually used when some outside influence has affected 
     the display.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     REFRESH

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Redraw(CHARTYPE *params)
#else
short Redraw(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Redraw");
#endif
 if (strcmp(params,"") != 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 restore_THE();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     refresh - refresh the contents of the current screen

SYNTAX
     REFRESH

DESCRIPTION
     The REFRESH command refreshes what is being displayed on the screen.
     This is usually used from within a macro to indicate the progress
     of the macro.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     REDRAW

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short THERefresh(CHARTYPE *params)
#else
short THERefresh(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_macro;
 extern CHARTYPE display_screens;
/*--------------------------- local data ------------------------------*/
 bool save_in_macro=in_macro;
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   THERefresh");
#endif
 if (strcmp(params,"") != 0)
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 getyx(CURRENT_WINDOW,y,x);
 in_macro = FALSE;
 if (display_screens > 1)
   {
    build_other_screen();
    display_other_screen();
   }
 show_footing();
 build_current_screen(); 
 display_current_screen();
 wmove(CURRENT_WINDOW,y,x);
 wnoutrefresh(CURRENT_WINDOW);
 doupdate();
 in_macro = save_in_macro;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     repeat - repeat the last command

SYNTAX
     REPEat [target]

DESCRIPTION
     The REPEAT command advances the current line and executes the
     last command. It is equivalent to NEXT 1 (or UP 1) and = for
     the specified number of times.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Incomplete
**man-end**********************************************************************/
#ifdef PROTO
short Repeat(CHARTYPE *params)
#else
short Repeat(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short len_params=0,rc=RC_OK;
 LINETYPE true_line=0L;
 LINE *curr=NULL;
 SELECTTYPE current_select=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Repeat");
#endif

 display_error(0,(CHARTYPE *)"This command has yet to be implemented",FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     replace - replace the current line with supplied text

SYNTAX
     Replace [text]

DESCRIPTION
     The REPLACE command replaces the current line with the supplied
     text.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Replace(CHARTYPE *params)
#else
short Replace(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short len_params=0,rc=RC_OK;
 LINETYPE true_line=0L;
 LINE *curr=NULL;
 SELECTTYPE current_select=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Replace");
#endif

 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if (CURRENT_VIEW->hex)
   {
    if ((len_params = convert_hex_strings(params)) == (-1))
      {
       display_error(32,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   len_params = strlen(params);
 true_line = get_true_line();
 if (TOF(true_line)
 ||  BOF(true_line))
   {
    display_error(38,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY); /* ?? */
   }
 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
 current_select = curr->select;
 add_to_recovery_list(curr->line,curr->length);
 curr = delete_line(CURRENT_FILE->first_line,curr,DIRECTION_FORWARD);
 curr = curr->prev;
 if ((curr = add_line(CURRENT_FILE->first_line,curr,
                          params,len_params,current_select)) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);

 build_current_screen(); 
 display_current_screen();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     reset - cancel the marked block or prefix commands or both

SYNTAX
     RESet ALL|Block|Prefix

DESCRIPTION
     The RESET command unmarks any marked block or outstanding prefix
     commands or both.

COMPATIBILITY
     XEDIT: Adds Block and All options.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Reset(CHARTYPE *params)
#else
short Reset(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern VIEW_DETAILS *vd_mark;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
/*--------------------------- local data ------------------------------*/
#define RES_PARAMS  1
 CHARTYPE *word[RES_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"1";
 unsigned short num_params=0;
 PPC *curr_ppc=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Reset");
#endif
 num_params = param_split(params,word,RES_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params > 1)
   {
    display_error(1,word[1],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Reset the marked block, if any.                                     */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"block",word[0],1)
 ||  equal((CHARTYPE *)"all",word[0],3)
 ||  num_params == 0)
   {
    if (MARK_VIEW != (VIEW_DETAILS *)NULL)
      {
       MARK_VIEW->marked_line = MARK_VIEW->marked_col = FALSE;
       MARK_VIEW = (VIEW_DETAILS *)NULL;
       build_current_screen(); 
       display_current_screen();
      }
   }
/*---------------------------------------------------------------------*/
/* Reset the pending prefix commands, if any.                          */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"prefix",word[0],1)
 ||  equal((CHARTYPE *)"all",word[0],3)
 ||  num_params == 0)
   {
    curr_ppc = CURRENT_FILE->first_ppc;
    while(curr_ppc != NULL)
       curr_ppc = delete_pending_prefix_command(curr_ppc,CURRENT_FILE,(LINE *)NULL);
    memset(pre_rec,' ',PREFIX_WIDTH+1);
    pre_rec_len = 0;
    build_current_screen(); 
    display_current_screen();
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     rgtleft - scroll the screen to the left or right

SYNTAX
     RGTLEFT [n]

DESCRIPTION
     The RGTLEFT command scrolls the screen n columns to the right
     if the value of VERSHIFT is less than or equal to 0, or if
     the value of VERSHIFT is greater than 0, the screen is
     scrolled n columns to the left.
     If n is not specified, the screen scrolls by three quarters the
     number of columns displayed.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     LEFT, RIGHT

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Rgtleft(CHARTYPE *params)
#else
short Rgtleft(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 LINETYPE shift_val=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Rgtleft");
#endif
/*---------------------------------------------------------------------*/
/* Validate only parameter, a positive integer. 3/4 if no argument.    */
/*---------------------------------------------------------------------*/
 if (blank_field(params))
    shift_val = min(CURRENT_SCREEN.cols[WINDOW_MAIN],
                  1 + CURRENT_VIEW->verify_end - CURRENT_VIEW->verify_start)*3/4;
 else
   {
    if (valid_positive_integer(params))
       shift_val = atol(params);
   }
 if (shift_val == (-1))                            /* invalid argument */
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if ((LINETYPE)CURRENT_VIEW->verify_col - (LINETYPE)CURRENT_VIEW->verify_start > 0)
    shift_val = -shift_val;
 CURRENT_VIEW->verify_col = max(1,CURRENT_VIEW->verify_col+shift_val);
 build_current_screen();
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     right - scroll the screen to the right

SYNTAX
     RIght [n|HALF]

DESCRIPTION
     The RIGHT command scrolls the screen n columns to the right.
     If no parameter is supplied, the screen is scrolled by 1
     column. If HALF is specified the screen is scrolled by half 
     the number of columns in the FILEAREA.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     LEFT, RGTLEFT

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Right(CHARTYPE *params)
#else
short Right(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 LINETYPE shift_val=-1L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Right");
#endif
/*---------------------------------------------------------------------*/
/* Validate only parameter, HALF or positive integer. 1 if no argument.*/
/*---------------------------------------------------------------------*/
 if (equal(params,(CHARTYPE *)"half",4))
    shift_val = CURRENT_SCREEN.cols[WINDOW_MAIN]/2;
 if (blank_field(params))
    shift_val = 1L;
 if (shift_val == (-1))              /* argument not HALF or empty ... */
   {
    if (valid_positive_integer(params))
      {
       shift_val = atol(params);
       if (shift_val != 0)
          shift_val = shift_val;
      }
   }
 if (shift_val == (-1))                            /* invalid argument */
   {
    display_error(1,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the argument is 0, restore the original verify columns display.  */
/*---------------------------------------------------------------------*/
 if (shift_val == 0L)
    CURRENT_VIEW->verify_col = CURRENT_VIEW->verify_start;
 else
    CURRENT_VIEW->verify_col = max(1,CURRENT_VIEW->verify_col+shift_val);
#ifdef MSWIN
 Win31HScroll(CURRENT_VIEW->verify_col);
#endif
 build_current_screen();
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     save - save changes to current file

SYNTAX
     SAVE [fileid]

DESCRIPTION
     The SAVE command writes the current file to disk. If a fileid is
     supplied, the current file is saved in that file, unless the file
     already exists which will result in an error message being
     displayed. The 'Alterations' counter on the heading line is 
     reset to zero.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SSAVE, FILE, FFILE

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Save(CHARTYPE *params)
#else
short Save(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Save");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if ((rc = save_file(CURRENT_FILE,params,FALSE,CURRENT_FILE->number_lines,1L,FALSE,0,max_line_length,TRUE,FALSE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Only set the alteration count to zero if save was successful.       */
/*---------------------------------------------------------------------*/
 CURRENT_FILE->autosave_alt = CURRENT_FILE->save_alt = 0;
/*---------------------------------------------------------------------*/
/* If autosave is on at the time of Saving, remove the .aus file...    */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->autosave > 0)
    rc = remove_aus_file(CURRENT_FILE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     schange - selectively change strings

SYNTAX
     SCHange /string1/string2/ [target] [n] [m]

DESCRIPTION
     The SCHANGE command changes one string of text to another.

     The first parameter to the change command is the old and new
     string values, seperated by delimiters.
     The allowable delimiters are '/' '\' and '@'.

     The second parameter is the target; how many lines are to be
     searched for occurrences of the first string to be changed.

     The third parameter determines how many occurrences of 'string1'
     are to be changed on each line.

     The fourth parameter determines at which occurrences of 'string1'
     on the line are changes to commence.

COMPATIBILITY
     XEDIT: Functionally compatible, but syntax different.
     KEDIT: Compatible.

DEFAULT
     1 1 1

SEE ALSO
     CHANGE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Schange(CHARTYPE *params)
#else
short Schange(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Schange");
#endif
 rc = execute_change_command(params,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     set - execute various set commands

SYNTAX
     SET set_command [set_command parameter(s) ...]

DESCRIPTION
     The SET command is a front end to existing 'set' commands. It treats
     the first parameter it receives as a command and executes it.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Set(CHARTYPE *params)
#else
short Set(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Set");
#endif
 rc = execute_set_sos_command(TRUE,params);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     shift - move text left or right

SYNTAX
     SHift Left|Right [n] [target]

DESCRIPTION
     The SHIFT command moves text in the direction specified the number
     of columns [n] for the specified target lines.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Shift(CHARTYPE *params)
#else
short Shift(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define SHI_PARAMS  3
 CHARTYPE *word[SHI_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"100";
 short shift_left=(-1);
 LINETYPE num_lines=0L,true_line=0L;
 short num_cols=0,num_params=0;
 short rc=RC_OK;
 short target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL;
 TARGET target;
 bool num_lines_based_on_scope=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Shift");
#endif
 num_params = param_split(params,word,SHI_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)                                     /* no params */
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate first parameter:                                           */
/*    must be Left or Right                                            */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"left",word[0],1))
     shift_left = TRUE;
 if (equal((CHARTYPE *)"right",word[0],1))
     shift_left = FALSE;
 if (shift_left == (-1))
   {
    display_error(1,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate second parameter (if there is one)                         */
/*    If present, must be valid positive integer.                      */
/*    If not present, default to 1.                                    */
/*---------------------------------------------------------------------*/
 if (num_params < 2)
    num_cols = 1;
 else
   {
    if (!valid_positive_integer(strtrunc(word[1])))
      {
       display_error(4,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    num_cols = atoi(word[1]);
   }
/*---------------------------------------------------------------------*/
/* Validate third  parameter (if there is one)                         */
/*    If present, must be valid target.                                */
/*    If not present, default to 1.                                    */
/*---------------------------------------------------------------------*/
 if (num_params < 3)                                      /* no target */
   {
    num_lines = 1L;
    true_line = get_true_line();
   }
 else
   {
    initialise_target(&target);
    if ((rc = validate_target(word[2],&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
      {
       free_target(&target);
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
    num_lines = target.num_lines;
    true_line = target.true_line;
    num_lines_based_on_scope = (target.rt[0].target_type == TARGET_BLOCK_CURRENT) ? FALSE : TRUE;
    free_target(&target);
   }
/*---------------------------------------------------------------------*/
/* Now we are here, everything's OK, do the actual shift...            */
/*---------------------------------------------------------------------*/
 rc = execute_shift_command(shift_left,num_cols,true_line,num_lines,num_lines_based_on_scope);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     showkey - display current key value and command assignation

SYNTAX
     SHOWkey

DESCRIPTION
     The SHOWKEY command prompts the user to enter a key and responds
     with the key name and associated command (if applicable).
     To exit from SHOWKEY, press the space bar.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short ShowKey(CHARTYPE *params)
#else
short ShowKey(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int key=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   ShowKey");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 display_prompt("Press the key to be translated...spacebar to exit");
 key = 0;
 while(key != ' ')
   {
    key = my_getch(stdscr);
    clear_msgline();
    display_prompt(get_key_definition(key));
   }
 clear_msgline();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     sort - sort selected lines in a file

SYNTAX
     SORT target [[[Ascending|Descending] left_col right_col] [...]]

DESCRIPTION
     The SORT command sort a portion of a file based on the sort field
     specifications.
     target can be any valid target including ALL, *, -*, and BLOCK.
     Each sort field specification consists of an optional ordering
     flag and a left and right column.

     Only 10 sort fields are allowed.

COMPATIBILITY
     XEDIT: XEDIT only allows ordering flag for all fields
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sort(CHARTYPE *params)
#else
short Sort(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Sort");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 rc = execute_sort(params);
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen(); 
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     sos - execute various sos commands

SYNTAX
     SOS sos_command [sos_command ...]

DESCRIPTION
     The SOS command is a front end to existing 'sos' commands. It treats
     each parameter it receives as a command and executes it.

COMPATIBILITY
     XEDIT: XEDIT only permits 1 command
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Sos(CHARTYPE *params)
#else
short Sos(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
#define SOS_PARAMS  10
 CHARTYPE *strip=(CHARTYPE *)"1111111111";
 CHARTYPE *word[SOS_PARAMS+1];
 short num_params=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Sos");
#endif
 num_params = param_split(params,word,SOS_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)                                     /* no params */
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* For each "command" go an execute it.                                */
/*---------------------------------------------------------------------*/
 for (i=0;i<num_params;i++)
   {
    if ((rc = execute_set_sos_command(FALSE,word[i])) != RC_OK)
       break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     split - split a line into two lines

SYNTAX
     SPlit [ALigned]

DESCRIPTION
     The SPLIT command splits the focus line into two lines.
     If ALIGNED is specified, the first non-blank character of the new
     line is positioned under the first non-blank character of the
     focus line. If ALIGNED is not specified, the text of the new line
     starts in column 1.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     JOIN, SPLTJOIN

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Split(CHARTYPE *params)
#else
short Split(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0,col=0;
 short rc=RC_OK;
 bool aligned=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Split");
#endif
/*---------------------------------------------------------------------*/
/* Check here for parameter value of 'Aligned'.                        */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"aligned",params,2))
    aligned = TRUE;
 else
    if (strcmp(params,"") == 0)
       aligned = FALSE;
    else
      {
       display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
 if (curses_started)
   {
    getyx(CURRENT_WINDOW,y,x);
    col = (x+CURRENT_VIEW->verify_col-1);
   }
 else
    col = 0;
 rc = execute_split_join(TRUE,aligned,col);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     spltjoin - split/join two lines

SYNTAX
     spltjoin

DESCRIPTION
     The SPLTJOIN command splits the focus line into two or joins the
     focus line with the next line depending on the position of the
     cursor. If the cursor is after the last column of a line, the JOIN
     command is executed, otherwise the SPLIT command is executed.
     The text in the new line is aligned with the text in the focus line.

     This command can only be used by assigning it to a function key.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     JOIN, SPLIT

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Spltjoin(CHARTYPE *params)
#else
short Spltjoin(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
 extern LENGTHTYPE rec_len;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0,col=0;
 short rc=RC_OK;
 bool split=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Spltjoin");
#endif
 if (curses_started)
   {
    getyx(CURRENT_WINDOW,y,x);
    col = (x+CURRENT_VIEW->verify_col-1);
   }
 else
    col = 0;
/*---------------------------------------------------------------------*/
/* Determine whether a split or join is to be performed.               */
/* If after end of line, join, else split the line.                    */
/*---------------------------------------------------------------------*/
 split = (col >= rec_len) ? FALSE : TRUE;
 rc = execute_split_join(split,TRUE,col);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     ssave - force SAVE to specified file

SYNTAX
     SSave [fileid]

DESCRIPTION
     The SSAVE command writes the current file to disk. If a fileid is
     supplied, the current file is saved in that file, otherwise the
     current name of the file is used.
     If a fileid is supplied and that fileid already exists, the previous
     contents of that fileid will be replaced with the current file.
     The 'Alterations' counter on the heading line is reset to zero.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     SAVE, FILE, FFILE

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Ssave(CHARTYPE *params)
#else
short Ssave(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Ssave");
#endif
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 if ((rc = save_file(CURRENT_FILE,params,TRUE,CURRENT_FILE->number_lines,1L,FALSE,0,max_line_length,TRUE,FALSE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Only set the alteration count to zero if save was successful.       */
/*---------------------------------------------------------------------*/
 CURRENT_FILE->autosave_alt = CURRENT_FILE->save_alt = 0;
/*---------------------------------------------------------------------*/
/* If autosave is on at the time of SSaving, remove the .aus file...   */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->autosave > 0)
    rc = remove_aus_file(CURRENT_FILE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     status - display current settings of various variables

SYNTAX
     STATus [filename]

DESCRIPTION
     The STATUS command, without the optional filename, displays a full
     screen of current settings for various variables. With the filename,
     the STATUS command creates a file containing a series of SET commands
     with the current values of these settings.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible. KEDIT does not support [filename] option.

SEE ALSO
     QUERY, EXTRACT, MODIFY

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Status(CHARTYPE *params)
#else
short Status(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_profile;
 extern bool in_macro;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Status");
#endif
 if (strcmp(params,"") == 0)
   {
    if (in_profile || in_macro)
      {
       display_error(24,(CHARTYPE *)"status",FALSE);
       rc = RC_INVALID_ENVIRON;
      }
    else
       rc = show_status();
   }
 else
    rc = save_status(params);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     suspend - suspend THE and return to operating system

SYNTAX
     SUSPend

DESCRIPTION
     The SUSPEND command suspends the current editing session and 
     returns control to the operating system. Under DOS and OS/2 this
     is the equivalent of OSNowait. Under UNIX, the process gets placed
     in the background until it is brought to the foreground.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     OSNOWAIT

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Suspend(CHARTYPE *params)
#else
short Suspend(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
#ifdef UNIX
 void (*func)();
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Suspend");
#endif
 if (strcmp(params,"") != 0)
   {
    display_error(2,params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
#if defined(UNIX) && !defined(ATT) && !defined(M_XENIX)
 if (strcmp("/bin/sh",getenv("SHELL")) == 0)
   {
    display_error(40,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 suspend_curses();
 func = signal(SIGTSTP,SIG_DFL);
 kill(0,SIGTSTP);
 signal(SIGTSTP,func);
 resume_curses();
 Redraw((CHARTYPE *)"");
#else
 rc = execute_os_command(params,FALSE,FALSE);
#endif
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
