/***********************************************************************/
/* COMMSET2.C - SET commands O-Z                                       */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * 36 David Road                 Phone:                    +617 3849 7731
 * Holland Park                  http://www.gu.edu.au/gwis/the/markh.html
 * Brisbane                      **** Maintainer PDCurses & REXX/SQL ****
 * QLD 4121                      ************* Author of THE ************
 * Australia                     ************* Member RexxLA ************
 */

/*
$Id: commset2.c 2.1 1995/06/24 16:29:10 MH Rel MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/

/*--------------------------- global data -----------------------------*/
bool rexx_output=FALSE;
/*man-start*********************************************************************
COMMAND
     pending - set status of pending prefix commands

SYNTAX
     [SET] PENDing ON string
     [SET] PENDing OFF
     [SET] PENDing BLOCK string

DESCRIPTION
     The PENDING command allows the user to insert or remove commands
     from the pending prefix list.

     ON string, simulates the user typing 'string' in the PREFIX area
     of the focus line.

     OFF, removes any pending prefix command from the focus line.

     BLOCK string, simulates the user typing 'string' in the PREFIX
     area of the focus line and identifies the prefix command to be 
     a BLOCK command.

COMPATIBILITY
     XEDIT: Does not support ERROR option.
     KEDIT: N/A

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Pending(CHARTYPE *params)
#else
short Pending(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
#define PEND_ON    1
#define PEND_OFF   2
#define PEND_BLOCK 3
/*------------------------- external data -----------------------------*/
 extern bool in_profile;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
/*--------------------------- local data ------------------------------*/
#define PEN_PARAMS  3
 CHARTYPE *word[PEN_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"111";
 unsigned short num_params=0;
 short rc=RC_OK;
 LINE *curr=NULL;
 LINETYPE true_line=0L;
 short command=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Pending");
#endif
/*---------------------------------------------------------------------*/
/* Validate parameters.                                                */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,PEN_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
/*---------------------------------------------------------------------*/
/* If no arguments, error.                                             */
/*---------------------------------------------------------------------*/
 if (num_params == 0)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If more than 2 arguments, error.                                    */
/*---------------------------------------------------------------------*/
 if (num_params > 2)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate first parameter...                                         */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"off",word[0],3))
    command = PEND_OFF;
 else
    if (equal((CHARTYPE *)"on",word[0],2))
       command = PEND_ON;
    else
       if (equal((CHARTYPE *)"block",word[0],5))
          command = PEND_BLOCK;
       else
         {
          display_error(1,word[0],FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
 true_line = get_true_line();
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 switch(command)
   {
/*---------------------------------------------------------------------*/
/* PENDING ON and PENDING BLOCK...                                     */
/*---------------------------------------------------------------------*/
    case PEND_ON:
    case PEND_BLOCK:
/*---------------------------------------------------------------------*/
/* The second argument must be present and <= PREFIX_WIDTH.            */
/*---------------------------------------------------------------------*/
         if (num_params != 2)
           {
            display_error(3,(CHARTYPE *)"",FALSE);
            rc = RC_INVALID_OPERAND;
            break;
           }
         if (strlen(word[1]) > PREFIX_WIDTH)
           {
            display_error(1,word[1],FALSE); /* different error ?? */
            rc = RC_INVALID_OPERAND;
            break;
           }
/*---------------------------------------------------------------------*/
/* Copy the string into pre_rec and set its length.                    */
/*---------------------------------------------------------------------*/
         memset(pre_rec,' ',PREFIX_WIDTH);
         strcpy(pre_rec,word[1]);
         pre_rec_len = strlen(word[1]);
         pre_rec[pre_rec_len] = ' ';
         pre_rec[PREFIX_WIDTH] = '\0';
         curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
         if (command == PEND_BLOCK)
            add_prefix_command(curr,true_line,TRUE);
         else
            add_prefix_command(curr,true_line,FALSE);
         break;
/*---------------------------------------------------------------------*/
/* PENDING OFF...                                                      */
/*---------------------------------------------------------------------*/
    case PEND_OFF:
         if (num_params != 1)
           {
            display_error(2,(CHARTYPE *)"",FALSE);
            rc = RC_INVALID_OPERAND;
            break;
           }
         curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
         (void)delete_pending_prefix_command(curr->pre,CURRENT_FILE,curr);
         memset(pre_rec,' ',PREFIX_WIDTH);
         pre_rec_len = 0;
         break;
   }
 if (rc == RC_OK)
   {
    build_current_screen();
    display_current_screen();
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     point - assign a name to the current line

SYNTAX
     [SET] Point .name [OFF]

DESCRIPTION
     The POINT command assignes the specified name to the current
     line, or removes the name from the line with the specified name.
     A valid line name must start with a '.' followed by an alphabetic
     character. eg. .a .fred and .GG are valid names; fred and .3 are
     invalid line names.

     When a line is moved within the same file, its line name stays
     with the line.

COMPATIBILITY
     XEDIT: Compatible. See below.
     KEDIT: Compatible. See below.
     Does not allow for multiple names for the same line.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Point(CHARTYPE *params)
#else
short Point(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external date -----------------------------*/
/*--------------------------- local data ------------------------------*/
#define POI_PARAMS  2
 CHARTYPE *word[POI_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 unsigned short num_params=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Point");
#endif
/*---------------------------------------------------------------------*/
/* Validate parameters.                                                */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,POI_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Turning on line name...                                             */
/*---------------------------------------------------------------------*/
 if (num_params == 1)
   {
    if (word[0][0] != '.'
    ||  !isalpha(word[0][1]))
      {
       display_error(18,word[0],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if ((rc = execute_set_point(word[0],get_true_line(),TRUE)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
/*---------------------------------------------------------------------*/
/* Turning off line name...                                            */
/*---------------------------------------------------------------------*/
 else
   {
    if (!equal((CHARTYPE *)"off",word[1],3))
      {
       display_error(1,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if ((rc = execute_set_point(word[0],get_true_line(),FALSE)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     position - determine if LINE/COL is displayed on IDLINE

SYNTAX
     [SET] POSition ON|OFF

DESCRIPTION
     The POSITION command allows the user to turn on or off the
     display of LINE/COL on the IDLINE.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     ON

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Position(CHARTYPE *params)
#else
short Position(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external date -----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Position");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->position_status);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     prefix - set prefix area attributes

SYNTAX
     [SET] PREfix ON [Left|Right]
     [SET] PREfix Nulls [Left|Right]
     [SET] PREfix OFF
     [SET] PREfix Synonym newname oldname

DESCRIPTION
     The first form of the PREFIX command allows the user to display
     the prefix area and optionally to select the position the prefix
     should be displayed at.

     The second form of the PREFIX command is functionally the same
     as the first form. The difference is that when the prefix area
     is displayed with NUMBER ON, numbers are displyed with leading
     spaces rather than zeros; with NUMBER OFF, blanks are displayed
     instead of equal signs.

     The third form, turns the display of the prefix area off.
     Executed from within the profile, the only effect is that the
     defaults for all files is changed.
     Executed from the command line, the PREFIX command changes the
     current window displays to reflect the required options.

     The fourth form of the PREFIX command allows the users to specify
     a synonym for a prefix command or REXX prefix macro. The 'newname'
     is the command entered in the prefix area and 'oldname' corresponds
     to an existing prefix command or a REXX macro file in the MACROPATH
     ending in .the or whatever the value of MACROEXT is at the time the
     prefix command is executed. The 'oldname' can also be the fully
     qualified filename of a REXX macro.

COMPATIBILITY
     XEDIT: Compatible. See below.
     KEDIT: Compatible. See below.

DEFAULT
     ON LEFT

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Prefix(CHARTYPE *params)
#else
short Prefix(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
 extern short prefix_width;
/*--------------------------- local data ------------------------------*/
#define PRE_PARAMS  4
 CHARTYPE *word[PRE_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"1111";
 CHARTYPE prefix=PREFIX_OFF;
 CHARTYPE previous_prefix=CURRENT_VIEW->prefix;
 unsigned short num_params=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Prefix");
#endif
 num_params = param_split(params,word,PRE_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (equal((CHARTYPE *)"on",word[0],2)
 ||  equal((CHARTYPE *)"nulls",word[0],1))
   {
    if (num_params > 2)
      {
       display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if (equal((CHARTYPE *)"left",word[1],1))
       prefix = PREFIX_LEFT;
    else
       if (equal((CHARTYPE *)"right",word[1],1))
          prefix = PREFIX_RIGHT;
       else
          if (num_params == 1)       /* no left/right, default to left */
             prefix = PREFIX_LEFT;
          else
            {
             display_error(1,word[1],FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
    if (equal((CHARTYPE *)"on",word[0],2))
       CURRENT_VIEW->prefix = prefix | PREFIX_ON;
    else
       CURRENT_VIEW->prefix = prefix | PREFIX_NULLS;
   }
 else
   {
    if (equal((CHARTYPE *)"off",word[0],3))
      {
       if (num_params > 1)
         {
          display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
       if (strcmp(word[1],"") == 0)
          CURRENT_VIEW->prefix = PREFIX_OFF;
       else
         {
          display_error(2,word[1],FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
      }
    else
       if (equal((CHARTYPE *)"synonym",word[0],1))
         {
          if (num_params < 3)
            {
             display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
          if (num_params > 3)
            {
             display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
          if (strlen(word[1]) > PREFIX_WIDTH)
            {
             display_error(37,word[1],FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
          rc = add_prefix_synonym(word[1],word[2]);
#ifdef TRACE
          trace_return();
#endif
          return(rc);
         }
       else
         {
          display_error(1,word[0],FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
   }
/*---------------------------------------------------------------------*/
/* If the new setting for PREFIX is identical with the previous values */
/* don't do anything more.                                             */
/*---------------------------------------------------------------------*/
 if (previous_prefix == CURRENT_VIEW->prefix)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If the new location for PREFIX is the same as the previous value,   */
/* then the only difference MUST be the status, so just rebuild and    */
/* display the screen.                                                 */
/*---------------------------------------------------------------------*/
 if ((previous_prefix&PREFIX_LOCATION_MASK) == (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK))
   {
    build_current_screen();
    display_current_screen();
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* To get here the location of the window has changed, so rebuild the  */
/* windows and display the screen.                                     */
/*---------------------------------------------------------------------*/
 set_screen_defaults();
 if (set_up_windows(current_screen) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 build_current_screen();
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     printer - define printer spooler name

SYNTAX
     [SET] PRINTER spooler

DESCRIPTION
     The PRINTER command sets up the print spooler name to determine
     where output from the PRINT command goes.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     LPT1 - DOS/OS2, lpr - Unix

SEE ALSO
     PRINT

STATUS
     Complete. 
**man-end**********************************************************************/
#ifdef PROTO
short THEPrinter(CHARTYPE *params)
#else
short THEPrinter(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external date -----------------------------*/
#if defined(UNIX) || defined(OS2)
 extern CHARTYPE *spooler_name;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:THEPrinter");
#endif
 if ((spooler_name = (CHARTYPE *)(*the_realloc)(spooler_name,strlen(params)+1)) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 strcpy(spooler_name,params);
#endif
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     reprofile - indicate if profile file to be executed for all files

SYNTAX
     [SET] REPROFile ON|OFF

DESCRIPTION
     The REPROFILE command allows the user to determine if the profile
     file is to reexecuted for files subsequenlty edited.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     OFF

SEE ALSO
     XEDIT

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Reprofile(CHARTYPE *params)
#else
short Reprofile(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool REPROFILEx;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Reprofile");
#endif
 rc = execute_set_on_off(params,&REPROFILEx);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     reserved - display a reserved line

SYNTAX
     [SET] RESERved +|-n [colour] [text|OFF]

DESCRIPTION
     The RESERVED command reseves a line for the display of arbitrary
     text by the user. The position is determined by +|-n.
     This number, if positive, specifies the line relative from the
     top of the display. A negative number is relative from the
     bottom of the display.

     By specifying a line, say +3, then THREE lines from the top will
     be reserved, with the supplied text being displayed in line 3.

     The idline of a file will always be displayed after any reserved
     lines.

     The statusline is not considered part of the displayable area,
     so any positioning specifications ignore that line.

     A reserved line can only be turned off by identifying it in the
     same way that it was defined.  If a reserved line was added with
     the position specification of -1, it cannot be turned off with
     a position specification of 23, even though both position 
     specifiers result in the same display line.

     The colour option specifies the colours to use to display the
     reserved line.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SET COLOUR

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Reserved(CHARTYPE *params)
#else
short Reserved(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
#define RSR_PARAMS  2
 CHARTYPE *word[RSR_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"10";
 unsigned short num_params=0;
 short base=0,off=0;
 COLOUR_ATTR attr,save_attr;              /* initialisation done below */
 CHARTYPE *string=NULL;
 short rc=RC_OK;
 unsigned short x=0,y=0;
 LINETYPE new_focus_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Reserved");
#endif
/*---------------------------------------------------------------------*/
/* Initialise attr and save_sttr...                                    */
/*---------------------------------------------------------------------*/
 memset((char *)&attr,0,sizeof(COLOUR_ATTR));
 memset((char *)&save_attr,0,sizeof(COLOUR_ATTR));
 num_params = param_split(params,word,RSR_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
    {
     display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Parse the position parameter...                                     */
/*---------------------------------------------------------------------*/
 rc = execute_set_row_position(word[0],&base,&off);
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 if (equal((CHARTYPE *)"OFF",word[1],3))
    rc = delete_reserved_line(base,off);
 else
   {
/*---------------------------------------------------------------------*/
/* Parse the colour arguments (if any)...                              */
/*---------------------------------------------------------------------*/
    if ((rc = parse_colours(word[1],&attr,&string,TRUE)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
/*---------------------------------------------------------------------*/
/* If no colours were specified, use the default colour as set by any  */
/* SET COLOUR RESERVED... command.                                     */
/*---------------------------------------------------------------------*/
    if (memcmp(&attr,&save_attr,sizeof(COLOUR_ATTR)) == 0)
       rc = add_reserved_line(word[0],string,base,off,CURRENT_FILE->attr+ATTR_RESERVED);
    else
       rc = add_reserved_line(word[0],string,base,off,&attr);
   }
 build_current_screen();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    if (curses_started)
       getyx(CURRENT_WINDOW,y,x);
    new_focus_line = get_focus_line_in_view(CURRENT_VIEW->focus_line,y);
    if (new_focus_line != CURRENT_VIEW->focus_line)
      {
       CURRENT_VIEW->focus_line = new_focus_line;
       y = get_row_for_focus_line(CURRENT_VIEW->focus_line,CURRENT_VIEW->current_row);
       if (curses_started)
          wmove(CURRENT_WINDOW,y,x);
       pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
      }
   }
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     rexxoutput - indicate where REXX output is to go

SYNTAX
     [SET] REXXOUTput File|Display n

DESCRIPTION
     The REXXOUTPUT command indicates where output from the REXX
     interpreter is to go; either captured to a file in the ring
     or displayed in a scrolling fashion on the screen.

     Also specified is the maximum number of lines from the REXX
     interpreter that are to be displayed or captured. This is
     particularly useful when a REXX macro gets into an infinite
     loop.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     DISPLAY 1000

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Rexxoutput(CHARTYPE *params)
#else
short Rexxoutput(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern LINETYPE CAPREXXMAXx;
 extern bool CAPREXXOUTx;
 extern bool rexx_output;
/*--------------------------- local data ------------------------------*/
#define REX_PARAMS  2
 CHARTYPE *word[REX_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 unsigned short num_params=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Rexxoutput");
#endif
 if (rexx_output)
   {
    display_error(0,(CHARTYPE *)"Error: Unable to alter REXXOUTPUT settings",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 num_params = param_split(params,word,REX_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 2)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (equal((CHARTYPE *)"file",word[0],1))
    CAPREXXOUTx = TRUE;
 else
    if (equal((CHARTYPE *)"display",word[0],1))
       CAPREXXOUTx = FALSE;
    else
      {
       display_error(1,(CHARTYPE *)word[0],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 if (!valid_positive_integer(word[1]))
   {
    display_error(4,(CHARTYPE *)word[1],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CAPREXXMAXx =  atol(word[1]);
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     scale - set position and status of scale line on screen

SYNTAX
     [SET] SCALe ON|OFF [M[+n|-n]|[+|-]n]

DESCRIPTION
     The SCALE command sets the position and status of the scale line
     for the current view.

     The two forms of the position parameters are:
     M[+n|-n] - this sets the scale line to be relative to the
                middle of the screen. A positive value adds to the
                middle line number, a negative subtracts from it.
                eg. M+3 on a 24 line screen will be line 15
                    M-5 on a 24 line screen will be line 7

     [+|-]n   - this sets the scale line to be relative to the
                top of the screen (if positive or no sign) or
                relative to the bottom of the screen if negative.
                eg. +3 or 3 will set scale line to line 3
                    -3 on a 24 line screen will be line 21

     If the resulting line is outside the bounds of the screen
     the position of the current line will become the middle line
     on the screen.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     OFF M+1

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Scale(CHARTYPE *params)
#else
short Scale(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
#define SCA_PARAMS  2
 CHARTYPE *word[SCA_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 short num_params=0;
 short rc=RC_OK;
 short base=(short)CURRENT_VIEW->scale_base;
 short off=CURRENT_VIEW->scale_off;
 bool scalests=FALSE;
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Scale");
#endif
 num_params = param_split(params,word,SCA_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the status parameter...                                       */
/*---------------------------------------------------------------------*/
 rc = execute_set_on_off(word[0],&scalests);
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Parse the position parameter...                                     */
/*---------------------------------------------------------------------*/
 if (num_params > 1)
   {
    rc = execute_set_row_position(word[1],&base,&off);
    if (rc != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
 CURRENT_VIEW->scale_base = (CHARTYPE)base;
 CURRENT_VIEW->scale_off = off;
 CURRENT_VIEW->scale_on = scalests;
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    if (curses_started)
       getyx(CURRENT_WINDOW,y,x);
    CURRENT_VIEW->focus_line = get_focus_line_in_view(CURRENT_VIEW->focus_line,y);
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,CURRENT_VIEW->current_row);
    if (curses_started)
       wmove(CURRENT_WINDOW,y,x);
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
   }
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     scope - sets which lines are to be excluded from commands

SYNTAX
     [SET] SCOPE All|Display

DESCRIPTION
     The SCOPE command indicates whether lines not displayed as
     the result of a SET DISPLAY or ALL command are included in
     the scope of lines to be acted upon by other THE commands.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     DISPLAY

SEE ALSO
     SET DISPLAY, SET SELECT, ALL

STATUS
     Completed.
**man-end**********************************************************************/
#ifdef PROTO
short Scope(CHARTYPE *params)
#else
short Scope(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Scope");
#endif
 if (equal("all",params,1))
    CURRENT_VIEW->scope_all = TRUE;
 else
    if (equal("display",params,1))
       CURRENT_VIEW->scope_all = FALSE;
    else
      {
       display_error(1,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     screen - specifiy number of screens displayed

SYNTAX
     [SET] SCReen n [Horizontal|Vertical]

DESCRIPTION
     The SCREEN command specifies the number of views of file(s) to
     display on screen at once. If the number of views specified is 2
     and only one file is currently in the ring, two views of the
     same file are displayed.

COMPATIBILITY
     XEDIT: Does not support Size,Width or Define options.
     KEDIT: Does not support Size option.
     Only 2 screens are supported.

DEFAULT
     1

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short THEScreen(CHARTYPE *params)
#else
short THEScreen(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern WINDOW *divider;
 extern VIEW_DETAILS *vd_first;
 extern CHARTYPE number_of_views;
 extern CHARTYPE display_screens;
 extern bool horizontal;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
#define SCR_PARAMS  2
 CHARTYPE *word[SCR_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 register short i=0;
 unsigned short num_params=0,num_views=0;
 CHARTYPE save_display_screens=0;
 bool save_horizontal=FALSE;
 int horiz=(-1);
 VIEW_DETAILS *save_current_view=NULL;
 CHARTYPE save_current_screen=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:THEScreen");
#endif
 num_params = param_split(params,word,SCR_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (!valid_positive_integer(word[0]))
   {
    display_error(1,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if ((num_views = atoi(word[0])) > MAX_SCREENS)
   {
    display_error(6,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_views == 1)
    horiz = TRUE;
 else
   {
    if (equal((CHARTYPE *)"horizontal",word[1],1)
    || strcmp(word[1],"") == 0)
       horiz = TRUE;
    if (equal((CHARTYPE *)"vertical",word[1],1))
       horiz = FALSE;
    if (horiz == (-1))
      {
       display_error(1,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
/*---------------------------------------------------------------------*/
/* Set the global variable display_screens to indicate the number of   */
/* screens currently displayed and the orientation of those screens    */
/* Save the old values first so we know how the screens were oriented. */
/*---------------------------------------------------------------------*/
 save_display_screens = display_screens;
 save_horizontal = horizontal;

 display_screens = (CHARTYPE)num_views;
 horizontal=(bool)horiz;
/*---------------------------------------------------------------------*/
/* If there is no change to the screens, exit.                         */
/*---------------------------------------------------------------------*/
 if (display_screens == save_display_screens
 &&  horizontal == save_horizontal)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Save the screen coordinates for later retrieval.                    */
/*---------------------------------------------------------------------*/
 if (curses_started)
   {
    getyx(CURRENT_WINDOW_MAIN,CURRENT_VIEW->y[WINDOW_MAIN],CURRENT_VIEW->x[WINDOW_MAIN]);
    if (CURRENT_WINDOW_PREFIX != NULL)
       getyx(CURRENT_WINDOW_PREFIX,CURRENT_VIEW->y[WINDOW_PREFIX],CURRENT_VIEW->x[WINDOW_PREFIX]);
   }
/*---------------------------------------------------------------------*/
/* Set up the screen views correctly so that when we create the new    */
/* window for each screen, the screens have a view to point to...      */
/*---------------------------------------------------------------------*/
 switch(display_screens)
   {
    case 1:                         /* now have only 1 screen (from 2) */
           save_current_view = CURRENT_VIEW;
           if (CURRENT_SCREEN.screen_view->file_for_view == OTHER_SCREEN.screen_view->file_for_view)
             {
              CURRENT_VIEW = OTHER_SCREEN.screen_view;
              CURRENT_FILE->file_views--;
              free_a_view();
              CURRENT_VIEW = save_current_view;
             }
           if (divider == (WINDOW *)NULL)
             {
              delwin(divider);
              divider = NULL;
             }
           current_screen = 0;
           CURRENT_SCREEN.screen_view = CURRENT_VIEW = save_current_view;
           OTHER_SCREEN.screen_view = NULL;
           break;
    case 2:                             /* now have 2 screens (from 1) */
           save_current_view = CURRENT_VIEW;
           save_current_screen = current_screen;
           current_screen = (current_screen==0)?1:0; /* make other screen current */
           if (number_of_views == 1)
             {
              if ((rc = defaults_for_other_files(TRUE)) != RC_OK)
                {
#ifdef TRACE
                 trace_return();
#endif
                 return(rc);
                }
              CURRENT_FILE = save_current_view->file_for_view;
              CURRENT_FILE->file_views++;
             }
           else
             {
              if (NEXT_VIEW == (VIEW_DETAILS *)NULL)
                 CURRENT_VIEW = vd_first;
              else
                 CURRENT_VIEW = NEXT_VIEW;
             }
           CURRENT_SCREEN.screen_view = CURRENT_VIEW;
           CURRENT_VIEW = save_current_view;
           current_screen = save_current_screen;
           break;
   }

 set_screen_defaults();
 if (curses_started)
   {
    for (i=0;i<display_screens;i++)
      {
       if ((rc = set_up_windows(i)) != RC_OK)
         {
#ifdef TRACE
          trace_return();
#endif
          return(rc);
         }
       if (screen[i].screen_view->prefix)
          wmove(screen[i].win[WINDOW_PREFIX],
                screen[i].screen_view->y[WINDOW_PREFIX],screen[i].screen_view->x[WINDOW_PREFIX]);
       wmove(screen[i].win[WINDOW_MAIN],
             screen[i].screen_view->y[WINDOW_MAIN],screen[i].screen_view->x[WINDOW_MAIN]);
      }
   }

 if (!horizontal
 && display_screens > 1
 && curses_started)
   {
    redraw_window(divider);
    touchwin(divider);
    wnoutrefresh(divider);
   }

 if (display_screens > 1)
   {
    pre_process_line(OTHER_SCREEN.screen_view,OTHER_SCREEN.screen_view->focus_line);
    build_other_screen();
    display_other_screen();
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     select - sets the selection level for the specified lines

SYNTAX
     [SET] SELect [+|-]n [target]

DESCRIPTION
     The SELECT command sets the selection level for the indicated
     lines equal to n (if no signs are specified) or adds or subtracts
     n from the selection level currently set for the lines in the
     target.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     0

SEE ALSO
     SET SCOPE, SET DISPLAY, ALL

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Select(CHARTYPE *params)
#else
short Select(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
#define SEL_PARAMS  2
 CHARTYPE *word[SEL_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 LINETYPE num_lines=0L,true_line=0L,i=0L;
 short num_params=0;
 bool relative=FALSE;
 short direction=0;
 short off=0;
 LINE *curr=NULL;
 short target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL;
 TARGET target;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Select");
#endif
 num_params = param_split(params,word,SEL_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the first parameter...                                        */
/*---------------------------------------------------------------------*/
/* Determine if the selection level is relative to the existing value  */
/* or is an absolute value.                                            */
/*---------------------------------------------------------------------*/
 if (*(word[0]) == '-'
 ||  *(word[0]) == '+')
    relative = TRUE;
 else
    relative = FALSE;
/*---------------------------------------------------------------------*/
/* Get the value, positive or negative.                                */
/*---------------------------------------------------------------------*/
 if (!valid_integer(word[0]))
   {
    display_error(1,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 off = atoi(word[0]);
/*---------------------------------------------------------------------*/
/* Parse the next parameter...                                         */
/*---------------------------------------------------------------------*/
 true_line = get_true_line();
/*---------------------------------------------------------------------*/
/* If no target specified, just apply to the current line...           */
/*---------------------------------------------------------------------*/
 if (num_params == 1)
    word[1] = (CHARTYPE *)"1";
 initialise_target(&target);
 if ((rc = validate_target(word[1],&target,target_type,get_true_line(),TRUE,TRUE)) != RC_OK)
   {
    free_target(&target);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 num_lines = target.num_lines;
 true_line = target.true_line;
 direction = (target.num_lines<0) ? DIRECTION_BACKWARD : DIRECTION_FORWARD;
 free_target(&target);
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Get the current line from which to begin changing the select level. */
/*---------------------------------------------------------------------*/
 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
/*---------------------------------------------------------------------*/
/* For the number of lines affected, change the select level if the    */
/* line is in scope.                                                   */
/*---------------------------------------------------------------------*/
 for (i=0;i<num_lines;i++)
   {
    if (in_scope(CURRENT_VIEW,curr) || CURRENT_VIEW->scope_all)
      {
       if (relative)
         {
          if (((short)curr->select + off) > 255)
             curr->select = 255;
          else
             if (((short)curr->select + off) < 0)
                curr->select = 0;
             else
                curr->select = off;
         }
       else
          curr->select = off;
      }
    if (direction == DIRECTION_FORWARD)
       curr = curr->next;
    else
       curr = curr->prev;
   }
/*---------------------------------------------------------------------*/
/* If we are on the command line and the result of this statement means*/
/* that the current line is no longer in scope, we need to make the    */
/* current line and possibly the focus line the next line in scope.    */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
   {
    CURRENT_VIEW->current_line = find_next_in_scope(CURRENT_VIEW,NULL,true_line,DIRECTION_FORWARD);
    build_current_screen();
    if (!line_in_view(CURRENT_VIEW->focus_line))
      {
       CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
       pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
      }
   }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 display_current_screen();

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     shadow - determines if shadow lines are displayed or not

SYNTAX
     [SET] SHADOW ON|OFF

DESCRIPTION
     The SHADOW command indicates whether shadow lines are to be
     displayed.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     ON

SEE ALSO
     SET DISPLAY, SET SELECT, ALL

STATUS
     Completed.
**man-end**********************************************************************/
#ifdef PROTO
short Shadow(CHARTYPE *params)
#else
short Shadow(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Shadow");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->shadow);
 if (rc == RC_OK)
   {
    build_current_screen();
    display_current_screen();
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     statusline - set position of status line

SYNTAX
     [SET] STATUSLine Top|Bottom|Off|GUI

DESCRIPTION

     The STATUSLINE command determines the position of the status
     line for the editing session. TOP will place the status line on
     the first line of the screen; BOTTOM will place the status line
     on the last line of the screen; OFF turns off the display of
     the status line.

     The GUI option is only meaningful for those platforms that support
     a separate status line window. If specified for non-GUI ports, the
     GUI option is equivalent to OFF.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.
            Added GUI option.

DEFAULT
     BOTTOM

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Statusline(CHARTYPE *params)
#else
short Statusline(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern ROWTYPE STATUSLINEx;
 extern CHARTYPE display_screens;
 extern bool curses_started;
 extern bool horizontal;
 extern WINDOW *divider;
/*--------------------------- local data ------------------------------*/
 CHARTYPE stat_place='?';
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Statusline");
#endif
 if (equal((CHARTYPE *)"top",params,1))
    stat_place='T';
 if (equal((CHARTYPE *)"bottom",params,1))
    stat_place='B';
 if (equal((CHARTYPE *)"off",params,2))
    stat_place='O';
 if (equal((CHARTYPE *)"gui",params,3))
   {
    stat_place='G';
#ifdef MSWIN
    SetStatusBar(1);
#endif
   }
#ifdef MSWIN
 else
   {
    SetStatusBar(0);
   }
#endif
 if (stat_place=='?')
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the setting supplied is the same as the current setting, just    */
/* return without doing anything.                                      */
/*---------------------------------------------------------------------*/
 if (stat_place == STATUSLINEx)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Now we need to move the windows around.                             */
/*---------------------------------------------------------------------*/
 STATUSLINEx = stat_place;
/*---------------------------------------------------------------------*/
/* Redefine the screen sizes and recreate statusline window...         */
/*---------------------------------------------------------------------*/
 set_screen_defaults();
 if (curses_started)
   {
    if (create_statusline_window() != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
/*---------------------------------------------------------------------*/
/* Recreate windows for other screen (if there is one)...              */
/*---------------------------------------------------------------------*/
 if (display_screens > 1)
   {
    if (curses_started)
      {
       if (set_up_windows((current_screen==0) ? 1 : 0) != RC_OK)
         {
#ifdef TRACE
          trace_return();
#endif
          return(rc);
         }
       if (!horizontal)
         {
          redraw_window(divider);
          touchwin(divider);
          wnoutrefresh(divider);
         }
      }
    build_other_screen();
    display_other_screen();
   }
/*---------------------------------------------------------------------*/
/* Recreate windows for the current screen...                          */
/*---------------------------------------------------------------------*/
 if (curses_started)
   {
    if (set_up_windows(current_screen) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
 build_current_screen();
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     stay - set condition of cursor position after CHANGE/LOCATE commands

SYNTAX
     [SET] STAY ON|OFF

DESCRIPTION
     The STAY set command determines what line is displayed as the current
     line after an unsuccessful LOCATE or successful CHANGE command.
     With STAY ON, the current line remains where it currently is. With
     STAY OFF, after an unsuccessful LOCATE, the current line is "Bottom
     of File". After a successful CHANGE, the current line is the last
     line affected by the CHANGE command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     ON

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Stay(CHARTYPE *params)
#else
short Stay(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Stay");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->stay);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     synonym - define synonyms for commands

SYNTAX
     [SET] SYNonym ON|OFF
     [SET] SYNonym [LINEND char] newname [n] oldname

DESCRIPTION
     The SYNONYM command allows the user to define synonyms for commands
     or macros.

     The first format indicates if synonym processing is to be performed.

     The second format specifies the name to be associated with the
     following commands.

COMPATIBILITY
     XEDIT: Does not support the formatting option.
     KEDIT: Compatible.

DEFAULT
     ON

STATUS
     Incomplete.
**man-end**********************************************************************/
#ifdef PROTO
short Synonym(CHARTYPE *params)
#else
short Synonym(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Synonym");
#endif
 display_error(0,"This command not yet implemented",FALSE);
/*---------------------------------------------------------------------*/
/* Do not allow the command COMMAND to be synonymed.                   */
/*---------------------------------------------------------------------*/
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     tabkey - set characteristics of the SOS TABF command

SYNTAX
     [SET] TABKey Tab|Character Tab|Character

DESCRIPTION
     The TABKEY sets the action to be taken when the SOS TABF command
     is executed. Depending on the insert mode, the SOS TABF command
     will either display a raw tab character or will move to the next
     tab column.
     The first operand refers to the behaviour of the SOS TABF command
     when INSERTMODE is OFF; the second operand specifies the behaviour
     when the SOS TABF command is executed when INSERTMODE is ON.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     TAB CHARACTER

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
short Tabkey(CHARTYPE *params)
#else
short Tabkey(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE tabkey_insert,tabkey_overwrite;
/*--------------------------- local data ------------------------------*/
#define TKY_PARAMS  3
 CHARTYPE *word[TKY_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"111";
 unsigned short num_params=0;
 CHARTYPE tabo=tabkey_overwrite;
 CHARTYPE tabi=tabkey_insert;
 short rc=RC_INVALID_OPERAND;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabkey");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,TKY_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* Too few parameters, error.                                          */
/*---------------------------------------------------------------------*/
    case 0:
    case 1:
         display_error(3,(CHARTYPE *)"",FALSE);
         break;
/*---------------------------------------------------------------------*/
/* 2 parameters, validate them...                                      */
/*---------------------------------------------------------------------*/
    case 2:
/*---------------------------------------------------------------------*/
/* Validate first parameter; overwrite mode setting...                 */
/*---------------------------------------------------------------------*/
         if (equal((CHARTYPE *)"character",word[0],1))
           {
            tabo = 'C';
            rc = RC_OK;
           }
         if (equal((CHARTYPE *)"tab",word[0],1))
           {
            tabo = 'T';
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid first parameter, display an error and exit.          */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[0],FALSE);
            break;
           }
         rc = RC_INVALID_OPERAND;
/*---------------------------------------------------------------------*/
/* Validate second parameter; insert mode setting...                   */
/*---------------------------------------------------------------------*/
         if (equal((CHARTYPE *)"character",word[1],1))
           {
            tabi = 'C';
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* Validate second parameter; insert mode setting...                   */
/*---------------------------------------------------------------------*/
         if (equal((CHARTYPE *)"tab",word[1],1))
           {
            tabi = 'T';
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid second parameter, display an error and exit.         */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[1],FALSE);
            break;
           }
         rc = RC_OK;
         break;
/*---------------------------------------------------------------------*/
/* Too many parameters...                                              */
/*---------------------------------------------------------------------*/
    default:
         display_error(2,(CHARTYPE *)"",FALSE);
         break;
   }
/*---------------------------------------------------------------------*/
/* If valid parameters, change the settings...                         */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    tabkey_insert = tabi;
    tabkey_overwrite = tabo;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     tabline - set position and status of tab line on screen

SYNTAX
     [SET] TABLine ON|OFF [M[+n|-n]|[+|-]n]

DESCRIPTION
     The TABLINE command sets the position and status of the tab line
     for the current view.

     The two forms of the position parameters are:
     M[+n|-n] - this sets the tab line to be relative to the
                middle of the screen. A positive value adds to the
                middle line number, a negative subtracts from it.
                eg. M+3 on a 24 line screen will be line 15
                    M-5 on a 24 line screen will be line 7

     [+|-]n   - this sets the tab line to be relative to the
                top of the screen (if positive or no sign) or
                relative to the bottom of the screen if negative.
                eg. +3 or 3 will set tab line to line 3
                    -3 on a 24 line screen will be line 21

     If the resulting line is outside the bounds of the screen
     the position of the current line will become the middle line
     on the screen.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     OFF -3

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Tabline(CHARTYPE *params)
#else
short Tabline(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
#define TBL_PARAMS  2
 CHARTYPE *word[TBL_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 short num_params=0;
 short rc=RC_OK;
 short base=(short)CURRENT_VIEW->tab_base;
 short off=CURRENT_VIEW->tab_off;
 bool tabsts=FALSE;
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabline");
#endif
 num_params = param_split(params,word,TBL_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 1)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the status parameter...                                       */
/*---------------------------------------------------------------------*/
 rc = execute_set_on_off(word[0],&tabsts);
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Parse the position parameter...                                     */
/*---------------------------------------------------------------------*/
 if (num_params > 1)
   {
    rc = execute_set_row_position(word[1],&base,&off);
    if (rc != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
 CURRENT_VIEW->tab_base = (CHARTYPE)base;
 CURRENT_VIEW->tab_off = off;
 CURRENT_VIEW->tab_on = tabsts;
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
 build_current_screen();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    if (curses_started)
       getyx(CURRENT_WINDOW,y,x);
    CURRENT_VIEW->focus_line = get_focus_line_in_view(CURRENT_VIEW->focus_line,y);
    y = get_row_for_focus_line(CURRENT_VIEW->focus_line,CURRENT_VIEW->current_row);
    if (curses_started)
       wmove(CURRENT_WINDOW,y,x);
    pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line);
   }
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     tabs - set tab length for TAB key

SYNTAX
     [SET] TABS n1 n2[n3...n32]
     [SET] TABS INCR n

DESCRIPTION
     The TABS command determines the position of tab columns when the
     SOS TABF or SOS TABB commands are executed.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     INCR 8

SEE ALSO
     SOS TABF, SOS TABB

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Tabs(CHARTYPE *params)
#else
short Tabs(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define TABS_PARAMS  MAX_NUMTABS
 CHARTYPE *word[TABS_PARAMS+1];
 CHARTYPE strip[TABS_PARAMS];
 LENGTHTYPE stops[TABS_PARAMS];
 register short i=0;
 unsigned short num_params=0;
 LENGTHTYPE tabinc=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabs");
#endif
 for (i=0;i<TABS_PARAMS;i++)
    strip[i] = '1';
 num_params = param_split(params,word,TABS_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
/*---------------------------------------------------------------------*/
/* If the INCR option is specified...                                  */
/*---------------------------------------------------------------------*/
 if (equal((CHARTYPE *)"incr",word[0],2))
   {
    if (num_params != 2)
      {
       display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if (!valid_positive_integer(word[1]))
      {
       display_error(4,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    tabinc = (LENGTHTYPE)atoi(word[1]);
    if (tabinc < 1)
      {
       display_error(5,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if (tabinc > 32)
      {
       display_error(6,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    for (i=0;i<MAX_NUMTABS;i++)
       CURRENT_VIEW->tabs[i] = 1 + (tabinc*i);
    CURRENT_VIEW->numtabs = MAX_NUMTABS;
   }
 else
/*---------------------------------------------------------------------*/
/* ... individual TAB stop settings.                                   */
/*---------------------------------------------------------------------*/
   {
    if (num_params > MAX_NUMTABS)
      {
       display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if (num_params == 0)
      {
       display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    for (i=0;i<num_params;i++)
      {
       if (!valid_positive_integer(word[i]))
         {
          display_error(4,word[i],FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
       tabinc = (LENGTHTYPE)atoi(word[i]);
       if (i > 0)
         {
          if (stops[i-1] >= tabinc)
            {
             display_error(5,word[i],FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
         }
       stops[i] = tabinc;
      }
    CURRENT_VIEW->numtabs = num_params;
    for (i=0;i<num_params;i++)
      {
       CURRENT_VIEW->tabs[i] = stops[i];
      }
   }
 build_current_screen();
 display_current_screen();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     tabsin - set tab processing on file input

SYNTAX
     [SET] TABSIn ON|OFF [n]

DESCRIPTION
     The TABSIN command determines if tabs read from a file are to be
     expanded to spaces and if so how many spaces.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Does not support TABQUOTE option.

DEFAULT
     OFF 8

SEE ALSO
     SET TABSOUT

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Tabsin(CHARTYPE *params)
#else
short Tabsin(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*---------------------------------------------------------------------*/
/* The settings for TABSIN is a global value, despite it supposedly    */
/* being a file level value.                                           */
/*---------------------------------------------------------------------*/
/*-------------------------- external data ----------------------------*/
 extern bool in_profile;
 extern CHARTYPE TABI_ONx;
 extern CHARTYPE TABI_Nx;
/*--------------------------- local data ------------------------------*/
#define TABI_PARAMS  3
 CHARTYPE *word[TABI_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"111";
 unsigned short num_params=0;
 short rc=RC_INVALID_OPERAND;
 CHARTYPE tabsts=TABI_ONx;
 CHARTYPE tabn  =TABI_Nx;
 bool save_scope=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabsin");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,TABI_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* Too few parameters, error.                                          */
/*---------------------------------------------------------------------*/
    case 0:
         display_error(3,(CHARTYPE *)"",FALSE);
         break;
/*---------------------------------------------------------------------*/
/* 1 or 2 parameters, validate them...                                 */
/*---------------------------------------------------------------------*/
    case 1:
    case 2:
/*---------------------------------------------------------------------*/
/* Validate first parameter; on or off...                              */
/*---------------------------------------------------------------------*/
         if (equal((CHARTYPE *)"on",word[0],2))
           {
            tabsts = TRUE;
            rc = RC_OK;
           }
         if (equal((CHARTYPE *)"off",word[0],3))
           {
            tabsts = FALSE;
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid first parameter, display an error and exit.          */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[0],FALSE);
            break;
           }
/*---------------------------------------------------------------------*/
/* For 1 parameter, don't check any more.                              */
/*---------------------------------------------------------------------*/
         if (num_params == 1)
            break;
/*---------------------------------------------------------------------*/
/* Validate second parameter; number of spaces for a TAB...            */
/*---------------------------------------------------------------------*/
         if (!valid_positive_integer(word[1]))
           {
            display_error(4,word[1],FALSE);
            break;
           }
         tabn = (CHARTYPE)atoi(word[1]);
/*---------------------------------------------------------------------*/
/* tabn must be between 1 and 32...                                    */
/*---------------------------------------------------------------------*/
         if (tabn < 1)
           {
            display_error(5,word[1],FALSE);
            break;
           }
         if (tabn > 32)
           {
            display_error(6,word[1],FALSE);
            break;
           }
         rc = RC_OK;
         break;
/*---------------------------------------------------------------------*/
/* Too many parameters...                                              */
/*---------------------------------------------------------------------*/
    default:
         display_error(2,(CHARTYPE *)"",FALSE);
         break;
   }
/*---------------------------------------------------------------------*/
/* If valid parameters, change the settings...                         */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    TABI_ONx = tabsts;
    TABI_Nx = tabn;
/*---------------------------------------------------------------------*/
/* If this command is issued from the profile file, we need to run     */
/* EXPAND ALL on it, as we have already read in the file.              */
/* We need to save the current setting of scope so that it can be      */
/* changed to ALL so that every line will be expanded.                 */
/* Of course if TABSIN OFF was set we DON'T run EXPAND ALL :-)         */
/*---------------------------------------------------------------------*/
    if (in_profile && tabsts)
      {
       save_scope = CURRENT_VIEW->scope_all;
       rc = execute_expand("ALL",FALSE,FALSE,FALSE);
       CURRENT_VIEW->scope_all = save_scope;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     tabsout - set tab processing on file output

SYNTAX
     [SET] TABSOut ON|OFF [n]

DESCRIPTION
     The TABSOUT command determines if spaces written to a file are to be
     compressed to tabs and if so how many spaces.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     OFF 8

SEE ALSO
     SET TABSIN

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Tabsout(CHARTYPE *params)
#else
short Tabsout(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define TABO_PARAMS  3
 CHARTYPE *word[TABO_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"111";
 unsigned short num_params=0;
 short rc=RC_INVALID_OPERAND;
 bool tabsts=CURRENT_FILE->tabsout_on;
 CHARTYPE tabn=CURRENT_FILE->tabsout_num;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabsout");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,TABO_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* Too few parameters, error.                                          */
/*---------------------------------------------------------------------*/
    case 0:
         display_error(3,(CHARTYPE *)"",FALSE);
         break;
/*---------------------------------------------------------------------*/
/* 1 or 2 parameters, validate them...                                 */
/*---------------------------------------------------------------------*/
    case 1:
    case 2:
/*---------------------------------------------------------------------*/
/* Validate first parameter; on or off...                              */
/*---------------------------------------------------------------------*/
         if (equal((CHARTYPE *)"on",word[0],2))
           {
            tabsts = TRUE;
            rc = RC_OK;
           }
         if (equal((CHARTYPE *)"off",word[0],3))
           {
            tabsts = FALSE;
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid first parameter, display an error and exit.          */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[0],FALSE);
            break;
           }
/*---------------------------------------------------------------------*/
/* For 1 parameter, don't check any more.                              */
/*---------------------------------------------------------------------*/
         if (num_params == 1)
            break;
/*---------------------------------------------------------------------*/
/* Validate second parameter; number of spaces for a TAB...            */
/*---------------------------------------------------------------------*/
         if (!valid_positive_integer(word[1]))
           {
            display_error(4,word[1],FALSE);
            break;
           }
         tabn = (CHARTYPE)atoi(word[1]);
/*---------------------------------------------------------------------*/
/* tabn must be between 1 and 32...                                    */
/*---------------------------------------------------------------------*/
         if (tabn < 1)
           {
            display_error(5,word[1],FALSE);
            break;
           }
         if (tabn > 32)
           {
            display_error(6,word[1],FALSE);
            break;
           }
         rc = RC_OK;
         break;
/*---------------------------------------------------------------------*/
/* Too many parameters...                                              */
/*---------------------------------------------------------------------*/
    default:
         display_error(2,(CHARTYPE *)"",FALSE);
         break;
   }
/*---------------------------------------------------------------------*/
/* If valid parameters, change the settings...                         */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    CURRENT_FILE->tabsout_on = tabsts;
    CURRENT_FILE->tabsout_num = tabn;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     typeahead - set behavior of screen redraw

SYNTAX
     [SET] TYPEAhead ON|OFF

DESCRIPTION
     The TYPEAHEAD set command determines whether or not THE uses the
     curses screen display optimization techniques.
     With TYPEAHEAD ON, curses will abort screen display if a keystroke
     is pending.
     With TYPEAHEAD OFF, curses will not abort screen display if a
     keystroke is pending.
     For BSD based curses, this function has no effect.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     ON

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short THETypeahead(CHARTYPE *params)
#else
short THETypeahead(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 bool setting=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:THETypeahead");
#endif

#if !defined(BSD) && !defined(USE_NCURSES) && !defined(USE_EXTCURSES)
 if (curses_started)
   {
    rc = execute_set_on_off(params,&setting);
    if (rc == RC_OK)
      {
       if (setting)
#if defined(DOS) || defined(OS2)
          typeahead(1);
#else
          typeahead(stdin);
#endif
       else
          typeahead(-1);
      }
   }
#endif

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     verify - set column display limits

SYNTAX
     [SET] Verify first [last]

DESCRIPTION
     The VERIFY command sets the column limits for the display of the
     current file. 'first' specifies the first column to be displayed
     and 'last' specifies the last column to be displayed.

     If no 'last' option is specified '*' is assumed.

COMPATIBILITY
     XEDIT: Does not implement HEX display nor multiple column pairs.
     KEDIT: Does not implement HEX display nor multiple column pairs.

DEFAULT
     1 *

SEE ALSO
     SET ZONE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Verify(CHARTYPE *params)
#else
short Verify(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define VER_PARAMS  2
 CHARTYPE *word[VER_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 unsigned short num_params=0;
 LINETYPE col1=0L,col2=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Verify");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. One or two         */
/* parameters can be supplied. The first parameter MUST be a positive  */
/* integer. The second can be a positive integer or '*'. If no second  */
/* parameter is supplied, '*' is assumed. The second parameter MUST be */
/* >= first parameter. '*' is regarded as the biggest number and is    */
/* literally max_line_length.                                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,VER_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params == 0)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 2)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 col1 = atol(word[0]);
 if (num_params == 1)
     col2 = max_line_length;
 else
     if (strcmp(word[1],"*") == 0)
        col2 = max_line_length;
     else
        if (!valid_positive_integer(word[1]))
          {
           display_error(4,word[1],FALSE);
#ifdef TRACE
           trace_return();
#endif
           return(RC_INVALID_OPERAND);
          }
        else
           col2 = atol(word[1]);

 if (col2 > max_line_length)
    col2 = max_line_length;
 if (col1 > col2)
   {
    display_error(6,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->verify_start = (LENGTHTYPE)col1;
 CURRENT_VIEW->verify_col = (LENGTHTYPE)col1;
 CURRENT_VIEW->verify_end = (LENGTHTYPE)col2;

#ifdef MSWIN
 Win31HScroll(CURRENT_VIEW->verify_col);
#endif

 build_current_screen();
 display_current_screen();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     wordwrap - set wordwrap feature on or off

SYNTAX
     [SET] WORDWrap ON|OFF

DESCRIPTION
     The WORDWRAP set command determines whether wordwrap occurs when
     the cursor moves past the right margin (as set my [SET] MARGINS
     command).
     With WORDWRAP ON, the line, from the beginning of the word that
     exceeds the right margin, is wrapped onto the next line. The cursor
     position stays in the same position relative to the current word.
     WIth WORDWRAP OFF, no word wrap occurs.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     OFF

SEE ALSO
     SET MARGINS

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Wordwrap(CHARTYPE *params)
#else
short Wordwrap(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Wordwrap");
#endif
 rc = execute_set_on_off(params,&CURRENT_VIEW->wordwrap);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     zone - set column limits for editing

SYNTAX
     [SET] Zone first [last]

DESCRIPTION
     The ZONE command sets the column limits for various other editor
     commands, such as 'locate' and 'change'. It effectively restricts
     to the specified columns those parts of the file which can be
     acted upon.

     If no 'last' option is specified '*' is assumed.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     1 *

SEE ALSO
     SET VERIFY

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
short Zone(CHARTYPE *params)
#else
short Zone(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define ZON_PARAMS  2
 CHARTYPE *word[ZON_PARAMS+1];
 CHARTYPE *strip=(CHARTYPE *)"11";
 unsigned short num_params=0;
 LINETYPE col1=0L,col2=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Zone");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. One only           */
/* parameter MUST be supplied. The first parameter MUST be a positive  */
/* integer. The second can be a positive integer or '*'. If no second  */
/* parameter is supplied, ERROR.          The second parameter MUST be */
/* >= first parameter. '*' is regarded as the biggest number and is    */
/* literally max_line_length.                                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,ZON_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 if (num_params < 2)
   {
    display_error(3,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 2)
   {
    display_error(2,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 col1 = atol(word[0]);
 if (strcmp(word[1],"*") == 0)
    col2 = max_line_length;
 else
    if (!valid_positive_integer(word[1]))
      {
       display_error(4,word[1],FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    else
       col2 = atol(word[1]);

 if (col2 > max_line_length)
    col2 = max_line_length;
 if (col1 > col2)
   {
    display_error(6,word[0],FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->zone_start = (LENGTHTYPE)col1;
 CURRENT_VIEW->zone_end   = (LENGTHTYPE)col2;
/*---------------------------------------------------------------------*/
/* Change the current column pointer if it is outside the new zone     */
/* settings...                                                         */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_column < CURRENT_VIEW->zone_start)
    CURRENT_VIEW->current_column = max(1,CURRENT_VIEW->zone_start-1);
 if (CURRENT_VIEW->current_column > CURRENT_VIEW->zone_end)
    CURRENT_VIEW->current_column = min(max_line_length,CURRENT_VIEW->zone_end+1);
/*---------------------------------------------------------------------*/
/* If the SCALE line is currently displayed, display the page so that  */
/* any changes are reflected in the SCALE line.                        */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->scale_on)
   {
    build_current_screen();
    display_current_screen();
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
