{$B-,F-,I+,R+}

unit CTool;

{ Define TTool - a class for drawing tools }

{ Copyright 1989
  Scott Bussinger
  110 South 131st Street
  Tacoma, WA  98444
  (206)531-8944
  Compuserve 72247,2671 }

interface

uses CObject,CMouse,CWindow,CStyle,MSGraph,Crt;

type TToolPane = object(TPaneWindow)
       procedure Anchor;                         { Set the anchor point for the tool }
       procedure ClearAnchor;                    { The anchor should be broken }
       procedure Draw;                           { Draw using the tool }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for this pane }
       procedure Idle;                           { Let a text tool look at the keyboard }
       function Select: boolean; override;       { Select this pane }
       procedure SetCursor;                      { Set the mouse cursor for the tool }
       procedure Track;                          { Track the mouse }
       end;

type TEraserTool = object(TToolPane)
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for an eraser }
       function Select: boolean; override;       { Select an eraser }
       end;

type TQuitTool = object(TToolPane)
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for quitting }
       function Select: boolean; override;       { Select to quit }
       end;

type TModalToolPane = object(TToolPane)
       fAnchored: boolean;
       fAnchorPoint: _XYCoord;
       fDrawAnchored: boolean;
       fDrawPoint: _XYCoord;
       fTrackPoint: _XYCoord;
       procedure Init(Bordered: boolean;X1,Y1,X2,Y2: real); override; { Initialize a window }
       procedure Anchor; override;               { Set the anchor point for the tool }
       procedure ClearAnchor; override;          { The anchor should be broken }
       procedure Draw; override;                 { Draw using the tool }
       function Select: boolean; override;       { Select this pane }
       procedure Track; override;                { Track the mouse }
       end;

type TPenTool = object(TModalToolPane)
       procedure Anchor; override;               { Set the anchor point for a pen }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a pen }
       procedure Track; override;                { Track the mouse }
       end;

type TPaintBucketTool = object(TModalToolPane)
       procedure Draw; override;                 { Draw with a paint bucket }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a paint bucket }
       procedure SetCursor; override;            { Set the mouse cursor for a paint bucket }
       end;

type TRubberBandToolPane = object(TModalToolPane)
       fRectangleTrack: boolean;
       procedure Init(Bordered: boolean;X1,Y1,X2,Y2: real); override; { Initialize a rubberbanding tool }
       procedure ClearAnchor; override;          { The anchor should be broken }
       procedure Track; override;                { Track the mouse }
       end;

type TRectangleRubberBandToolPane = object(TRubberBandToolPane)
       procedure Init(Bordered: boolean;X1,Y1,X2,Y2: real); override; { Initialize a rectangular rubberband tool }
       end;

type TLineTool = object(TRubberBandToolPane)
       procedure Draw; override;                 { Draw using the line tool }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a line tool }
       end;

type TBoxTool = object(TRectangleRubberBandToolPane)
       procedure Draw; override;                 { Draw with a box }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a box tool }
       end;

type TFilledBoxTool = object(TRectangleRubberBandToolPane)
       procedure Draw; override;                 { Draw with a filled box }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a filled box tool }
       end;

type TCircleTool = object(TRubberBandToolPane)
       procedure Draw; override;                 { Draw with a circle }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a circle tool }
       end;

type TFilledCircleTool = object(TRubberBandToolPane)
       procedure Draw; override;                 { Draw with a filled circle }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a filled circle tool }
       end;

type TEllipseTool = object(TRectangleRubberBandToolPane)
       procedure Draw; override;                 { Draw with an ellipse }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for an ellipse tool }
       end;

type TFilledEllipseTool = object(TRectangleRubberBandToolPane)
       procedure Draw; override;                 { Draw with a filled ellipse }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a filled ellipse tool }
       end;

type TTextTool = object(TRectangleRubberBandToolPane)
       fEntryEnabled: boolean;
       procedure Draw; override;                 { Draw with a text tool }
       procedure DrawIcon(Marked: boolean); override; { Draw the icon for a text tool }
       procedure Idle; override;                 { Let the text tool look at the keyboard }
       function Select: boolean; override;       { Select the text tool }
       end;

type TToolWindow = object(TMultipanedWindow)
       procedure Init(Bordered: boolean;X1,Y1,X2,Y2: real); override; { Initialize a tool selection window }
       procedure ChangePane(Pane: integer); override; { Change to a new active pane }
       function CreatePane(Pane: integer): TPaneWindow; override; { Create a new tool pane }
       end;

implementation

var CurrentTrackLineStyle: word;

procedure TToolPane.Anchor;
  { Set the anchor point for the tool }
  begin
  CurrentCanvas.Activate
  end;

procedure TToolPane.ClearAnchor;
  { The anchor should be broken }
  begin
  end;

procedure TToolPane.Draw;
  { Draw using the tool }
  begin
  CurrentCanvas.Activate;
  self.ClearAnchor
  end;

procedure TToolPane.DrawIcon(Marked: boolean);
  { Draw the icon for this tool }
  begin
  inherited self.DrawIcon(Marked);
  if Marked                                      { Choose the background color for the icon }
   then
    _SetColor(SystemColor)
   else
    _SetColor(SystemBackground);
  _Rectangle_w(_GFillInterior,0.00,0.00,1.00,1.00);
  if Marked                                      { Choose the foreground color for the icon }
   then
    _SetColor(SystemBackground)
   else
    _SetColor(SystemColor)
  end;

procedure TToolPane.Idle;
  { Normally this does nothing }
  begin
  end;

function TToolPane.Select: boolean;
  { Select this pane }
  begin
  Select := inherited self.Select;
  Select := false;
  self.ClearAnchor
  end;

procedure TToolPane.SetCursor;
  { Set the mouse cursor for the tool }
  begin
  Mouse.SetCursor(PenCursor)                     { The default tool cursor is a pen }
  end;

procedure TToolPane.Track;
  { Track the mouse }
  begin
  CurrentCanvas.Activate
  end;

procedure TEraserTool.DrawIcon(Marked: boolean);
  { Draw the icon for an eraser tool }
  var DontCare: integer;
  begin
  inherited self.DrawIcon(Marked);
  FitText(Roman,'C')
  end;

function TEraserTool.Select: boolean;
  { Clear the drawing window }
  begin
  Select := inherited self.Select;
  Mouse.Hide;                                    { Keep the display clean }
  _ClearScreen(_GViewport)
  end;

procedure TQuitTool.DrawIcon(Marked: boolean);
  { Draw the icon for the quitting tool }
  var DontCare: integer;
  begin
  inherited self.DrawIcon(Marked);
  FitText(Roman,'Q')
  end;

function TQuitTool.Select: boolean;
  { Quit the program }
  begin
  Select := inherited self.Select;
  halt                                           { This one's easy -- just quit }
  end;

procedure TModalToolPane.Init(Bordered: boolean;
                              X1,Y1,X2,Y2: real);
  { Initialize a window }
  begin
  inherited self.Init(Bordered,X1,Y1,X2,Y2);
  self.ClearAnchor
  end;

procedure TModalToolPane.Anchor;
  { Set the anchor point for the tool }
  begin
  Mouse.DisableTextCursor;
  Mouse.Hide;                                    { Keep the display clean }
  inherited self.Anchor;
  _GetViewCoord(Mouse.GetLocationX,Mouse.GetLocationY,self.fAnchorPoint);
  self.fAnchored := true;
  self.fTrackPoint := self.fAnchorPoint
  end;

procedure TModalToolPane.ClearAnchor;
  { The anchor should be broken }
  begin
  inherited self.ClearAnchor;
  self.fAnchored := false
  end;

procedure TModalToolPane.Draw;
  { Draw using the tool }
  begin
  self.fDrawAnchored := self.fAnchored;
  Mouse.Hide;                                    { Keep the display clean }
  inherited self.Draw;
  _GetViewCoord(Mouse.GetLocationX,Mouse.GetLocationY,self.fDrawPoint)
  end;

function TModalToolPane.Select: boolean;
  { Is this a modal tool? }
  begin
  Select := inherited self.Select;
  Select := true;
  Mouse.DisableTextCursor                        { Turn the text cursor off }
  end;

procedure TModalToolPane.Track;
  { Track the mouse }
  begin
  Mouse.Hide;                                    { Keep the display clean }
  inherited self.Track;
  _GetViewCoord(Mouse.GetLocationX,Mouse.GetLocationY,self.fTrackPoint)
  end;

procedure TRubberBandToolPane.Init(Bordered: boolean;
                                   X1,Y1,X2,Y2: real);
  { Initialize a window }
  begin
  inherited self.Init(Bordered,X1,Y1,X2,Y2);
  self.fRectangleTrack := false
  end;

procedure XorRubberBand(Rectangular: boolean;
                        var StartPoint,EndPoint: _XYCoord);
  { Draw/Undraw a rubberband cursor }
  var SaveStatus: GraphicsStatus;
  begin
  Mouse.Hide;                                    { Keep the display clean }
  GetGraphicsStatus(SaveStatus);
  _SetLineStyle(CurrentTrackLineStyle);
  _SetWriteMode(_GXOR);                          { Temporarily switch to XOR mode }
  _SetColor(SystemWhite);
  if Rectangular                                 { Get rid of previous track }
   then
    _Rectangle(_GBorder,StartPoint.XCoord,StartPoint.YCoord,EndPoint.XCoord,EndPoint.YCoord)
   else
    begin
    _MoveTo(StartPoint.XCoord,StartPoint.YCoord);
    _LineTo(EndPoint.XCoord,EndPoint.YCoord)
    end;
  SetGraphicsStatus(SaveStatus)
  end;

procedure TRubberBandToolPane.ClearAnchor;
  { The anchor should be broken }
  begin
  if self.fAnchored then
    XorRubberBand(self.fRectangleTrack,self.fAnchorPoint,self.fTrackPoint);
  inherited self.ClearAnchor
  end;

procedure TRubberBandToolPane.Track;
  { Track the mouse }
  var PreviousTrackPoint: _XYCoord;
  begin
  PreviousTrackPoint := self.fTrackPoint;
  inherited self.Track;
  if self.fAnchored and not CompareXYCoord(PreviousTrackPoint,self.fTrackPoint) then { Have we moved? }
    begin
    XorRubberBand(self.fRectangleTrack,self.fAnchorPoint,PreviousTrackPoint);
    if not CompareXYCoord(self.fAnchorPoint,self.fTrackPoint) then
      XorRubberBand(self.fRectangleTrack,self.fAnchorPoint,self.fTrackPoint)
    end
  end;

procedure TRectangleRubberBandToolPane.Init(Bordered: boolean;
                                            X1,Y1,X2,Y2: real);
  { Initialize a window }
  begin
  inherited self.Init(Bordered,X1,Y1,X2,Y2);
  self.fRectangleTrack := true
  end;

procedure TPenTool.Anchor;
  { Move current position to mouse location }
  begin
  inherited self.Anchor;
  _MoveTo(self.fAnchorPoint.XCoord,self.fAnchorPoint.YCoord)
  end;

procedure TPenTool.DrawIcon(Marked: boolean);
  { Draw the icon for the pen tool }
  begin
  inherited self.DrawIcon(Marked);
  _MoveTo_w(0.13,0.87);
  _LineTo_w(0.33,0.30);
  _LineTo_w(0.48,0.30);
  _LineTo_w(0.58,0.50);
  _LineTo_w(0.63,0.50);
  _LineTo_w(0.87,0.13)
  end;

procedure TPenTool.Track;
  { Leave a trail of dots by drawing a line to the current location }
  begin
  inherited self.Track;
  if self.fAnchored then
    _LineTo(self.fTrackPoint.XCoord,self.fTrackPoint.YCoord)
  end;

procedure TPaintBucketTool.Draw;
  { Start a Flood Fill at the current location }
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    _FloodFill(self.fDrawPoint.XCoord,self.fDrawPoint.YCoord,_GetColor)
  end;

procedure TPaintBucketTool.DrawIcon(Marked: boolean);
  { Draw the icon for a paint bucket }
  begin
  inherited self.DrawIcon(Marked);
  _MoveTo_w(0.20,0.50);
  _LineTo_w(0.70,0.20);
  _LineTo_w(0.80,0.50);
  _LineTo_w(0.30,0.80);
  _LineTo_w(0.20,0.50);
  _LineTo_w(0.80,0.50);
  _LineTo_w(0.80,0.80);
  _FloodFill_w(0.30,0.60,_GetColor)
  end;

procedure TPaintBucketTool.SetCursor;
  { Set a special cursor for a paint bucket tool }
  begin
  Mouse.SetCursor(BucketCursor)
  end;

procedure TLineTool.Draw;
  { Draw a straight line }
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    begin
    _MoveTo(self.fAnchorPoint.XCoord,self.fAnchorPoint.YCoord);
    _LineTo(self.fDrawPoint.XCoord,self.fDrawPoint.YCoord)
    end
  end;

procedure TLineTool.DrawIcon(Marked: boolean);
  { Draw the icon for the line tool }
  begin
  inherited self.DrawIcon(Marked);
  _MoveTo_w(0.20,0.80);
  _LineTo_w(0.80,0.20);
  _MoveTo_w(0.20,0.50);
  _LineTo_w(0.80,0.50)
  end;

procedure TBoxTool.Draw;
  { Draw a rectangle }
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    _Rectangle(_GBorder,self.fAnchorPoint.XCoord,self.fAnchorPoint.YCoord,
                        self.fDrawPoint.XCoord,self.fDrawPoint.YCoord)
  end;

procedure TBoxTool.DrawIcon(Marked: boolean);
  { Draw the icon for the box tool }
  begin
  inherited self.DrawIcon(Marked);
  _Rectangle_w(_GBorder,0.20,0.20,0.80,0.80)
  end;

procedure TFilledBoxTool.Draw;
  { Draw a filled rectangle }
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    _Rectangle(_GFillInterior,self.fAnchorPoint.XCoord,self.fAnchorPoint.YCoord,
                              self.fDrawPoint.XCoord,self.fDrawPoint.YCoord)
  end;

procedure TFilledBoxTool.DrawIcon(Marked: boolean);
  { Draw the icon for the filled box tool }
  begin
  inherited self.DrawIcon(Marked);
  _Rectangle_w(_GFillInterior,0.20,0.20,0.80,0.80)
  end;

procedure TCircleTool.Draw;
  { Draw a circle }
  var CenterX,CenterY: integer;
      XRadius,YRadius: integer;
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    begin
    CenterX := (self.fAnchorPoint.XCoord+self.fDrawPoint.XCoord) div 2;
    CenterY := (self.fAnchorPoint.YCoord+self.fDrawPoint.YCoord) div 2;
    XRadius := round(sqrt(sqr(longint(self.fAnchorPoint.XCoord-self.fDrawPoint.XCoord)) +
                          sqr(AspectRatio*longint(self.fAnchorPoint.YCoord-self.fDrawPoint.YCoord))) / 2.0);
    YRadius := round(XRadius * AspectRatio);
    _Ellipse(_GBorder,CenterX-XRadius,CenterY-YRadius,CenterX+XRadius,CenterY+YRadius)
    end

  (* this is code for specifying circles by center point and radius
  XRadius := round(sqrt(sqr(self.fAnchorPoint.XCoord-self.fDrawPoint.XCoord) +
                        sqr(AspectRatio*(self.fAnchorPoint.YCoord-self.fDrawPoint.YCoord))));
  YRadius := round(XRadius * AspectRatio);
  _Ellipse(_GBorder,self.fAnchorPoint.XCoord-XRadius,self.fAnchorPoint.YCoord-YRadius,
                    self.fAnchorPoint.XCoord+XRadius,self.fAnchorPoint.YCoord+YRadius)
  *)
  end;

procedure TCircleTool.DrawIcon(Marked: boolean);
  { Draw the icon for a circle tool }
  const XRadius = 0.35;
  var YRadius: real;
  begin
  inherited self.DrawIcon(Marked);
  YRadius := XRadius * AspectRatioW;
  _Ellipse_w(_GBorder,0.50-XRadius,0.50-YRadius,0.50+XRadius,0.50+YRadius)
  end;

procedure TFilledCircleTool.Draw;
  { Draw a filled circle }
  var CenterX,CenterY: integer;
      XRadius,YRadius: integer;
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    begin
    CenterX := (self.fAnchorPoint.XCoord+self.fDrawPoint.XCoord) div 2;
    CenterY := (self.fAnchorPoint.YCoord+self.fDrawPoint.YCoord) div 2;
    XRadius := round(sqrt(sqr(longint(self.fAnchorPoint.XCoord-self.fDrawPoint.XCoord)) +
                          sqr(AspectRatio*longint(self.fAnchorPoint.YCoord-self.fDrawPoint.YCoord))) / 2.0);
    YRadius := round(XRadius * AspectRatio);
    _Ellipse(_GFillInterior,CenterX-XRadius,CenterY-YRadius,CenterX+XRadius,CenterY+YRadius)
    end

  (* this is code for specifying circles by center point and radius
  XRadius := round(sqrt(sqr(self.fAnchorPoint.XCoord-self.fDrawPoint.XCoord) +
                        sqr(AspectRatio*(self.fAnchorPoint.YCoord-self.fDrawPoint.YCoord))));
  YRadius := round(XRadius * AspectRatio);
  _Ellipse(_GFillInterior,self.fAnchorPoint.XCoord-XRadius,self.fAnchorPoint.YCoord-YRadius,
                          self.fAnchorPoint.XCoord+XRadius,self.fAnchorPoint.YCoord+YRadius)
  *)
  end;

procedure TFilledCircleTool.DrawIcon(Marked: boolean);
  { Draw the icon for a filled circle tool }
  const XRadius = 0.35;
  var YRadius: real;
  begin
  inherited self.DrawIcon(Marked);
  YRadius := XRadius * AspectRatioW;
  _Ellipse_w(_GFillInterior,0.50-XRadius,0.50-YRadius,0.50+XRadius,0.50+YRadius)
  end;

procedure TEllipseTool.Draw;
  { Draw an ellipse }
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    _Ellipse(_GBorder,self.fAnchorPoint.XCoord,self.fAnchorPoint.YCoord,
                      self.fDrawPoint.XCoord,self.fDrawPoint.YCoord)
  end;

procedure TEllipseTool.DrawIcon(Marked: boolean);
  { Draw the icon for the ellipse tool }
  begin
  inherited self.DrawIcon(Marked);
  _Ellipse_w(_GBorder,0.30,0.20,0.70,0.80)
  end;

procedure TFilledEllipseTool.Draw;
  { Draw a filled ellipse }
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    _Ellipse(_GFillInterior,self.fAnchorPoint.XCoord,self.fAnchorPoint.YCoord,
                            self.fDrawPoint.XCoord,self.fDrawPoint.YCoord)
  end;

procedure TFilledEllipseTool.DrawIcon(Marked: boolean);
  { Draw the icon for a filled ellipse tool }
  begin
  inherited self.DrawIcon(Marked);
  _Ellipse_w(_GFillInterior,0.30,0.20,0.70,0.80)
  end;

procedure TTextTool.Draw;
  { Setup for text entry }
  var DontCare: integer;
      Ch: char;
      Height: integer;
      Left: integer;
      Top: integer;
      Width: integer;
  begin
  inherited self.Draw;
  if self.fDrawAnchored then
    begin
    Height := abs(self.fAnchorPoint.YCoord-self.fDrawPoint.YCoord);
    if Height < 5 then                           { Always pick a minimum height }
      Height := 5;
    Width := abs(self.fAnchorPoint.XCoord-self.fDrawPoint.XCoord);
    if Width < 5 then                            { Always pick a minimum width }
      Width := 5;
    SetFont(CurrentFont,Height,Width);
    if self.fAnchorPoint.XCoord < self.fDrawPoint.XCoord
     then
      Left := self.fAnchorPoint.XCoord
     else
      Left := self.fDrawPoint.XCoord;
    if self.fAnchorPoint.YCoord < self.fDrawPoint.YCoord
     then
      Top := self.fAnchorPoint.YCoord
     else
      Top := self.fDrawPoint.YCoord;
    _MoveTo(Left,Top);                           { Move the current position to the top left corner }
    Mouse.SetTextCursor(Height);                 { Turn on the text cursor }
    Mouse.EnableTextCursor;
    self.fEntryEnabled := true;                  { Allow text entry now }
    while keypressed do                          { Clear the keyboard buffer }
      Ch := ReadKey
    end
  end;

procedure TTextTool.DrawIcon(Marked: boolean);
  { Draw the icon for the text tool }
  begin
  inherited self.DrawIcon(Marked);
  FitText(Modern,'Abc');
  end;

procedure TTextTool.Idle;
  { Watch the keyboard }
  var Ch: char;
  begin
  inherited self.Idle;
  if self.fEntryEnabled then
    Mouse.EnableTextCursor;
  if keypressed then
    begin
    Ch := ReadKey;
    if Ch = #0                                   { Clear function keys }
     then
      Ch := ReadKey
     else
      if (Ch>=' ') and self.fEntryEnabled then
        begin
        Mouse.Hide;
        CurrentCanvas.Activate;
        _OutGText(Ch)
        end
    end
  end;

function TTextTool.Select: boolean;
  { Turn off text entry until a size is selected }
  begin
  Select := inherited self.Select;
  self.fEntryEnabled := false
  end;

procedure TToolWindow.Init(Bordered: boolean;
                           X1,Y1,X2,Y2: real);
  { Initialize a tool selection window }
  begin
  inherited self.Init(false,X1,Y1,X2,Y2);
  self.Partition(Bordered,X1,Y1,X2,Y2,2,6)
  end;

procedure TToolWindow.ChangePane(Pane: integer);
  { Change to a new active pane }
  begin
  self.fPane[self.fCurrentPane].DrawIcon(false); { Turn off the previous icon }
  inherited self.ChangePane(Pane);               { Change the current pane }
  self.fPane[self.fCurrentPane].DrawIcon(true)   { Turn on the new icon }
  end;

function TToolWindow.CreatePane(Pane: integer): TPaneWindow;
  { Create a new tool pane }
  var Temp: record
        case integer of
          0: (PenTool: TPenTool);
          1: (LineTool: TLineTool);
          2: (BoxTool: TBoxTool);
          3: (FilledBoxTool: TFilledBoxTool);
          4: (CircleTool: TCircleTool);
          5: (FilledCircleTool: TFilledCircleTool);
          6: (EllipseTool: TEllipseTool);
          7: (FilledEllipseTool: TFilledEllipseTool);
          8: (TextTool: TTextTool);
          9: (PaintBucketTool: TPaintBucketTool);
          10: (EraserTool: TEraserTool);
          11: (QuitTool: TQuitTool)
        end;
  begin
  case Pane of
    0: begin
       new(Temp.PenTool);
       CreatePane := Temp.PenTool
       end;
    1: begin
       new(Temp.LineTool);
       CreatePane := Temp.LineTool
       end;
    2: begin
       new(Temp.BoxTool);
       CreatePane := Temp.BoxTool
       end;
    3: begin
       new(Temp.FilledBoxTool);
       CreatePane := Temp.FilledBoxTool
       end;
    4: begin
       new(Temp.CircleTool);
       CreatePane := Temp.CircleTool
       end;
    5: begin
       new(Temp.FilledCircleTool);
       CreatePane := Temp.FilledCircleTool
       end;
    6: begin
       new(Temp.EllipseTool);
       CreatePane := Temp.EllipseTool
       end;
    7: begin
       new(Temp.FilledEllipseTool);
       CreatePane := Temp.FilledEllipseTool
       end;
    8: begin
       new(Temp.TextTool);
       CreatePane := Temp.TextTool
       end;
    9: begin
       new(Temp.PaintBucketTool);
       CreatePane := Temp.PaintBucketTool
       end;
    10: begin
        new(Temp.EraserTool);
        CreatePane := Temp.EraserTool
        end;
    11: begin
        new(Temp.QuitTool);
        CreatePane := Temp.QuitTool
        end;
    end
  end;

begin
CurrentTrackLineStyle := $6666
end.
