//
// Written by: Robert C. Pendleton
// 
// Copyright 1993 by Robert C. Pendleton
//
// Non-commercial use by individuals
// is permitted.
//

#include <dos.h>
#include <conio.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "vbe.h"

#define FORCE_PTYPES
#include "ptypes.h"
#undef FORCE_PTYPES

typedef struct {
    int32 edi;
    int32 esi;
    int32 ebp;
    int32 reserved_by_system;
    int32 ebx;
    int32 edx;
    int32 ecx;
    int32 eax;
    int16 flags;
    int16 es,ds,fs,gs,ip,cs,sp,ss;
} rminfo;

//---------------------------------------------------
//
// Color palette addresses
//

#define PAL_WRITE_ADDR (0x3c8)      // palette write address
#define PAL_READ_ADDR  (0x3c7)      // palette write address
#define PAL_DATA       (0x3c9)      // palette data register

//---------------------------------------------------
//
// Set one entry in the color palatte
//

void
vgaSetColor(int16 index, uint16 r, uint16 g, uint16 b)
{
    if (index < 0 || index > 255)
    {
        return;
    }

    while(!(inp(0x3da) & 0x01));    // wait for blanking

    outp(PAL_WRITE_ADDR, index);
    outp(PAL_DATA, r);
    outp(PAL_DATA, g);
    outp(PAL_DATA, b);
}

//---------------------------------------------------
//
// Set a range of entries in the color table
//

void
vgaSetPalette(int16 start, int16 count, vgaColor *p)
{
    int16 i;

    if (start < 0 || (start + count - 1) > 255)
    {
        return;
    }

    while(!(inp(0x3da) & 0x08));    // wait vertical retrace

    outp(PAL_WRITE_ADDR, start);
    for (i = 0; i < count; i++)
    {
        outp(PAL_DATA, p->red);
        outp(PAL_DATA, p->green);
        outp(PAL_DATA, p->blue);
        p++;
    }
}

//---------------------------------------------------
//
// Ask dos for memory in the low part of memory
//

static uint32
allocDosMem(int16 size, uint32 *segment, uint32 *selector)
{
    union REGS reg;

    reg.w.ax = 0x0100;
    reg.w.bx = (size + 15) >> 4;

    int386(0x31, &reg, &reg);

    if (reg.x.cflag)
    {
        return 0;
    }

    *segment = reg.w.ax;
    *selector = reg.w.dx;
    return (uint32) ((reg.w.ax & 0xffff) << 4);
}

//---------------------------------------------------
//
// free memory allocated using allocDosMem()
//

static void
freeDosMem(uint32 selector)
{
    union REGS reg;

    reg.w.ax = 0x0101;
    reg.w.dx = selector;

    int386(0x31, &reg, &reg);
}

//---------------------------------------------------
//
// Convert a real mode pointer into a protected mode
// pointer. 
//

static uint32
rmp2pmp(uint32 addr)
{
    return ((addr & 0xffff0000) >> 12) + (addr & 0xffff);
}

//---------------------------------------------------
//
// use DPMI translation services to simulate a real
// mode interrupt
//

static inline
int386rm(uint16 inter, rminfo *inout)
{
    union REGS reg;
    struct SREGS sreg;

    segread(&sreg);
    memset(&reg, 0, sizeof(reg));

    reg.w.ax = 0x0300;
    reg.h.bl = inter;
    reg.h.bh = 0;
    reg.w.cx = 0;
    sreg.es = FP_SEG(inout);
    reg.x.edi = FP_OFF(inout);

    int386x(0x31, &reg, &reg, &sreg);
}

//---------------------------------------------------
//
// Use the VBE to get information about the video card
//

int16
vbeGetInfo(vbeInfo *infoPtr)
{
    rminfo rmi;

    uint32 dosmem;
    uint32 seg;
    uint32 sel;

    int16 len;
    uint16 *modes;

    dosmem = allocDosMem(sizeof(vbeInfo), &seg, &sel);

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f00;
    rmi.es = (dosmem >> 4) & 0xffff;
    rmi.edi = 0;

    int386rm(0x10, &rmi);

    memcpy(infoPtr, (void *)dosmem, sizeof(vbeInfo));
    freeDosMem(sel);

    if (rmi.eax == 0x004f)
    {
        infoPtr->vendorName = 
            strdup((char *)rmp2pmp((uint32)infoPtr->vendorName));

        modes = (uint16 *) rmp2pmp((uint32)infoPtr->modes);
        len = 0;

        while ((*modes) != 0xffff)
        {
            len++;
            modes++;
        }
        modes = (uint16 *) rmp2pmp((uint32)infoPtr->modes);
        infoPtr->modes = (uint16 *)malloc(sizeof(uint16) * (len + 1));
        memcpy(infoPtr->modes, modes, sizeof(uint16) * (len + 1));

        return TRUE;
    }

    return FALSE;
}

//---------------------------------------------------
//
// Print the info you get using vbeGetInfo
//

void
vbePrintInfo(FILE *file, vbeInfo *info)
{
    uint16 *mode;

    fprintf(file, "name=%c%c%c%c\n", 
            info->vesa[0],
            info->vesa[1],
            info->vesa[2],
            info->vesa[3]);

    fprintf(file, "Version %d.%d\n", info->majorMode, info->minorMode);
    fprintf(file, "Vendor %s\n", info->vendorName);
    fprintf(file, "Capabilities %lX\n", info->capabilities);
    fprintf(file, "Memory %dk\n", info->memory * 64);

    mode = info->modes;
    while ((*mode) != 0xffff)
    {
        fprintf(file, "mode=%x\n", *mode);
        mode++;
    }
    fflush(file);
}

//---------------------------------------------------
//
// Get details about a specific SVGA video mode
//

int16
vbeGetModeInfo(int16 mode, vbeModeInfo *infoPtr)
{
    union REGS reg;
    struct SREGS sreg;
    rminfo rmi;

    uint32 dosmem;
    uint32 seg;
    uint32 sel;

    dosmem = allocDosMem(sizeof(vbeModeInfo), &seg, &sel);

    segread(&sreg);
    memset(&reg, 0, sizeof(reg));

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f01;
    rmi.ecx = mode;
    rmi.es = (dosmem >> 4) & 0xffff;
    rmi.edi = 0;

    int386rm(0x10, &rmi);

    memcpy(infoPtr, (void *)dosmem, sizeof(vbeModeInfo));
    freeDosMem(sel);

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Print the info we get from vbeGetModeInfo
//

void
vbePrintModeInfo(FILE *file, vbeModeInfo *info)
{
    char *memoryModels[] = 
    {
        "Text mode",
        "CGA graphics",
        "Hercules graphics",
        "4-plane planar",
        "Packed pixel",
        "Non-chain 4, 256 color",
        "Direct Color",
        "YUV",
    };

    // mode attributes

    if (info->modeAttr & (1 << 0))
    {
        fprintf(file, "mode supported in hardware\n");
    }
    else
    {
        fprintf(file, "mode not supported in hardware\n");
    }

    if (info->modeAttr & (1 << 1))
    {
        fprintf(file, "optional info available\n");
    }
    else
    {
        fprintf(file, "no optional info\n");
    }

    if (info->modeAttr & (1 << 2))
    {
        fprintf(file, "BIOS output\n");
    }
    else
    {
        fprintf(file, "no BIOS output\n");
    }

    if (info->modeAttr & (1 << 3))
    {
        fprintf(file, "color mode\n");
    }
    else
    {
        fprintf(file, "monochrome mode\n");
    }

    if (info->modeAttr & (1 << 4))
    {
        fprintf(file, "graphics mode\n");
    }
    else
    {
        fprintf(file, "text mode\n");
    }

    // bank A attributes

    if (info->bankAAttr & (1 << 0))
    {
        fprintf(file, "bank A exists\n");
    }
    else
    {
        fprintf(file, "bank A does not exist\n");
    }

    if (info->bankAAttr & (1 << 1))
    {
        fprintf(file, "bank A readable\n");
    }
    else
    {
        fprintf(file, "bank A not readable\n");
    }

    if (info->bankAAttr & (1 << 2))
    {
        fprintf(file, "bank A writable\n");
    }
    else
    {
        fprintf(file, "bank A not writable\n");
    }

    // bank B attributes

    if (info->bankBAttr & (1 << 0))
    {
        fprintf(file, "bank B exists\n");
    }
    else
    {
        fprintf(file, "bank B does not exist\n");
    }

    if (info->bankBAttr & (1 << 1))
    {
        fprintf(file, "bank B readable\n");
    }
    else
    {
        fprintf(file, "bank B not readable\n");
    }

    if (info->bankBAttr & (1 << 2))
    {
        fprintf(file, "bank B writable\n");
    }
    else
    {
        fprintf(file, "bank B not writable\n");
    }

    // bank attributes

    fprintf(file, "bank granularity=%d\n", info->bankGranularity);
    fprintf(file, "bank size=%d\n", info->bankSize);
    fprintf(file, "bank A start seg=%x\n", info->bankASegment);
    fprintf(file, "bank B start seg=%x\n", info->bankBSegment);

    fprintf(file, "positioning function=%p\n", info->posFuncPtr);

    fprintf(file, "bytes/scanline=%d\n", info->bytesPerScanLine);


    fprintf(file, "width =%d\n", info->width);
    fprintf(file, "height=%d\n", info->height);

    fprintf(file, "char width =%d\n", (int) (info->charWidth));
    fprintf(file, "char height=%d\n", (int) (info->charHeight));

    fprintf(file, "number of planes=%d\n", (int) (info->numberOfPlanes));
    fprintf(file, "bits/pixel=%d\n", (int) (info->bitsPerPixel));
    fprintf(file, "number of banks=%d\n", (int) (info->numberOfBanks));

    if (0x00 <= (int)info->memoryModel && (int)info->memoryModel <= 0x07)
    {
        fprintf(file, "memory model=%d, %s\n",
                (int) (info->memoryModel), memoryModels[info->memoryModel]);
    }
    else if (0x08 <= (int)info->memoryModel && (int)info->memoryModel <= 0x0f)
    {
        fprintf(file, "memory model=%d, Reserved by VESA\n", 
                (int) (info->memoryModel));
    }
    else if (0x10 <= (int)info->memoryModel && (int)info->memoryModel <= 0xff)
    {
        fprintf(file, "memory model=%d, OEM custom mode\n",
                (int) (info->memoryModel));
    }

    fprintf(file, "video bank size=%d\n", (int) (info->videoBankSize));
    fprintf(file, "image pages=%d\n", (int) (info->imagePages));

    fprintf(file, "Red Mask Size=%d\n", (int) (info->redMaskSize));
    fprintf(file, "Green Mask Size=%d\n", (int) (info->greenMaskSize));
    fprintf(file, "Blue Mask Size=%d\n", (int) (info->blueMaskSize));
    fprintf(file, "Rsvd Mask Size=%d\n", (int) (info->rsvdMaskSize));

    fprintf(file, "Red Field Pos=%d\n", (int) (info->redFieldPos));
    fprintf(file, "Green Field Pos=%d\n", (int) (info->greenFieldPos));
    fprintf(file, "Blue Field Pos=%d\n", (int) (info->blueFieldPos));
    fprintf(file, "Rsvd Field Pos=%d\n", (int) (info->rsvdFieldPos));

    if (info->DirectColorInfo & (1 << 0))
    {
        fprintf(file, "color ramp is fixed\n");
    }
    else
    {
        fprintf(file, "color ramp is programmable\n");
    }

    if (info->DirectColorInfo & (1 << 1))
    {
        fprintf(file, "Bits in Rsvd field are reserved\n");
    }
    else
    {
        fprintf(file, "Bits in Rsvd field are usable\n");
    }

    fflush(file);
}

//---------------------------------------------------
//
// Set the video mode. This works with normal vga modes
// and with vbe modes.
//

int16
vbeSetMode(int16 mode)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f02;
    rmi.ebx = mode;

    int386rm(0x10, &rmi);

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Return the current video mode. Works with vga and
// vbe modes.
//

int16
vbeGetMode(int16 *mode)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f03;

    int386rm(0x10, &rmi);

    *mode = rmi.ebx;

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Position bank A in video memory
//

int16
vbeSetBankAAddr(uint16 addr)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f05;
    rmi.ebx = 0x0000;
    rmi.edx = addr;

    int386rm(0x10, &rmi);

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Position bank B in video memory
//

int16
vbeSetBankBAddr(uint16 addr)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f05;
    rmi.ebx = 0x0001;
    rmi.edx = addr;

    int386rm(0x10, &rmi);

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Get the current position of bank A in video memory
//

int16
vbeGetBankAAddr(uint16 *addr)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f05;
    rmi.ebx = 0x0100;

    int386rm(0x10, &rmi);

    *addr = rmi.edx;

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Get the current position of bank B in video memory
//

int16
vbeGetBankBAddr(uint16 *addr)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f05;
    rmi.ebx = 0x0101;

    int386rm(0x10, &rmi);

    *addr = rmi.edx;

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Set the logical scan line length
//

int16
vbeSetScanLineLength(uint16 length)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f06;
    rmi.ebx = 0x0000;
    rmi.ecx = length;

    int386rm(0x10, &rmi);

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Get the logical scan line length
//

int16
vbeGetScanLineLength(uint16 *bytesPerScanLine,
                     uint16 *pixelsPerScanLine,
                     uint16 *maxScanLines)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f06;
    rmi.ebx = 0x0001;

    int386rm(0x10, &rmi);

    *bytesPerScanLine = rmi.ebx;
    *pixelsPerScanLine = rmi.ecx;
    *maxScanLines = rmi.edx;

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Used for panning, scrolling and page flipping
//

int16
vbeSetDisplayStart(uint16 pixel, uint16 line)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f07;
    rmi.ebx = 0x0000;
    rmi.ecx = pixel;
    rmi.edx = line;

    int386rm(0x10, &rmi);

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Find out where the current start of the display is.
//

int16
vbeGetDisplayStart(uint16 *pixel, uint16 *line)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f07;
    rmi.ebx = 0x0001;

    int386rm(0x10, &rmi);

    *pixel = rmi.ecx;
    *line = rmi.edx;

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Set the desired number of bits of color in the
// palette. Normal VGA is 6. Many SVGAs support 8.
//

int16
vbeSetDACPaletteControl(uint16 paletteWidth)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f08;
    rmi.ebx = paletteWidth << 8;

    int386rm(0x10, &rmi);

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
//
// Get the current width of colors
//

int16
vbeGetDACPaletteControl(uint16 *paletteWidth)
{
    rminfo rmi;

    memset(&rmi, 0, sizeof(rmi));
    rmi.eax = 0x4f08;
    rmi.ebx = 0x0001;

    int386rm(0x10, &rmi);

    *paletteWidth = rmi.ebx >> 8;

    return (rmi.eax == 0x004f);
}

//---------------------------------------------------
