/* LABELS - A Program to Produce Mailing Labels from a Text File */
/* Version 1.0 by Richard Conn */

/* Usage: labels [-L#][-C#][-B#] filename.typ */

/* Instructions:
   The mailing labels file consists of a sequence of lines, where
   lines beginning in column 1 (non-space - no blank or tab in
   column 1) indicate a new label.  This program assumes a standard
   mailing label of 8 lines by 40 columns with a blank line between
   labels.  Lines beginning with a semicolon (;) are interpreted as
   comments and are not displayed in the output.  Lines beginning with
   a double quote (") are interpreted literally and are output
   without centering.  Blank lines are ignored.

   The options can be used to change the default values of the label
   parameters.  The -L# sets the number of lines on the label,
   -C# sets the number of columns, and -B# sets the number of blank
   lines between labels.

   Diagnostics are recorded in the log file ERRLOG (defined below).

*/

#include <stdio.h>

#define MAXLINES 20
#define MAXCOLS  80
#define NLINES 8
#define NCOLS 40
#define NBLANKS 1
#define LINELEN 200
#define ERRLOG "labels.log"
#define COMMENT ';'
#define QUOTE '"'

char line[MAXLINES][MAXCOLS+2];
FILE *fdo;
int emptylabel;
int labelcnt;
int columns = NCOLS;
int lines = NLINES;
int blanks = NBLANKS;
int opterr = 0;

main (argc, argv)
int argc;
char **argv;
{
	char inline[LINELEN];
        char *curchar;
	int curline;
        FILE *fd, *fopen();
	int firsttime = 1;
	int msgprinted = 0;
	int i;
	int fnindex = 0;

	labelcnt = 0;
	if (argc < 2) {
		help (argv[0]);
		exit (0);
	}
	for (i=1; i<argc; i++) {
		switch (*argv[i]) {
			case '-' : option (&argv[i][1]);
				   break;
			default  : fnindex = i;
				   break;
		}
	}
	if ((fnindex == 0) || (opterr)) {
		help (argv[0]);
		if (opterr > 1) {
			switch (opterr) {
				case 2 : printf (" Max Columns Exceeded\n");
					 break;
				case 3 : printf (" Max Lines Exceeded\n");
					 break;
				default: break;
			}
		}
		exit (0);
	}
	if ((fd = fopen (argv[fnindex], "r")) == NULL) {
		printf (" Cannot open %s\n", argv[fnindex]);
		exit (0);
	}
	if ((fdo = fopen (ERRLOG, "w")) == NULL) {
		printf (" Cannot create log file %s\n", ERRLOG);
		exit(0);
	}
	fprintf (fdo, " Label Parameters:");
	fprintf (fdo, "  Lines = %d, Columns = %d, Blanks = %d\n",
		lines, columns, blanks);
	while (fgets (inline, LINELEN, fd) != NULL) {
		if ((inline[0] != COMMENT) && (inline[0] != '\n')) {
			if (inline[0] > ' ') {
				if (firsttime) firsttime--;
				else printlabel();
				newlabel();
				msgprinted = 0;
				curline = -1;
			}
			curline++;
			curchar = inline;
			while ((*curchar == ' ') || (*curchar == '\t'))
				curchar++;
			if (curline >= lines) {
				if (!msgprinted) {
				fprintf (fdo, " Number of Lines Exceeded\n");
				for (i=0; i<lines; i++)
					fprintf (fdo, "   %s", line[i]);
				msgprinted = 1;
				}
			} else
				setline (curline, curchar);
        	}
        }
	fclose (fd);
	if (!emptylabel) printlabel();
	fprintf (fdo, " %d Labels Printed\n", labelcnt);
	fclose (fdo);
}

/* Print help message */
help (str)
char *str;
{
	printf (" Syntax: %s [-L#] [-C#] [-B#] filename.typ\n",
		str);
	printf ("   Where:\n");
	printf ("     -L# is the number of text lines on the label\n");
	printf ("     -C# is the number of columns on the label\n");
	printf ("     -B# is the number of blank lines between labels\n");
	printf ("   Default values are L=%d, C=%d, B=%d\n",
		NLINES, NCOLS, NBLANKS);
	printf ("   Maximum values are L=%d, C=%d\n", MAXLINES, MAXCOLS);
}

/* Process command-line options */
option (str)
char *str;
{
	switch (*str) {
		case 'b' :
		case 'B' : blanks = atoi (++str);
			   break;
		case 'c' :
		case 'C' : columns = atoi (++str);
			   if (columns > MAXCOLS) opterr = 2;
			   break;
		case 'l' :
		case 'L' : lines = atoi (++str);
			   if (lines > MAXLINES) opterr = 3;
			   break;
		default  : opterr = 1;
			   break;
	}
}

/* Store the next line into the label image */
setline (number, inline)
int number;
char *inline;
{
	int spacein;
        int i;

	if (inline[0] == QUOTE) {
		strcpy (line[number], &inline[1]);
	} else {
		if (strlen(inline) > columns) {
			spacein = 0;
			fprintf (fdo, " Truncation of:\n    %s", inline);
			inline[columns] = '\n';
			inline[columns+1] = '\0';
		} else
			spacein = (columns - strlen(inline) - 1) / 2;
		for (i=0; i<spacein; i++) line[number][i] = ' ';
		strcpy (&line[number][spacein], inline);
	}
	emptylabel = 0;
}

/* Initialize the label */
newlabel ()
{
	int i;

	for (i=0; i<lines; i++) {
		line[i][0] = '\n';
		line[i][1] = '\0';
	}
	emptylabel = 1;
}

/* Print the label */
printlabel ()
{
	int i;

	if (emptylabel)
		fprintf (fdo, " Attempt to print empty label\n");
	else {
		for (i=0; i<lines; i++) printf ("%s", line[i]);
		for (i=0; i<blanks; i++) printf ("\n");
		labelcnt++;
	}
}
